﻿//-------------------------------------------------------------------------------------------------
// File : asdxResPMD.h
// Desc : Polygon Model Data Format Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>
#include <asdxMath.h>
#include <asdxILoadable.h>
#include <asdxIDisposable.h>

#include <vector>
#include <string>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_EDGE_FLAG 
///////////////////////////////////////////////////////////////////////////////////////////////////
enum PMD_EDGE_FLAG
{
    PMD_EDGE_FLAG_DEFAULT = 0,  //!< 通常.
    PMD_EDGE_FLAG_DISABLE = 1   //!< エッジ無効.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_BONE_TYPE enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum PMD_BONE_TYPE
{
    PMD_BONE_TYPE_ROTATE                    = 0,        //!< 回転.
    PMD_BONE_TYPE_ROTATE_AND_TARNSLATE      = 1,        //!< 回転と移動.
    PMD_BONE_TYPE_IK                        = 2,        //!< IK
    PMD_BONE_TYPE_UNKNOWN                   = 3,        //!< 不明.
    PMD_BONE_TYPE_UNDER_INFLUENCE_OF_IK     = 4,        //!< IK影響下.
    PMD_BONE_TYPE_UNDER_INFLUENCE_OF_ROTATE = 5,        //!< 回転影響下.
    PMD_BONE_TYPE_TARGET_IK                 = 6,        //!< IK接続先.
    PMD_BONE_TYPE_INVISIBLE                 = 7,        //!< 非表示.
    PMD_BONE_TYPE_TWIST                     = 8,        //!< 捻り.
    PMD_BONE_TYPE_ROTATRY_MOTION            = 9,        //!< 回転運動.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_MORPH_TYPE enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum PMD_MORPH_TYPE
{
    PMD_MORPH_TYPE_BASE     = 0,        //!< base
    PMD_MORPH_TYPE_EYEBROW  = 1,        //!< 眉.
    PMD_MORPH_TYPE_EYE      = 2,        //!< 目.
    PMD_MORPH_TYPE_LIP      = 3,        //!< 唇.
    PMD_MORPT_TYPE_OTHER    = 4,        //!< その他.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_COLLISION_SHAPE_TYPE enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum PMD_COLLISION_SHAPE_TYPE
{
    PMD_COLLISION_SHAPE_TYPE_SPHERE  = 0,       //!< 球.
    PMD_COLLISION_SHAPE_TYPE_BOX     = 1,       //!< 箱.
    PMD_COLLISION_SHAPE_TYPE_CAPSULE = 2,       //!< カプセル.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_RIGIDBODY_TYPE
///////////////////////////////////////////////////////////////////////////////////////////////////
enum PMD_RIGIDBODY_TYPE
{
    PMD_RIGIDBODY_TYPE_FLLOW_BONE             = 0,  //!< ボーン追従.
    PMD_RIGIDBODY_TYPE_PHYSICS                = 1,  //!< 物理演算.
    PMD_RIGIDBODY_TYPE_PHYSICS_FLLOW_POSITION = 2,  //!< 物理演算(ボーン位置追従).
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_HEADER strcture
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_HEADER
{
    u8      Magic[3];           //!< ファイルマジック "Pmd"
    f32     Version;            //!< ファイルバージョン.
    char8   ModelName[20];      //!< モデル名.
    char8   Comment[256];       //!< コメント.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_VERTEX structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_VERTEX
{
    asdx::Vector3   Position;       //!< 位置座標.
    asdx::Vector3   Normal;         //!< 法線ベクトル.
    asdx::Vector2   TexCoord;       //!< テクスチャ座標.
    u16             BoneIndex[2];   //!< ボーン番号. モデル変形(頂点移動)時に影響.
    u8              BoneWeight;     //!< ボーン0へ与える影響度 min: 0, max : 100 (ボーン1への影響度は 100 - BoneWeight).
    u8              EdgeFlag;       //!< エッジフラグ.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_MATERIAL structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_MATERIAL
{
    asdx::Vector3   Diffuse;            //!< 拡散反射
    f32             Alpha;              //!< 透過度.
    f32             Power;              //!< 鏡面反射強度.
    asdx::Vector3   Specular;           //!< 鏡面反射.
    asdx::Vector3   Emissive;           //!< 自己照明.
    u8              ToonIndex;          //!< トゥーンテクスチャ番号.
    u8              VisualFlag;         //!< 輪郭. 影
    u32             VertexCount;        //!< 頂点数.
    char8           TextureName[20];    //!< テクスチャファイル名(非NULL終端であることに注意).
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_BONE structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_BONE
{
    char8           Name[20];           //!< ボーン名.
    u16             ParentIndex;        //!< 親ボーンの番号(親がいない場合は 0xFFFF).
    u16             TailIndex;          //!< 末尾のボーン番号
    u8              BoneType;           //!< ボーンの種類.
    u16             IKBoneIndex;        //!< IKボーン番号（影響IKボーン。無い場合は0)
    asdx::Vector3   Position;           //!< ボーンのヘッド位置.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_IK structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_IK
{
    u16     BoneIndex;          //!< IKボーン番号.
    u16     TargetBoneIndex;    //!< IKターゲットボーン番号（IKボーンが最初に接続するボーン)
    u8      ChainCount;         //!< IKチェーンの長さ（子供の数)
    u16     RecursiveCount;     //!< 再帰演算回数.
    f32     ControlWeight;      //!< 演算1回当たりの制限角度.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_MORPH_VERTEX structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct PMD_MORPH_VERTEX
{
    u32           Index;        //!< 表情用の頂点番号.
    asdx::Vector3 Position;     //!< 表情用の頂点の位置座標(頂点自体の座標).
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_MORPH structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_MORPH
{
    char8   Name[20];       //!< 表情名.
    u32     VertexCount;    //!< 表情用の頂点数.
    u8      Type;           //!< 表情の種類.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_BONE_LABEL_INDEX structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_BONE_LABEL_INDEX
{
    u16     BoneIndex;      //!< 表示枠用ボーン番号.
    u8      FrameIndex;     //!< 表示枠番号
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_LOCALIZE_HEADER structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_LOCALIZE_HEADER
{
    u8      LocalizeFlag;       //!< ローカライズフラグ(1 : 英名対応あり)
    char8   ModelName[20];      //!< 英語名モデル.
    char8   Comment[256];       //!< 英語コメント.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_TOON_TEXTURE_LIST structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_TOON_TEXTURE_LIST
{
    char8 FileName[10][100];    //!< トゥーンテクスチャファイル名.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_RIGIDBODY structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_RIGIDBODY
{
    char8           Name[20];               //!< 名前. (例：頭)
    u16             RelationBoneIndex;      //!< 関連ボーン番号.
    u8              GroupIndex;             //!< グループ番号.
    u16             GroupTarget;            //!< グループ対象.
    u8              ShapeType;              //!< 形状.
    asdx::Vector3   ShapeSize;              //!< 形状サイズ.
    asdx::Vector3   Position;               //!< 位置座標.
    asdx::Vector3   Angle;                  //!< 回転角(ラジアン).
    f32             Mass;                   //!< 質量.
    f32             DampingTranslate;       //!< 移動減衰.
    f32             DampingRotation;        //!< 回転減衰.
    f32             Elastictiy;             //!< 反発力.
    f32             Friction;               //!< 摩擦力.
    u8              Type;                   //!< タイプ.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// PMD_PHYSICS_JOINT structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct PMD_PHYSICS_JOINT
{
    char8           Name[20];           //!< 名前.
    u32             RigidBodyA;         //!< 剛体A.
    u32             RigidBodyB;         //!< 剛体B.
    asdx::Vector3   Position;           //!< 位置座標.
    asdx::Vector3   Angle;              //!< 回転角.
    asdx::Vector3   LimitPosition[2];   //!< 移動制限.
    asdx::Vector3   LimitAngle[2];      //!< 回転制限.
    asdx::Vector3   SpringPosition;     //!< ばね 位置座標.
    asdx::Vector3   SpringAngle;        //!< ばな 回転角.
};
#pragma pack( pop )


///////////////////////////////////////////////////////////////////////////////////////////////////
// ResPmdIK structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct ResPmdIK
{
    u16                 BoneIndex;          //!< IKボーン番号.
    u16                 TargetBoneIndex;    //!< IKターゲットボーン番号.
    u16                 RecursiveCount;     //!< 再帰演算回数.
    f32                 ControlWeight;      //!< 演算1会当たりの制限角度.
    std::vector<u16>    ChildBoneIndices;   //!< IK影響下のボーン番号.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// ResPmdMorph structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct ResPmdMorph
{
    std::string                     Name;       //!< 名前.
    u8                              Type;       //!< 表情タイプ.
    std::vector<PMD_MORPH_VERTEX>   Vertices;   //!< 表情用の頂点データ.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// ResPmd class
///////////////////////////////////////////////////////////////////////////////////////////////////
class ResPmd : public ILoadable, public IDisposable
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    std::string                         Name;                   //!< モデル名.
    std::string                         Comment;                //!< コメント.
    f32                                 Version;                //!< バージョン.
    std::vector<PMD_VERTEX>             Vertices;               //!< 頂点データ.
    std::vector<u16>                    Indices;                //!< 頂点インデックスデータ.
    std::vector<PMD_MATERIAL>           Materials;              //!< マテリアルデータ.
    std::vector<PMD_BONE>               Bones;                  //!< ボーンデータ.
    std::vector<ResPmdIK>               IKs;                    //!< IKデータ.
    std::vector<ResPmdMorph>            Morphes;                //!< モーフデータ.
    std::vector<u16>                    MorphLabelIndices;      //!< 表情枠用表示リスト.
    std::vector<std::string>            BoneLabels;             //!< ボーン枠用枠名リスト.
    std::vector<PMD_BONE_LABEL_INDEX>   BoneLabelIndices;       //!< ボーン枠用表示リスおｔ.
    PMD_TOON_TEXTURE_LIST               ToonTextureList;        //!< トゥーンテクスチャリスト.
    std::vector<PMD_RIGIDBODY>          RigidBodies;            //!< 剛体データ.
    std::vector<PMD_PHYSICS_JOINT>      Joints;                 //!< ジョイントデータ.

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです
    //---------------------------------------------------------------------------------------------
    ResPmd();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~ResPmd();

    //---------------------------------------------------------------------------------------------
    //! @brief      ファイルから読み込みを行います.
    //!
    //! @param[in]      filename        ファイル名です.
    //! @retval true    読み込みに成功.
    //! @retval false   読み込みに失敗.
    //---------------------------------------------------------------------------------------------
    bool Load( const char16* filename ) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      破棄処理を行います.
    //---------------------------------------------------------------------------------------------
    void Dispose() override;
};



} // namespace asdx
