﻿//--------------------------------------------------------------------------------------------------
// File : asdxResTGA.h
// Desc : Targa Module.
// Copyright(c) Project Asura. All right reserved.
//--------------------------------------------------------------------------------------------------

#ifndef __ASDX_RES_TGA_H__
#define __ASDX_RES_TGA_H__

//--------------------------------------------------------------------------------------------------
// Includes
//--------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>
#include <asdxILoadable.h>


namespace asdx {

////////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_FORMA_TYPE enum
////////////////////////////////////////////////////////////////////////////////////////////////////
enum TGA_FORMAT_TYPE
{
    TGA_FORMAT_NONE             = 0,        //!< イメージなし.
    TGA_FORMAT_INDEXCOLOR       = 1,        //!< インデックスカラー(256色).
    TGA_FORMAT_FULLCOLOR        = 2,        //!< フルカラー
    TGA_FORMAT_GRAYSCALE        = 3,        //!< 白黒.
    TGA_FORMAT_RLE_INDEXCOLOR   = 9,        //!< RLE圧縮インデックスカラー.
    TGA_FORMAT_RLE_FULLCOLOR    = 10,       //!< RLE圧縮フルカラー.
    TGA_FORMAT_RLE_GRAYSCALE    = 11,       //!< RLE圧縮白黒.
};


////////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_HEADER structure
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct TGA_HEADER
{
    u8  IdFieldLength;      // IDフィードのサイズ(範囲は0～255).
    u8  HasColorMap;        // カラーマップ有無(0=なし, 1=あり)
    u8  Format;             // 画像形式.
    u16 ColorMapEntry;      // カラーマップエントリー.
    u16 ColorMapLength;     // カラーマップのエントリーの総数.
    u8  ColorMapEntrySize;  // カラーマップの1エントリー当たりのビット数.
    u16 OffsetX;            // 画像のX座標.
    u16 OffsetY;            // 画像のY座標.
    u16 Width;              // 画像の横幅.
    u16 Height;             // 画像の縦幅.
    u8  BitPerPixel;        // ビットの深さ.
    u8  ImageDescriptor;    // (0~3bit) : 属性, 4bit : 格納方向(0=左から右,1=右から左), 5bit : 格納方向(0=下から上, 1=上から下), 6~7bit : インタリーブ(使用不可).
};
#pragma pack( pop )


////////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_FOOTER structure
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct TGA_FOOTER
{
    u32  OffsetExt;      // 拡張データへのオフセット(byte数) [オフセットはファイルの先頭から].
    u32  OffsetDev;      // ディベロッパーエリアへのオフセット(byte数)[オフセットはファイルの先頭から].
    char Tag[18];        // 'TRUEVISION-XFILE.\0'
};
#pragma pack( pop )


///////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_EXTENSION structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct TGA_EXTENSION
{
    u16     Size;                       //!< サイズ.
    char    AuthorName[ 41 ];           //!< 著作者名.
    char    AuthorComment[ 324 ];       //!< 著作者コメント.
    u16     StampMonth;                 //!< タイムスタンプ　月(1-12).
    u16     StampDay;                   //!< タイムスタンプ　日(1-31).
    u16     StampYear;                  //!< タイムスタンプ　年(4桁, 例1989).
    u16     StampHour;                  //!< タイムスタンプ　時(0-23).
    u16     StampMinute;                //!< タイムスタンプ　分(0-59).
    u16     StampSecond;                //!< タイムスタンプ　秒(0-59).
    char    JobName[ 41 ];              //!< ジョブ名 (最後のバイトはゼロが必須).
    u16     JobHour;                    //!< ジョブ時間  時(0-65535)
    u16     JobMinute;                  //!< ジョブ時間　分(0-59)
    u16     JobSecond;                  //!< ジョブ時間　秒(0-59)
    char    SoftwareId[ 41 ];           //!< ソフトウェアID (最後のバイトはゼロが必須).
    u16     VersionNumber;              //!< ソフトウェアバージョン    VersionNumber * 100になる.
    u8      VersionLetter;              //!< ソフトウェアバージョン
    u32     KeyColor;                   //!< キーカラー.
    u16     PixelNumerator;             //!< ピクセル比分子　ピクセル横幅.
    u16     PixelDenominator;           //!< ピクセル比分母　ピクセル縦幅.
    u16     GammaNumerator;             //!< ガンマ値分子.
    u16     GammaDenominator;           //!< ガンマ値分母
    u32     ColorCorrectionOffset;      //!< 色補正テーブルへのオフセット.
    u32     StampOffset;                //!< ポステージスタンプ画像へのオフセット.
    u32     ScanLineOffset;             //!< スキャンラインオフセット.
    u8      AttributeType;              //!< アルファチャンネルデータのタイプ
};
#pragma pack( pop )


////////////////////////////////////////////////////////////////////////////////////////////////////
// ResTGA class
////////////////////////////////////////////////////////////////////////////////////////////////////
class ResTGA : public ILoadable
{
    //==============================================================================================
    // list of friend classes and methods.
    //==============================================================================================
    /* NOTHING */

public:
    //==============================================================================================
    // public variables.
    //==============================================================================================
    /* NOTHING */

    //==============================================================================================
    // public methods.
    //==============================================================================================

    //----------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------------
    ResTGA();

    //----------------------------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param[in]      value       コピー元の値です.
    //----------------------------------------------------------------------------------------------
    ResTGA( const ResTGA& value );

    //----------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //----------------------------------------------------------------------------------------------
    virtual ~ResTGA();

    //----------------------------------------------------------------------------------------------
    //! @brief      ファイルから読み込みします.
    //!
    //! @param[in]      filename        ファイル名です.
    //! @retval true    読み込み成功.
    //! @retval false   読み込み失敗.
    //----------------------------------------------------------------------------------------------
    bool Load( const char16* filename ) override;

    //----------------------------------------------------------------------------------------------
    //! @brief      メモリを解放します.
    //----------------------------------------------------------------------------------------------
    void Release();

    //----------------------------------------------------------------------------------------------
    //! @brief      画像の横幅を取得します.
    //!
    //! @return     画像の横幅を返却します.
    //----------------------------------------------------------------------------------------------
    const u32 GetWidth() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      画像の縦幅を取得します.
    //!
    //! @return     画像の縦幅を返却します.
    //----------------------------------------------------------------------------------------------
    const u32 GetHeight() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      画像の形式を取得します.
    //!
    //! @return     画像の形式を返却します.
    //----------------------------------------------------------------------------------------------
    const TGA_FORMAT_TYPE GetFormat() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      ビットの深さを取得します.
    //!
    //! @return     ビットの深さを返却します.
    //----------------------------------------------------------------------------------------------
    const u8 GetBitPerPixel() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      ピクセルデータを取得します.
    //!
    //! @return     ピクセルデータを返却します.
    //----------------------------------------------------------------------------------------------
    const u8* GetPixels() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param[in]      value       代入する値です.
    //! @return     代入結果を返却します.
    //----------------------------------------------------------------------------------------------
    ResTGA& operator = ( const ResTGA& value );

    //----------------------------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param[in]      value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //! @note   ハッシュキーを用いた比較を行います.
    //----------------------------------------------------------------------------------------------
    bool operator == ( const ResTGA& value ) const;

    //----------------------------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param[in]      value       比較する値.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //! @note   ハッシュキーを用いた比較を行います.
    //----------------------------------------------------------------------------------------------
    bool operator != ( const ResTGA& value ) const;

protected:
    //==============================================================================================
    // protected variables.
    //==============================================================================================
    u32             m_Width;            //!< 画像の横幅です.
    u32             m_Height;           //!< 画像の縦幅です.
    u8              m_BitPerPixel;      //!< 1ピクセルあたりのビットの深さです.
    TGA_FORMAT_TYPE m_Format;           //!< 画像形式です.
    u8*             m_pPixels;          //!< ピクセルデータです.
    u32             m_HashKey;          //!< ファイル名から作成されるハッシュキーです.

    //==============================================================================================
    // protected methods.
    //==============================================================================================
    /* NOTHING */

private:
    //==============================================================================================
    // private variables.
    //==============================================================================================
    /* NOTHING */

    //==============================================================================================
    // private methods.
    //==============================================================================================
    /* NOTHING */
};


} // namespace asdx


#endif//__ASDX_RES_TGA_H__
