#include "Value.h"
#include "Function.h"

//-----------------------------------------------------------------------------
// Heap access functions
//-----------------------------------------------------------------------------
#if defined(WINVER)
// malloc() and free() don't work correctly when they're used across
// Windows' DLL boundary. see KB190799 in MSDN.
void *operator new (size_t bytes)
{
	return ::LocalAlloc(LMEM_FIXED, bytes);
}

void operator delete (void *p)
{
	::LocalFree(p);
}
#endif

namespace AScript {

const int MAX_STACK_LEVEL = 2048;

//-----------------------------------------------------------------------------
// String
//-----------------------------------------------------------------------------
const StringList StringList::Null;

int GetEscaped(int ch)
{
	static const struct {
		int ch;
		int chConv;
	} tbl[] = {
		{ _T('a'),	_T('\a')	},
		{ _T('b'),	_T('\b')	},
		{ _T('f'),	_T('\f')	},
		{ _T('n'),	_T('\n')	},
		{ _T('r'),	_T('\r')	},
		{ _T('t'),	_T('\t')	},
		{ _T('v'),	_T('\v')	},
		{ _T('\\'),	_T('\\')	},
		{ _T('\''),	_T('\'')	},
		{ _T('"'),	_T('"')		},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].ch == ch) return tbl[i].chConv;
	}
	return ch;
}

String EscapeString(const TCHAR *str)
{
	String rtn;
	bool shiftFlag = false;
	for (const TCHAR *p = str; *p != _T('\0'); p++) {
		TCHAR ch = *p;
		bool sjisFirstFlag = IsSJISFirst(static_cast<unsigned char>(ch));
		if (ch == _T('\n')) {
			rtn += _T("\\n");
		} else if (ch == _T('\r')) {
			rtn += _T("\\r");
		} else if (ch == _T('\t')) {
			rtn += _T("\\t");
		} else if (ch == _T('"')) {
			rtn += _T("\\\"");
		} else if (ch == _T('\\') && !shiftFlag) {
			rtn += _T("\\\\");
		} else if (sjisFirstFlag || shiftFlag) {
			rtn += ch;
		} else if (ch < 0x20 || ch >= 0x7f) {
			TCHAR tmp[16];
			::_stprintf(tmp, _T("\\x%02x"), ch);
			rtn += tmp;
		} else {
			rtn += ch;
		}
		shiftFlag = shiftFlag? false : sjisFirstFlag;
	}
	return rtn;
}

const TCHAR *FindString(const TCHAR *str, const TCHAR *sub, bool ignoreCaseFlag)
{
	for ( ; *str != _T('\0'); str++) {
		const TCHAR *p1 = str, *p2 = sub;
		for ( ; *p2 != _T('\0'); p1++, p2++) {
			if (CompareChar(*p1, *p2, ignoreCaseFlag) != 0) break;
		}
		if (*p2 == _T('\0')) return str;
	}
	return NULL;
}

String Capitalize(const TCHAR *str)
{
	String rtn;
	const TCHAR *p = str;
	if (*p != _T('\0')) {
		rtn += ('a' <= *p && *p <= 'z')? *p - 'a' + 'A' : *p;
		p++;
	}
	for ( ; *p != _T('\0'); p++) {
		rtn += ('A' <= *p && *p <= 'Z')? *p - 'A' + 'a' : *p;
	}
	return rtn;
}

String Lower(const TCHAR *str)
{
	String rtn;
	for (const TCHAR *p = str; *p != _T('\0'); p++) {
		rtn += ('A' <= *p && *p <= 'Z')? *p - 'A' + 'a' : *p;
	}
	return rtn;
}

String Upper(const TCHAR *str)
{
	String rtn;
	for (const TCHAR *p = str; *p != _T('\0'); p++) {
		rtn += ('a' <= *p && *p <= 'z')? *p - 'a' + 'A' : *p;
	}
	return rtn;
}

String Strip(const TCHAR *str, const SymbolSet &attrs)
{
	bool stripLeftFlag = true, stripRightFlag = true;
	if (attrs.IsSet(AScript_Symbol(left)) && !attrs.IsSet(AScript_Symbol(right))) {
		stripRightFlag = false;
	} else if (!attrs.IsSet(AScript_Symbol(left)) && attrs.IsSet(AScript_Symbol(right))) {
		stripLeftFlag = false;
	}
	size_t len = ::_tcslen(str);
	if (len == 0) return String(_T(""));
	const TCHAR *p1 = str;
	const TCHAR *p2 = str + len - 1;
	if (stripLeftFlag) {
		for ( ; IsSpace(*p1); p1++) ;
	}
	if (stripRightFlag) {
		for ( ; p2 != p1 && IsSpace(*p2); p2--) ;
	}
	return String(p1, p2 - p1 + 1);
}

String Left(const TCHAR *str, size_t len)
{
	return (len < ::_tcslen(str))? String(str, 0, len) : String(str);
}

String Right(const TCHAR *str, size_t len)
{
	size_t lenSrc = ::_tcslen(str);
	return (len < lenSrc)? String(str + lenSrc - len) : String(str);
}

String Middle(const TCHAR *str, int start, int len)
{
	int lenSrc = static_cast<int>(::_tcslen(str));
	if (start < 0) {
		start += lenSrc;
		if (start < 0) start = 0;
	}
	if (start >= lenSrc || len == 0) {
		return String(_T(""));
	} else if (len > 0 && start + len < lenSrc) {
		return String(str + start, len);
	} else {
		return String(str + start);
	}
}

String Join(const ValueList &valList, const TCHAR *str)
{
	Signal sig;
	ValueList::const_iterator pValue = valList.begin();
	if (pValue == valList.end()) return _T("");
	String rtn = pValue->GetString();
	pValue++;
	for ( ; pValue != valList.end(); pValue++) {
		rtn += str;
		rtn += pValue->ToString(sig, false);
	}
	return rtn;
}

Value FindString(Environment &env, Signal sig,
		const TCHAR *str, const TCHAR *sub, int start, const SymbolSet &attrs)
{
	bool ignoreCaseFlag = attrs.IsSet(AScript_Symbol(icase));
	do {
		int len = static_cast<int>(::_tcslen(str));
		if (start < 0) {
			start += len;
			if (start < 0) start = 0;
		}
		if (start > len) return Value::Null;
	} while (0);
	if (attrs.IsSet(AScript_Symbol(rev))) {
		const TCHAR *p = FindString(str + start, sub, ignoreCaseFlag);
		if (attrs.IsSet(AScript_Symbol(list))) {
			ValueList valListOrg;
			for ( ; p != NULL; p = FindString(p + 1, sub, ignoreCaseFlag)) {
				valListOrg.push_back(Value(static_cast<Number>(p - str)));
			}
			Value result;
			ValueList &valList = result.InitAsList(env);
			foreach_reverse (ValueList, pValue, valListOrg) {
				valList.push_back(*pValue);
			}
			return result;
		} else {
			const TCHAR *pLast = NULL;
			for ( ; p != NULL; p = FindString(p + 1, sub, ignoreCaseFlag)) {
				pLast = p;
			}
			return (pLast == NULL)? Value::Null :
							Value(static_cast<Number>(pLast - str));
		}
	} else {
		const TCHAR *p = FindString(str + start, sub, ignoreCaseFlag);
		if (attrs.IsSet(AScript_Symbol(list))) {
			Value result;
			ValueList &valList = result.InitAsList(env);
			for ( ; p != NULL; p = FindString(p + 1, sub, ignoreCaseFlag)) {
				valList.push_back(Value(static_cast<Number>(p - str)));
			}
			return result;
		} else {
			return (p == NULL)? Value::Null :
							Value(static_cast<Number>(p - str));
		}
	}
}

String Replace(const TCHAR *str, const TCHAR *sub, const TCHAR *replace,
										int nMaxReplace, const SymbolSet &attrs)
{
	bool ignoreCaseFlag = attrs.IsSet(AScript_Symbol(icase));
	String result;
	int lenSub = ::_tcslen(sub);
	if (lenSub == 0) {
		if (nMaxReplace != 0) {
			result += replace;
			nMaxReplace--;
		}
		const TCHAR *p = str;
		for ( ; *p != _T('\0') && nMaxReplace != 0; p++, nMaxReplace--) {
			result += *p;
			result += replace;
		}
		result += p;
	} else {
		const TCHAR *pLeft = str;
		const TCHAR *pRight = FindString(pLeft, sub, ignoreCaseFlag);
		for ( ; pRight != NULL && nMaxReplace != 0;
					pRight = FindString(pLeft, sub, ignoreCaseFlag), nMaxReplace--) {
			result.append(pLeft, pRight - pLeft);
			result += replace;
			pLeft = pRight + lenSub;
		}
		result += pLeft;
	}
	return result;
}

void SplitPathList(Environment &env, const TCHAR *str, ValueList &valList)
{
	enum {
		STAT_SkipSpace, STAT_Field,
	} stat = STAT_SkipSpace;
	TCHAR ch = _T('\0');
	String field;
	int cntSpace = 0;
	bool eatNextFlag = true;
	do {
		if (eatNextFlag) ch = *str++;
		eatNextFlag = true;
		if (stat == STAT_SkipSpace) {
			if (IsSpace(ch)) {
				// nothing to do
			} else {
				field.clear();
				cntSpace = 0;
				eatNextFlag = false;
				stat = STAT_Field;
			}
		} else if (stat == STAT_Field) {
			if (IsPathSeparator(ch) || ch == _T('\0')) {
				valList.push_back(Value(env, field.c_str()));
				stat = STAT_SkipSpace;
			} else if (ch == _T(' ')) {
				cntSpace++;
			} else {
				while (cntSpace-- > 0) field.push_back(_T(' '));
				field.push_back(ch);
			}
		}
	} while (ch != _T('\0'));
}

//-----------------------------------------------------------------------------
// global functions
//-----------------------------------------------------------------------------
const Symbol *GetValueTypeSymbol(ValueType valType)
{
	return
		(valType == VTYPE_Invalid)?		AScript_Symbol(nil) :
		(valType == VTYPE_Number)?		AScript_Symbol(number) :
		(valType == VTYPE_Boolean)?		AScript_Symbol(boolean) :
		(valType == VTYPE_Symbol)?		AScript_Symbol(symbol) :
		(valType == VTYPE_String)?		AScript_Symbol(string) :
		(valType == VTYPE_Complex)?		AScript_Symbol(complex) :
		(valType == VTYPE_Module)?		AScript_Symbol(Module) :
		(valType == VTYPE_Class)?		AScript_Symbol(Class) :
		(valType == VTYPE_Object)?		AScript_Symbol(Object) :
		(valType == VTYPE_Function)?	AScript_Symbol(Function) :
		(valType == VTYPE_List)?		AScript_Symbol(List) :
		(valType == VTYPE_Dict)?		AScript_Symbol(Dict) :
		(valType == VTYPE_File)?		AScript_Symbol(File) :
		(valType == VTYPE_FileStat)?	AScript_Symbol(FileStat) :
		(valType == VTYPE_DateTime)?	AScript_Symbol(DateTime) :
		(valType == VTYPE_Expr)?		AScript_Symbol(Expr) :
		(valType == VTYPE_Environment)?	AScript_Symbol(Environment) :
		(valType == VTYPE_Error)?		AScript_Symbol(Error) :
		(valType == VTYPE_Struct)?		AScript_Symbol(Struct) :
		AScript_Symbol(unknown);
}

const Symbol *GetOccurPatternSymbol(OccurPattern occurPattern)
{
	return
		(occurPattern == OCCUR_ZeroOrOnce)? AScript_Symbol(Char_Question) :
		(occurPattern == OCCUR_ZeroOrMore)? AScript_Symbol(Char_Multiply) :
		(occurPattern == OCCUR_OnceOrMore)? AScript_Symbol(Char_Plus) :
		AScript_Symbol(Str_Empty);
}

const TCHAR *GetSignalTypeName(SignalType sigType)
{
	static const struct {
		SignalType sigType;
		const TCHAR *name;
	} tbl[] = {
		{ SIGTYPE_None,		_T("none"),		},
		{ SIGTYPE_Error,	_T("error"),	},
		{ SIGTYPE_Break,	_T("break"),	},
		{ SIGTYPE_Continue,	_T("continue"),	},
		{ SIGTYPE_Return,	_T("return"),	},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].sigType == sigType) return tbl[i].name;
	}
	return _T("unknown");
}

const TCHAR *GetEnvTypeName(EnvType envType)
{
	static const struct {
		EnvType envType;
		const TCHAR *name;
	} tbl[] = {
		{ ENVTYPE_Invalid,	_T("invalid"),	},
		{ ENVTYPE_Root,		_T("root"),		},
		{ ENVTYPE_Local,	_T("local"),	},
		{ ENVTYPE_Block,	_T("block"),	},
		{ ENVTYPE_Module,	_T("module"),	},
		{ ENVTYPE_Class,	_T("class"),	},
		{ ENVTYPE_Instance,	_T("instance"),	},
		{ ENVTYPE_Method,	_T("method"),	},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].envType == envType) return tbl[i].name;
	}
	return _T("unknown");
}

const TCHAR *GetErrorTypeName(ErrorType errType)
{
	static const struct {
		ErrorType errType;
		const TCHAR *name;
	} tbl[] = {
		{ ERR_None,					_T("None")				},
		{ ERR_SyntaxError,			_T("SyntaxError")		},
		{ ERR_ArithmeticError,		_T("ArithmeticError")	},
		{ ERR_TypeError,			_T("TypeError")			},
		{ ERR_ZeroDivisionError,	_T("ZeroDivisionError")	},
		{ ERR_ValueError,			_T("ValueError")		},
		{ ERR_SystemError,			_T("SyntaxError")		},
		{ ERR_IOError,				_T("IOError")			},
		{ ERR_IndexError,			_T("IndexError")		},
		{ ERR_KeyError,				_T("KeyError")			},
		{ ERR_ImportError,			_T("ImportError")		},
		{ ERR_AttributeError,		_T("AttributeError")	},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].errType == errType) return tbl[i].name;
	}
	return _T("unknown");
}

//-----------------------------------------------------------------------------
// Signal
//-----------------------------------------------------------------------------
Signal::Signal() : _pMsg(new Message()), _stackLevel(0)
{
}

Signal::Signal(const Signal &sig) :
						_pMsg(sig._pMsg), _stackLevel(sig._stackLevel + 1)
{
	if (_stackLevel > MAX_STACK_LEVEL) {
		SetError(ERR_SystemError, _T("stack level exceeds maximum (%d)"), MAX_STACK_LEVEL);
	}
}

Signal::Message::Message() : sigType(SIGTYPE_None), pValue(new Value())
{
}

String Signal::GetErrString() const
{
	String str(GetErrorName());
	str += _T(": ");
	str += _pMsg->str.c_str();
	return str;
}

void Signal::SetError(ErrorType errType, const TCHAR *format, ...)
{
	va_list list;
	va_start(list, format);
	SetErrorV(errType, format, list);
	va_end(list);
}

void Signal::SetErrorV(ErrorType errType, const TCHAR *format, va_list list)
{
	TCHAR str[2048];
	::_vstprintf(str, format, list);
	_pMsg->sigType = SIGTYPE_Error;
	_pMsg->errType = errType;
	_pMsg->str = str;
	*_pMsg->pValue = Value::Null;
}

//-----------------------------------------------------------------------------
// Symbol
//-----------------------------------------------------------------------------
Symbol::Symbol(UniqNumber uniqNum, const TCHAR *name) : _uniqNum(uniqNum)
{
	_name = new TCHAR[::_tcslen(name) + 1];
	::_tcscpy(_name, name);
}

Symbol::~Symbol()
{
	delete[] _name;
}

const Symbol *Symbol::Add(const TCHAR *name)
{
	return SymbolPool::GetInstance()->Add(name);
}

bool Symbol::IsFlowControlSymbol() const
{
	return 
		IsIdentical(AScript_Symbol(if_)) ||
		IsIdentical(AScript_Symbol(elsif)) ||
		IsIdentical(AScript_Symbol(repeat)) ||
		IsIdentical(AScript_Symbol(while_)) ||
		IsIdentical(AScript_Symbol(for_));
}

//-----------------------------------------------------------------------------
// SymbolSet
//-----------------------------------------------------------------------------
const SymbolSet SymbolSet::Null;

SymbolSet::SymbolSet(const SymbolSet &symbolSet)
{
	foreach_const (SymbolSet, ppSymbol, symbolSet) Insert(*ppSymbol);
}

void SymbolSet::operator=(const SymbolSet &symbolSet)
{
	foreach_const (SymbolSet, ppSymbol, symbolSet) Insert(*ppSymbol);
}

//-----------------------------------------------------------------------------
// SymbolPool
//-----------------------------------------------------------------------------
SymbolPool *SymbolPool::_pInst = NULL;

SymbolPool::~SymbolPool()
{
	foreach (Content, ppSymbol, _content) {
		Symbol::Delete(*ppSymbol);
	}
}

void SymbolPool::Initialize()
{
	if (_pInst == NULL) {
		_pInst = new SymbolPool();
		_pInst->_Initialize();
	}
}

void SymbolPool::_Initialize()
{
	AScript_RealizeSymbolEx(Str_Empty,		_T(""));
	AScript_RealizeSymbolEx(Char_Plus,		_T("+"));
	AScript_RealizeSymbolEx(Char_Multiply,	_T("*"));
	AScript_RealizeSymbolEx(Char_Question,	_T("?"));
	AScript_RealizeSymbolEx(Char_Modulo,	_T("%"));
	AScript_RealizeSymbolEx(Char_And,		_T("&"));
	AScript_RealizeSymbol(unknown);
	AScript_RealizeSymbol(number);
	AScript_RealizeSymbol(boolean);
	AScript_RealizeSymbol(symbol);
	AScript_RealizeSymbol(string);
	AScript_RealizeSymbol(complex);
	AScript_RealizeSymbol(Module);
	AScript_RealizeSymbol(Class);
	AScript_RealizeSymbol(Object);
	AScript_RealizeSymbol(Function);
	AScript_RealizeSymbol(List);
	AScript_RealizeSymbol(Dict);
	AScript_RealizeSymbol(File);
	AScript_RealizeSymbol(FileStat);
	AScript_RealizeSymbol(DateTime);
	AScript_RealizeSymbol(Expr);
	AScript_RealizeSymbol(Environment);
	AScript_RealizeSymbol(Error);
	AScript_RealizeSymbol(Struct);
	AScript_RealizeSymbol(e);
	AScript_RealizeSymbol(pi);
	AScript_RealizeSymbol(nil);
	AScript_RealizeSymbol(zero);
	AScript_RealizeSymbol(raise);
	AScript_RealizeSymbolEx(true_,			_T("true"));
	AScript_RealizeSymbolEx(false_,			_T("false"));
	AScript_RealizeSymbol(rem);				// dummy for MS-DOS batch
	AScript_RealizeSymbol(j);
	AScript_RealizeSymbolEx(if_,			_T("if"));
	AScript_RealizeSymbol(elsif);
	AScript_RealizeSymbolEx(else_,			_T("else"));
	AScript_RealizeSymbol(repeat);
	AScript_RealizeSymbolEx(while_,			_T("while"));
	AScript_RealizeSymbolEx(for_,			_T("for"));
	AScript_RealizeSymbolEx(break_,			_T("break"));
	AScript_RealizeSymbolEx(continue_,		_T("continue"));
	AScript_RealizeSymbolEx(except_,		_T("except"));
	AScript_RealizeSymbol(__init__);
	AScript_RealizeSymbol(__del__);
	AScript_RealizeSymbol(__str__);
	AScript_RealizeSymbol(super);
	AScript_RealizeSymbol(self);
	AScript_RealizeSymbolEx(static_,		_T("static"));
	AScript_RealizeSymbolEx(const_,			_T("const"));
	AScript_RealizeSymbolEx(_anonymous_,	_T("<>"));
	AScript_RealizeSymbolEx(public_,		_T("public"));
	AScript_RealizeSymbolEx(private_,		_T("private"));
	AScript_RealizeSymbolEx(protected_,		_T("protected"));
	AScript_RealizeSymbol(dynamic_scope);
	AScript_RealizeSymbol(inside_scope);
	AScript_RealizeSymbol(map);
	AScript_RealizeSymbol(nomap);
	AScript_RealizeSymbol(strict);
	AScript_RealizeSymbol(loose);
	AScript_RealizeSymbol(block);
	AScript_RealizeSymbol(list);
	AScript_RealizeSymbol(xlist);
	AScript_RealizeSymbol(set);
	AScript_RealizeSymbol(xset);
	AScript_RealizeSymbol(rev);
	AScript_RealizeSymbol(and);
	AScript_RealizeSymbol(or);
	AScript_RealizeSymbol(xor);
	AScript_RealizeSymbolEx(void_,			_T("void"));
	AScript_RealizeSymbol(icase);
	AScript_RealizeSymbol(index);
	AScript_RealizeSymbol(last_index);
	AScript_RealizeSymbol(indices);
	AScript_RealizeSymbol(ascend);
	AScript_RealizeSymbol(descend);
	AScript_RealizeSymbol(stable);
	AScript_RealizeSymbol(up);
	AScript_RealizeSymbol(down);
	AScript_RealizeSymbol(left);
	AScript_RealizeSymbol(right);
	AScript_RealizeSymbol(prev);
	AScript_RealizeSymbol(next);
	AScript_RealizeSymbol(name);
	AScript_RealizeSymbol(parent);
	AScript_RealizeSymbol(parents);
	AScript_RealizeSymbol(sibling);
	AScript_RealizeSymbol(siblings);
	AScript_RealizeSymbol(child);
	AScript_RealizeSymbol(children);
	AScript_RealizeSymbol(path);
	AScript_RealizeSymbol(x);
	AScript_RealizeSymbol(y);
	AScript_RealizeSymbol(z);
}

const Symbol *SymbolPool::Add(const TCHAR *name)
{
	Symbol *pSymbol = new Symbol(_uniqNum, name);
	std::pair<Content::iterator, bool> rtn = _content.insert(pSymbol);
	if (rtn.second) {
		_uniqNum++;
	} else {
		Symbol::Delete(pSymbol);
	}
	pSymbol = *rtn.first;
	return pSymbol;
}

}
