#include "Expr.h"
#include "Function.h"
#include "Module.h"
#include "Object_List.h"
#include "Object_Function.h"

namespace AScript {

//-----------------------------------------------------------------------------
// ExprType
//-----------------------------------------------------------------------------
const char *GetExprTypeName(ExprType exprType)
{
	static const struct {
		ExprType exprType;
		const char *name;
	} tbl[] = {
		{ EXPRTYPE_UnaryOp,		"unaryop",		},
		{ EXPRTYPE_Quote,		"quote",		},
		{ EXPRTYPE_Force,		"force",		},
		{ EXPRTYPE_Prefix,		"prefix",		},
		{ EXPRTYPE_Suffix,		"suffix",		},
		{ EXPRTYPE_BinaryOp,	"binaryop",		},
		{ EXPRTYPE_Assign,		"assign",		},
		{ EXPRTYPE_DictAssign,	"dictassign",	},
		{ EXPRTYPE_Field,		"field",		},
		{ EXPRTYPE_BlockParam,	"blockparam",	},
		{ EXPRTYPE_Block,		"block",		},
		{ EXPRTYPE_Lister,		"lister",		},
		{ EXPRTYPE_Value,		"value",		},
		{ EXPRTYPE_Symbol,		"symbol",		},
		{ EXPRTYPE_Indexer,		"indexer",		},
		{ EXPRTYPE_Function,	"function",		},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].exprType == exprType) return tbl[i].name;
	}
	return "unknown";
}

//-----------------------------------------------------------------------------
// Expr
// [class hierarchy under Expr]
// Expr <-+- Expr_Unary <-----+- Expr_UnaryOp
//        |                   +- Expr_Quote
//        |                   +- Expr_Force
//        |                   +- Expr_Prefix
//        |                   `- Expr_Suffix
//        +- Expr_Binary <----+- Expr_BinaryOp
//        |                   +- Expr_Assign
//        |                   +- Expr_DictAssign
//        |                   `- Expr_Field
//        +- Expr_Container <-+- Expr_BlockParam
//        |                   +- Expr_Block
//        |                   `- Expr_Lister
//        +- Expr_Complex <---+- Expr_Indexer
//        |                   `- Expr_Caller
//        +- Expr_Value
//        `- Expr_Symbol
//-----------------------------------------------------------------------------
Expr::~Expr()
{
}

Expr *Expr::IncRef() const
{
	Expr *pExpr = const_cast<Expr *>(this);
	pExpr->_cntRef++;
	return pExpr;
}

Expr *Expr::CloneSubstituted(Environment &env, Signal sig) const
{
	return Clone();
}

ValueType Expr::GetStaticValueType() const
{
	return VTYPE_AnyType;
}

Value Expr::DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const
{
	SetError(sig, ERR_SyntaxError, "l-value is required");
	return Value::Null;
}

bool Expr::FindSymbol(const Symbol *pSymbol) const
{
	ExprVisitor_FindSymbol visitor(pSymbol);
	Accept(visitor);
	return visitor.GetFlag();
}

bool Expr::ExprVisitor_FindSymbol::Visit(const Expr *pExpr)
{
	if (_flag) return false;
	if (pExpr->IsSymbol()) {
		const Expr_Symbol *pExprSym = dynamic_cast<const Expr_Symbol *>(pExpr);
		if (pExprSym->GetSymbol()->IsIdentical(_pSymbol)) {
			_flag = true;
			return false;
		}
	}
	return true;
}

void Expr::GatherSymbol(SymbolSet &symbolSet) const
{
	ExprVisitor_GatherSymbol visitor(symbolSet);
	Accept(visitor);
}

bool Expr::ExprVisitor_GatherSymbol::Visit(const Expr *pExpr)
{
	if (pExpr->IsSymbol()) {
		const Expr_Symbol *pExprSym = dynamic_cast<const Expr_Symbol *>(pExpr);
		_symbolSet.insert(pExprSym->GetSymbol());
	}
	return true;
}

const Expr *Expr::Unquote() const
{
	return this;
}

bool Expr::IsValue() const			{ return false; }
bool Expr::IsSymbol() const			{ return false; }
bool Expr::IsConstructor() const	{ return false; }
bool Expr::IsComplex() const		{ return false; }
bool Expr::IsCaller() const			{ return false; }
bool Expr::IsIndexer() const		{ return false; }
bool Expr::IsContainer() const		{ return false; }
bool Expr::IsBlock() const			{ return false; }
bool Expr::IsBlockParam() const		{ return false; }
bool Expr::IsLister() const			{ return false; }

bool Expr::IsUnary() const			{ return false; }
bool Expr::IsQuote() const			{ return false; }
bool Expr::IsForce() const			{ return false; }
bool Expr::IsPrefix() const			{ return false; }
bool Expr::IsSuffix() const			{ return false; }
bool Expr::IsField() const			{ return false; }
bool Expr::IsUnaryOp() const		{ return false; }

bool Expr::IsBinary() const			{ return false; }
bool Expr::IsAssign() const			{ return false; }
bool Expr::IsDictAssign() const		{ return false; }
bool Expr::IsContainCheck() const	{ return false; }
bool Expr::IsBinaryOp() const		{ return false; }
bool Expr::IsNeg() const			{ return false; }
bool Expr::IsPlus() const			{ return false; }
bool Expr::IsMinus() const			{ return false; }
bool Expr::IsMultiply() const		{ return false; }
bool Expr::IsDivide() const			{ return false; }
bool Expr::IsPower() const			{ return false; }


bool Expr::IsConstZero() const		{ return false; }
bool Expr::IsConstOne() const		{ return false; }
bool Expr::IsConstMinusOne() const	{ return false; }
bool Expr::IsConstMinus() const		{ return false; }

void Expr::SetError(Signal sig, ErrorType errType, const char *format, ...) const
{
	va_list list;
	va_start(list, format);
	sig.SetErrorV(errType, format, list);
	sig.AddExprCause(this);
	va_end(list);
}

void Expr::SetError_NotAssignableSymbol(Signal sig, const Symbol *pSymbol) const
{
	SetError(sig, ERR_ValueError,
		"symbol '%s' cannot be assigned in this object", pSymbol->GetName());
}

//-----------------------------------------------------------------------------
// Expr_Unary
//-----------------------------------------------------------------------------
bool Expr_Unary::IsUnary() const { return true; }

Expr_Unary::~Expr_Unary()
{
}

Expr *Expr_Unary::IncRef() const
{
	GetChild()->IncRef();
	return Expr::IncRef();
}

void Expr_Unary::Accept(ExprVisitor &visitor) const
{
	if (visitor.Visit(this)) {
		GetChild()->Accept(visitor);
	}
}

//-----------------------------------------------------------------------------
// Expr_Binary
//-----------------------------------------------------------------------------
bool Expr_Binary::IsBinary() const { return true; }

Expr_Binary::~Expr_Binary()
{
}

Expr *Expr_Binary::IncRef() const
{
	GetLeft()->IncRef();
	GetRight()->IncRef();
	return Expr::IncRef();
}

void Expr_Binary::Accept(ExprVisitor &visitor) const
{
	if (visitor.Visit(this)) {
		GetLeft()->Accept(visitor);
		GetRight()->Accept(visitor);
	}
}

//-----------------------------------------------------------------------------
// Expr_Complex
//-----------------------------------------------------------------------------
bool Expr_Complex::IsComplex() const { return true; }

Expr_Complex::~Expr_Complex()
{
	Expr::Delete(_pExprCar);
	Expr::Delete(_pExprLister);
}

Expr *Expr_Complex::IncRef() const
{
	_pExprCar->IncRef();
	_pExprLister->IncRef();
	return Expr::IncRef();
}

//-----------------------------------------------------------------------------
// Expr_Value
//-----------------------------------------------------------------------------
bool Expr_Value::IsValue() const { return true; }

Expr_Value::~Expr_Value()
{
}

Expr *Expr_Value::IncRef() const
{
	return Expr::IncRef();
}

Expr *Expr_Value::Clone() const
{
	return new Expr_Value(*this);
}

Value Expr_Value::Exec(Environment &env, Signal sig) const
{
	if (_value.IsExpr()) return _value.GetExpr()->Exec(env, sig);
	if (sig.IsSignalled()) sig.AddExprCause(this);
	return _value;
}

ValueType Expr_Value::GetStaticValueType() const
{
	return _value.GetType();
}

void Expr_Value::Accept(ExprVisitor &visitor) const
{
	visitor.Visit(this);
}

bool Expr_Value::IsConstZero() const
{
	return _value.IsNumber() && _value.GetNumber() == 0.;
}

bool Expr_Value::IsConstOne() const
{
	return _value.IsNumber() && _value.GetNumber() == 1.;
}

bool Expr_Value::IsConstMinusOne() const
{
	return _value.IsNumber() && _value.GetNumber() == -1.;
}

bool Expr_Value::IsConstMinus() const
{
	return _value.IsNumber() && _value.GetNumber() < 0.;
}

String Expr_Value::ToString(Signal sig) const
{
	return _value.ToString(sig);
}

//-----------------------------------------------------------------------------
// Expr_Symbol
//-----------------------------------------------------------------------------
bool Expr_Symbol::IsSymbol() const { return true; }

Expr_Symbol::~Expr_Symbol()
{
}

Expr *Expr_Symbol::IncRef() const
{
	return Expr::IncRef();
}

Expr *Expr_Symbol::Clone() const
{
	return new Expr_Symbol(*this);
}

Expr *Expr_Symbol::CloneSubstituted(Environment &env, Signal sig) const
{
	const Value *pValue = env.LookupValue(GetSymbol(), false);
	if (pValue != NULL && pValue->IsExpr()) {
		// substitute with argument's expression
		return pValue->GetExpr()->Clone();
	}
	return Clone();
}

Value Expr_Symbol::Exec(Environment &env, Signal sig) const
{
	const Value *pValue = env.LookupValue(GetSymbol(), true);
	if (pValue != NULL) return *pValue;
	bool evaluatedFlag = false;
	Value result = env.EvalGetter(sig, GetSymbol(), evaluatedFlag);
	if (evaluatedFlag) return result;
	if (env.IsModule()) {
		SetError(sig, ERR_ValueError,
				"%s module does not have a symbol '%s'",
				dynamic_cast<Module *>(&env)->GetName(),
				GetSymbol()->GetName());
	} else if (env.IsClass()) {
		SetError(sig, ERR_ValueError,
				"%s class does not have a property '%s'",
				dynamic_cast<Class *>(&env)->GetName(),
				GetSymbol()->GetName());
	} else if (env.IsObject()) {
		SetError(sig, ERR_ValueError,
				"the object of %s class does not have a property '%s'",
				dynamic_cast<Object *>(&env)->GetClass()->GetName(),
				GetSymbol()->GetName());
	} else {
		SetError(sig, ERR_ValueError,
				"undefined symbol '%s'", GetSymbol()->GetName());
	}
	return Value::Null;
}

Value Expr_Symbol::DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const
{
	bool evaluatedFlag = false;
	Value result = env.EvalSetter(sig, GetSymbol(), value, evaluatedFlag);
	if (evaluatedFlag) return result;
	if (pSymbolsAssignable != NULL && !pSymbolsAssignable->IsSet(GetSymbol())) {
		SetError_NotAssignableSymbol(sig, GetSymbol());
		return Value::Null;
	}
	if (value.IsModule()) {
		Module *pModule = value.GetModule();
		if (pModule->IsAnonymous()) {
			pModule->SetSymbol(GetSymbol());
		}
	} else if (value.IsFunction()) {
		Function *pFunc = value.GetFunction();
		if (pFunc->IsAnonymous()) {
			pFunc->SetSymbol(GetSymbol());
		}
		Class *pClassToConstruct = pFunc->GetClassToConstruct();
		if (pClassToConstruct != NULL && pClassToConstruct->IsAnonymous()) {
			pClassToConstruct->SetSymbol(_pSymbol);
		}
	}
	env.AssignValue(GetSymbol(), value, true);
	return value;
}

void Expr_Symbol::Accept(ExprVisitor &visitor) const
{
	visitor.Visit(this);
}

String Expr_Symbol::ToString(Signal sig) const
{
	return GetSymbol()->GetName();
}

//-----------------------------------------------------------------------------
// Expr_Container
//-----------------------------------------------------------------------------
bool Expr_Container::IsContainer() const { return true; }

Expr_Container::~Expr_Container()
{
}

Expr *Expr_Container::IncRef() const
{
	_exprOwner.IncRef();
	return Expr::IncRef();
}

void Expr_Container::Accept(ExprVisitor &visitor) const
{
	if (visitor.Visit(this)) {
		_exprOwner.Accept(visitor);
	}
}

//-----------------------------------------------------------------------------
// Expr_Block
//-----------------------------------------------------------------------------
bool Expr_Block::IsBlock() const { return true; }

Expr_Block::~Expr_Block()
{
	Expr::Delete(_pExprBlockParam);
}

Expr *Expr_Block::Clone() const
{
	return new Expr_Block(*this);
}

Value Expr_Block::Exec(Environment &env, Signal sig) const
{
	if (_pExprBlockParam != NULL) {} // needs to do something here?
	if (env.IsType(ENVTYPE_Lister)) {
		return _exprOwner.ExecForList(env, sig, false);
	}
	return _exprOwner.Exec(env, sig);
}

void Expr_Block::Accept(ExprVisitor &visitor) const
{
	if (_pExprBlockParam != NULL) _pExprBlockParam->Accept(visitor);
	Expr_Container::Accept(visitor);
}

String Expr_Block::ToString(Signal sig) const
{
	String str;
	str += "{";
	if (_pExprBlockParam != NULL) {
		str += _pExprBlockParam->ToString(sig);
	}
	str += " ";
	str += GetExprList().ToString(sig);
	if (!GetExprList().empty()) str += " ";
	str += "}";
	return str;
}

//-----------------------------------------------------------------------------
// Expr_BlockParam
//-----------------------------------------------------------------------------
bool Expr_BlockParam::IsBlockParam() const { return true; }

Expr_BlockParam::~Expr_BlockParam()
{
}

Expr *Expr_BlockParam::Clone() const
{
	return new Expr_BlockParam(*this);
}

Value Expr_BlockParam::Exec(Environment &env, Signal sig) const
{
	return _exprOwner.Exec(env, sig);
}

String Expr_BlockParam::ToString(Signal sig) const
{
	String str;
	str += "|";
	str += GetExprList().ToString(sig);
	str += "|";
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Lister
//-----------------------------------------------------------------------------
bool Expr_Lister::IsLister() const { return true; }

Expr_Lister::~Expr_Lister()
{
}

Expr *Expr_Lister::Clone() const
{
	return new Expr_Lister(*this);
}

Value Expr_Lister::Exec(Environment &env, Signal sig) const
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ExprOwner, ppExpr, _exprOwner) {
		Value value = (*ppExpr)->Exec(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		if (value.IsIterator()) {
			Iterator *pIterator = value.CreateIterator(sig);
			if (sig.IsSignalled()) return Value::Null;
			if (pIterator->IsInfinite()) {
				sig.SetError(ERR_IteratorError,
							"infinite iterator cannot be realized as list");
				Iterator::Delete(pIterator);
				sig.AddExprCause(this);
				return Value::Null;
			}
			Value value;
			while (pIterator->Next(sig, value)) {
				valList.push_back(value);
			}
			Iterator::Delete(pIterator);
			if (sig.IsSignalled()) {
				sig.AddExprCause(this);
				return Value::Null;
			}
		} else {
			valList.push_back(value);
		}
	}
	return result;
}

Value Expr_Lister::DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const
{
	const ExprList &exprList = GetExprList();
	if (value.IsList()) {
		ValueList *pValList = NULL;
		ExprList::const_iterator ppExpr = exprList.begin();
		foreach (ValueList, pValue, value.GetList()) {
			if (pValList != NULL) {
				pValList->push_back(*pValue);
				continue;
			}
			if (ppExpr == exprList.end()) break;
			const Expr *pExpr = *ppExpr;
			OccurPattern occurPattern = OCCUR_Once;
			if (pExpr->IsSuffix()) {
				const Expr_Suffix *pExprSuffix =
								dynamic_cast<const Expr_Suffix *>(pExpr);
				pExpr = pExprSuffix->GetChild();
				occurPattern = pExprSuffix->GetOccurPattern();
				if (occurPattern == OCCUR_Invalid) {
					sig.SetError(ERR_SyntaxError,
								"invalid expression of array assignment");
					return Value::Null;
				}
			}
			if (occurPattern == OCCUR_ZeroOrMore || occurPattern == OCCUR_OnceOrMore) {
				Value value;
				pValList = &value.InitAsList(env);
				pExpr->DoAssign(env, sig, value, pSymbolsAssignable);
				if (sig.IsSignalled()) return Value::Null;
				pValList->push_back(*pValue);
			} else {
				pExpr->DoAssign(env, sig, *pValue, pSymbolsAssignable);
				if (sig.IsSignalled()) return Value::Null;
			}
			ppExpr++;
		}
		for ( ; ppExpr != exprList.end(); ppExpr++) {
			const Expr *pExpr = *ppExpr;
			OccurPattern occurPattern = OCCUR_Once;
			if (pExpr->IsSuffix()) {
				const Expr_Suffix *pExprSuffix =
								dynamic_cast<const Expr_Suffix *>(pExpr);
				pExpr = pExprSuffix->GetChild();
				occurPattern = pExprSuffix->GetOccurPattern();
			}
			if (occurPattern == OCCUR_ZeroOrMore) {
				Value value;
				pValList = &value.InitAsList(env);
				pExpr->DoAssign(env, sig, value, pSymbolsAssignable);
				if (sig.IsSignalled()) return Value::Null;
			} else if (occurPattern == OCCUR_Once || occurPattern == OCCUR_OnceOrMore) {
				sig.SetError(ERR_ValueError,
						"not enough value to initialize arrayed variables");
				return Value::Null;
			}
		}
	} else {
		foreach_const (ExprList, ppExpr, exprList) {
			const Expr *pExpr = *ppExpr;
			pExpr->DoAssign(env, sig, value, pSymbolsAssignable);
			if (sig.IsSignalled()) return Value::Null;
		}
	}
	return value;
}

String Expr_Lister::ToString(Signal sig) const
{
	String str;
	str += "[";
	str += GetExprList().ToString(sig);
	str += "]";
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Indexer
//-----------------------------------------------------------------------------
bool Expr_Indexer::IsIndexer() const { return true; }

Expr_Indexer::~Expr_Indexer()
{
}

Expr *Expr_Indexer::IncRef() const
{
	return Expr_Complex::IncRef();
}

Expr *Expr_Indexer::Clone() const
{
	return new Expr_Indexer(*this);
}

Value Expr_Indexer::Exec(Environment &env, Signal sig) const
{
	Value valueCar = GetCar()->Exec(env, sig);
	if (sig.IsSignalled()) {
		return Value::Null;
	}
	const ExprList &exprList = GetLister()->GetExprList();
	Object &objCar = *valueCar.GetObject();
	Value valueIdx = exprList.ExecForList(env, sig, true);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	ValueList &valIdxList = valueIdx.GetList();
	if (valIdxList.size() == 0) return Value::Null;
	if (!valueCar.IsObject()) {
		SetError(sig, ERR_ValueError, "object should be specified as l-value of indexer");
		return Value::Null;
	}
	Value result;
	if (valIdxList.size() == 1) {
		const Value &valueIdx = valIdxList.front();
		if (valueIdx.IsIterator()) {
			ValueList &valListDst = result.InitAsList(env);
			Iterator *pIterator = valueIdx.CreateIterator(sig);
			if (sig.IsSignalled()) {
				sig.AddExprCause(this);
				return Value::Null;
			}
			for (;;) {
				Value valueIdx;
				if (!pIterator->Next(sig, valueIdx)) {
					if (sig.IsSignalled()) {
						sig.AddExprCause(this);
						Iterator::Delete(pIterator);
						return Value::Null;
					}
					break;
				}
				Value value = objCar.GetByIndex(sig, valueIdx);
				if (sig.IsSignalled()) {
					if (sig.GetErrorType() == ERR_IndexError &&
														pIterator->IsInfinite()) {
						sig.ClearSignal();
						break;
					}
					sig.AddExprCause(this);
					return Value::Null;
				}
				valListDst.push_back(value);
			}
			Iterator::Delete(pIterator);
		} else {
			result = objCar.GetByIndex(sig, valueIdx);
		}
	} else {
		ValueList &valListDst = result.InitAsList(env);
		foreach_const (ValueList, pValueIdx, valIdxList) {
			if (pValueIdx->IsIterator()) {
				Iterator *pIterator = pValueIdx->CreateIterator(sig);
				if (sig.IsSignalled()) {
					sig.AddExprCause(this);
					return Value::Null;
				}
				for (;;) {
					Value valueIdx;
					if (!pIterator->Next(sig, valueIdx)) {
						if (sig.IsSignalled()) {
							sig.AddExprCause(this);
							Iterator::Delete(pIterator);
							return Value::Null;
						}
						break;
					}
					Value value = objCar.GetByIndex(sig, valueIdx);
					if (sig.IsSignalled()) {
						if (sig.GetErrorType() == ERR_IndexError &&
													pIterator->IsInfinite()) {
							sig.ClearSignal();
							break;
						}
						sig.AddExprCause(this);
						return Value::Null;
					}
					valListDst.push_back(value);
				}
				Iterator::Delete(pIterator);
			} else {
				Value value = objCar.GetByIndex(sig, *pValueIdx);
				if (sig.IsSignalled()) {
					sig.AddExprCause(this);
					return Value::Null;
				}
				valListDst.push_back(value);
			}
		}
	}
	return result;
}

Value Expr_Indexer::DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const
{
	Value valueDst = GetCar()->Exec(env, sig);
	if (sig.IsSignalled()) {
		return Value::Null;
	}
	if (!valueDst.IsObject()) {
		SetError(sig, ERR_ValueError, "object is expected as l-value of indexer");
		return Value::Null;
	}
	const ExprList &exprList = GetLister()->GetExprList();
	Object &objDst = *valueDst.GetObject();
	if (!value.IsList()) {
		// obj[idx] = v  /  obj[idx, idx, ..] = v
		ValueVisitor_AssignValue visitor(objDst, value);
		foreach_const (ExprList, ppExprCdr, exprList) {
			Value valueIdx = (*ppExprCdr)->Exec(env, sig);
			if (sig.IsSignalled()) {
				return Value::Null;
			}
			valueIdx.Accept(sig, visitor);
		}
	} else if (exprList.size() == 1) {
		// obj[idx] = (v, v, v, ..)
		Value valueIdx = exprList.front()->Exec(env, sig);
		if (sig.IsSignalled()) {
			return Value::Null;
		}
		if (valueIdx.IsList()) {
			ValueVisitor_AssignList visitor(objDst, value.GetList());
			valueIdx.Accept(sig, visitor);
		} else {
			objDst.SetByIndex(sig, valueIdx, value);
		}
	} else {
		// obj[idx, idx, ..] = (v, v, v, ..)
		const ValueList &valListCar = value.GetList();
		ValueVisitor_AssignList visitor(objDst, valListCar);
		foreach_const (ExprList, ppExprCdr, exprList) {
			Value valueIdx = (*ppExprCdr)->Exec(env, sig);
			if (sig.IsSignalled()) {
				return Value::Null;
			}
			valueIdx.Accept(sig, visitor);
		}
	}
	return value;
}

void Expr_Indexer::Accept(ExprVisitor &visitor) const
{
	if (visitor.Visit(this)) {
		GetCar()->Accept(visitor);
		GetLister()->Accept(visitor);
	}
}

void Expr_Indexer::ValueVisitor_AssignList::Visit(Signal sig, const Value &valueIdx)
{
	if (_pValueSrc != _valListSrc.end()) {
		_objDst.SetByIndex(sig, valueIdx, *_pValueSrc);
		_pValueSrc++;
	}
}

void Expr_Indexer::ValueVisitor_AssignValue::Visit(Signal sig, const Value &valueIdx)
{
	_objDst.SetByIndex(sig, valueIdx, _value);
}

String Expr_Indexer::ToString(Signal sig) const
{
	String str;
	str += GetCar()->ToString(sig);
	str += "[";
	str += GetLister()->GetExprList().ToString(sig);
	str += "]";
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Caller
//-----------------------------------------------------------------------------
bool Expr_Caller::IsCaller() const { return true; }

Expr_Caller::Expr_Caller(Expr *pExprCar, Expr_Lister *pExprLister, Expr_Block *pExprBlock) :
		Expr_Complex(EXPRTYPE_Function,
				pExprCar, (pExprLister == NULL)? new Expr_Lister() : pExprLister),
		_pExprBlock(pExprBlock), _pExprCallerSucc(NULL)
{
}

Expr_Caller::Expr_Caller(const Function *pFunc, Expr *pExprArg) :
		Expr_Complex(EXPRTYPE_Function,
				new Expr_Symbol(pFunc->GetSymbol()), new Expr_Lister()),
		_pExprBlock(NULL), _pExprCallerSucc(NULL)
{
	_pExprLister->AddExpr(pExprArg);
}

Expr_Caller::Expr_Caller(const Function *pFunc, Expr *pExprArg1, Expr *pExprArg2) :
		Expr_Complex(EXPRTYPE_Function,
				new Expr_Symbol(pFunc->GetSymbol()), new Expr_Lister()),
		_pExprBlock(NULL), _pExprCallerSucc(NULL)
{
	_pExprLister->AddExpr(pExprArg1);
	_pExprLister->AddExpr(pExprArg2);
}

Expr_Caller::Expr_Caller(const Expr_Caller &expr) : Expr_Complex(expr),
		_pExprBlock((expr._pExprBlock == NULL)? NULL :
					dynamic_cast<Expr_Block *>(expr._pExprBlock->Clone())),
		_pExprCallerSucc((expr._pExprCallerSucc == NULL)? NULL :
					dynamic_cast<Expr_Caller *>(expr._pExprCallerSucc->Clone())),
		_attrs(expr._attrs), _attrsOpt(expr._attrsOpt)
{
}

Expr_Caller::~Expr_Caller()
{
	Expr::Delete(_pExprBlock);
	Expr::Delete(_pExprCallerSucc);
}

Expr *Expr_Caller::IncRef() const
{
	if (_pExprBlock != NULL) _pExprBlock->IncRef();
	if (_pExprCallerSucc != NULL) _pExprCallerSucc->IncRef();
	return Expr_Complex::IncRef();
}

Expr *Expr_Caller::Clone() const
{
	return new Expr_Caller(*this);
}

Value Expr_Caller::Exec(Environment &env, Signal sig) const
{
	Value result;
	const Function *pFuncSuccRequester = NULL;
	for (const Expr_Caller *pExprCaller = this; pExprCaller != NULL;
									pExprCaller = pExprCaller->GetSucceeding()) {
		result = pExprCaller->DoExec(env, sig, &pFuncSuccRequester);
		if (pFuncSuccRequester == NULL) break;
	}
	// if there's an error suspended by try() function, it would be resumed below.
	// otherwise, nothing would happen and any error would be kept intact.
	sig.ResumeError();
	return result;
}

Value Expr_Caller::DoExec(Environment &env, Signal sig,
									const Function **ppFuncSuccRequester) const
{
	// Expr_Caller::Exec(), Expr_Field::Exec() and Expr_Field::DoAssign()
	// correspond to method-calling, property-getting and property-setting.
	if (!_pExprCar->IsField()) {
		Value valueCar = _pExprCar->Exec(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		ICallable *pCallable = valueCar.GetObject();
		if (pCallable == NULL) {
			SetError(sig, ERR_TypeError, "object is not callable");
			return Value::Null;
		}
		return pCallable->EvalExpr(env, sig, Value::Null, this, ppFuncSuccRequester);
	}
	const Expr_Field *pExprField = dynamic_cast<Expr_Field *>(_pExprCar);
	Value valueSelf = pExprField->GetLeft()->Exec(env, sig);
	if (sig.IsSignalled()) {
		return Value::Null;
	}
	if (pExprField->GetMode() == Expr_Field::MODE_Normal) {
		return EvalEach(env, sig, valueSelf, ppFuncSuccRequester);
	}
	ObjectBase *pObj = valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	if (pExprField->GetMode() == Expr_Field::MODE_Mapping) {
		Iterator *pIterator = pObj->CreateIterator(sig);
		if (pIterator == NULL) {
			if (sig.IsSignalled()) return Value::Null;
			return EvalEach(env, sig, valueSelf, ppFuncSuccRequester);
		}
		Value result;
		ValueList *pValList = NULL;
		for (size_t n = 0; ; n++) {
			Value valueSelfEach;
			if (!pIterator->Next(sig, valueSelfEach)) {
				if (sig.IsSignalled()) {
					sig.AddExprCause(this);
					Iterator::Delete(pIterator);
					return Value::Null;
				}
				if (n == 0) result.InitAsList(env);
				break;
			}
			Value valueEach = EvalEach(env, sig, valueSelfEach, ppFuncSuccRequester);
			if (sig.IsSignalled()) {
				Iterator::Delete(pIterator);
				return Value::Null;
			}
			if (valueEach.IsValid()) {
				if (pValList == NULL) {
					pValList = &result.InitAsList(env, n, Value::Null);
				}
				pValList->push_back(valueEach);
			} else if (pValList != NULL) {
				pValList->push_back(valueEach);
			}
		}
		Iterator::Delete(pIterator);
		return result;
	} else {	// pExprField->GetMode() == Expr_Field::MODE_MapToIter
		Iterator *pIterator = pObj->CreateIterator(sig);
		if (pIterator == NULL) {
			if (sig.IsSignalled()) return Value::Null;
			return EvalEach(env, sig, valueSelf, ppFuncSuccRequester);
		}
		pIterator = new Iterator_MethodMapping(env,
							pIterator, dynamic_cast<Expr_Caller *>(IncRef()));
		return Value(env, pIterator);
	}
}

Value Expr_Caller::EvalEach(Environment &env, Signal sig,
				Value &valueSelf, const Function **ppFuncSuccRequester) const
{
	const Expr_Field *pExprField = dynamic_cast<Expr_Field *>(_pExprCar);
	ASSUME(env, pExprField != NULL);
	const Expr *pExprRight = pExprField->GetRight();
	ObjectBase *pObj = valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) return Value::Null;
	ICallable *pCallable = NULL;
	if (pExprRight->IsSymbol()) {
		pCallable = pObj->GetCallable(sig,
						dynamic_cast<const Expr_Symbol *>(pExprRight)->GetSymbol());
		if (sig.IsSignalled()) return Value::Null;
	}
	if (pCallable == NULL) {
		Value valueCar = pExprRight->Exec(*pObj, sig);
		if (sig.IsSignalled()) return Value::Null;
		pCallable = valueCar.GetObject();
	}
	if (pCallable == NULL) {
		SetError(sig, ERR_TypeError, "object is not callable");
		return Value::Null;
	}
	return pCallable->EvalExpr(env, sig, valueSelf, this, ppFuncSuccRequester);
}

Value Expr_Caller::DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const
{
	if (!value.IsExpr()) {
		SetError(sig, ERR_SyntaxError, "invalid function expression");
		return Value::Null;
	}
	Expr *pExprBody = value.GetExpr()->IncRef();
	// get symbol part of function's declaration
	const Symbol *pSymbol;
	if (GetCar()->IsField()) {
		const Expr_Field *pExprField =
						dynamic_cast<const Expr_Field *>(GetCar());
		if (pExprField->GetRight()->IsSymbol()) {
			const Expr_Symbol *pExprSym =
						dynamic_cast<const Expr_Symbol *>(pExprField->GetRight());
			pSymbol = pExprSym->GetSymbol();
		}
	} else if (GetCar()->IsSymbol()) {
		const Expr_Symbol *pExprSym =
						dynamic_cast<const Expr_Symbol *>(GetCar());
		pSymbol = pExprSym->GetSymbol();
	} else {
		SetError(sig, ERR_SyntaxError, "invalid function expression");
		return Value::Null;
	}
	if (pSymbolsAssignable != NULL && !pSymbolsAssignable->IsSet(pSymbol)) {
		SetError_NotAssignableSymbol(sig, pSymbol);
		return Value::Null;
	}
	FunctionCustom *pFunc =
			new FunctionCustom(env, pSymbol, pExprBody, NULL, FUNCTYPE_Function);
	ContextExpr contextExpr(GetAttrs(), GetAttrsOpt(), _pExprBlock, NULL,
												Value::Null, GetExprList());
	if (!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
		sig.AddExprCause(this);
		Function::Delete(pFunc);
		return Value::Null;
	}
	Value valueFunc;
	valueFunc.InitAsFunction(env, pFunc);
	GetCar()->DoAssign(env, sig, valueFunc, pSymbolsAssignable);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	return valueFunc;
}

void Expr_Caller::Accept(ExprVisitor &visitor) const
{
	if (visitor.Visit(this)) {
		GetCar()->Accept(visitor);
		GetExprList().Accept(visitor);
	}
}

String Expr_Caller::ToString(Signal sig) const
{
	String str;
	str += _pExprCar->ToString(sig);
	bool argListFlag = !GetExprList().empty() ||
									!_attrs.empty() || _pExprBlock == NULL;
	if (_pExprCar->IsSymbol()) {
		const Symbol *pSymbol = dynamic_cast<Expr_Symbol *>(_pExprCar)->GetSymbol();
		if (pSymbol->IsFlowControlSymbol() && argListFlag) {
			str += " ";
		}
	}
	if (argListFlag) {
		str += "(";
		str += GetExprList().ToString(sig);
		str += ")";
	}
	foreach_const (SymbolSet, ppSymbol, _attrs) {
		const Symbol *pSymbol = *ppSymbol;
		str += ":";
		str += pSymbol->GetName();
	}
	if (_pExprBlock != NULL) {
		str += " ";
		str += _pExprBlock->ToString(sig);
	}
	if (_pExprCallerSucc != NULL) {
		str += " ";
		str += _pExprCallerSucc->ToString(sig);
	}
	return str;
}

//-----------------------------------------------------------------------------
// Expr_UnaryOp
//-----------------------------------------------------------------------------
bool Expr_UnaryOp::IsUnaryOp() const { return true; }
bool Expr_UnaryOp::IsNeg() const { return _func.IsNeg(); }

Expr_UnaryOp::~Expr_UnaryOp()
{
}

Value Expr_UnaryOp::Exec(Environment &env, Signal sig) const
{
	ContextExpr contextExpr(GetExprList());
	Value result = _func.EvalExpr(env, sig, contextExpr);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	return result;
}

Expr *Expr_UnaryOp::Clone() const
{
	return new Expr_UnaryOp(*this);
}

String Expr_UnaryOp::ToString(Signal sig) const
{
	String str;
	if (!_suffixSymbolFlag) str += _func.GetMathSymbol();
	if (GetChild()->IsUnaryOp() || GetChild()->IsBinaryOp()) {
		str += "(";
		str += GetChild()->ToString(sig);
		str += ")";
	} else {
		str += GetChild()->ToString(sig);
	}
	if (_suffixSymbolFlag) str += _func.GetMathSymbol();
	return str;
}

//-----------------------------------------------------------------------------
// Expr_BinaryOp
//-----------------------------------------------------------------------------
bool Expr_BinaryOp::IsBinaryOp() const { return true; }
bool Expr_BinaryOp::IsPlus() const { return _func.IsPlus(); }
bool Expr_BinaryOp::IsMinus() const { return _func.IsMinus(); }
bool Expr_BinaryOp::IsMultiply() const { return _func.IsMultiply(); }
bool Expr_BinaryOp::IsDivide() const { return _func.IsDivide(); }
bool Expr_BinaryOp::IsPower() const { return _func.IsPower(); }
bool Expr_BinaryOp::IsContainCheck() const { return _func.IsContainCheck(); }

Expr_BinaryOp::~Expr_BinaryOp()
{
}

Value Expr_BinaryOp::Exec(Environment &env, Signal sig) const
{
	ContextExpr contextExpr(GetExprList());
	Value result = _func.EvalExpr(env, sig, contextExpr);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	return result;
}

Expr *Expr_BinaryOp::Clone() const
{
	return new Expr_BinaryOp(*this);
}

String Expr_BinaryOp::ToString(Signal sig) const
{
	String str;
	if (GetLeft()->IsUnaryOp() || GetLeft()->IsBinaryOp()) {
		str += "(";
		str += GetLeft()->ToString(sig);
		str += ")";
	} else {
		str += GetLeft()->ToString(sig);
	}
	str += " ";
	str += _func.GetMathSymbol();
	str += " ";
	if (GetRight()->IsUnaryOp() || GetRight()->IsBinaryOp()) {
		str += "(";
		str += GetRight()->ToString(sig);
		str += ")";
	} else {
		str += GetRight()->ToString(sig);
	}
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Quote
//-----------------------------------------------------------------------------
bool Expr_Quote::IsQuote() const { return true; }

Expr_Quote::~Expr_Quote()
{
}

Expr *Expr_Quote::Clone() const
{
	return new Expr_Quote(*this);
}

const Expr *Expr_Quote::Unquote() const
{
	return GetChild();
}

Value Expr_Quote::Exec(Environment &env, Signal sig) const
{
	Value value;
	if (GetChild()->IsSymbol()) {
		const Expr_Symbol *pExprSym =
						dynamic_cast<const Expr_Symbol *>(GetChild());
		value.SetSymbol(pExprSym->GetSymbol());
	} else {
		value.InitAsExpr(env, GetChild()->IncRef());
	}
	return value;
}

ValueType Expr_Quote::GetStaticValueType() const
{
	return GetChild()->IsSymbol()? VTYPE_Symbol : VTYPE_Expr;
}

String Expr_Quote::ToString(Signal sig) const
{
	String str;
	str += "`";
	str += GetChild()->ToString(sig);
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Force
//-----------------------------------------------------------------------------
bool Expr_Force::IsForce() const { return true; }

Expr_Force::~Expr_Force()
{
}

Expr *Expr_Force::Clone() const
{
	return new Expr_Force(*this);
}

Value Expr_Force::Exec(Environment &env, Signal sig) const
{
	Value result = GetChild()->Exec(env, sig);
	return result;
}

String Expr_Force::ToString(Signal sig) const
{
	String str;
	str += "!!";
	str += GetChild()->ToString(sig);
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Prefix
//-----------------------------------------------------------------------------
bool Expr_Prefix::IsPrefix() const { return true; }

Expr_Prefix::~Expr_Prefix()
{
}

Expr *Expr_Prefix::Clone() const
{
	return new Expr_Prefix(*this);
}

Value Expr_Prefix::Exec(Environment &env, Signal sig) const
{
	SetError(sig, ERR_SyntaxError, "invalid expression");
	return Value::Null;
}

String Expr_Prefix::ToString(Signal sig) const
{
	String str;
	str += _pSymbol->GetName();
	str += GetChild()->ToString(sig);
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Suffix
//-----------------------------------------------------------------------------
bool Expr_Suffix::IsSuffix() const { return true; }

Expr_Suffix::~Expr_Suffix()
{
}

Expr *Expr_Suffix::Clone() const
{
	return new Expr_Suffix(*this);
}

Value Expr_Suffix::Exec(Environment &env, Signal sig) const
{
	SetError(sig, ERR_SyntaxError, "invalid expression");
	return Value::Null;
}

OccurPattern Expr_Suffix::GetOccurPattern() const
{
	return
		(_pSymbol->IsIdentical(AScript_Symbol(Char_Multiply)))?	OCCUR_ZeroOrMore :
		(_pSymbol->IsIdentical(AScript_Symbol(Char_Plus)))?		OCCUR_OnceOrMore :
		(_pSymbol->IsIdentical(AScript_Symbol(Char_Question)))?	OCCUR_ZeroOrOnce :
		OCCUR_Invalid;
}

String Expr_Suffix::ToString(Signal sig) const
{
	String str;
	str += GetChild()->ToString(sig);
	str += _pSymbol->GetName();
	return str;
}

//-----------------------------------------------------------------------------
// Expr_Assign
//-----------------------------------------------------------------------------
bool Expr_Assign::IsAssign() const { return true; }

Expr_Assign::~Expr_Assign()
{
}

Expr *Expr_Assign::Clone() const
{
	return new Expr_Assign(*this);
}

Value Expr_Assign::Exec(Environment &env, Signal sig) const
{
	return Exec(env, sig, env, NULL);
}

Value Expr_Assign::Exec(Environment &env, Signal sig,
					Environment &envDst, const SymbolSet *pSymbolsAssignable) const
{
	Value value;
	const Expr *pExpr = GetLeft();
	bool funcAssignFlag = false;
	if (pExpr->IsCaller()) {
		if (_pFuncToApply != NULL) {
			SetError(sig, ERR_SyntaxError, "invalid operation");
			return Value::Null;
		}
		if (GetRight()->IsForce()) {
			value = GetRight()->Exec(env, sig);
			if (sig.IsSignalled()) {
				return Value::Null;
			}
		} else {
			Expr *pExprBody = GetRight()->Unquote()->IncRef();
			value.InitAsExpr(env, pExprBody);
		}
	} else {
		value = GetRight()->Exec(env, sig);
		if (sig.IsSignalled()) {
			return Value::Null;
		}
		if (_pFuncToApply != NULL) {
			Value valueLeft = pExpr->Exec(env, sig);
			if (sig.IsSignalled()) {
				return Value::Null;
			}
			ValueList valListArg(valueLeft, value);
			Context context(valListArg);
			value = _pFuncToApply->IsApplicable(valListArg)?
							_pFuncToApply->Eval(env, sig, context) :
							_pFuncToApply->EvalMap(env, sig, context);
			if (sig.IsSignalled()) {
				sig.AddExprCause(this);
				return Value::Null;
			}
		}
	}
	return GetLeft()->DoAssign(envDst, sig, value, pSymbolsAssignable);
}

String Expr_Assign::ToString(Signal sig) const
{
	String str;
	str += GetLeft()->ToString(sig);
	str += " ";
	if (_pFuncToApply != NULL) str += _pFuncToApply->GetMathSymbol();
	str += "= ";
	str += GetRight()->ToString(sig);
	return str;
}

//-----------------------------------------------------------------------------
// Expr_DictAssign
//-----------------------------------------------------------------------------
bool Expr_DictAssign::IsDictAssign() const { return true; }

Expr_DictAssign::~Expr_DictAssign()
{
}

Expr *Expr_DictAssign::Clone() const
{
	return new Expr_DictAssign(*this);
}

Value Expr_DictAssign::Exec(Environment &env, Signal sig) const
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	const Expr *pExprLeft = GetLeft()->Unquote();
	if (pExprLeft->IsValue()) {
		// value => value
		const Expr_Value *pExprValue =
						dynamic_cast<const Expr_Value *>(pExprLeft);
		valList.push_back(pExprValue->GetValue());
	} else if (pExprLeft->IsSymbol()) {
		// symbol => value
		const Expr_Symbol *pExprSymbol =
						dynamic_cast<const Expr_Symbol *>(pExprLeft);
		valList.push_back(Value(pExprSymbol->GetSymbol()));
	} else {
		SetError(sig, ERR_KeyError, "key must be a value or symbol");
		return Value::Null;
	}
	Value value = GetRight()->Exec(env, sig);
	if (sig.IsSignalled()) {
		return Value::Null;
	}
	valList.push_back(value);
	return result;
}

String Expr_DictAssign::ToString(Signal sig) const
{
	String str;
	str += GetLeft()->Unquote()->ToString(sig);
	str += " => ";
	str += GetRight()->ToString(sig);
	return str;
}

Value Expr_DictAssign::GetKey(Signal sig) const
{
	const Expr *pExpr = GetLeft()->Unquote();
	if (pExpr->IsSymbol()) {
		const Symbol *pSymbol = dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
		return Value(pSymbol);
	} else if (pExpr->IsValue()) {
		return dynamic_cast<const Expr_Value *>(pExpr)->GetValue();
	} else {
		SetError(sig, ERR_KeyError,
				"l-value of dictionary assignment must be a symbol or a constant value");
		return Value::Null;
	}
}

//-----------------------------------------------------------------------------
// Expr_Field
//-----------------------------------------------------------------------------
bool Expr_Field::IsField() const { return true; }

Expr_Field::~Expr_Field()
{
}

Expr *Expr_Field::Clone() const
{
	return new Expr_Field(*this);
}

Value Expr_Field::Exec(Environment &env, Signal sig) const
{
	// Expr_Caller::Exec(), Expr_Field::Exec() and Expr_Field::DoAssign()
	// correspond to method-calling, property-getting and property-setting.
	Value valueSelf = GetLeft()->Exec(env, sig);
	ObjectBase *pObj = valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	if (GetMode() == MODE_Normal) {
		Value result = GetRight()->Exec(*pObj, sig);
		if (result.IsFunction()) {
			Object *pObj = new Object_WrappedMethod(env.GetClass_Function(),
									result.GetFunction()->IncRef(), valueSelf);
			return Value(pObj, VTYPE_Function, true);
		} else {
			return result;
		}
	} else if (GetMode() == MODE_Mapping) {
		Iterator *pIterator = pObj->CreateIterator(sig);
		if (pIterator == NULL) {
			if (sig.IsSignalled()) return Value::Null;
			return GetRight()->Exec(*pObj, sig);
		}
		Value result;
		ValueList *pValList = NULL;
		for (size_t n = 0; ; n++) {
			Value valueSelfEach;
			if (!pIterator->Next(sig, valueSelfEach)) {
				if (sig.IsSignalled()) {
					sig.AddExprCause(this);
					Iterator::Delete(pIterator);
					return Value::Null;
				}
				if (n == 0) result.InitAsList(env);
				break;
			}
			ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
			if (sig.IsSignalled()) {
				sig.AddExprCause(this);
				Iterator::Delete(pIterator);
				return Value::Null;
			}
			Value resultEach = GetRight()->Exec(*pObjEach, sig);
			if (sig.IsSignalled()) {
				Iterator::Delete(pIterator);
				return Value::Null;
			}
			if (resultEach.IsValid()) {
				if (pValList == NULL) {
					pValList = &result.InitAsList(env, n, Value::Null);
				}
				if (resultEach.IsFunction()) {
					Object *pObj = new Object_WrappedMethod(env.GetClass_Function(),
								resultEach.GetFunction()->IncRef(), valueSelfEach);
					pValList->push_back(Value(pObj, VTYPE_Function, true));
				} else {
					pValList->push_back(resultEach);
				}
			} else if (pValList != NULL) {
				pValList->push_back(Value::Null);
			}
		}
		Iterator::Delete(pIterator);
		return result;
	} else { // GetMode() == MODE_MapToIter
		Iterator *pIterator = pObj->CreateIterator(sig);
		if (pIterator == NULL) {
			if (sig.IsSignalled()) return Value::Null;
			return GetRight()->Exec(*pObj, sig);
		}
		pIterator = new Iterator_MemberMapping(pIterator, GetRight()->IncRef());
		return Value(env, pIterator);
	}
}

Value Expr_Field::DoAssign(Environment &env, Signal sig,
					Value &value, const SymbolSet *pSymbolsAssignable) const
{
	// Expr_Caller::Exec(), Expr_Field::Exec() and Expr_Field::DoAssign()
	// correspond to method-calling, property-getting and property-setting.
	Value valueSelf = GetLeft()->Exec(env, sig);
	if (sig.IsSignalled()) {
		return Value::Null;
	}
	ObjectBase *pObj = valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) {
		sig.AddExprCause(this);
		return Value::Null;
	}
	if (GetMode() == MODE_Normal) {
		return GetRight()->DoAssign(*pObj, sig, value, pSymbolsAssignable);
	} else if (GetMode() == MODE_Mapping) {
		Iterator *pIterator = pObj->CreateIterator(sig);
		if (pIterator == NULL) {
			if (sig.IsSignalled()) return Value::Null;
			return GetRight()->DoAssign(*pObj, sig, value, pSymbolsAssignable);
		}
		if (value.IsList()) {
			ValueList &valListSrc = value.GetList();
			ValueList::iterator pValueSrc = valListSrc.begin();
			for ( ; pValueSrc != valListSrc.end(); pValueSrc++) {
				Value valueSelfEach;
				if (!pIterator->Next(sig, valueSelfEach)) {
					if (sig.IsSignalled()) {
						sig.AddExprCause(this);
						Iterator::Delete(pIterator);
						return Value::Null;
					}
					break;
				}
				ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
				if (sig.IsSignalled()) {
					sig.AddExprCause(this);
					Iterator::Delete(pIterator);
					return Value::Null;
				}
				GetRight()->DoAssign(*pObjEach, sig, *pValueSrc, pSymbolsAssignable);
				if (sig.IsSignalled()) return Value::Null;
			}
		} else {
			for (;;) {
				Value valueSelfEach;
				if (!pIterator->Next(sig, valueSelfEach)) {
					if (sig.IsSignalled()) {
						sig.AddExprCause(this);
						Iterator::Delete(pIterator);
						return Value::Null;
					}
					break;
				}
				ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
				if (sig.IsSignalled()) {
					sig.AddExprCause(this);
					Iterator::Delete(pIterator);
					return Value::Null;
				}
				GetRight()->DoAssign(*pObjEach, sig, value, pSymbolsAssignable);
				if (sig.IsSignalled()) {
					Iterator::Delete(pIterator);
					return Value::Null;
				}
			}
		}
		Iterator::Delete(pIterator);
	} else { // GetMode() == MODE_MapToIter
		SetError(sig, ERR_ValueError, "assignment to iterator is not supported");
		return Value::Null;
	}
	return value;
}

String Expr_Field::ToString(Signal sig) const
{
	String str;
	str += GetLeft()->ToString(sig);
	str +=
		(_mode == MODE_Normal)? "." :
		(_mode == MODE_Mapping)? "::" :
		(_mode == MODE_MapToIter)? ":*" : "????";
	str += GetRight()->ToString(sig);
	return str;
}

//-----------------------------------------------------------------------------
// ExprList
//-----------------------------------------------------------------------------
const ExprList ExprList::Null;

void ExprList::IncRef() const
{
	foreach (ExprList, ppExpr, *const_cast<ExprList *>(this)) {
		(*ppExpr)->IncRef();
	}
}

Value ExprList::Exec(Environment &env, Signal sig) const
{
	Value result;
	foreach_const (ExprList, ppExpr, *this) {
		result = (*ppExpr)->Exec(env, sig);
		if (sig.IsSignalled()) {
			break;
		}
	}
	return result;
}

Value ExprList::ExecForList(Environment &env, Signal sig, bool flattenFlag) const
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ExprList, ppExpr, *this) {
		Value value = (*ppExpr)->Exec(env, sig);
		if (sig.IsSignalled()) {
			return Value::Null;
		}
		if (flattenFlag && value.IsList()) {
			ValueVisitorEx visitor(valList);
			value.Accept(sig, visitor);
		} else {
			valList.push_back(value);
		}
	}
	return result;
}

void ExprList::Accept(ExprVisitor &visitor) const
{
	foreach_const (ExprList, ppExpr, *this) {
		(*ppExpr)->Accept(visitor);
	}
}

String ExprList::ToString(Signal sig, const char *sep) const
{
	String str;
	foreach_const (ExprList, ppExpr, *this) {
		if (ppExpr != begin()) str += sep;
		str += (*ppExpr)->ToString(sig);
	}
	return str;
}

void ExprList::ValueVisitorEx::Visit(Signal sig, const Value &value)
{
	_valList.push_back(value);
}

//-----------------------------------------------------------------------------
// ExprOwner
//-----------------------------------------------------------------------------
ExprOwner::ExprOwner(const ExprList &exprList)
{
	foreach_const (ExprList, ppExpr, exprList) {
		push_back((*ppExpr)->Clone());
	}
}

ExprOwner::ExprOwner(const ExprOwner &exprOwner)
{
	foreach_const (ExprOwner, ppExpr, exprOwner) {
		push_back((*ppExpr)->Clone());
	}
}

ExprOwner::~ExprOwner()
{
	foreach (ExprOwner, ppExpr, *this) {
		Expr::Delete(const_cast<Expr *>(*ppExpr));
	}
}

}
