#include "Iterator.h"
#include "Object.h"
#include "Expr.h"
#include "OAL.h"
#include "Function.h"
#include "Object_Function.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
Iterator::~Iterator()
{
}

Value Iterator::Eval(Environment &env, Signal sig, Context &context)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value, result;
	Function::ResultListComposer resultListComposer(env, context, result);
	while (Next(sig, value)) {
		resultListComposer.Store(value);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Eval(Environment &envBlock, Signal sig, Context &context,
												const Function *pFuncBlock)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value, result;
	Function::ResultListComposer resultListComposer(envBlock, context, result);
	for (int idx = 0; Next(sig, value); idx++) {
		ValueList valListArg(value, Value(static_cast<Number>(idx)));
		Context contextSub(valListArg);
		Value resultElem = pFuncBlock->Eval(envBlock, sig, contextSub);
		AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
		resultListComposer.Store(resultElem);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

String Iterator::ToString(Signal sig) const
{
	return String("<iterator>");
}

Value Iterator::MinMax(Environment &env, Signal sig,
									bool maxFlag, const SymbolSet &attrs)
{
	Value valueHit;
	if (!Next(sig, valueHit)) return Value::Null;
	Value result;
	if (attrs.IsSet(AScript_Symbol(index))) {
		size_t idxHit = 0;
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
				idxHit = idx;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result.SetNumber(idxHit);
	} else if (attrs.IsSet(AScript_Symbol(last_index))) {
		size_t idxHit = 0;
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp >= 0) {
				valueHit = value;
				idxHit = idx;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result.SetNumber(idxHit);
	} else if (attrs.IsSet(AScript_Symbol(indices))) {
		ValueList &resultList = result.InitAsList(env);
		resultList.push_back(Value(static_cast<Number>(0)));
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
				resultList.clear();
				resultList.push_back(Value(static_cast<Number>(idx)));
			} else if (cmp == 0) {
				resultList.push_back(Value(static_cast<Number>(idx)));
			}
		}
		if (sig.IsSignalled()) return Value::Null;
	} else {
		Value value;
		while (Next(sig, value)) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result = valueHit;
	}
	return result;
}

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
Iterator_Constant::~Iterator_Constant()
{
}

bool Iterator_Constant::Rewind()
{
	return false;
}

bool Iterator_Constant::Next(Signal sig, Value &value)
{
	value = _value;
	return true;
}

String Iterator_Constant::ToString(Signal sig) const
{
	String rtn = "<iterator:";
	rtn += _value.ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Fill
//-----------------------------------------------------------------------------
Iterator_Fill::~Iterator_Fill()
{
}

bool Iterator_Fill::Rewind()
{
	return false;
}

bool Iterator_Fill::Next(Signal sig, Value &value)
{
	if (_idx >= _cnt) return false;
	if (_cnt > 0) _idx++;
	value = _value;
	return true;
}

String Iterator_Fill::ToString(Signal sig) const
{
	String rtn = "<iterator:fill(";
	rtn += NumberToString(_cnt);
	if (_value.IsValid()) {
		rtn += ", ";
		rtn += _value.ToString(sig, true);
	}
	rtn += ")>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Rand
//-----------------------------------------------------------------------------
Iterator_Rand::~Iterator_Rand()
{
}

bool Iterator_Rand::Rewind()
{
	return false;
}

bool Iterator_Rand::Next(Signal sig, Value &value)
{
	if (_idx >= _cnt) return false;
	if (_cnt > 0) _idx++;
	value = Value((_range > 0)?
		static_cast<Number>(static_cast<int>(::genrand_real2() * _range)) :
		static_cast<Number>(::genrand_real2()));
	return true;
}

String Iterator_Rand::ToString(Signal sig) const
{
	String rtn = "<iterator:rands(";
	rtn += NumberToString(_cnt);
	if (_range > 0) {
		rtn += ", ";
		rtn += NumberToString(_range);
	}
	rtn += ")>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Range
//-----------------------------------------------------------------------------
Iterator_Range::~Iterator_Range()
{
}

bool Iterator_Range::Rewind()
{
	return false;
}

bool Iterator_Range::Next(Signal sig, Value &value)
{
	if (!((_numStep > 0)? (_num < _numEnd) : (_num > _numEnd))) return false;
	value = Value(_num);
	_num += _numStep;
	return true;
}

String Iterator_Range::ToString(Signal sig) const
{
	String rtn;
	if (_numStep == 1 || _numStep == -1) {
		if (_numBegin == 0) {
			rtn = "<iterator:range(";
			rtn += NumberToString(_numEnd);
			rtn += ")>";
		} else {
			rtn = "<iterator:range(";
			rtn += NumberToString(_numBegin);
			rtn += ", ";
			rtn += NumberToString(_numEnd);
			rtn += ")>";
		}
	} else {
		rtn = "<iterator:range(";
		rtn += NumberToString(_numBegin);
		rtn += ", ";
		rtn += NumberToString(_numEnd);
		rtn += ", ";
		rtn += NumberToString(_numStep);
		rtn += ")>";
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Sequence
//-----------------------------------------------------------------------------
Iterator_Sequence::~Iterator_Sequence()
{
}

bool Iterator_Sequence::Rewind()
{
	return false;
}

bool Iterator_Sequence::Next(Signal sig, Value &value)
{
	if (!((_numStep > 0)? (_num <= _numEnd) : (_num >= _numEnd))) return false;
	value = Value(_num);
	_num += _numStep;
	return true;
}

String Iterator_Sequence::ToString(Signal sig) const
{
	String rtn;
	if (_numStep == 1. || _numStep == -1.) {
		rtn = "<iterator:";
		rtn += NumberToString(_numBegin);
		rtn += "..";
		rtn += NumberToString(_numEnd);
		rtn += ">";
	} else {
		rtn = "<iterator:seq(";
		rtn += NumberToString(_numBegin);
		rtn += ", ";
		rtn += NumberToString(_numEnd);
		rtn += ", ";
		rtn += NumberToString(_numStep);
		rtn += ")>";
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SequenceInf
//-----------------------------------------------------------------------------
Iterator_SequenceInf::~Iterator_SequenceInf()
{
}

bool Iterator_SequenceInf::Rewind()
{
	return false;
}

bool Iterator_SequenceInf::Next(Signal sig, Value &value)
{
	value = Value(_num);
	_num += 1;
	return true;
}

String Iterator_SequenceInf::ToString(Signal sig) const
{
	String rtn;
	rtn = "<iterator:";
	rtn += NumberToString(_numBegin);
	rtn += "..>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Interval
//-----------------------------------------------------------------------------
Iterator_Interval::~Iterator_Interval()
{
}

bool Iterator_Interval::Rewind()
{
	return false;
}

bool Iterator_Interval::Next(Signal sig, Value &value)
{
	if (_idx >= _numSamples) return false;
	value = Value((_numEnd - _numBegin) * _iFactor / _numDenom + _numBegin);
	_iFactor++, _idx++;
	return true;
}

String Iterator_Interval::ToString(Signal sig) const
{
	return String("<iterator:interval>");
}

//-----------------------------------------------------------------------------
// Iterator_Mapping
//-----------------------------------------------------------------------------
Iterator_Mapping::Iterator_Mapping(Environment &env, Signal sig,
									Function *pFunc, Context &context) :
	Iterator(false), _env(env), _pFunc(pFunc), _valueSelf(context.GetSelf())
{
	pFunc->PrepareIteratorsForMap(sig, _iterOwner, context.GetArgs());
}

Iterator_Mapping::~Iterator_Mapping()
{
	Function::Delete(_pFunc);
}

bool Iterator_Mapping::Rewind()
{
	return false;
}

bool Iterator_Mapping::Next(Signal sig, Value &value)
{
	ValueList valList;
	if (!_iterOwner.Next(sig, valList)) return false;
	Context context(valList);
	context.SetSelf(_valueSelf);
	value = _pFunc->Eval(_env, sig, context);
	if (sig.IsSignalled()) return false;
	return true;
}

String Iterator_Mapping::ToString(Signal sig) const
{
	return String("<iterator:mapping>");
}

//-----------------------------------------------------------------------------
// Iterator_MemberMapping
//-----------------------------------------------------------------------------
Iterator_MemberMapping::~Iterator_MemberMapping()
{
	Expr::Delete(_pExpr);
	Iterator::Delete(_pIterator);
}

bool Iterator_MemberMapping::Rewind()
{
	return false;
}

bool Iterator_MemberMapping::Next(Signal sig, Value &value)
{
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
	if (sig.IsSignalled()) return false;
	Environment &env = *pObjEach;
	value = _pExpr->Exec(env, sig);
	if (value.IsFunction()) {
		Object *pObj = new Object_WrappedMethod(env.GetClass_Function(),
							value.GetFunction()->IncRef(), valueSelfEach);
		value = Value(pObj, VTYPE_Function, true);
	}
	return true;
}

String Iterator_MemberMapping::ToString(Signal sig) const
{
	return String("<iterator:member_mapping>");
}

//-----------------------------------------------------------------------------
// Iterator_MethodMapping
//-----------------------------------------------------------------------------
Iterator_MethodMapping::~Iterator_MethodMapping()
{
	Expr::Delete(_pExprCaller);
	Iterator::Delete(_pIterator);
}

bool Iterator_MethodMapping::Rewind()
{
	return false;
}

bool Iterator_MethodMapping::Next(Signal sig, Value &value)
{
	const Function *pFuncSuccRequester = NULL;
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	value = _pExprCaller->EvalEach(_env, sig, valueSelfEach, &pFuncSuccRequester);
	return true;
}

String Iterator_MethodMapping::ToString(Signal sig) const
{
	return String("<iterator:method_mapping>");
}

//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
Iterator_Fork::Iterator_Fork(Environment &env, Signal sig,
									Function *pFunc, Context &context) :
	Iterator(false), _env(env), _pFunc(pFunc), _doneFlag(false)
{
	pFunc->PrepareIteratorsForMap(sig, _iterOwner, context.GetArgs());
	_pValListToWrite = &_valListA;
	_pValListToRead = &_valListB;
	_pValueRead = _pValListToRead->begin();
	_readBlock.blockedFlag = false;
	_writeBlock.blockedFlag = false;
}

Iterator_Fork::~Iterator_Fork()
{
	Function::Delete(_pFunc);
}

bool Iterator_Fork::Rewind()
{
	return false;
}

bool Iterator_Fork::Next(Signal sig, Value &value)
{
	if (_pValueRead == _pValListToRead->end()) {
		for (;;) {
			SwapList();
			if (_pValueRead != _pValListToRead->end()) {
				break;
			} else if (_doneFlag) {
				return false;
			} else {
				_readBlock.blockedFlag = true;
				_readBlock.event.Wait();
			}
		}
	}
	value = *_pValueRead++;
	return true;
}

String Iterator_Fork::ToString(Signal sig) const
{
	return String("<iterator:fork>");
}

void Iterator_Fork::SwapList()
{
	_semaphore.Wait();
	ValueList *pValList = _pValListToWrite;
	_pValListToWrite = _pValListToRead;
	_pValListToRead = pValList;
	_pValListToWrite->clear();
	_pValueRead = _pValListToRead->begin();
	if (_writeBlock.blockedFlag) {
		_writeBlock.blockedFlag = false;
		_writeBlock.event.Notify();
	}
	_semaphore.Release();
}

void Iterator_Fork::ForkProcess()
{
	IncRef();
	Start();
}

void Iterator_Fork::Run()
{
	Signal sig;
	ValueList valList;
	while (_iterOwner.Next(sig, valList)) {
		Context context(valList);
		Value value = _pFunc->Eval(_env, sig, context);
		if (sig.IsSignalled()) break;
		_semaphore.Wait();
		_pValListToWrite->push_back(value);
		_semaphore.Release();
		if (_readBlock.blockedFlag) {
			_readBlock.blockedFlag = false;
			_readBlock.event.Notify();
		}
	}
	_doneFlag = true;
}

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
Iterator_Delay::~Iterator_Delay()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Delay::Rewind()
{
	return false;
}

bool Iterator_Delay::Next(Signal sig, Value &value)
{
	OAL::Sleep(_delay);
	return _pIterator->Next(sig, value);
}

String Iterator_Delay::ToString(Signal sig) const
{
	return String("<iterator:delay>");
}

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
Iterator_Skip::~Iterator_Skip()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Skip::Rewind()
{
	return false;
}

bool Iterator_Skip::Next(Signal sig, Value &value)
{
	bool flag = _pIterator->Next(sig, value);
	for (int i = 0; flag && i < _nSkip; i++) {
		Value valueTmp;
		flag = _pIterator->Next(sig, valueTmp);
	}
	return flag;
}

String Iterator_Skip::ToString(Signal sig) const
{
	return String("<iterator:skip>");
}

//-----------------------------------------------------------------------------
// Iterator_SkipInvalid
//-----------------------------------------------------------------------------
Iterator_SkipInvalid::~Iterator_SkipInvalid()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_SkipInvalid::Rewind()
{
	return false;
}

bool Iterator_SkipInvalid::Next(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		if (value.IsValid()) return true;
	}
	return false;
}

String Iterator_SkipInvalid::ToString(Signal sig) const
{
	return String("<iterator:skip_invalid>");
}

//-----------------------------------------------------------------------------
// Iterator_Filter
//-----------------------------------------------------------------------------
Iterator_Filter::~Iterator_Filter()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_Filter::Rewind()
{
	return false;
}

bool Iterator_Filter::Next(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		ValueList valList(value);
		Context context(valList);
		Value valueCriteria = _pObjFunc->Eval(_env, sig, context);
		if (valueCriteria.GetBoolean()) return true;
	}
	return false;
}

String Iterator_Filter::ToString(Signal sig) const
{
	return String("<iterator:filter>");
}

//-----------------------------------------------------------------------------
// Iterator_Map
//-----------------------------------------------------------------------------
Iterator_Map::~Iterator_Map()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_Map::Rewind()
{
	return false;
}

bool Iterator_Map::Next(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	ValueList valList(value);
	Context context(valList);
	value = _pObjFunc->Eval(_env, sig, context);
	return true;
}

String Iterator_Map::ToString(Signal sig) const
{
	return String("<iterator:map>");
}

//-----------------------------------------------------------------------------
// Iterator_Format
//-----------------------------------------------------------------------------
Iterator_Format::~Iterator_Format()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Format::Rewind()
{
	return false;
}

bool Iterator_Format::Next(Signal sig, Value &value)
{
	Value valueSrc;
	if (!_pIterator->Next(sig, valueSrc)) return false;
	String str;
	if (valueSrc.IsList()) {
		str = Formatter::Format(sig, _format.c_str(), valueSrc.GetList());
	} else {
		ValueList valList(valueSrc);
		str = Formatter::Format(sig, _format.c_str(), valList);
	}
	value = Value(_env, str.c_str());
	return true;
}

String Iterator_Format::ToString(Signal sig) const
{
	return String("<iterator:format>");
}

//-----------------------------------------------------------------------------
// Iterator_Zip
//-----------------------------------------------------------------------------
Iterator_Zip::~Iterator_Zip()
{
}

bool Iterator_Zip::Rewind()
{
	return false;
}

bool Iterator_Zip::Next(Signal sig, Value &value)
{
	ValueList &valList = value.InitAsList(_env);
	return _iterOwner.Next(sig, valList);
}

String Iterator_Zip::ToString(Signal sig) const
{
	return String("<iterator:zip>");
}

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
Iterator_Align::~Iterator_Align()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Align::Rewind()
{
	return false;
}

bool Iterator_Align::Next(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	if (!_pIterator->Next(sig, value)) value = _valueFill;
	return true;
}

String Iterator_Align::ToString(Signal sig) const
{
	return String("<iterator:align>");
}

//-----------------------------------------------------------------------------
// Iterator_Head
//-----------------------------------------------------------------------------
Iterator_Head::~Iterator_Head()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Head::Rewind()
{
	return false;
}

bool Iterator_Head::Next(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	return _pIterator->Next(sig, value);
}

String Iterator_Head::ToString(Signal sig) const
{
	return String("<iterator:head>");
}

//-----------------------------------------------------------------------------
// Iterator_EachFold
//-----------------------------------------------------------------------------
Iterator_EachFold::~Iterator_EachFold()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_EachFold::Rewind()
{
	return false;
}

bool Iterator_EachFold::Next(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	value = _valueNext;
	_cnt--;
	if (_cnt > 0 && !_pIterator->Next(sig, _valueNext)) _cnt = 0;
	return true;
}

String Iterator_EachFold::ToString(Signal sig) const
{
	return String("<iterator:each_fold>");
}

//-----------------------------------------------------------------------------
// Iterator_Fold
//-----------------------------------------------------------------------------
Iterator_Fold::~Iterator_Fold()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Fold::Rewind()
{
	return false;
}

bool Iterator_Fold::Next(Signal sig, Value &value)
{
	Value valueNext;
	if (!_pIterator->Next(sig, valueNext)) return false;
	value.InitAsIterator(_env,
				new Iterator_EachFold(_pIterator->IncRef(), _cnt, valueNext));
	return true;
}

String Iterator_Fold::ToString(Signal sig) const
{
	return String("<iterator:fold>");
}

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
IteratorOwner::IteratorOwner(const IteratorOwner &iterOwner)
{
	reserve(iterOwner.size());
	foreach_const (IteratorOwner, ppIterator, iterOwner) {
		push_back((*ppIterator)->IncRef());
	}
}

IteratorOwner::~IteratorOwner()
{
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator::Delete(*ppIterator);
	}
}

bool IteratorOwner::Next(Signal sig, ValueList &valList)
{
	valList.clear();
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator *pIterator = *ppIterator;
		Value value;
		if (!pIterator->Next(sig, value)) return false;
		valList.push_back(value);
	}
	return true;
}

bool IteratorOwner::IsInfinite() const
{
	foreach_const (IteratorOwner, ppIterator, *this) {
		if (!(*ppIterator)->IsInfinite()) return false;
	}
	return true;
}

}
