#ifndef __ITERATOR_H__
#define __ITERATOR_H__

#include "Environment.h"
#include "Signal.h"
#include "Value.h"
#include "OAL.h"

namespace AScript {

class Expr_Caller;
class Context;

//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
class Iterator {
private:
	int _cntRef;
	bool _infiniteFlag;
public:
	inline Iterator(bool infiniteFlag) : _cntRef(1), _infiniteFlag(infiniteFlag) {}
	inline Iterator *IncRef() { _cntRef++; return this; }
	inline int DecRef() { _cntRef--; return _cntRef; }
	inline int GetRefCnt() const { return _cntRef; }
	inline static void Delete(Iterator *pIterator) {
		if (pIterator != NULL && pIterator->DecRef() <= 0) delete pIterator;
	}
	inline bool IsInfinite() const { return _infiniteFlag; }
	virtual bool Rewind() = 0;
	virtual bool Next(Signal sig, Value &value) = 0;
	virtual Value Eval(Environment &env, Signal sig, Context &context);
	virtual Value Eval(Environment &envBlock, Signal sig, Context &context,
									const Function *pFuncBlock);
	Value MinMax(Environment &env, Signal sig,
									bool maxFlag, const SymbolSet &attrs);
	virtual String ToString(Signal sig) const;
protected:
	virtual ~Iterator();
private:
	inline Iterator(const Iterator &iterator) {}
	inline void operator=(const Iterator &iterator) {}
};

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
class IteratorOwner : public std::vector<Iterator *> {
public:
	inline IteratorOwner() {}
	IteratorOwner(const IteratorOwner &iterOwner);
	virtual ~IteratorOwner();
	bool Next(Signal sig, ValueList &valList);
	bool IsInfinite() const;
};

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
class Iterator_Constant : public Iterator {
private:
	Value _value;
public:
	inline Iterator_Constant(const Value &value) : Iterator(true), _value(value) {}
	~Iterator_Constant();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Fill
//-----------------------------------------------------------------------------
class Iterator_Fill : public Iterator {
private:
	int _cnt;
	Value _value;
	int _idx;
public:
	inline Iterator_Fill(int cnt, const Value &value) :
						Iterator(false), _cnt(cnt), _value(value), _idx(0) {}
	~Iterator_Fill();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Rand
//-----------------------------------------------------------------------------
class Iterator_Rand : public Iterator {
private:
	int _cnt;
	int _range;
	int _idx;
public:
	inline Iterator_Rand(int cnt, int range) :
						Iterator(false), _cnt(cnt), _range(range), _idx(0) {}
	~Iterator_Rand();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Range
//-----------------------------------------------------------------------------
class Iterator_Range : public Iterator {
private:
	Number _num;
	Number _numBegin;
	Number _numEnd;
	Number _numStep;
public:
	inline Iterator_Range(Number numBegin, Number numEnd, Number numStep) :
			Iterator(false), _num(numBegin),
			_numBegin(numBegin), _numEnd(numEnd), _numStep(numStep) {}
	~Iterator_Range();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Sequence
//-----------------------------------------------------------------------------
class Iterator_Sequence : public Iterator {
private:
	Number _num;
	Number _numBegin;
	Number _numEnd;
	Number _numStep;
public:
	inline Iterator_Sequence(Number numBegin, Number numEnd, Number numStep) :
			Iterator(false), _num(numBegin),
			_numBegin(numBegin), _numEnd(numEnd), _numStep(numStep) {}
	~Iterator_Sequence();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_SequenceInf
//-----------------------------------------------------------------------------
class Iterator_SequenceInf : public Iterator {
private:
	Number _num;
	Number _numBegin;
public:
	inline Iterator_SequenceInf(Number numBegin) :
					Iterator(true), _num(numBegin), _numBegin(numBegin) {}
	~Iterator_SequenceInf();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Interval
//-----------------------------------------------------------------------------
class Iterator_Interval : public Iterator {
private:
	Number _numBegin;
	Number _numEnd;
	Number _numDenom;
	int _numSamples;
	int _iFactor;
	int _idx;
public:
	inline Iterator_Interval(Number numBegin, Number numEnd,
								Number numDenom, int numSamples, int iFactor) :
		Iterator(false), _numBegin(numBegin), _numEnd(numEnd), _numDenom(numDenom),
		_numSamples(numSamples), _iFactor(iFactor), _idx(0) {}
	~Iterator_Interval();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Mapping
//-----------------------------------------------------------------------------
class Iterator_Mapping : public Iterator {
private:
	Environment _env;
	Function *_pFunc;
	Value _valueSelf;
	IteratorOwner _iterOwner;
public:
	Iterator_Mapping(Environment &env, Signal sig,
									Function *pFunc, Context &context);
	~Iterator_Mapping();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_MemberMapping
//-----------------------------------------------------------------------------
class Iterator_MemberMapping : public Iterator {
private:
	Iterator *_pIterator;
	Expr *_pExpr;
public:
	inline Iterator_MemberMapping(Iterator *pIterator, Expr *pExpr) :
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _pExpr(pExpr) {}
	~Iterator_MemberMapping();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_MethodMapping
//-----------------------------------------------------------------------------
class Iterator_MethodMapping : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Expr_Caller *_pExprCaller;
public:
	inline Iterator_MethodMapping(Environment &env, Iterator *pIterator, Expr_Caller *pExprCaller) :
		Iterator(pIterator->IsInfinite()), _env(env), _pIterator(pIterator), _pExprCaller(pExprCaller) {}
	~Iterator_MethodMapping();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
class Iterator_Fork : public Iterator, public OAL::Thread {
private:
	Environment _env;
	Function *_pFunc;
	IteratorOwner _iterOwner;
	ValueList _valListA;
	ValueList _valListB;
	ValueList *_pValListToWrite;
	ValueList *_pValListToRead;
	ValueList::iterator _pValueRead;
	OAL::Semaphore _semaphore;
	bool _doneFlag;
	struct {
		bool blockedFlag;
		OAL::Event event;
	} _readBlock;
	struct {
		bool blockedFlag;
		OAL::Event event;
	} _writeBlock;
	public:
	Iterator_Fork(Environment &env, Signal sig,
									Function *pFunc, Context &context);
	~Iterator_Fork();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
	virtual void Run();
	void SwapList();
	void ForkProcess();
};

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
class Iterator_Delay : public Iterator {
private:
	Iterator *_pIterator;
	Number _delay;
public:
	inline Iterator_Delay(Iterator *pIterator, Number delay) : 
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _delay(delay) {}
	~Iterator_Delay();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
class Iterator_Skip : public Iterator {
private:
	Iterator *_pIterator;
	int _nSkip;
public:
	inline Iterator_Skip(Iterator *pIterator, int nSkip) :
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _nSkip(nSkip) {}
	~Iterator_Skip();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_SkipInvalid
//-----------------------------------------------------------------------------
class Iterator_SkipInvalid : public Iterator {
private:
	Iterator *_pIterator;
public:
	inline Iterator_SkipInvalid(Iterator *pIterator) :
				Iterator(pIterator->IsInfinite()), _pIterator(pIterator) {}
	~Iterator_SkipInvalid();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Filter
//-----------------------------------------------------------------------------
class Iterator_Filter : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_Filter(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	~Iterator_Filter();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Map
//-----------------------------------------------------------------------------
class Iterator_Map : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_Map(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	~Iterator_Map();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Format
//-----------------------------------------------------------------------------
class Iterator_Format : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	String _format;
public:
	inline Iterator_Format(Environment &env,
							Iterator *pIterator, const char *format) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _format(format) {}
	~Iterator_Format();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Zip
//-----------------------------------------------------------------------------
class Iterator_Zip : public Iterator {
private:
	Environment _env;
	IteratorOwner _iterOwner;
public:
	inline Iterator_Zip(Environment &env, const IteratorOwner &iterOwner) :
		Iterator(iterOwner.IsInfinite()), _env(env), _iterOwner(iterOwner) {}
	~Iterator_Zip();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
class Iterator_Align : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
	Value _valueFill;
public:
	inline Iterator_Align(Iterator *pIterator, int cnt, const Value &valueFill) :
		Iterator(false), _pIterator(pIterator), _cnt(cnt), _valueFill(valueFill) {}
	virtual ~Iterator_Align();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Head
//-----------------------------------------------------------------------------
class Iterator_Head : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
public:
	inline Iterator_Head(Iterator *pIterator, int cnt) :
						Iterator(false), _pIterator(pIterator), _cnt(cnt) {}
	virtual ~Iterator_Head();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_EachFold
//-----------------------------------------------------------------------------
class Iterator_EachFold : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
	Value _valueNext;
public:
	inline Iterator_EachFold(Iterator *pIterator, int cnt, const Value &valueNext) :
			Iterator(false), _pIterator(pIterator),
			_cnt(cnt), _valueNext(valueNext) {}
	virtual ~Iterator_EachFold();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Fold
//-----------------------------------------------------------------------------
class Iterator_Fold : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	int _cnt;
public:
	inline Iterator_Fold(Environment &env, Iterator *pIterator, int cnt) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _cnt(cnt) {}
	virtual ~Iterator_Fold();
	virtual bool Rewind();
	virtual bool Next(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

}

#endif
