//
// Object_Iterator
//

#include "Object_Iterator.h"
#include "Object_Function.h"
#include "Object_List.h"
#include "Expr.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Iterator
//-----------------------------------------------------------------------------
Object_Iterator::Object_Iterator(const Object_Iterator &obj) :
							Object(obj), _pIterator(obj._pIterator->IncRef())
{
}

Object_Iterator::~Object_Iterator()
{
	Iterator::Delete(_pIterator);
}

Object *Object_Iterator::Clone() const
{
	return new Object_Iterator(*this);
}

Iterator *Object_Iterator::CreateIterator(Signal sig)
{
	return _pIterator->IncRef();
}

String Object_Iterator::ToString(Signal sig, bool exprFlag)
{
	return _pIterator->ToString(sig);
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Iterator
//-----------------------------------------------------------------------------
// Iterator#is_infinite()
AScript_DeclareMethod(Iterator, is_infinite)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, is_infinite)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	return Value(pSelf->GetIterator()->IsInfinite());
}

// Iterator#next()
AScript_DeclareMethod(Iterator, next)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, next)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Value value;
	if (pSelf->GetIterator()->Next(sig, value)) return value;
	return Value::Null;
}

// Iterator#len()
AScript_DeclareMethod(Iterator, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, len)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Value value;
	int cnt;
	for (cnt = 0; pSelf->GetIterator()->Next(sig, value); cnt++) ;
	return Value(static_cast<Number>(cnt));
}

// Iterator#min():[index,last_index,indices]
AScript_DeclareMethod(Iterator, min)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementMethod(Iterator, min)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	return pSelf->GetIterator()->MinMax(env, sig, false, context.GetAttrs());
}

// Iterator#max():[index,last_index,indices]
AScript_DeclareMethod(Iterator, max)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementMethod(Iterator, max)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	return pSelf->GetIterator()->MinMax(env, sig, true, context.GetAttrs());
}

// Iterator#filter(func:Function) {block?}
AScript_DeclareMethod(Iterator, filter)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "func", VTYPE_Function);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, filter)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_Filter(env,
			pSelf->GetIterator()->IncRef(),
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef()));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#map(func:Function) {block?}
AScript_DeclareMethod(Iterator, map)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "func", VTYPE_Function);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, map)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_Map(env,
			pSelf->GetIterator()->IncRef(),
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef()));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#format(format:string) {block?}
AScript_DeclareMethod(Iterator, format)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, format)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_Format(env,
			pSelf->GetIterator()->IncRef(), context.GetString(0));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#eval()
AScript_DeclareMethod(Iterator, eval)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, eval)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator();
	return pIterator->Eval(env, sig, context);
}

// Iterator#each() {block}
AScript_DeclareMethod(Iterator, each)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementMethod(Iterator, each)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator();
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	return pIterator->Eval(env, sig, context, pFuncBlock);
}

// Iterator#print(file?:File)
AScript_DeclareMethod(Iterator, print)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, print)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	File *pFile = context.IsFile(0)?
				&context.GetFile(0) : env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL) return Value::Null;
	Iterator *pIterator = pSelf->GetIterator();
	Value value;
	while (pIterator->Next(sig, value)) {
		String str(value.ToString(sig, false));
		if (sig.IsSignalled()) return Value::Null;
		pFile->PutString(str.c_str());
	}
	return Value::Null;
}

// Iterator#println(file?:File)
AScript_DeclareMethod(Iterator, println)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, println)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	File *pFile = context.IsFile(0)?
				&context.GetFile(0) : env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL) return Value::Null;
	Iterator *pIterator = pSelf->GetIterator();
	Value value;
	while (pIterator->Next(sig, value)) {
		String str(value.ToString(sig, false));
		if (sig.IsSignalled()) return Value::Null;
		pFile->PutString(str.c_str());
		pFile->PutString("\n");
	}
	return Value::Null;
}

// Iterator#printf(format:string, file?:File)
AScript_DeclareMethod(Iterator, printf)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "file", VTYPE_File, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, printf)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	const char *format = context.GetString(0);
	File *pFile = context.IsFile(1)?
				&context.GetFile(1) : env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL) return Value::Null;
	Iterator *pIterator = pSelf->GetIterator();
	Value value;
	while (pIterator->Next(sig, value)) {
		if (value.IsList()) {
			pFile->Printf(sig, format, value.GetList());
		} else {
			pFile->Printf(sig, format, ValueList(value));
		}
		if (sig.IsSignalled()) return Value::Null;
	}
	return Value::Null;
}

// List#join(sep?:string)
AScript_DeclareMethod(Iterator, join)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "sep", VTYPE_String, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, join)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	const char *sep = context.IsString(0)? context.GetString(0) : "";
	Iterator *pIterator = pSelf->GetIterator();
	String rtn;
	Value value;
	if (pIterator->Next(sig, value)) {
		rtn += value.ToString(sig, false);
		if (sig.IsSignalled()) return Value::Null;
		while (pIterator->Next(sig, value)) {
			rtn += sep;
			rtn += value.ToString(sig, false);
			if (sig.IsSignalled()) return Value::Null;
		}
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, rtn.c_str());
}

// Iterator#delay(delay:number) {block?}
AScript_DeclareMethod(Iterator, delay)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "delay", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, delay)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator =
		new Iterator_Delay(pSelf->GetIterator()->IncRef(), context.GetNumber(0));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#skip(n:number) {block?}
AScript_DeclareMethod(Iterator, skip)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, skip)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator =
		new Iterator_Skip(pSelf->GetIterator()->IncRef(), static_cast<int>(context.GetNumber(0)));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#skip_nil() {block?}
AScript_DeclareMethod(Iterator, skip_nil)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, skip_nil)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_SkipInvalid(pSelf->GetIterator()->IncRef());
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#align(n:number, value?) {block?}
AScript_DeclareMethod(Iterator, align)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, align)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_Align(pSelf->GetIterator()->IncRef(),
			static_cast<int>(context.GetNumber(0)), context.GetValue(1));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#head(n:number) {block?}
AScript_DeclareMethod(Iterator, head)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, head)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_Head(pSelf->GetIterator()->IncRef(),
									static_cast<int>(context.GetNumber(0)));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#offset(n:number) {block?}
AScript_DeclareMethod(Iterator, offset)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, offset)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator()->IncRef();
	for (int cnt = static_cast<int>(context.GetNumber(0)); cnt > 0; cnt--) {
		Value value;
		pIterator->Next(sig, value);
	}
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#fold(n:number) {block?}
AScript_DeclareMethod(Iterator, fold)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, fold)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	int cnt = static_cast<int>(context.GetNumber(0));
	Iterator *pIterator = pSelf->GetIterator()->IncRef();
	return ReturnIterator(env, sig, context, new Iterator_Fold(env, pIterator, cnt));
}

// Iterator#reverse() {block?}
AScript_DeclareMethod(Iterator, reverse)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, reverse)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator();
	Value value = pIterator->Eval(env, sig, context);
	if (sig.IsSignalled()) return Value::Null;
	ASSUME(env, value.IsList());
	Object_List *pObj = dynamic_cast<Object_List *>(value.GetListObj()->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorReverse(pObj));
}

// Iterator#round(n:number) {block?}
AScript_DeclareMethod(Iterator, round)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, round)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	int cnt = static_cast<int>(context.GetNumber(0));
	Iterator *pIterator = pSelf->GetIterator();
	Value value = pIterator->Eval(env, sig, context);
	if (sig.IsSignalled()) return Value::Null;
	ASSUME(env, value.IsList());
	Object_List *pObj = dynamic_cast<Object_List *>(value.GetListObj()->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorRound(pObj, cnt));
}

// assignment
Class_Iterator::Class_Iterator(Class *pClassSuper, const Symbol *pSymbol) :
												Class(pClassSuper, pSymbol)
{
	AScript_AssignMethod(Iterator, is_infinite);
	AScript_AssignMethod(Iterator, next);
	AScript_AssignMethod(Iterator, len);
	AScript_AssignMethod(Iterator, min);
	AScript_AssignMethod(Iterator, max);
	AScript_AssignMethod(Iterator, filter);
	AScript_AssignMethod(Iterator, map);
	AScript_AssignMethod(Iterator, format);
	AScript_AssignMethod(Iterator, eval);
	AScript_AssignMethod(Iterator, each);
	AScript_AssignMethod(Iterator, print);
	AScript_AssignMethod(Iterator, println);
	AScript_AssignMethod(Iterator, printf);
	AScript_AssignMethod(Iterator, join);
	AScript_AssignMethod(Iterator, delay);
	AScript_AssignMethod(Iterator, skip);
	AScript_AssignMethod(Iterator, skip_nil);
	AScript_AssignMethod(Iterator, align);
	AScript_AssignMethod(Iterator, head);
	AScript_AssignMethod(Iterator, offset);
	AScript_AssignMethod(Iterator, fold);
	AScript_AssignMethod(Iterator, reverse);
	AScript_AssignMethod(Iterator, round);
}

Object *Class_Iterator::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
