#include "Signal.h"
#include "Symbol.h"
#include "Value.h"
#include "Expr.h"

namespace AScript {

//-----------------------------------------------------------------------------
// SignalType
//-----------------------------------------------------------------------------
const char *GetSignalTypeName(SignalType sigType)
{
	static const struct {
		SignalType sigType;
		const char *name;
	} tbl[] = {
		{ SIGTYPE_None,				"none",				},
		{ SIGTYPE_Error,			"error",			},
		{ SIGTYPE_ErrorSuspended,	"error_suspended",	},
		{ SIGTYPE_Break,			"break",			},
		{ SIGTYPE_Continue,			"continue",			},
		{ SIGTYPE_Return,			"return",			},
		{ SIGTYPE_DetectEncoding,	"detect_encoding",	},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].sigType == sigType) return tbl[i].name;
	}
	return "unknown";
}

//-----------------------------------------------------------------------------
// ErrorType
//-----------------------------------------------------------------------------
const char *GetErrorTypeName(ErrorType errType)
{
	static const struct {
		ErrorType errType;
		const char *name;
	} tbl[] = {
		{ ERR_None,					"None"					},
		{ ERR_SyntaxError,			"SyntaxError"			},
		{ ERR_ArithmeticError,		"ArithmeticError"		},
		{ ERR_TypeError,			"TypeError"				},
		{ ERR_ZeroDivisionError,	"ZeroDivisionError"		},
		{ ERR_ValueError,			"ValueError"			},
		{ ERR_SystemError,			"SystemError"			},
		{ ERR_IOError,				"IOError"				},
		{ ERR_IndexError,			"IndexError"			},
		{ ERR_KeyError,				"KeyError"				},
		{ ERR_ImportError,			"ImportError"			},
		{ ERR_AttributeError,		"AttributeError"		},
		{ ERR_StopIteration,		"StopIteration"			},
		{ ERR_RuntimeError,			"RuntimeError"			},
		{ ERR_NameError,			"NameError"				},
		{ ERR_NotImplementedError,	"NotImplementedError"	},
		{ ERR_IteratorError,		"IteratorError"			},
		{ ERR_CodecError,			"CodecError"			},
		{ ERR_CommandError,			"CommandError"			},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].errType == errType) return tbl[i].name;
	}
	return "unknown";
}

//-----------------------------------------------------------------------------
// Signal
//-----------------------------------------------------------------------------
Signal::Signal() : _pMsg(new Message()), _stackLevel(0)
{
}

Signal::Signal(const Signal &sig) :
						_pMsg(sig._pMsg), _stackLevel(sig._stackLevel + 1)
{
	if (_stackLevel > MAX_STACK_LEVEL) {
		SetError(ERR_SystemError, "stack level exceeds maximum (%d)", MAX_STACK_LEVEL);
	}
}

void Signal::SetValue(const Value &value) const
{
	*_pMsg->pValue = value;
}

void Signal::ClearSignal()
{
	_pMsg->sigType = SIGTYPE_None;
	_pMsg->errType = ERR_None;
	_pMsg->fileName.clear();
	_pMsg->pExprCauseList->clear();
}

void Signal::SetSignal(SignalType sigType, const Value &value)
{
	_pMsg->sigType = sigType, *_pMsg->pValue = value;
	_pMsg->fileName.clear();
}

void Signal::AddExprCause(const Expr *pExpr)
{
	_pMsg->pExprCauseList->push_back(const_cast<Expr *>(pExpr));
}

Signal::Message::Message() : sigType(SIGTYPE_None),
		errType(ERR_None), pValue(new Value()), pExprCauseList(new ExprList())
{
}

String Signal::GetErrString() const
{
	String str;
	const ExprList &exprCauseList = GetExprCauseList();
	str += GetErrorName();
	if (exprCauseList.empty()) {
		str += ": ";
	} else {
		//foreach_const (ExprList, ppExprCause, exprCauseList) {
		//	const Expr *pExprCause = *ppExprCause;
		//	char buff[32];
		//	::sprintf(buff, " at line.%d", pExprCause->GetLineNo());
		//	str += buff;
		//}
		const Expr *pExprCause = exprCauseList.front();
		char buff[32];
		str += " at";
		if (::strlen(GetFileName()) > 0) {
			str += " ";
			str += File::ExtractBaseName(GetFileName());
		}
		::sprintf(buff, " line.%d", pExprCause->GetLineNo());
		str += buff;
		str += ": ";
	}
	str += _pMsg->str.c_str();
	return str;
}

void Signal::SetError(ErrorType errType, const char *format, ...)
{
	va_list list;
	va_start(list, format);
	SetErrorV(errType, format, list);
	va_end(list);
}

void Signal::SetErrorV(ErrorType errType,
					const char *format, va_list list, const char *textPre)
{
	char *str = new char [2048];
	::strcpy(str, textPre);
	::vsprintf(str + ::strlen(textPre), format, list);
	_pMsg->sigType = SIGTYPE_Error;
	_pMsg->errType = errType;
	_pMsg->str = str;
	*_pMsg->pValue = Value::Null;
	delete [] str;
}

}
