#include "Module_builtins.h"
#include "Object_Custom.h"
#include "Object_List.h"
#include "Object_Dict.h"
#include "Object_Struct.h"
#include "Object_Bytes.h"
#include "Object_Iterator.h"
#include "Object_Function.h"
#include "Object_File.h"
#include "Object_Error.h"
#include "Object_Semaphore.h"

AScript_BeginModule(__builtins__)

static bool PrepareRepeaterIterators(Environment &env, Signal sig,
					const ValueList &valListArg, SymbolList &symbolList,
					IteratorOwner &iteratorOwner, ValueList &valListSave);

// declaration of symbols privately used in this module
AScript_DeclarePrivSymbol(open);
AScript_DeclarePrivSymbol(open_l);
AScript_DeclarePrivSymbol(open_r);

//-----------------------------------------------------------------------------
// AScript module functions: __builtins__
//-----------------------------------------------------------------------------
// num = int(value):map
AScript_DeclareFunctionEx(int_, "int")
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementFunction(int_)
{
	const Value &value = context.GetValue(0);
	Value result;
	if (value.IsNumber()) {
		result.SetNumber(value.GetLong());
	} else if (value.IsComplex()) {
		result.SetNumber(static_cast<long>(std::abs(value.GetComplex())));
	} else if (value.IsString()) {
		bool successFlag;
		Number num = value.ToNumber(true, successFlag);
		if (!successFlag) {
			sig.SetError(ERR_ValueError, "failed to convert to a number");
			return Value::Null;
		}
		result.SetNumber(static_cast<long>(num));
	} else if (value.IsValid()) {
		SetError_InvalidValType(sig, value);
	}
	return result;
}

// func = class(super_class?) {block?}
AScript_DeclareFunctionEx(class_, "class")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "super_class", VTYPE_Function, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(static_));
}

AScript_ImplementFunction(class_)
{
	const Expr_Block *pExprBlock = context.GetBlock();
	Class *pClassSuper = env.LookupClass(VTYPE_Object);
	const Value &valueSuper = context.GetValue(0);
	if (valueSuper.IsFunction()) {
		pClassSuper = valueSuper.GetFunction()->GetClassToConstruct();
		if (pClassSuper == NULL) {
			valueSuper.GetFunction()->SetError_NotConstructor(sig);
			return Value::Null;
		}
	}
	Class_Custom *pClassCustom =
					new Class_Custom(pClassSuper, AScript_Symbol(_anonymous_));
	if (pExprBlock != NULL && !pClassCustom->BuildContent(env, sig, pExprBlock)) {
		Class::Delete(pClassCustom);
		return Value::Null;
	}
	ClassPrototype *pFunc = NULL;
	FunctionType funcType = context.IsSet(AScript_Symbol(static_))?
											FUNCTYPE_Class : FUNCTYPE_Function;
	FunctionCustom *pFuncInit = dynamic_cast<FunctionCustom *>(
					pClassCustom->LookupFunction(AScript_Symbol(__init__), false));
	if (pFuncInit == NULL) {
		pFunc = new ClassPrototype(env, AScript_Symbol(_anonymous_),
									new Expr_Block(), pClassCustom, funcType);
		ContextExpr contextExpr(ExprList::Null);
		if (!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
			Function::Delete(pFunc);
			return Value::Null;
		}
	} else {
		pFunc = new ClassPrototype(env, AScript_Symbol(_anonymous_),
					pFuncInit->GetExprBody()->IncRef(), pClassCustom, funcType);
		pFunc->CopyDeclare(*pFuncInit);
	}
	pClassCustom->SetConstructor(pFunc->IncRef());
	Value result;
	result.InitAsFunction(env, pFunc);
	return result;
}

// func = struct(`args*) {block?}
// if :loose attribute is specified, arguments in the generated function
// will get the following modification.
// - Once attribute will be modified to ZeroOrOnce.
// - OnceOrMore attribute will be modified to ZeroOrMore
AScript_DeclareFunctionEx(struct_, "struct")
{
	DeclareArg(env, "args", VTYPE_Quote, false, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(loose));
}

AScript_ImplementFunction(struct_)
{
	const Expr_Block *pExprBlock = context.GetBlock();
	Class *pClassSuper = env.LookupClass(VTYPE_Struct);
	ExprList exprArgs;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		exprArgs.push_back(const_cast<Expr *>(pValue->GetExpr()));
	}
	Value result;
	Class_Custom *pClassCustom =
					new Class_Custom(pClassSuper, AScript_Symbol(_anonymous_));
	if (pExprBlock != NULL && !pClassCustom->BuildContent(env, sig, pExprBlock)) {
		Class::Delete(pClassCustom);
		return Value::Null;
	}
	StructPrototype *pFunc = new StructPrototype(pClassCustom);
	pClassCustom->SetConstructor(pFunc->IncRef());
	ContextExpr contextExpr(context.GetAttrs(), SymbolSet::Null,
										NULL, NULL, Value::Null, exprArgs);
	if (!pFunc->CustomDeclare(env, sig, _attrsOpt, contextExpr)) {
		Function::Delete(pFunc);
		return false;
	}
	if (context.IsSet(AScript_Symbol(loose))) {
		pFunc->GetDeclList().SetAsLoose();
	}
	result.InitAsFunction(env, pFunc);
	return result;
}

// val = module {block}
AScript_DeclareFunction(module)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(module)
{
	Value result;
	Module *pModule = new Module(&env, AScript_Symbol(_anonymous_));
	result.InitAsModule(pModule);
	context.GetBlock()->Exec(*pModule, sig);
	return result;
}

// mixin(module:Module)
AScript_DeclareFunction(mixin)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "module", VTYPE_Module);
}

AScript_ImplementFunction(mixin)
{
	
	return Value::Null;
}

// import(`module, `name?) {symbols?}
AScript_DeclareFunctionEx(import_, "import")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "module", VTYPE_Quote);
	DeclareArg(env, "name", VTYPE_Quote, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(import_)
{
	SymbolSet symbolsToMixIn;
	SymbolSet *pSymbolsToMixIn = NULL;
	if (context.IsBlockSpecified()) {
		foreach_const (ExprList, ppExpr, context.GetBlock()->GetExprList()) {
			if (!(*ppExpr)->IsSymbol()) {
				sig.SetError(ERR_SyntaxError,
					"wrong format for an element in import list");
				return Value::Null;
			}
			const Expr_Symbol *pExprSymbol =
							dynamic_cast<const Expr_Symbol *>(*ppExpr);
			symbolsToMixIn.insert(pExprSymbol->GetSymbol());
		}
		pSymbolsToMixIn = &symbolsToMixIn;
	}
	const Symbol *pSymbol = NULL;
	if (!context.IsExpr(1)) {
		// nothing to do
	} else if (!context.GetExpr(1)->IsSymbol()) {
		sig.SetError(ERR_ValueError, "symbol is expected as a module name");
		return Value::Null;
	} else {
		pSymbol =
			dynamic_cast<const Expr_Symbol *>(context.GetExpr(1))->GetSymbol();
	}
	if (!env.ImportModule(sig, context.GetExpr(0), pSymbol, pSymbolsToMixIn)) {
		return Value::Null;
	}
	return Value::Null;
}

// execfile(filename:string):map
AScript_DeclareFunction(execfile)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "filename", VTYPE_String);
}

AScript_ImplementFunction(execfile)
{
	File file;
	file.Open(sig, context.GetString(0), "r", NULL);
	if (!sig.IsSignalled()) {
		Parser().ExecFile(env, sig, file, true);
	}
	return Value::Null;
}

// expr = parse(code:string):map
AScript_DeclareFunction(parse)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "code", VTYPE_String);
}

AScript_ImplementFunction(parse)
{
	Value result;
	const char *str = context.GetString(0);
	Expr *pExpr = Parser().ParseString(env, sig, str, ::strlen(str));
	if (pExpr == NULL) return Value::Null;
	result.InitAsExpr(env, pExpr);
	return result;
}

// val = eval(expr:Expr):map
AScript_DeclareFunction(eval)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "expr", VTYPE_Expr);
}

AScript_ImplementFunction(eval)
{
	return context.GetExpr(0)->Exec(env, sig);
}

// val = locals()
AScript_DeclareFunction(locals)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(locals)
{
	Value value;
	value.InitAsEnvironment(env);
	return value;
}

// val = outers()
AScript_DeclareFunction(outers)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(outers)
{
	Value value;
	Environment envOuter(&env, ENVTYPE_Outer);
	value.InitAsEnvironment(envOuter);
	return value;
}

// try () {block}
class Func_try : public Function {
public:
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
	virtual bool IsSucceedable(const ICallable *pCallable) const;
	Func_try(Environment &env, const char *name = "try");
};
Func_try::Func_try(Environment &env, const char *name) : Function(name, FUNCTYPE_Function)
{
	DeclareBlock(OCCUR_Once);
}

Value Func_try::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result = context.GetBlock()->Exec(env, sig);
	sig.SuspendError();
	context.RequestSucceeding(this);
	return result;
}

bool Func_try::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// except (error*:Error) {block}
class Func_except : public Function {
private:
	SymbolSet _symbolsSucceed;
public:
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
	virtual bool IsSucceedable(const ICallable *pCallable) const;
	Func_except(Environment &env, const char *name = "except");
};
Func_except::Func_except(Environment &env, const char *name) : Function(name, FUNCTYPE_Function)
{
	DeclareArg(env, "error", VTYPE_Error, false, OCCUR_ZeroOrMore);
	DeclareBlock(OCCUR_Once);
	_symbolsSucceed.Insert(AScript_Symbol(except_));
}

Value Func_except::DoEval(Environment &env, Signal sig, Context &context) const
{
	bool handleFlag = false;
	if (!sig.IsErrorSuspended()) {
		// nothing to do
	} else if (context.GetList(0).empty()) {
		handleFlag = true;
	} else {
		foreach_const (ValueList, pValue, context.GetList(0)) {
			if (pValue->GetErrorType() == sig.GetErrorType()) {
				handleFlag = true;
				break;
			}
		}
	}
	if (!handleFlag) {
		context.RequestSucceeding(this);
		return Value::Null;
	}
	Value value;
	value.InitAsError(env, sig.GetErrorType(),
							sig.GetErrString().c_str(), sig.GetValue());
	ValueList valListArg(value);
	sig.ClearSignal(); // clear even the suspended state
	const Function *pFuncBlock = GetBlockFunction(env, sig, context);
	if (sig.IsSignalled()) return Value::Null;
	Environment envBlock(&env, ENVTYPE_Block);
	Context contextSub(valListArg);
	return pFuncBlock->Eval(envBlock, sig, contextSub);
}

bool Func_except::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// finally () {block}
class Func_finally : public Function {
public:
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
	Func_finally(Environment &env, const char *name = "finally");
};
Func_finally::Func_finally(Environment &env, const char *name) : Function(name, FUNCTYPE_Function)
{
	DeclareBlock(OCCUR_Once);
}

Value Func_finally::DoEval(Environment &env, Signal sig, Context &context) const
{
	return context.GetBlock()->Exec(env, sig);
}

// if (`cond) {block}
class Func_if : public Function {
public:
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
	virtual bool IsSucceedable(const ICallable *pCallable) const;
	Func_if(Environment &env, const char *name = "if");
};
Func_if::Func_if(Environment &env, const char *name) : Function(name, FUNCTYPE_Function)
{
	DeclareArg(env, "cond", VTYPE_Quote);
	DeclareBlock(OCCUR_Once);
}

Value Func_if::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value value = context.GetExpr(0)->Exec(env, sig);
	if (value.GetBoolean()) {
		return context.GetBlock()->Exec(env, sig);
	}
	context.RequestSucceeding(this);
	return Value::Null;
}

bool Func_if::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// elsif (`cond) {block}
class Func_elsif : public Function {
public:
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
	virtual bool IsSucceedable(const ICallable *pCallable) const;
	Func_elsif(Environment &env, const char *name = "elsif");
};
Func_elsif::Func_elsif(Environment &env, const char *name) : Function(name, FUNCTYPE_Function)
{
	DeclareArg(env, "cond", VTYPE_Quote);
	DeclareBlock(OCCUR_Once);
}

Value Func_elsif::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value value = context.GetExpr(0)->Exec(env, sig);
	if (value.GetBoolean()) {
		return context.GetBlock()->Exec(env, sig);
	}
	context.RequestSucceeding(this);
	return Value::Null;
}

bool Func_elsif::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// else () {block}
class Func_else : public Function {
public:
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
	Func_else(Environment &env, const char *name = "else");
};
Func_else::Func_else(Environment &env, const char *name) : Function(name, FUNCTYPE_Function)
{
	DeclareBlock(OCCUR_Once);
}

Value Func_else::DoEval(Environment &env, Signal sig, Context &context) const
{
	// this function works as a terminater of if-else and try-except
	if (sig.IsErrorSuspended()) return Value::Null;
	return context.GetBlock()->Exec(env, sig);
}

// repeat (n?:number) {block}
AScript_DeclareFunction(repeat)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(repeat)
{
	Value result;
	const Value &value = context.GetValue(0);
	const Function *pFuncBlock = GetBlockFunction(env, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		if (!value.IsNumber()) {
			sig.SetError(ERR_AttributeError,
				"infinite loop is not allowed with list, xlist, set or xset");
			return Value::Null;
		}
		ResultListComposer resultListComposer(env, context, result);
		int cnt = value.GetInt();
		for (int idx = 0; cnt > 0; cnt--, idx++) {
			ValueList valListArg(Value(static_cast<Number>(idx)));
			Context contextSub(valListArg);
			Value resultElem = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
			resultListComposer.Store(resultElem);
		}
	} else {
		int cnt, cntStep;
		if (value.IsNumber()) {
			cnt = value.GetInt();
			cntStep = -1;
		} else {
			cnt = 1;
			cntStep = 0;
		}
		for (int idx = 0; cnt > 0; cnt += cntStep, idx++) {
			ValueList valListArg(Value(static_cast<Number>(idx)));
			Context contextSub(valListArg);
			result = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		}
	}
	return result;
}

// while (`cond) {block}
AScript_DeclareFunctionEx(while_, "while")
{
	DeclareArg(env, "cond", VTYPE_Quote);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(while_)
{
	Value result;
	const Expr *pExpr = context.GetExpr(0);
	const Function *pFuncBlock = GetBlockFunction(env, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	std::auto_ptr<ResultListComposer> pResultListComposer;
	if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		pResultListComposer.reset(new ResultListComposer(env, context, result));
	}
	for (int idx = 0; ; idx++) {
		bool rtn = pExpr->Exec(env, sig).GetBoolean();
		if (sig.IsSignalled()) return Value::Null;
		if (!rtn) break;
		ValueList valListArg(Value(static_cast<Number>(idx)));
		Context contextSub(valListArg);
		if (pResultListComposer.get() == NULL) {
			result = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		} else {
			Value resultElem = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
			pResultListComposer->Store(resultElem);
		}
	}
	return result;
}

// for (`iter+) {block}
AScript_DeclareFunctionEx(for_, "for")
{
	DeclareArg(env, "iter", VTYPE_Quote, false, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(for_)
{
	Environment envBlock(&env, ENVTYPE_Block);
	SymbolList symbolList;
	IteratorOwner iteratorOwner;
	ValueList valListSave;	// save value to prevent them from being deleted
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	if (!PrepareRepeaterIterators(envBlock, sig, context.GetList(0),
				symbolList, iteratorOwner, valListSave)) return Value::Null;
	Value result;
	std::auto_ptr<ResultListComposer> pResultListComposer;
	if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		pResultListComposer.reset(new ResultListComposer(env, context, result));
	}
	for (int idx = 0; ; idx++) {
		SymbolList::iterator ppSymbol = symbolList.begin();
		bool doneFlag = false;
		foreach_const (IteratorOwner, ppIterator, iteratorOwner) {
			Iterator *pIterator = *ppIterator;
			Value value;
			if (!pIterator->Next(sig, value)) {
				if (sig.IsSignalled()) return Value::Null;
				doneFlag = true;
				break;
			}
			envBlock.AssignValue(*ppSymbol, value, false);
			ppSymbol++;
		}
		if (doneFlag) break;
		ValueList valListArg(Value(static_cast<Number>(idx)));
		Context contextSub(valListArg);
		if (pResultListComposer.get() == NULL) {
			result = pFuncBlock->Eval(envBlock, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		} else {
			Value resultElem = pFuncBlock->Eval(envBlock, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
			pResultListComposer->Store(resultElem);
		}
	}
	return result;
}

// cross (`iter+) {block}
AScript_DeclareFunction(cross)
{
	DeclareArg(env, "iter", VTYPE_Quote, false, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(cross)
{
	Environment envBlock(&env, ENVTYPE_Block);
	ValueList valListArg;
	ValueList valListSave;	// save value to prevent them from being deleted
	SymbolList symbolList;
	IteratorOwner iteratorOwner, iteratorOwnerOrg;
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	if (!PrepareRepeaterIterators(envBlock, sig, context.GetList(0),
				symbolList, iteratorOwnerOrg, valListSave)) return Value::Null;
	valListArg.reserve(iteratorOwnerOrg.size());
	iteratorOwner.reserve(iteratorOwnerOrg.size());
	bool continueFlag = false;
	do {
		SymbolList::iterator ppSymbol = symbolList.begin();
		foreach (IteratorOwner, ppIteratorOrg, iteratorOwnerOrg) {
			Iterator *pIterator = (*ppIteratorOrg)->Clone();
			Value value;
			bool rtn = pIterator->Next(sig, value);
			if (sig.IsSignalled()) return Value::Null;
			if (rtn) {
				continueFlag = true;
				valListArg.push_back(Value(0));
				iteratorOwner.push_back(pIterator);
			} else {
				iteratorOwner.push_back(NULL);
				valListArg.push_back(Value::Null);
				Iterator::Delete(pIterator);
			}
			envBlock.AssignValue(*ppSymbol, value, false);
			ppSymbol++;
		}
	} while (0);
	Value result;
	std::auto_ptr<ResultListComposer> pResultListComposer;
	if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		pResultListComposer.reset(new ResultListComposer(env, context, result));
	}
	while (continueFlag) {
		Context contextSub(valListArg);
		if (pResultListComposer.get() == NULL) {
			result = pFuncBlock->Eval(envBlock, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		} else {
			Value resultElem = pFuncBlock->Eval(envBlock, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
			pResultListComposer->Store(resultElem);
		}
		SymbolList::reverse_iterator ppSymbol = symbolList.rbegin();
		ValueList::reverse_iterator pValueArg = valListArg.rbegin();
		IteratorOwner::reverse_iterator ppIteratorOrg = iteratorOwnerOrg.rbegin();
		foreach_reverse (IteratorOwner, ppIterator, iteratorOwner) {
			Iterator *pIterator = *ppIterator;
			Value value;
			if (pIterator != NULL) {
				if (pIterator->Next(sig, value)) {
					*pValueArg = Value(pValueArg->GetNumber() + 1);
					envBlock.AssignValue(*ppSymbol, value, false);
					break;
				}
				if (sig.IsSignalled()) return Value::Null;
				Iterator::Delete(pIterator);
				pIterator = (*ppIteratorOrg)->Clone();
				pIterator->Next(sig, value);
				if (sig.IsSignalled()) return Value::Null;
				envBlock.AssignValue(*ppSymbol, value, false);
				*pValueArg = Value(0);
				*ppIterator = pIterator;
			}
			if (ppIterator + 1 == iteratorOwner.rend()) {
				continueFlag = false;
				break;
			}
			ppSymbol++;
			pValueArg++;
			ppIteratorOrg++;
		}
	}
	return result;
}

// break(value?):symbol_func
AScript_DeclareFunctionEx(break_, "break")
{
	SetAsSymbolFunc();
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(break_)
{
	sig.SetSignal(SIGTYPE_Break, context.GetValue(0));
	return Value::Null;
}

// continue(value?):symbol_func
AScript_DeclareFunctionEx(continue_, "continue")
{
	SetAsSymbolFunc();
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(continue_)
{
	sig.SetSignal(SIGTYPE_Continue, context.GetValue(0));
	return Value::Null;
}

// return(value?):symbol_func
AScript_DeclareFunctionEx(return_, "return")
{
	SetAsSymbolFunc();
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(return_)
{
	sig.SetSignal(SIGTYPE_Return, context.GetValue(0));
	return Value::Null;
}

// raise(error, msg?:string, value?)
AScript_DeclareFunction(raise)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "error", VTYPE_Error);
	DeclareArg(env, "msg", VTYPE_String, false, OCCUR_ZeroOrOnce);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(raise)
{
	sig.SetError(context.GetErrorType(0), "%s",
			context.IsString(1)? context.GetString(1) : "error");
	sig.SetValue(context.GetValue(2));
	return Value::Null;
}

// list = zip(value+)
AScript_DeclareFunction(zip)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(zip)
{
	IteratorOwner iterOwner;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Iterator *pIteratorArg = NULL;
		if (pValue->IsNumber() || pValue->IsString()) {
			pIteratorArg = new Iterator_Constant(*pValue);
		} else {
			pIteratorArg = pValue->CreateIterator(sig);
			if (sig.IsSignalled()) return Value::Null;
		}
		iterOwner.push_back(pIteratorArg);
	}
	Iterator *pIterator = new Iterator_Zip(env, iterOwner);
	return ReturnIterator(env, sig, context, pIterator);
}

// list = sort(elem[], directive?):[stable]
AScript_DeclareFunction(sort)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
	DeclareArg(env, "directive", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(stable));
}

AScript_ImplementFunction(sort)
{
	Object_List *pObj = context.GetListObj(0);
	return pObj->SortRank(sig, context.GetValue(1),
							false, context.IsSet(AScript_Symbol(stable)));
}

// list = rank(elem[], directive?):[stable]
AScript_DeclareFunction(rank)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
	DeclareArg(env, "directive", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(stable));
}

AScript_ImplementFunction(rank)
{
	Object_List *pObj = context.GetListObj(0);
	return pObj->SortRank(sig, context.GetValue(1),
							true, context.IsSet(AScript_Symbol(stable)));
}

// iter = reverse(elem[])
AScript_DeclareFunction(reverse)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(reverse)
{
	Object_List *pObj = dynamic_cast<Object_List *>(context.GetListObj(0)->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorReverse(pObj));
}

// iter = round(elem[], n?:number)
AScript_DeclareFunction(round)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
	DeclareArg(env, "n", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(round)
{
	int cnt = context.IsNumber(1)? context.GetInt(1) : -1;
	Object_List *pObj = dynamic_cast<Object_List *>(context.GetListObj(0)->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorRound(pObj, cnt));
}

// num = len(iter:iterator)
AScript_DeclareFunction(len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(len)
{
	int cnt = 0;
	Value value;
	for (Iterator *pIterator = context.GetIterator(0);
										pIterator->Next(sig, value); cnt++) ;
	return Value(cnt);
}

// list = list(iter+:iterator)
AScript_DeclareFunction(list)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator, false, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(list)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Value value;
		for (Iterator *pIterator = pValue->GetIterator(); pIterator->Next(sig, value); ) {
			valList.push_back(value);
		}
		if (sig.IsSignalled()) return Value::Null;
	}
	return result;
}

// list = xlist(iter+:iterator)
AScript_DeclareFunction(xlist)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator, false, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(xlist)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Value value;
		for (Iterator *pIterator = pValue->GetIterator(); pIterator->Next(sig, value); ) {
			if (value.IsValid()) valList.push_back(value);
		}
		if (sig.IsSignalled()) return Value::Null;
	}
	return result;
}

// list = set(iter+:iterator)
// list = xset(iter+:iterator)
// attrs: and, xor, or
AScript_DeclareFunctionBegin(set_xset)
	bool _acceptInvalidFlag;
AScript_DeclareFunctionEnd(set_xset)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	_acceptInvalidFlag = (::strcmp(GetName(), "set") == 0);
	DeclareArg(env, "iter", VTYPE_Iterator, false, OCCUR_OnceOrMore);
	DeclareAttr(AScript_Symbol(or));
	DeclareAttr(AScript_Symbol(and));
	DeclareAttr(AScript_Symbol(xor));
}

AScript_ImplementFunction(set_xset)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	if (context.IsSet(AScript_Symbol(and))) {			// AND combination
		ValueList valList1, valList2;
		ValueList::const_reverse_iterator pValueArg = context.GetList(0).rbegin();
		Value value;
		for (Iterator *pIterator = pValueArg->GetIterator();
											pIterator->Next(sig, value); ) {
			if ((_acceptInvalidFlag || value.IsValid()) &&
											!valList1.IsContain(value)) {
				valList1.push_back(value);
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		pValueArg++;
		ValueList *pValListAnd = &valList1;
		ValueList *pValListWork = &valList2;
		for ( ; pValueArg != context.GetList(0).rend() &&
										!pValListAnd->empty(); pValueArg++) {
			Value value;
			for (Iterator *pIterator = pValueArg->GetIterator();
											pIterator->Next(sig, value); ) {
				if (pValListAnd->IsContain(value) && !pValListWork->IsContain(value)) {
					pValListWork->push_back(value);
				}
			}
			ValueList *pValListTmp = pValListAnd;
			pValListAnd = pValListWork, pValListWork = pValListTmp;
			pValListWork->clear();
		}
		foreach_const (ValueList, pValue, *pValListAnd) {
			valList.push_back(*pValue);
		}
	} else if (context.IsSet(AScript_Symbol(xor))) {	// XOR combination
		ValueList valList1, valList2;
		ValueList *pValListAnd = &valList1;
		ValueList valListOr;
		ValueList::const_iterator pValueArg = context.GetList(0).begin();
		Value value;
		for (Iterator *pIterator = pValueArg->GetIterator();
										pIterator->Next(sig, value); ) {
			if (!valList1.IsContain(value)) valList1.push_back(value);
			if ((_acceptInvalidFlag || value.IsValid()) &&
											!valListOr.IsContain(value)) {
				valListOr.push_back(value);
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		pValueArg++;
		ValueList *pValListWork = &valList2;
		for ( ; pValueArg != context.GetList(0).end() &&
										!pValListAnd->empty(); pValueArg++) {
			Value value;
			for (Iterator *pIterator = pValueArg->GetIterator();
										pIterator->Next(sig, value); ) {
				if (pValListAnd->IsContain(value)) pValListWork->push_back(value);
				if ((_acceptInvalidFlag || value.IsValid()) &&
												!valListOr.IsContain(value)) {
					valListOr.push_back(value);
				}
			}
			if (sig.IsSignalled()) return Value::Null;
			ValueList *pValListTmp = pValListAnd;
			pValListAnd = pValListWork, pValListWork = pValListTmp;
			pValListWork->clear();
		}
		foreach_const (ValueList, pValue, valListOr) {
			if (!pValListAnd->IsContain(*pValue)) {
				valList.push_back(*pValue);
			}
		}
	} else {										// OR combination
		foreach_const (ValueList, pValue, context.GetList(0)) {
			Value value;
			for (Iterator *pIterator = pValue->GetIterator();
												pIterator->Next(sig, value); ) {
				if ((_acceptInvalidFlag || value.IsValid()) &&
												!valList.IsContain(value)) {
					valList.push_back(value);
				}
			}
			if (sig.IsSignalled()) return Value::Null;
		}
	}
	return result;
}

// replace(elem[], value, replace)
AScript_DeclareFunction(replace)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
	DeclareArg(env, "value", VTYPE_AnyType);
	DeclareArg(env, "replace", VTYPE_AnyType);
}

AScript_ImplementFunction(replace)
{
	const Value &value = context.GetValue(1);
	const Value &replace = context.GetValue(2);
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (Value::Compare(*pValue, value) == 0) {
			valList.push_back(replace);
		} else {
			valList.push_back(*pValue);
		}
	}
	return result;
}

// nilto(elem[], replace)
AScript_DeclareFunction(nilto)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, true);
	DeclareArg(env, "replace", VTYPE_AnyType);
}

AScript_ImplementFunction(nilto)
{
	const Value &replace = context.GetValue(1);
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (pValue->IsValid()) {
			valList.push_back(*pValue);
		} else {
			valList.push_back(replace);
		}
	}
	return result;
}

// filter(criteria, iter:iterator) {block?}
AScript_DeclareFunction(filter)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "criteria", VTYPE_AnyType);
	DeclareArg(env, "iter", VTYPE_Iterator);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(filter)
{
	Iterator *pIterator;
	Iterator *pIteratorSrc = context.GetIterator(1)->Clone();
	if (context.IsFunction(0)) {
		Object_Function *pFuncObjCriteria = 
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef());
		pIterator = new Iterator_Filter(env, pIteratorSrc, pFuncObjCriteria);
	} else if (context.IsList(0) || context.IsIterator(0)) {
		Iterator *pIteratorCriteria = context.GetValue(0).CreateIterator(sig);
		pIterator = new Iterator_FilterEach(pIteratorSrc, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for filter");
		return Value::Null;
	}
	return ReturnIterator(env, sig, context, pIterator);
}

// map(func:Function, iter:iterator) {block?}
AScript_DeclareFunction(map)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "func", VTYPE_Function);
	DeclareArg(env, "iter", VTYPE_Iterator);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(map)
{
	Iterator *pIterator = new Iterator_Map(env,
			context.GetIterator(1)->Clone(),
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef()));
	return ReturnIterator(env, sig, context, pIterator);
}

// v = min(iter:iterator):[index,last_index,indices]
AScript_DeclareFunction(min)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementFunction(min)
{
	Iterator *pIterator = context.GetIterator(0);
	return pIterator->MinMax(env, sig, false, context.GetAttrs());
}

// v = max(iter:iterator):[index,last_index,indices]
AScript_DeclareFunction(max)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementFunction(max)
{
	Iterator *pIterator = context.GetIterator(0);
	return pIterator->MinMax(env, sig, true, context.GetAttrs());
}

// v = choosemin(values+):map / v = choosemax(values+):map
AScript_DeclareFunctionBegin(choosemin_max)
	bool _maxFlag;
AScript_DeclareFunctionEnd(choosemin_max)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	_maxFlag = (::strcmp(GetName(), "choosemax") == 0);
	DeclareArg(env, "values", VTYPE_AnyType, false, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(choosemin_max)
{
	const ValueList &valList = context.GetList(0);
	ValueList::const_iterator pValue = valList.begin();
	Value result = *pValue++;
	for ( ; pValue != valList.end(); pValue++) {
		int cmp = Value::Compare(result, *pValue);
		if (_maxFlag) cmp = -cmp;
		if (cmp > 0) result = *pValue;
	}
	return result;
}

// v = choose(index:number, values+):map
AScript_DeclareFunction(choose)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "index", VTYPE_Number);
	DeclareArg(env, "values", VTYPE_AnyType, false, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(choose)
{
	size_t index = context.GetSizeT(0);
	const ValueList &valList = context.GetList(1);
	if (index >= valList.size()) {
		sig.SetError(ERR_IndexError, "index is out of range");
		return Value::Null;
	}
	return valList[index];
}

// v = chooseif(flag, value1, vlaue2):map
AScript_DeclareFunction(chooseif)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "flag", VTYPE_Boolean);
	DeclareArg(env, "value1", VTYPE_AnyType);
	DeclareArg(env, "value2", VTYPE_AnyType);
}

AScript_ImplementFunction(chooseif)
{
	return context.GetBoolean(0)? context.GetValue(1) : context.GetValue(2);
}

// num = sum(iter:iterator)
AScript_DeclareFunction(sum)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(sum)
{
	Iterator *pIterator = context.GetIterator(0);
	Value value;
	if (!pIterator->Next(sig, value)) return Value::Null;
	Value result = value;
	while (pIterator->Next(sig, value)) {
		ValueList valListArg(result, value);
		Context contextSub(valListArg);
		result = env.GetFunc_Plus().Eval(env, sig, contextSub);
		if (sig.IsSignalled()) return Value::Null;
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// num = average(iter:iterator)
AScript_DeclareFunction(average)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(average)
{
	int cnt = 0;
	Iterator *pIterator = context.GetIterator(0);
	Value value;
	if (!pIterator->Next(sig, value)) return Value::Null;
	Value result = value;
	for (cnt = 1; pIterator->Next(sig, value); cnt++) {
		ValueList valListArg(result, value);
		Context contextSub(valListArg);
		result = env.GetFunc_Plus().Eval(env, sig, contextSub);
		if (sig.IsSignalled()) return Value::Null;
	}
	if (sig.IsSignalled()) return Value::Null;
	ValueList valListArg(result, Value(static_cast<Number>(cnt)));
	Context contextSub(valListArg);
	return env.GetFunc_Divide().Eval(env, sig, contextSub);
}

// flag = and(iter:iterator)
AScript_DeclareFunctionEx(and_, "and")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(and_)
{
	Iterator *pIterator = context.GetIterator(0);
	Value value;
	while (pIterator->Next(sig, value)) {
		if (!value.GetBoolean()) return Value(false);
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(true);
}

// flag = or(iter:iterator)
AScript_DeclareFunctionEx(or_, "or")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(or_)
{
	Iterator *pIterator = context.GetIterator(0);
	Value value;
	while (pIterator->Next(sig, value)) {
		if (value.GetBoolean()) return Value(true);
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(false);
}

// text = tostring(value):map
AScript_DeclareFunction(tostring)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementFunction(tostring)
{
	return Value(env, context.GetValue(0).ToString(sig, false).c_str());
}

// num = tonumber(value):map:[strict,raise,zero,nil]
AScript_DeclareFunction(tonumber)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
	DeclareAttr(AScript_Symbol(strict));
	DeclareAttr(AScript_Symbol(raise));
	DeclareAttr(AScript_Symbol(zero));
	DeclareAttr(AScript_Symbol(nil));
}

AScript_ImplementFunction(tonumber)
{
	bool allowPartFlag = !context.IsSet(AScript_Symbol(strict));
	bool successFlag;
	Number num = context.GetValue(0).ToNumber(allowPartFlag, successFlag);
	if (successFlag) {
		return Value(num);
	} else if (context.IsSet(AScript_Symbol(raise))) {
		sig.SetError(ERR_ValueError, "failed to convert to a number");
		return Value::Null;
	} else if (context.IsSet(AScript_Symbol(zero))) {
		return Value(0.);
	} else { // context.IsSet(AScript_PrivSymbol(nil)
		return Value::Null;
	}
}

// symbol = tosymbol(str:string):map
AScript_DeclareFunction(tosymbol)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(tosymbol)
{
	return Value(Symbol::Add(context.GetString(0)));
}

// num = rand(num?)
AScript_DeclareFunction(rand)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "num", VTYPE_Number, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(rand)
{
	if (context.IsInvalid(0)) {
		return Value(::genrand_real2());
	}
	unsigned long num = context.GetULong(0);
	Number result = static_cast<unsigned long>(::genrand_real2() * num);
	return Value(result);
}

// ord(str:string):map
AScript_DeclareFunction(ord)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(ord)
{
	const char *str = context.GetString(0);
	unsigned long num = static_cast<unsigned char>(*str);
	if (IsUTF8First(*str)) {
		str++;
		for ( ; IsUTF8Follower(*str); str++) {
			num = (num << 8) | static_cast<unsigned char>(*str);
		}
	}
	return Value(num);
}

// chr(num:number):map
AScript_DeclareFunction(chr)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "num", VTYPE_Number);
}

AScript_ImplementFunction(chr)
{
	unsigned long num = context.GetULong(0);
	int i = 0;
	char buff[4];
	buff[i++] = static_cast<unsigned char>(num & 0xff);
	num >>= 8;
	for ( ; num != 0; num >>= 8) {
		buff[i++] = static_cast<unsigned char>(num & 0xff);
	}
	String str;
	for ( ; i > 0; i--) {
		str.push_back(buff[i - 1]);
	}
	return Value(env, str.c_str());
}

// hex(num:number, digits?:number):map:[upper]
AScript_DeclareFunction(hex)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "num", VTYPE_Number);
	DeclareArg(env, "digits", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(upper));
}

AScript_ImplementFunction(hex)
{
	int digits = context.IsNumber(1)? context.GetInt(1) : 0;
	bool upperFlag = context.IsSet(AScript_Symbol(upper));
	String str;
	if (digits <= 0) {
		str = Formatter::Format(sig, upperFlag? "%X" : "%x",
						ValueList(context.GetValue(0)));
	} else {
		str = Formatter::Format(sig, upperFlag? "%0*X" : "%0*x",
						ValueList(Value(digits), context.GetValue(0)));
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, str.c_str());
}

// print(value*):map:void
AScript_DeclareFunction(print)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(print)
{
	File *pFile = env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL || !pFile->IsWritable()) return Value::Null;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pFile->PutString(pValue->ToString(sig, false).c_str());
		if (sig.IsSignalled()) break;
	}
	pFile->Flush();
	return Value::Null;
}

// println(value*):map:void
AScript_DeclareFunction(println)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(println)
{
	File *pFile = env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL || !pFile->IsWritable()) return Value::Null;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pFile->PutString(pValue->ToString(sig, false).c_str());
		if (sig.IsSignalled()) break;
	}
	pFile->PutString("\n");
	pFile->Flush();
	return Value::Null;
}

// printf(format, values*):map:void
AScript_DeclareFunction(printf)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "values", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(printf)
{
	File *pFile = env.GetConsole(AScript_Symbol(stdout));
	if (pFile == NULL || !pFile->IsWritable()) return Value::Null;
	pFile->Printf(sig, context.GetString(0), context.GetList(1));
	pFile->Flush();
	return Value::Null;
}

// dir(obj?)
AScript_DeclareFunction(dir)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "obj", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(dir)
{
	const Environment::Frame *pFrame = &env.GetBottomFrame();
	const Value &value = context.GetValue(0);
	if (value.IsModule()) {
		pFrame = &value.GetModule()->GetTopFrame();
	} else if (value.IsFunction() &&
						value.GetFunction()->GetClassToConstruct() != NULL) {
		const Class *pClass = value.GetFunction()->GetClassToConstruct();
		pFrame = &pClass->GetTopFrame();
	} else if (value.IsObject()) {
		const Object *pObj = value.GetObject();
		Value result;
		ValueList &valList = result.InitAsList(env);
		foreach_const (Environment::FrameList, ppFrame, pObj->GetFrameList()) {
			const Environment::Frame *pFrame = *ppFrame;
			if (!pFrame->IsType(ENVTYPE_Class) && !pFrame->IsType(ENVTYPE_Instance)) {
				break;
			}
			foreach_const (ValueMap, iter, pFrame->GetValueMap()) {
				valList.push_back(Value(env, iter->first->GetName()));
			}
		}
		return result;
	} else if (value.IsValid()) {
		sig.SetError(ERR_TypeError, "invalid argument");
		return Value::Null;
	}
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueMap, iter, pFrame->GetValueMap()) {
		valList.push_back(Value(env, iter->first->GetName()));
	}
	return result;
}

// isdefined(`symbol)
AScript_DeclareFunction(isdefined)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "symbol", VTYPE_Quote);
}

AScript_ImplementFunction(isdefined)
{
	bool definedFlag = false;
	const Expr *pExpr = context.GetExpr(0);
	if (pExpr->IsSymbol() || pExpr->IsField()) {
		pExpr->Exec(env, sig);
		if (sig.IsSignalled() && !sig.IsError()) return Value::Null;
		definedFlag = !sig.IsError();
		sig.ClearSignal();
	} else {
		sig.SetError(ERR_ValueError, "argument must be a symbol");
		return Value::Null;
	}
	return Value(definedFlag);
}

// istype*(value)
class AScript_Function(istype_) : public Function {
private:
	ValueType _valType;
public:
	AScript_Function(istype_)(Environment &env, const char *name, ValueType valType);
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
};
AScript_Function(istype_)::AScript_Function(istype_)(
					Environment &env, const char *name, ValueType valType) :
	Function(name, FUNCTYPE_Function), _valType(valType)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementFunction(istype_)
{
	return Value(context.GetValue(0).GetType() == _valType);
}

// isinstance(value, type:expr):map
AScript_DeclareFunction(isinstance)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
	DeclareArg(env, "type", VTYPE_Expr);
}

AScript_ImplementFunction(isinstance)
{
	SymbolList symbolList;
	if (!context.GetExpr(1)->GetChainedSymbolList(symbolList)) {
		sig.SetError(ERR_TypeError, "invalid type name");
		return Value::Null;
	}
	const ValueTypeInfo *pValueTypeInfo = env.LookupValueType(symbolList);
	if (pValueTypeInfo == NULL) {
		sig.SetError(ERR_ValueError, "invalid type name");
		return Value::Null;
	}
	return context.GetValue(0).IsInstanceOf(pValueTypeInfo->GetValueType());
}

// typename(`value)
AScript_DeclareFunctionEx(typename_, "typename")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Quote);
}

AScript_ImplementFunction(typename_)
{
	const Expr *pExpr = context.GetExpr(0);
	const char *typeName = "unknown";
	if (pExpr->IsSymbol() || pExpr->IsField()) {
		Value value = pExpr->Exec(env, sig);
		if (sig.IsSignalled() && !sig.IsError()) return Value::Null;
		typeName = sig.IsError()? "undefined" : value.GetTypeName();
		sig.ClearSignal();
	} else {
		Value value = pExpr->Exec(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		typeName = value.GetTypeName();
	}
	return Value(env, typeName);
}

// undef(`value+)
AScript_DeclareFunctionEx(undef_, "undef")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Quote, false, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(undef_)
{
	foreach_const (ValueList, pValueArg, context.GetList(0)) {
		const Expr *pExpr = pValueArg->GetExpr();
		Environment *pEnv = &env;
		const Symbol *pSymbol = NULL;
		if (pExpr->IsSymbol()) {
			pSymbol = dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
		} else {
			SymbolList symbolList;
			if (!pExpr->GetChainedSymbolList(symbolList)) {
				sig.SetError(ERR_ValueError, "invalid symbol name");
				return Value::Null;
			}
			for (SymbolList::iterator ppSymbol = symbolList.begin();
								ppSymbol + 1 != symbolList.end(); ppSymbol++) {
				Value *pValue = pEnv->LookupValue(*ppSymbol, false);
				if (pValue == NULL) {
					sig.SetError(ERR_ValueError, "undefined symbol");
					return Value::Null;
				}
				if (pValue->IsModule()) {
					pEnv = pValue->GetModule();
				} else if (pValue->IsClass()) {
					pEnv = pValue->GetClass();
				} else if (pValue->IsObject()) {
					pEnv = pValue->GetObject();
				} else {
					sig.SetError(ERR_ValueError, "undefined symbol");
					return Value::Null;
				}
			}
			pSymbol = symbolList.back();
		}
		if (!pEnv->LookupValue(pSymbol, false)) {
			sig.SetError(ERR_ValueError, "undefined symbol");
			return Value::Null;
		}
		pEnv->RemoveValue(pSymbol);
	}
	return Value::Null;
}

// iter = range(num:number, num_end?:number, step?:number):map {block?}
AScript_DeclareFunction(range)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "num",		VTYPE_Number);
	DeclareArg(env, "num_end",	VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareArg(env, "step",		VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(range)
{
	Number numBegin = 0.;
	Number numEnd = 0.;
	Number numStep = 1.;
	if (context.GetValue(1).IsInvalid()) {
		numEnd = context.GetValue(0).GetNumber();
		if (numBegin > numEnd) numStep = -1.;
	} else if (context.GetValue(2).IsInvalid()) {
		numBegin = context.GetValue(0).GetNumber();
		numEnd = context.GetValue(1).GetNumber();
		if (numBegin > numEnd) numStep = -1.;
	} else {
		numBegin = context.GetValue(0).GetNumber();
		numEnd = context.GetValue(1).GetNumber();
		numStep = context.GetValue(2).GetNumber();
		if (numStep == 0.) {
			sig.SetError(ERR_ValueError, "step cannot be specified as zero");
			return Value::Null;
		}
	}
	return ReturnIterator(env, sig, context,
						new Iterator_Range(numBegin, numEnd, numStep));
}

// iter = interval(a:number, b:number, samples:number):map {block?}
AScript_DeclareFunction(interval)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "a", VTYPE_Number);
	DeclareArg(env, "b", VTYPE_Number);
	DeclareArg(env, "samples", VTYPE_Number);
	DeclareAttr(AScript_PrivSymbol(open));
	DeclareAttr(AScript_PrivSymbol(open_l));
	DeclareAttr(AScript_PrivSymbol(open_r));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(interval)
{
	const ValueList &valListArg = context.GetArgs();
	Number numBegin = valListArg[0].GetNumber();
	Number numEnd = valListArg[1].GetNumber();
	int numSamples = valListArg[2].GetInt();
	if (numSamples <= 1) {
		sig.SetError(ERR_ValueError, "samples must be more than one");
		return Value::Null;
	}
	bool openFlag = context.IsSet(AScript_PrivSymbol(open));
	bool openLeftFlag = context.IsSet(AScript_PrivSymbol(open_l));
	bool openRightFlag = context.IsSet(AScript_PrivSymbol(open_r));
	int iFactor = 0;
	Number numDenom = numSamples - 1;
	if (openFlag || (openLeftFlag && openRightFlag)) {
		numDenom = numSamples + 1;
		iFactor = 1;
	} else if (openLeftFlag) {
		numDenom = numSamples;
		iFactor = 1;
	} else if (openRightFlag) {
		numDenom = numSamples;
		iFactor = 0;
	}
	return ReturnIterator(env, sig, context,
		new Iterator_Interval(numBegin, numEnd, numDenom, numSamples, iFactor));
}

// iter = fill(n:number, value?) {block?}
AScript_DeclareFunction(fill)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(fill)
{
	Iterator *pIterator = new Iterator_Fill(context.GetInt(0), context.GetValue(1));
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = rands(n:number, range?:number) {block?}
AScript_DeclareFunction(rands)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "range", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(rands)
{
	Iterator *pIterator = new Iterator_Rand(context.GetInt(0),
							context.IsNumber(1)? context.GetInt(1) : 0);
	return ReturnIterator(env, sig, context, pIterator);
}

// bytes(buff*)
AScript_DeclareFunction(bytes)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "buff", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(bytes)
{
	Value result;
	Object_Bytes *pObjBytes = result.InitAsBytes(env);
	Bytes &bytes = pObjBytes->GetBytes();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (pValue->IsString()) {
			bytes += pValue->GetString();
		} else if (pValue->IsBytes()) {
			bytes += pValue->GetBytes();
		} else {
			sig.SetError(ERR_ValueError, "string or bytes is expected");
			return Value::Null;
		}
	}
	return result;
}

// pack(format:string, value*):map
AScript_DeclareFunction(pack)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(pack)
{
	Value result;
	Object_Bytes *pObjBytes = result.InitAsBytes(env);
	size_t offset = 0;
	pObjBytes->Pack(sig, offset, context.GetString(0), context.GetList(1));
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// object()
AScript_DeclareFunction(object)
{
}

AScript_ImplementFunction(object)
{
	Value result;
	result.InitAsObject(env);
	return result;
}

// @(func?:Function) {block?}
AScript_DeclareFunction(ListInit)
{
	DeclareArg(env, "func", VTYPE_Function, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(ListInit)
{
	const Expr_Block *pExprBlock = context.GetBlock();
	const Value &valueFunc = context.GetValue(0);
	Value result;
	if (pExprBlock == NULL) {
		result.InitAsList(env);
	} else if (valueFunc.IsFunction()) {
		const Function *pFunc = valueFunc.GetFunction();
		size_t cntArgs = pFunc->GetDeclList().size();
		if (cntArgs == 0) {
			sig.SetError(ERR_TypeError, "function '%s' needs no argument", pFunc->GetName());
			return Value::Null;
		}
		Environment envLister(&env, ENVTYPE_Lister);
		Value valueRaw =
				pExprBlock->GetExprList().ExecForList(envLister, sig, false, false);
		if (sig.IsSignalled() || !valueRaw.IsList()) return Value::Null;
		ValueList &valList = result.InitAsList(env);
		foreach_const (ValueList, pValue, valueRaw.GetList()) {
			if (!pValue->IsList()) {
				sig.SetError(ERR_SyntaxError, "invalid format in list initializer");
				return Value::Null;
			}
			Context contextSub(pValue->GetList());
			Value valueElem = pFunc->Eval(env, sig, contextSub);
			valList.push_back(valueElem);
		}
	} else {
		Environment envLister(&env, ENVTYPE_Lister);
		result = pExprBlock->GetExprList().ExecForList(envLister, sig, false, false);
	}
	return result;
}

// func = lambda(`args*) {block}
class ExprVisitor_GatherArgs : public ExprVisitor {
private:
	SymbolSet _symbolSet;
	ExprList &_exprList;
public:
	inline ExprVisitor_GatherArgs(ExprList &exprList) : _exprList(exprList) {}
	virtual bool Visit(const Expr *pExpr);
};

AScript_DeclareFunction(lambda)
{
	DeclareArg(env, "args", VTYPE_Quote, false, OCCUR_ZeroOrMore);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(lambda)
{
	const Expr *pExprBlock = context.GetBlock();
	ExprList exprArgs;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		exprArgs.push_back(const_cast<Expr *>(pValue->GetExpr()));
	}
	if (exprArgs.empty()) {
		ExprVisitor_GatherArgs visitor(exprArgs);
		pExprBlock->Accept(visitor);
	}
	FunctionCustom *pFunc = new FunctionCustom(env, AScript_Symbol(_anonymous_),
									pExprBlock->IncRef(), FUNCTYPE_Function);
	ContextExpr contextExpr(context.GetAttrs(), SymbolSet::Null,
									NULL, NULL, Value::Null, exprArgs);
	if (!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
		Function::Delete(pFunc);
		return Value::Null;
	}
	Value result;
	result.InitAsFunction(env, pFunc);
	return result;
}

bool ExprVisitor_GatherArgs::Visit(const Expr *pExpr)
{
	if (pExpr->IsCaller()) {
		// avoid searching in a simple lambda inside
		const Expr *pExprCar =
					dynamic_cast<const Expr_Caller *>(pExpr)->GetCar();
		if (pExprCar->IsSymbol()) {
			const Symbol *pSymbol =
					dynamic_cast<const Expr_Symbol *>(pExprCar)->GetSymbol();
			if (::strcmp(pSymbol->GetName(), "@") == 0) return false;
		}
		
	} else if (pExpr->IsSymbol()) {
		const Symbol *pSymbol =
						dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
		if (pSymbol->GetName()[0] == '$' &&
								_symbolSet.find(pSymbol) == _symbolSet.end()) {
			_exprList.push_back(const_cast<Expr *>(pExpr));
			_symbolSet.insert(pSymbol);
		}
	}
	return true;
}

// matrix() {block?}
AScript_DeclareFunction(matrix)
{
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(matrix)
{
	Environment envLister(&env, ENVTYPE_Lister);
	const Expr_Block *pExprBlock = context.GetBlock();
	Value result = pExprBlock->GetExprList().ExecForList(envLister, sig, false, false);
	if (!result.GetList().CheckMatrix(NULL, NULL)) {
		sig.SetError(ERR_ValueError, "invalid matrix initialization");
		return Value::Null;
	}
	return result;
}

// dict(elem[]?) {block?}
AScript_DeclareFunction(dict)
{
	DeclareArg(env, "elem", VTYPE_AnyType, true, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(dict)
{
	Object_Dict *pObj = new Object_Dict(env.LookupClass(VTYPE_Dict));
	Value result(pObj, VTYPE_Dict);
	ValueDict &valDict = pObj->GetDict();
	if (context.GetValue(0).IsList()) {
		if (!valDict.Store(sig, context.GetList(0))) return Value::Null;
	}
	if (context.IsBlockSpecified()) {
		Environment envLister(&env, ENVTYPE_Lister);
		Value valueList =
			context.GetBlock()->GetExprList().ExecForList(envLister, sig, false, false);
		if (sig.IsSignalled() || !valueList.IsList()) return Value::Null;
		if (!valDict.Store(sig, valueList.GetList())) return Value::Null;
	}
	return result;
}

// open(filename, mode?, encoding?):map {block?}
AScript_DeclareFunction(open)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "filename", VTYPE_String);
	DeclareArg(env, "mode", VTYPE_String, false, OCCUR_ZeroOrOnce);
	DeclareArg(env, "encoding", VTYPE_String, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(open)
{
	const char *fileName = context.GetString(0);
	const char *mode = context.IsString(1)? context.GetString(1) : "rt";
	const char *encoding = context.IsString(2)? context.GetString(2) : "iso-8859-1";
	Value result;
	Object_File *pObj = result.InitAsFile(env);
	pObj->GetFile().Open(sig, fileName, mode, encoding);
	if (sig.IsSignalled()) return Value::Null;
	if (context.IsBlockSpecified()) {
		do { // do-while is required for AScript_BlockSignalHandlerInLoop()
			const Function *pFuncBlock = GetBlockFunction(env, sig, context);
			if (pFuncBlock == NULL) return Value::Null;
			ValueList valListArg(result);
			Context contextSub(valListArg);
			pFuncBlock->Eval(env, sig, contextSub);
			result = Value::Null;	// object is destroyed here
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		} while (0);
	}
	return result;
}

// semaphore()
AScript_DeclareFunction(semaphore)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(semaphore)
{
	return Value(new Object_Semaphore(env.LookupClass(VTYPE_Semaphore)), VTYPE_Semaphore);
}

// Module entry
AScript_ModuleEntry()
{
	AScript_RealizePrivSymbol(open);
	AScript_RealizePrivSymbol(open_l);
	AScript_RealizePrivSymbol(open_r);
	AScript_AssignFunction(class_);
	AScript_AssignFunction(struct_);
	AScript_AssignFunction(int_);
	env.AssignFunction(new Func_try(env));
	env.AssignFunction(new Func_except(env));
	env.AssignFunction(new Func_finally(env));
	env.AssignFunction(new Func_if(env));
	env.AssignFunction(new Func_elsif(env));
	env.AssignFunction(new Func_else(env));
	AScript_AssignFunction(repeat);
	AScript_AssignFunction(while_);
	AScript_AssignFunction(for_);
	AScript_AssignFunction(cross);
	AScript_AssignFunction(break_);
	AScript_AssignFunction(continue_);
	AScript_AssignFunction(return_);
	AScript_AssignFunction(raise);
	AScript_AssignFunction(len);
	AScript_AssignFunction(list);
	AScript_AssignFunction(xlist);
	AScript_AssignFunctionEx(set_xset, "set");
	AScript_AssignFunctionEx(set_xset, "xset");
	AScript_AssignFunction(replace);
	AScript_AssignFunction(nilto);
	AScript_AssignFunction(filter);
	AScript_AssignFunction(map);
	AScript_AssignFunction(zip);
	AScript_AssignFunction(sort);
	AScript_AssignFunction(rank);
	AScript_AssignFunction(reverse);
	AScript_AssignFunction(round);
	AScript_AssignFunction(min);
	AScript_AssignFunction(max);
	AScript_AssignFunctionEx(choosemin_max, "choosemin");
	AScript_AssignFunctionEx(choosemin_max, "choosemax");
	AScript_AssignFunction(choose);
	AScript_AssignFunction(chooseif);
	AScript_AssignFunction(sum);
	AScript_AssignFunction(average);
	AScript_AssignFunctionEx(and_, "and");
	AScript_AssignFunctionEx(or_, "or");
	AScript_AssignFunction(module);
	AScript_AssignFunction(mixin);
	AScript_AssignFunction(import_);
	AScript_AssignFunction(execfile);
	AScript_AssignFunction(parse);
	AScript_AssignFunction(eval);
	AScript_AssignFunction(locals);
	AScript_AssignFunction(outers);
	AScript_AssignFunction(tostring);
	AScript_AssignFunction(tonumber);
	AScript_AssignFunction(tosymbol);
	AScript_AssignFunction(rand);
	AScript_AssignFunction(ord);
	AScript_AssignFunction(chr);
	AScript_AssignFunction(hex);
	AScript_AssignFunction(print);
	AScript_AssignFunction(println);
	AScript_AssignFunction(printf);
	AScript_AssignFunction(dir);
	AScript_AssignFunction(isdefined);
	AScript_AssignFunction(typename_);
	AScript_AssignFunction(undef_);
	AScript_AssignFunctionExx(istype_, "issymbol", VTYPE_Symbol);
	AScript_AssignFunctionExx(istype_, "isboolean", VTYPE_Boolean);
	AScript_AssignFunctionExx(istype_, "isnumber", VTYPE_Number);
	AScript_AssignFunctionExx(istype_, "iscomplex", VTYPE_Complex);
	AScript_AssignFunctionExx(istype_, "isstring", VTYPE_String);
	AScript_AssignFunctionExx(istype_, "ismodule", VTYPE_Module);
	AScript_AssignFunctionExx(istype_, "isclass", VTYPE_Class);
	AScript_AssignFunction(isinstance);
	AScript_AssignFunction(range);
	AScript_AssignFunction(interval);
	AScript_AssignFunction(fill);
	AScript_AssignFunction(rands);
	AScript_AssignFunction(bytes);
	AScript_AssignFunction(pack);
	AScript_AssignFunction(object);
	AScript_AssignFunctionEx(ListInit, "@");
	AScript_AssignFunction(lambda);
	AScript_AssignFunctionEx(lambda, "&");
	AScript_AssignFunction(matrix);
	AScript_AssignFunctionEx(matrix, "@@");
	AScript_AssignFunction(dict);
	AScript_AssignFunctionEx(dict, "%");
	AScript_AssignFunction(open);
	AScript_AssignFunction(semaphore);
}

AScript_ModuleTerminate()
{
}

// miscellaneous functions
bool PrepareRepeaterIterators(Environment &env, Signal sig,
					const ValueList &valListArg, SymbolList &symbolList,
					IteratorOwner &iteratorOwner, ValueList &valListSave)
{
	foreach_const (ValueList, pValue, valListArg) {
		ASSUME(env, pValue->IsExpr());
		const Symbol *pSymbol;
		const Expr *pExpr = pValue->GetExpr();
		Value value;
		if (!pExpr->IsContainCheck()) {
			sig.SetError(ERR_TypeError, "iteratable must be specified as arguments");
			return false;
		}
		const Expr_BinaryOp *pExprBin =
						dynamic_cast<const Expr_BinaryOp *>(pExpr);
		if (!pExprBin->GetLeft()->IsSymbol()) {
			sig.SetError(ERR_ValueError, "l-value of assignment must be a symbol");
			return false;
		}
		const Expr_Symbol *pExprSym =
						dynamic_cast<const Expr_Symbol *>(pExprBin->GetLeft());
		pSymbol = pExprSym->GetSymbol();
		value = pExprBin->GetRight()->Exec(env, sig);
		if (sig.IsSignalled()) return false;
		valListSave.push_back(value);
		Iterator *pIterator = value.CreateIterator(sig);
		if (pIterator == NULL) return false;
		symbolList.push_back(pSymbol);
		iteratorOwner.push_back(pIterator);
	}
	return true;
}

AScript_EndModule(__builtins__)

AScript_DLLModuleEntry(__builtins__)
