//
// Object_String
//

#include "Object_String.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_String
//-----------------------------------------------------------------------------
Object_String::Object_String(const Object_String &obj) : Object(obj), _str(obj._str)
{
}

Object_String::~Object_String()
{
}

Object *Object_String::Clone() const
{
	return new Object_String(*this);
}

Value Object_String::GetByIndex(Signal sig, const Value &valueIdx)
{
	if (!valueIdx.IsNumber()) {
		sig.SetError(ERR_IndexError, "index must be a number for string");
		return Value::Null;
	}
	int idx = valueIdx.GetInt();
	int len = Length(_str.c_str());
	if (idx >= 0) {
		if (idx >= len) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return Value::Null;
		}
		return Value(*this, PickChar(_str, idx).c_str());
	} else {
		if (-idx > len) {
			sig.SetError(ERR_IndexError, "index is out of range");
			return Value::Null;
		}
		return Value(*this, PickChar(_str, len + idx).c_str());
	}
}

void Object_String::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	sig.SetError(ERR_IndexError, "replacement of string is not supported yet");
}

Iterator *Object_String::CreateIterator(Signal sig)
{
	return new IteratorChar(dynamic_cast<Object_String *>(IncRef()), -1);
}

String Object_String::ToString(Signal sig, bool exprFlag)
{
	if (exprFlag) {
		String str;
		str += '"';
		//str += EscapeString(_str.c_str());
		EscapeString(str, _str.c_str());
		str += '"';
		return str;
	}
	return _str;
}

//-----------------------------------------------------------------------------
// Object_String::IteratorChar
//-----------------------------------------------------------------------------
Object_String::IteratorChar::IteratorChar(Object_String *pObj, int cntMax) :
			Iterator(false), _pObj(pObj), _cnt(cntMax), _cntMax(cntMax),
			_pCur(pObj->_GetString().begin())
{
}

Object_String::IteratorChar::~IteratorChar()
{
	Object::Delete(_pObj);
}

bool Object_String::IteratorChar::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObj;
	const String &str = _pObj->_GetString();
	if (_pCur == str.end()) return false;
	String::const_iterator pLeft = _pCur;
	if (_cnt == 0) {
		_pCur = str.end();
	} else {
		_pCur = NextChar(str, _pCur);
		if (_cnt > 0) _cnt--;
	}
	value = Value(env, String(pLeft, _pCur).c_str());
	return true;
}

String Object_String::IteratorChar::ToString(Signal sig) const
{
	return String("<iterator>");
}

//-----------------------------------------------------------------------------
// Object_String::IteratorLine
//-----------------------------------------------------------------------------
Object_String::IteratorLine::IteratorLine(Object_String *pObj,
											int cntMax, bool includeEOLFlag) :
			Iterator(false), _pObj(pObj), _cnt(cntMax), _cntMax(cntMax),
			_includeEOLFlag(includeEOLFlag), _pCur(pObj->_GetString().begin())
{
}

Object_String::IteratorLine::~IteratorLine()
{
	Object::Delete(_pObj);
}

bool Object_String::IteratorLine::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObj;
	const String &str = _pObj->_GetString();
	if (_pCur == str.end() || _cnt == 0) return false;
	String::const_iterator pLeft = _pCur;
	String::const_iterator pNext = str.end();
	for ( ; _pCur != str.end(); _pCur++) {
		if (*_pCur == '\r') {
			pNext = _pCur + 1;
			if (pNext != str.end() && *pNext == '\n') pNext++;
			break;
		} else if (*_pCur == '\n') {
			pNext = _pCur + 1;
			break;
		}
	}
	value = Value(env, String(pLeft, _includeEOLFlag? pNext : _pCur).c_str());
	_pCur = pNext;
	if (_cnt > 0) _cnt--;
	return true;
}

String Object_String::IteratorLine::ToString(Signal sig) const
{
	return String("<iterator>");
}

//-----------------------------------------------------------------------------
// Object_String::IteratorSplit
//-----------------------------------------------------------------------------
Object_String::IteratorSplit::IteratorSplit(Object_String *pObj, const char *sep,
											int cntMax, bool ignoreCaseFlag) :
	Iterator(false), _pObj(pObj), _sep(sep), _cnt(cntMax), _cntMax(cntMax),
	_ignoreCaseFlag(ignoreCaseFlag), _doneFlag(false),
	_pCur(pObj->_GetString().begin())
{
}

Object_String::IteratorSplit::~IteratorSplit()
{
	Object::Delete(_pObj);
}

bool Object_String::IteratorSplit::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObj;
	const String &str = _pObj->_GetString();
	if (_doneFlag) {
		return false;
	} else if (_pCur == str.end()) {
		value = Value(env, "");
		_doneFlag = true;
	} else if (_cnt == 0) {
		value = Value(env, String(_pCur, str.end()).c_str());
		_pCur = str.end();
		_doneFlag = true;
	} else {
		String::const_iterator pRight =
							FindString(_pCur, str.end(), _sep, _ignoreCaseFlag);
		value = Value(env, String(_pCur, pRight).c_str());
		if (pRight == str.end()) {
			_pCur = str.end();
			_doneFlag = true;
		} else {
			_pCur = pRight + _sep.size();
		}
		if (_cnt > 0) _cnt--;
	}
	return true;
}

String Object_String::IteratorSplit::ToString(Signal sig) const
{
	return String("<iterator>");
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_String
//-----------------------------------------------------------------------------
// string#len()
AScript_DeclareMethod(String, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(String, len)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(Length(pSelf->GetString()));
}

// string#capitalize()
AScript_DeclareMethod(String, capitalize)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(String, capitalize)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Capitalize(pSelf->GetString()).c_str());
}

// string#lower()
AScript_DeclareMethod(String, lower)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(String, lower)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Lower(pSelf->GetString()).c_str());
}

// string#upper()
AScript_DeclareMethod(String, upper)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(String, upper)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Upper(pSelf->GetString()).c_str());
}

// string#strip():[both,left,right]
AScript_DeclareMethod(String, strip)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(both));
	DeclareAttr(AScript_Symbol(left));
	DeclareAttr(AScript_Symbol(right));
}

AScript_ImplementMethod(String, strip)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Strip(pSelf->GetString(), context.GetAttrs()).c_str());
}

// string#align(len:number, padding?:string):map:[left,right,center]
AScript_DeclareMethod(String, align)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "len", VTYPE_Number);
	DeclareArg(env, "padding", VTYPE_String, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(left));
	DeclareAttr(AScript_Symbol(right));
	DeclareAttr(AScript_Symbol(center));
}

AScript_ImplementMethod(String, align)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	size_t len = context.GetSizeT(0);
	const char *padding = context.IsString(1)? context.GetString(1) : " ";
	if (Length(padding) != 1) {
		sig.SetError(ERR_ValueError, "padding must consist of a single character");
		return Value::Null;
	}
	String str;
	if (context.IsSet(AScript_Symbol(right))) {
		str = RJust(pSelf->GetString(), len, padding);
	} else if (context.IsSet(AScript_Symbol(center))) {
		str = Center(pSelf->GetString(), len, padding);
	} else {
		str = LJust(pSelf->GetString(), len, padding);
	}
	return Value(env, str.c_str());
}

// string#left(len?:number):map
AScript_DeclareMethod(String, left)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "len", VTYPE_Number, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(String, left)
{
	if (!context.IsNumber(0)) return context.GetSelf();
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Left(pSelf->GetString(), context.GetSizeT(0)).c_str());
}

// string#right(len?:number):map
AScript_DeclareMethod(String, right)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "len", VTYPE_Number, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(String, right)
{
	if (!context.IsNumber(0)) return context.GetSelf();
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Right(pSelf->GetString(), context.GetSizeT(0)).c_str());
}

// string#mid(start?, len?):map
AScript_DeclareMethod(String, mid)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "start", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareArg(env, "len", VTYPE_Number, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(String, mid)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Middle(pSelf->GetString(),
			context.IsNumber(0)? context.GetInt(0) : 0,
			context.IsNumber(1)? context.GetInt(1) : -1).c_str());
}

// string#find(sub:string, start?:number):map
AScript_DeclareMethod(String, find)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "sub",		VTYPE_String);
	DeclareArg(env, "start",	VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_Symbol(rev));
}

AScript_ImplementMethod(String, find)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return FindString(env, sig, pSelf->GetString(), context.GetString(0),
			context.IsNumber(1)? context.GetInt(1) : 0, context.GetAttrs());
}

// string#replace(sub:string, new:string, maxreplace?:number):map
AScript_DeclareMethod(String, replace)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "sub",			VTYPE_String);
	DeclareArg(env, "replace",		VTYPE_String);
	DeclareArg(env, "maxreplace",	VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
}

AScript_ImplementMethod(String, replace)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	String result = Replace(pSelf->GetString(),
			context.GetString(0), context.GetString(1),
			context.IsNumber(2)? context.GetInt(2) : -1, context.GetAttrs());
	return Value(env, result.c_str());
}

// string#splitlines(nlines?:number):[chop] {block?}
// conrresponding to file#readlines()
AScript_DeclareMethod(String, splitlines)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "nlines", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(chop));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(String, splitlines)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	Object_String *pObj = dynamic_cast<Object_String *>(pSelf->IncRef());
	int maxSplit = context.IsNumber(0)? context.GetInt(0) : -1;
	bool includeEOLFlag = !context.IsSet(AScript_Symbol(chop));
	return ReturnIterator(env, sig, context,
				new Object_String::IteratorLine(pObj, maxSplit, includeEOLFlag));
}

// string#split(sep?:string, maxnumber?:number):map:[icase] {block?}
AScript_DeclareMethod(String, split)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "sep", VTYPE_String, false, OCCUR_ZeroOrOnce);
	DeclareArg(env, "maxnumber", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(String, split)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	Object_String *pObj = dynamic_cast<Object_String *>(pSelf->IncRef());
	int maxSplit = context.IsNumber(1)? context.GetInt(1) : -1;
	Iterator *pIterator = NULL;
	if (context.IsString(0) && *context.GetString(0) != '\0') {
		const char *sep = context.GetString(0);
		bool ignoreCaseFlag = context.IsSet(AScript_Symbol(icase));
		pIterator = new Object_String::IteratorSplit(
										pObj, sep, maxSplit, ignoreCaseFlag);
	} else {
		pIterator = new Object_String::IteratorChar(pObj, maxSplit);
	}
	return ReturnIterator(env, sig, context, pIterator);
}

// str = String#format(values*):map
AScript_DeclareMethod(String, format)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "values", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementMethod(String, format)
{
	Object_String *pSelf = Object_String::GetSelfObj(context);
	return Value(env, Formatter::Format(sig,
						pSelf->GetString(), context.GetList(0)).c_str());
}

// Assignment
Class_String::Class_String(Class *pClassSuper) : Class(pClassSuper)
{
	AScript_AssignMethod(String, len);
	AScript_AssignMethod(String, capitalize);
	AScript_AssignMethod(String, lower);
	AScript_AssignMethod(String, upper);
	AScript_AssignMethod(String, splitlines);
	AScript_AssignMethod(String, strip);
	AScript_AssignMethod(String, align);
	AScript_AssignMethod(String, left);
	AScript_AssignMethod(String, right);
	AScript_AssignMethod(String, mid);
	AScript_AssignMethod(String, find);
	AScript_AssignMethod(String, replace);
	AScript_AssignMethod(String, split);
	AScript_AssignMethod(String, format);
}

Object *Class_String::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object_String((pClass == NULL)? this : pClass);
}

}
