#include "Value.h"
#include "Expr.h"
#include "Environment.h"
#include "Object.h"
#include "Module.h"
#include "Iterator.h"
#include "Object_Function.h"
#include "Object_String.h"
#include "Object_Bytes.h"
#include "Object_List.h"
#include "Object_Matrix.h"
#include "Object_Dict.h"
#include "Object_DateTime.h"
#include "Object_Iterator.h"
#include "Object_File.h"
#include "Object_FileStat.h"
#include "Object_Expr.h"
#include "Object_Environment.h"
#include "Object_Error.h"
#include "Object_Semaphore.h"
#include "Object_Struct.h"

namespace AScript {

//-----------------------------------------------------------------------------
// ValueTypeInfo
//-----------------------------------------------------------------------------
ValueTypeInfo::~ValueTypeInfo()
{
	Class::Delete(_pClass);
}

void ValueTypeInfo::SetClass(Class *pClass)
{
	_pClass = pClass;
	pClass->SetSymbol(_pSymbol);
}

//-----------------------------------------------------------------------------
// ValueTypePool
//-----------------------------------------------------------------------------
ValueTypePool *ValueTypePool::_pInst = NULL;

ValueTypePool::~ValueTypePool()
{
	foreach (ValueTypeList, ppValueTypeInfo, _valueTypeList) {
		ValueTypeInfo::Delete(*ppValueTypeInfo);
	}
}

void ValueTypePool::Initialize(Environment &env)
{
	if (_pInst == NULL) {
		_pInst = new ValueTypePool();
		_pInst->_Initialize(env);
	}
}

void ValueTypePool::_Initialize(Environment &env)
{
	AScript_RealizeVTYPE(Invalid,		"invalid");	// must be at first
	AScript_RealizeVTYPE(Quote,			"quote");	// for declaration
	AScript_RealizeVTYPE(Any,			"any");		// for declaration
	AScript_RealizeVTYPE(Symbol,		"symbol");
	AScript_RealizeVTYPE(Boolean,		"boolean");
	AScript_RealizeVTYPE(Number,		"number");
	AScript_RealizeVTYPE(Complex,		"complex");
	AScript_RealizeVTYPE(Module,		"module");
	AScript_RealizeVTYPE(Class,			"class");
	// object types
	AScript_RealizeVTYPE(Object,		"object");
	AScript_RealizeVTYPE(Function,		"function");
	AScript_RealizeVTYPE(String,		"string");
	AScript_RealizeVTYPE(StringEmb,		"string_emb");
	AScript_RealizeVTYPE(Bytes,			"bytes");
	AScript_RealizeVTYPE(BytesEmb,		"bytes_emb");
	AScript_RealizeVTYPE(BytesPtr,		"bytesptr");
	AScript_RealizeVTYPE(List,			"list");
	AScript_RealizeVTYPE(Matrix,		"matrix");
	AScript_RealizeVTYPE(Dict,			"dict");
	AScript_RealizeVTYPE(File,			"file");
	AScript_RealizeVTYPE(FileStat,		"filestat");
	AScript_RealizeVTYPE(DateTime,		"datetime");
	AScript_RealizeVTYPE(Iterator,		"iterator");
	AScript_RealizeVTYPE(Expr,			"expr");
	AScript_RealizeVTYPE(Environment,	"environment");
	AScript_RealizeVTYPE(Error,			"error");
	AScript_RealizeVTYPE(Semaphore,		"semaphore");
	AScript_RealizeVTYPE(Struct,		"struct");
	// Class must be initialized at 1st
	Class *pClass = new Class(&env);
	AScript_VTYPEInfo(Object)->SetClass(pClass);
	// Class_Function must be initialized at 2nd
	Class_Function *pClass_Function = new Class_Function(env);
	AScript_VTYPEInfo(Function)->SetClass(pClass_Function);
	pClass_Function->Prepare();	// methods of Function can only be initialized here
	pClass->Prepare();			// methods of Object can only be initialized here
	// other built-in object classes
	AScript_VTYPEInfo(String)->SetClass(new Class_String(env));
	AScript_VTYPEInfo(Bytes)->SetClass(new Class_Bytes(env));
	AScript_VTYPEInfo(BytesPtr)->SetClass(new Class_BytesPtr(env));
	AScript_VTYPEInfo(List)->SetClass(new Class_List(env));
	AScript_VTYPEInfo(Matrix)->SetClass(new Class_Matrix(env));
	AScript_VTYPEInfo(Dict)->SetClass(new Class_Dict(env));
	AScript_VTYPEInfo(File)->SetClass(new Class_File(env));
	AScript_VTYPEInfo(FileStat)->SetClass(new Class_FileStat(env));
	AScript_VTYPEInfo(DateTime)->SetClass(new Class_DateTime(env));
	AScript_VTYPEInfo(Iterator)->SetClass(new Class_Iterator(env));
	AScript_VTYPEInfo(Expr)->SetClass(new Class_Expr(env));
	AScript_VTYPEInfo(Environment)->SetClass(new Class_Environment(env));
	AScript_VTYPEInfo(Error)->SetClass(new Class_Error(env));
	AScript_VTYPEInfo(Semaphore)->SetClass(new Class_Semaphore(env));
	AScript_VTYPEInfo(Struct)->SetClass(new Class_Struct(env));
}

ValueTypeInfo *ValueTypePool::Add(const Symbol *pSymbol, Class *pClass)
{
	ValueType valType = static_cast<ValueType>(_valueTypeList.size());
	ValueTypeInfo *pValTypeInfo = new ValueTypeInfo(valType, pSymbol, pClass);
	_valueTypeList.push_back(pValTypeInfo);
	return pValTypeInfo;
}

ValueTypeInfo *ValueTypePool::Lookup(ValueType valType)
{
	return _valueTypeList[valType];
}

//-----------------------------------------------------------------------------
// Value
//-----------------------------------------------------------------------------
const Complex Value::_compZero;
const Value Value::Null;

Value::Value(const Value &value) : _valType(value._valType), _ownFlag(value._ownFlag)
{
	if (value.IsNumber()) {
		_u.num = value._u.num;
	} else if (value.IsBoolean()) {
		_u.flag = value._u.flag;
	} else if (value.IsSymbol()) {
		_u.pSymbol = value._u.pSymbol;
	} else if (value.IsComplex()) {
		_u.pComp = new Complex(*value._u.pComp);
	} else if (value.IsModule()) {
		_u.pModule = value._u.pModule->IncRef();
	} else if (value.IsClass()) {
		_u.pClass = value._u.pClass->IncRef();
	} else if (value.IsObject()) {
		_u.pObj = value._u.pObj->IncRef();
	} else {
		// nothing to do
	}
}

Value::~Value()
{
	FreeResource();
}

Value::Value(Environment &env, const char *str)
{
	_valType = static_cast<char>(VTYPE_String);
	_ownFlag = true;
	_u.pObj = new Object_String(env.LookupClass(VTYPE_String), str);
}

Value::Value(Environment &env, const char *str, size_t len)
{
	_valType = static_cast<char>(VTYPE_String);
	_ownFlag = true;
	_u.pObj = new Object_String(env.LookupClass(VTYPE_String), str, len);
}

Value::Value(Environment &env, ValueType valType, const char *str)
{
	_valType = static_cast<char>(valType);
	_ownFlag = true;
	_u.pObj = new Object_String(env.LookupClass(VTYPE_String), str);
}

Value::Value(Environment &env, ValueType valType, const char *str, size_t len)
{
	_valType = static_cast<char>(valType);
	_ownFlag = true;
	_u.pObj = new Object_String(env.LookupClass(VTYPE_String), str, len);
}

Value::Value(Environment &env, Iterator *pIterator)
{
	_valType = static_cast<char>(VTYPE_Iterator);
	_ownFlag = true;
	_u.pObj = new Object_Iterator(env.LookupClass(VTYPE_Iterator), pIterator);
}

Value::Value(Environment &env, Function *pFunc)
{
	_valType = static_cast<char>(VTYPE_Function);
	_ownFlag = true;
	_u.pObj = new Object_Function(env.LookupClass(VTYPE_Function), pFunc);
}

Value::Value(Object *pObj, ValueType valType, bool ownFlag)
{
	_valType = static_cast<char>(valType);
	_ownFlag = ownFlag;
	_u.pObj = pObj;
}

void Value::FreeResource()
{
	if (IsComplex()) {
		delete _u.pComp;
		_u.pComp = NULL;
	} else if (IsModule()) {
		if (_ownFlag) Module::Delete(_u.pModule);
		_u.pModule = NULL;
	} else if (IsClass()) {
		if (_ownFlag) Class::Delete(_u.pClass);
		_u.pClass = NULL;
	} else if (IsObject()) {
		if (_ownFlag) Object::Delete(_u.pObj);
		_u.pObj = NULL;
	} else { // Number, Boolean
		// nothing to do
	}
	_valType = static_cast<char>(VTYPE_Invalid);
}

Value &Value::operator=(const Value &value)
{
	FreeResource();
	_valType = value._valType;
	_ownFlag = value._ownFlag;
	if (value.IsNumber()) {
		_u.num = value._u.num;
	} else if (value.IsBoolean()) {
		_u.flag = value._u.flag;
	} else if (value.IsSymbol()) {
		_u.pSymbol = value._u.pSymbol;
	} else if (value.IsComplex()) {
		_u.pComp = new Complex(*value._u.pComp);
	} else if (value.IsModule()) {
		_u.pModule = value._u.pModule->IncRef();
	} else if (value.IsClass()) {
		_u.pClass = value._u.pClass->IncRef();
	} else if (value.IsObject()) {
		_u.pObj = value._u.pObj->IncRef();
	} else {
		_valType = static_cast<char>(VTYPE_Invalid);
	}
	return *this;
}

const char *Value::GetTypeName() const
{
	return GetValueTypeSymbol(GetType())->GetName();
}

ObjectBase *Value::ExtractObject(Signal sig)
{
	ObjectBase *pObjBase = NULL;
	if (IsModule()) {
		pObjBase = GetModule();
	} else if (IsClass()) {
		pObjBase = GetClass();
	} else if (IsObject()) {
		pObjBase = GetObject();
		if (pObjBase->IsFunction()) {
			const Object_Function *pObjFunc =
								dynamic_cast<const Object_Function *>(pObjBase);
			Class *pClass = pObjFunc->GetFunction()->GetClassToConstruct();
			if (pClass != NULL) {
				InitAsClass(pClass->IncRef());
				pObjBase = pClass;
			}
		}
	} else {
		sig.SetError(ERR_ValueError,
			"%s can not be specified as l-value of field", GetTypeName());
		return NULL;
	}
	return pObjBase;
}

bool Value::IsFlatList() const
{
	return IsList() && GetList().IsFlat();
}

bool Value::IsInstanceOf(ValueType valType)
{
	if (_valType == valType) return true;
	const ValueTypeInfo *pValueTypeInfo =
							ValueTypePool::GetInstance()->Lookup(_valType);
	if (pValueTypeInfo == NULL) return false;
	for (Class *pClass = pValueTypeInfo->GetClass(); pClass != NULL;
										pClass = pClass->GetClassSuper()) {
		if (pClass->GetValueType() == valType) return true;
	}
	return false;
}

Object_String *Value::GetStringObj()
{
	return dynamic_cast<Object_String *>(_u.pObj);
}

const Object_String *Value::GetStringObj() const
{
	return dynamic_cast<const Object_String *>(_u.pObj);
}

Object_File *Value::GetFileObj()
{
	return dynamic_cast<Object_File *>(_u.pObj);
}

const Object_File *Value::GetFileObj() const
{
	return dynamic_cast<const Object_File *>(_u.pObj);
}

const char *Value::GetString() const
{
	return dynamic_cast<Object_String *>(_u.pObj)->GetString();
}

const String &Value::_GetString() const
{
	return dynamic_cast<Object_String *>(_u.pObj)->_GetString();
}

const Bytes &Value::GetBytes() const
{
	return dynamic_cast<Object_Bytes *>(_u.pObj)->GetBytes();
}

Object_BytesPtr *Value::GetBytesPtr() const
{
	return dynamic_cast<Object_BytesPtr *>(_u.pObj);
}

const DateTime &Value::GetDateTime() const
{
	return dynamic_cast<Object_DateTime *>(_u.pObj)->GetDateTime();
}

ErrorType Value::GetErrorType() const
{
	return dynamic_cast<Object_Error *>(_u.pObj)->GetErrorType();
}

Object_List *Value::GetListObj()
{
	return dynamic_cast<Object_List *>(_u.pObj);
}

const Object_List *Value::GetListObj() const
{
	return dynamic_cast<const Object_List *>(_u.pObj);
}

Object_Matrix *Value::GetMatrixObj()
{
	return dynamic_cast<Object_Matrix *>(_u.pObj);
}

const Object_Matrix *Value::GetMatrixObj() const
{
	return dynamic_cast<const Object_Matrix *>(_u.pObj);
}

ValueList &Value::GetList()
{
	return dynamic_cast<Object_List *>(_u.pObj)->GetList();
}

const ValueList &Value::GetList() const
{
	return dynamic_cast<const Object_List *>(_u.pObj)->GetList();
}

ValueDict &Value::GetDict()
{
	return dynamic_cast<Object_Dict *>(_u.pObj)->GetDict();
}

const ValueDict &Value::GetDict() const
{
	return IsDict()?
		dynamic_cast<const Object_Dict *>(_u.pObj)->GetDict() : ValueDict::Null;
}

Iterator *Value::GetIterator() const
{
	return IsIterator()?
		dynamic_cast<const Object_Iterator *>(_u.pObj)->GetIterator() : NULL;
}

File &Value::GetFile()
{
	return dynamic_cast<Object_File *>(_u.pObj)->GetFile();
}

const Expr *Value::GetExpr() const
{
	return IsExpr()? dynamic_cast<Object_Expr *>(_u.pObj)->GetExpr() : NULL;
}

Function *Value::GetFunction()
{
	return IsFunction()? dynamic_cast<Object_Function *>(_u.pObj)->GetFunction() : NULL;
}

Object_Function *Value::GetFunctionObj()
{
	return IsFunction()? dynamic_cast<Object_Function *>(_u.pObj) : NULL;
}

Expr *Value::CloneExpr() const
{
	return IsExpr()? dynamic_cast<Object_Expr *>(_u.pObj)->GetExpr()->IncRef() :
			IsSymbol()? new Expr_Symbol(_u.pSymbol) : NULL;
}

Iterator *Value::CreateIterator(Signal sig) const
{
	if (IsObject()) {
		return _u.pObj->CreateIterator(sig);
	}
	sig.SetError(ERR_ValueError, "value of %s cannot generate iterator", GetTypeName());
	return NULL;
}

String Value::ToString(Signal sig, bool exprFlag) const
{
	if (IsNumber()) {
		return NumberToString(_u.num);
	} else if (IsBoolean()) {
		return String(_u.flag?
			AScript_Symbol(true_)->GetName() : AScript_Symbol(false_)->GetName());
	} else if (IsSymbol()) {
		String str;
		if (exprFlag) str += '`';
		str += _u.pSymbol->GetName();
		return str;
	} else if (IsComplex()) {
		const Complex &comp = *_u.pComp;
		String str;
		char buff[32];
		if (comp.real() == 0. && comp.imag() == 0.) {
			::sprintf(buff, GetNumberFormat(), 0.);
			str += buff;
		} else if (comp.real() == 0.) {
			if (_u.pComp->imag() < 0.) str += "-";
			::sprintf(buff, GetNumberFormat(), ::fabs(_u.pComp->imag()));
			str += buff;
			str += AScript_Symbol(j)->GetName();
		} else if (comp.imag() == 0.) {
			::sprintf(buff, GetNumberFormat(), _u.pComp->real());
			str += buff;
		} else {
			::sprintf(buff, GetNumberFormat(), _u.pComp->real());
			str += buff;
			str += (_u.pComp->imag() >= 0.)? "+" : "-";
			::sprintf(buff, GetNumberFormat(), ::fabs(_u.pComp->imag()));
			str += buff;
			str += AScript_Symbol(j)->GetName();
		}
		return str;
	} else if (IsModule()) {
		return _u.pModule->ToString(sig, exprFlag);
	} else if (IsClass()) {
		return _u.pClass->ToString(sig, exprFlag);
	} else if (IsObject()) {
		return _u.pObj->ToString(sig, exprFlag);
	}
	return String(exprFlag? "nil" : "");
}

Number Value::ToNumber(bool allowPartFlag, bool &successFlag) const
{
	successFlag = true;
	if (IsNumber()) {
		return _u.num;
	} else if (IsBoolean()) {
		return _u.flag? 1. : 0.;
	} else if (IsString()) {
		const char *str = GetString();
		Number num = 0.;
		char *p = const_cast<char *>(str);
		if (str[0] == '0' &&
				(str[1] == 'x' || str[1] == 'X' || IsOctDigit(str[1]))) {
			num = static_cast<Number>(::strtoul(str, &p, 0));
		} else {
			num = ::strtod(str, &p);
		}
		successFlag = (p > str && (allowPartFlag || *p == '\0'));
		return num;
	} else {
		successFlag = false;
		return 0.;
	}
}

int Value::Compare(const Value &value1, const Value &value2)
{
	int rtn = -1;
	if (value1.GetType() != value2.GetType()) {
		rtn = static_cast<int>(value1.GetType()) -
								static_cast<int>(value2.GetType());
	} else if (value1.IsNumber()) {
		rtn = (value1.GetNumber() == value2.GetNumber())? 0 :
				(value1.GetNumber() < value2.GetNumber())? -1 : +1;
	} else if (value1.IsBoolean()) {
		rtn = static_cast<int>(value1.GetBoolean()) -
								static_cast<int>(value2.GetBoolean());
	} else if (value1.IsSymbol()) {
		rtn = static_cast<int>(value1.GetSymbol()->GetUniqNum()) -
										value2.GetSymbol()->GetUniqNum();
	} else if (value1.IsString()) {
		rtn = ::strcmp(value1.GetString(), value2.GetString());
	} else if (value1.IsDateTime()) {
		rtn = DateTime::Compare(value1.GetDateTime(), value2.GetDateTime());
	} else if (value1.IsList()) {
		bool emptyFlag1 = value1.GetList().empty();
		bool emptyFlag2 = value2.GetList().empty();
		if (emptyFlag1 || emptyFlag2) {
			rtn = -(static_cast<int>(emptyFlag1) - static_cast<int>(emptyFlag2));
		} else {
			rtn = Value::Compare(value1.GetList().front(), value2.GetList().front());
		}
	} else if (value1.IsObject()) {
		rtn = value1.GetObject()->Compare(value2.GetObject());
	} else if (value1.IsInvalid() && value2.IsInvalid()) {
		rtn = 0;
	}
	return rtn;
}

void Value::Accept(Signal sig, ValueVisitor &visitor) const
{
	if (IsList()) {
		foreach_const (ValueList, pValue, GetList()) {
			pValue->Accept(sig, visitor);
			if (sig.IsSignalled()) break;
		}
	} else {
		visitor.Visit(sig, *this);
	}
}

void Value::InitAsModule(Module *pModule)
{
	FreeResource();
	_valType = static_cast<char>(VTYPE_Module), _u.pModule = pModule;
}

void Value::InitAsClass(Class *pClass)
{
	FreeResource();
	_valType = static_cast<char>(VTYPE_Class), _u.pClass = pClass;
}

void Value::InitAsObject(Object *pObj, ValueType valType, bool ownFlag)
{
	FreeResource();
	_valType = static_cast<char>(valType), _u.pObj = pObj;
	_ownFlag = ownFlag;
}

void Value::InitAsIterator(Environment &env, Iterator *pIterator)
{
	Object_Iterator *pObj = new Object_Iterator(env.LookupClass(VTYPE_Iterator), pIterator);
	InitAsObject(pObj, VTYPE_Iterator);
}

void Value::InitAsError(Environment &env,
						ErrorType errType, const char *text, const Value &value)
{
	Object_Error *pObj = new Object_Error(env.LookupClass(VTYPE_Error), errType);
	pObj->AssignValue(AScript_Symbol(text), Value(env, text), false);
	pObj->AssignValue(AScript_Symbol(value), value, false);
	InitAsObject(pObj, VTYPE_Error);
}

void Value::InitAsFunction(Environment &env, Function *pFunc)
{
	Object_Function *pObj = new Object_Function(env.LookupClass(VTYPE_Function), pFunc);
	InitAsObject(pObj, VTYPE_Function);
}

ValueList &Value::InitAsList(Environment &env)
{
	Object_List *pObj = new Object_List(env.LookupClass(VTYPE_List));
	InitAsObject(pObj, VTYPE_List);
	return pObj->GetList();
}

Object_Matrix *Value::InitAsMatrix(Environment &env,
								int nRows, int nCols, const Value &valueElem)
{
	Object_Matrix *pObj = new Object_Matrix(
					env.LookupClass(VTYPE_Matrix), nRows, nCols, valueElem);
	InitAsObject(pObj, VTYPE_Matrix);
	return pObj;
}

Object_Matrix *Value::InitAsMatrix(Environment &env, Signal sig, const ValueList &valList)
{
	size_t nRows, nCols;
	if (!valList.CheckMatrix(&nRows, &nCols)) {
		sig.SetError(ERR_ValueError, "invalid matrix initialization");
		return NULL;
	}
	Object_List *pObjList = new Object_List(env.LookupClass(VTYPE_List));
	ValueList &valListDst = pObjList->GetList();
	valListDst.reserve(nRows * nCols);
	foreach_const (ValueList, pValue, valList) {
		if (pValue->IsList()) {
			foreach_const (ValueList, pValueElem, pValue->GetList()) {
				valListDst.push_back(*pValueElem);
			}
		} else {
			valListDst.push_back(*pValue);
		}
	}
	Object_Matrix *pObj = new Object_Matrix(env.LookupClass(VTYPE_Matrix),
										pObjList, 0, 0, nRows, nCols, nCols);
	InitAsObject(pObj, VTYPE_Matrix);
	return pObj;
}

ValueList &Value::InitAsList(Environment &env, size_t n, const Value &value)
{
	Object_List *pObj = (n == 0)?
			new Object_List(env.LookupClass(VTYPE_List)) :
			new Object_List(env.LookupClass(VTYPE_List), n, value);
	InitAsObject(pObj, VTYPE_List);
	return pObj->GetList();
}

ValueDict &Value::InitAsDict(Environment &env)
{
	Object_Dict *pObj = new Object_Dict(env.LookupClass(VTYPE_Dict));
	InitAsObject(pObj, VTYPE_Dict);
	return pObj->GetDict();
}

Object_File *Value::InitAsFile(Environment &env)
{
	Object_File *pObj = new Object_File(env.LookupClass(VTYPE_File));
	InitAsObject(pObj, VTYPE_File);
	return pObj;
}

Object_Bytes *Value::InitAsBytes(Environment &env, ValueType valType)
{
	Object_Bytes *pObj = new Object_Bytes(env.LookupClass(VTYPE_Bytes));
	InitAsObject(pObj, valType);
	return pObj;
}

Object_Bytes *Value::InitAsBytes(Environment &env,
									const Bytes &bytes, ValueType valType)
{
	Object_Bytes *pObj = new Object_Bytes(env.LookupClass(VTYPE_Bytes), bytes);
	InitAsObject(pObj, valType);
	return pObj;
}

Object_BytesPtr *Value::InitAsBytesPtr(Environment &env,
									Object_Bytes *pObjBytes, size_t offset)
{
	Object_BytesPtr *pObj =
				new Object_BytesPtr(env.LookupClass(VTYPE_BytesPtr), pObjBytes, offset);
	InitAsObject(pObj, VTYPE_BytesPtr);
	return pObj;
}

void Value::InitAsExpr(Environment &env, Expr *pExpr)
{
	Object_Expr *pObj = new Object_Expr(env.LookupClass(VTYPE_Expr), pExpr);
	InitAsObject(pObj, VTYPE_Expr);
}

void Value::InitAsEnvironment(Environment &env)
{
	Object_Environment *pObj =
				new Object_Environment(env.LookupClass(VTYPE_Environment), env);
	InitAsObject(pObj, VTYPE_Environment);
}

//-----------------------------------------------------------------------------
// ValueList
//-----------------------------------------------------------------------------
const ValueList ValueList::Null;

ValueList::ValueList(const ValueList &valList)
{
	reserve(valList.size());
	foreach_const (ValueList, pValue, valList) {
		push_back(*pValue);
	}
}

bool ValueList::IsTypeMixed() const
{
	if (empty()) return false;
	ValueType valTypeFirst = front().GetType();
	if (valTypeFirst == VTYPE_Complex) valTypeFirst = VTYPE_Number;
	foreach_const (ValueList, pValue, *this) {
		ValueType valType = pValue->GetType();
		if (valType == VTYPE_Complex) valType = VTYPE_Number;
		if (valTypeFirst != valType) return true;
	}
	return false;
}

bool ValueList::IsFlat() const
{
	foreach_const (ValueList, pValue, *this) {
		if (pValue->IsList()) return false;
	}
	return true;
}

bool ValueList::IsContain(const Value &value) const
{
	foreach_const (ValueList, pValue, *this) {
		if (Value::Compare(*pValue, value) == 0) return true;
	}
	return false;
}

bool ValueList::IsContainIterator() const
{
	foreach_const (ValueList, pValue, *this) {
		if (pValue->IsIterator()) return true;
	}
	return false;
}

bool ValueList::CheckMatrix(size_t *pnRow, size_t *pnCol) const
{
	if (empty()) return false;
	size_t nRow = size();
	size_t nCol = 1;
	if (front().IsList()) {
		nCol = front().GetList().size();
		foreach_const (ValueList, pValueRow, *this) {
			if (!pValueRow->IsList()) {
				return false;
			} else if (pValueRow->GetList().size() != nCol) {
				return false;
			}
		}
	} else {
		foreach_const (ValueList, pValueCol, *this) {
			if (pValueCol->IsList()) return false;
		}
	}
	if (pnRow != NULL) *pnRow = nRow;
	if (pnCol != NULL) *pnCol = nCol;
	return true;
}

bool ValueList::AssumeSameLength(Signal sig,
						const ValueList &valList1, const ValueList &valList2)
{
	if (valList1.size() == valList2.size()) return true;
	sig.SetError(ERR_ValueError, "lists are different in length");
	return false;
}

void ValueList::Append(const ValueList &valList)
{
	foreach_const (ValueList, pValue, valList) {
		push_back(*pValue);
	}
}

void ValueList::Print(Signal sig, int indentLevel) const
{
	foreach_const (ValueList, pValue, *this) {
		if (pValue->IsList()) {
			pValue->GetList().Print(sig, indentLevel + 1);
		} else {
			::printf("%*s%s\n",
					indentLevel * 2, "", pValue->ToString(sig).c_str());
		}
	}
}

//-----------------------------------------------------------------------------
// ValueMap
//-----------------------------------------------------------------------------
const ValueMap ValueMap::Null;

//-----------------------------------------------------------------------------
// ValueDict
//-----------------------------------------------------------------------------
const ValueDict ValueDict::Null;

bool ValueDict::Store(Signal sig, const ValueList &valList)
{
	enum { FIELD_Key, FIELD_Value } field = FIELD_Key;
	Value valueIdx;
	const ValueList *pValList = &valList;
	if (valList.size() == 1 && valList[0].IsList()) {
		pValList = &valList[0].GetList();
	}
	foreach_const (ValueList, pValue, *pValList) {
		if (field == FIELD_Key) {
			Value value;
			valueIdx = *pValue;
			if (valueIdx.IsList()) {
				const ValueList &valListElem = valueIdx.GetList();
				if (valListElem.size() != 2) {
					sig.SetError(ERR_KeyError, "invalid key-value format");
					return false;
				}
				valueIdx = valListElem[0];
				value = valListElem[1];
			}
			if (!ValueDict::IsValidKey(valueIdx)) {
				sig.SetError(ERR_KeyError, "invalid value type for key");
				return false;
			} else if (find(valueIdx) != end()) {
				sig.SetError(ERR_KeyError, "duplicated key '%s'", valueIdx.ToString(sig).c_str());
				return false;
			}
			if (value.IsValid()) {
				insert(ValueDict::value_type(valueIdx, value));
			} else {
				field = FIELD_Value;
			}
		} else { // FIELD_Value
			insert(ValueDict::value_type(valueIdx, *pValue));
			field = FIELD_Key;
		}
	}
	if (field == FIELD_Value) {
		sig.SetError(ERR_KeyError, "unmatching key-value pair");
		return false;
	}
	return true;
}

bool ValueDict::Store(Signal sig, const Value &valueIdx, const Value &value)
{
	if (!ValueDict::IsValidKey(valueIdx)) {
		sig.SetError(ERR_KeyError, "invalid value type for key");
		return false;
	} else if (find(valueIdx) != end()) {
		sig.SetError(ERR_KeyError, "duplicated key '%s'", valueIdx.ToString(sig).c_str());
		return false;
	}
	insert(ValueDict::value_type(valueIdx, value));
	return true;
}

}
