#ifndef __DATETIME_H__
#define __DATETIME_H__
#include "Common.h"

namespace AScript {

class TimeDelta;

//-----------------------------------------------------------------------------
// DateTime
//-----------------------------------------------------------------------------
class DateTime {
private:
	short _year;
	char _month;
	char _day;
	long _sec;
	long _usec;
public:
	inline DateTime() : _year(0), _month(0), _day(0), _sec(0), _usec(0) {}
	inline DateTime(short year, char month, char day, long sec, long usec) :
		_year(year), _month(month), _day(day), _sec(sec), _usec(usec) {}
	inline DateTime(const DateTime &dt) :
		_year(dt._year), _month(dt._month), _day(dt._day),
		_sec(dt._sec), _usec(dt._usec) {}
#if defined(HAVE_WINDOWS_H)
	DateTime(const SYSTEMTIME &st);
	DateTime(const FILETIME &ft, bool toLocalFlag);
	void SetSystemTime(const SYSTEMTIME &st);
#else
	DateTime(const struct tm &tm);
	DateTime(time_t t, bool toLocalFlag);
	void SetTime(const struct tm &tm);
#endif
	String ToString(const char *format) const;
	inline short GetYear() const		{ return _year; }	// 0-9999
	inline char GetMonth() const		{ return _month; }	// 1-12
	inline char GetDay() const			{ return _day; }	// 1-31
	inline char GetHour() const			{ return static_cast<char>(_sec / 3600); }		// 0-23
	inline char GetMin() const			{ return static_cast<char>((_sec / 60) % 60); }	// 0-59
	inline char GetSec() const			{ return static_cast<char>(_sec % 60); }			// 0-59
	inline long GetSecRaw() const		{ return _sec; }	// 0 - (3600 * 24 - 1)
	inline long GetUSec() const			{ return _usec; }	// 0 - 999999
	inline bool IsLeapYear() const		{ return _IsLeapYear(_year); }
	inline long GetMSecOfDay() const	{ return _sec * 1000 + _usec / 1000; }
	inline short GetDaysOfYear() const	{ return _GetDaysOfYear(_year); }
	inline short GetDayOfYear() const	{ return _GetDayOfYear(_year, _month, _day); }
	inline char GetDayOfWeek() const	{ return _GetDayOfWeek(_year, _month, _day); }
	inline void ClearTime() { _sec = 0, _usec = 0; }
	void Plus(const TimeDelta &td);
	void Minus(const TimeDelta &td);
	TimeDelta Minus(const DateTime &dt) const;
	static inline bool _IsLeapYear(short year) {
		return (year % 4 == 0) && ((year % 100 != 0) || (year % 400 == 0));
	}
	static inline short _GetDaysOfYear(short year) {
		return _IsLeapYear(year)? 366 : 365;
	}
	static short _GetDayOfYear(short year, char month, char day);
	static void _GetMonthDay(short year, short dayOfYear, char &month, char &day);
	static char _GetDayOfWeek(short year, char month, char day);
	static char _GetDaysOfMonth(short year, char month);
	static int Compare(const DateTime &dt1, const DateTime &dt2);
};

//-----------------------------------------------------------------------------
// TimeDelta
//-----------------------------------------------------------------------------
class TimeDelta {
private:
	long _days;		// -999999999 <= _days <= 999999999
	long _secs;		// 0 <= _secs <= 3600 * 24 - 1
	long _usecs;	// 0 <= _usecs <= 999999
public:
	TimeDelta(long days, long secs, long usecs);
	inline TimeDelta(const TimeDelta &td) :
					_days(td._days), _secs(td._secs), _usecs(td._usecs) {}
	inline long GetDays() const		{ return _days; }
	inline char GetHours() const	{ return static_cast<char>(_secs / 3600); }		// 0-23
	inline char GetMins() const		{ return static_cast<char>((_secs / 60) % 60); }	// 0-59
	inline char GetSecs() const		{ return static_cast<char>(_secs % 60); }			// 0-59
	inline long GetSecsRaw() const	{ return _secs; }
	inline long GetUSecs() const	{ return _usecs; }
	void Plus(const TimeDelta &td);
	void Minus(const TimeDelta &td);
	void Regulate();
	String ToString(const char *format) const;
	static int Compare(const TimeDelta &td1, const TimeDelta &td2);
};

}

#endif
