//-----------------------------------------------------------------------------
// AScript string module
//-----------------------------------------------------------------------------
#include "Module_string.h"
#include "Object_String.h"
#include "Expr.h"

AScript_BeginModule(string)

//-----------------------------------------------------------------------------
// AScript module functions: string
//-----------------------------------------------------------------------------
// n = string.len(str:string):map
AScript_DeclareFunction(len)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(len)
{
	return Value(static_cast<Number>(Length(context.GetString(0))));
}

// str = string.capitalize(str:string):map
AScript_DeclareFunction(capitalize)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(capitalize)
{
	return Value(env, Capitalize(context.GetString(0)).c_str());
}

// str = string.upper(str:string):map
AScript_DeclareFunction(upper)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(upper)
{
	return Value(env, Upper(context.GetString(0)).c_str());
}

// str = string.lower(str:string):map
AScript_DeclareFunction(lower)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(lower)
{
	return Value(env, Lower(context.GetString(0)).c_str());
}

// str = string.strip(str:string):map:[both,left,right]
AScript_DeclareFunction(strip)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareAttr(AScript_Symbol(both));
	DeclareAttr(AScript_Symbol(left));
	DeclareAttr(AScript_Symbol(right));
}

AScript_ImplementFunction(strip)
{
	return Value(env, Strip(context.GetString(0), context.GetAttrs()).c_str());
}

// str = string.align(str:string, len:number, padding:string => " "):map
AScript_DeclareFunction(align)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "len", VTYPE_Number);
	DeclareArg(env, "padding", VTYPE_String, OCCUR_Once, false, new Expr_String(" "));
	DeclareAttr(AScript_Symbol(left));
	DeclareAttr(AScript_Symbol(right));
	DeclareAttr(AScript_Symbol(center));
}

AScript_ImplementFunction(align)
{
	size_t len = context.GetSizeT(1);
	const char *padding = context.GetString(2);
	if (Length(padding) != 1) {
		sig.SetError(ERR_ValueError, "padding must consist of a single character");
		return Value::Null;
	}
	String str;
	if (context.IsSet(AScript_Symbol(right))) {
		str = RJust(context.GetString(0), len, padding);
	} else if (context.IsSet(AScript_Symbol(center))) {
		str = Center(context.GetString(0), len, padding);
	} else {
		str = LJust(context.GetString(0), len, padding);
	}
	return Value(env, str.c_str());
}

// str = string.left(str:string, len?:number):map
AScript_DeclareFunction(left)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "len", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(left)
{
	if (context.IsInvalid(1)) return context.GetValue(0);
	return Value(env, Left(context.GetString(0), context.GetSizeT(1)).c_str());
}

// str = string.right(str, len?):map
AScript_DeclareFunction(right)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "len", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(right)
{
	if (context.IsInvalid(1)) return context.GetValue(0);
	return Value(env, Right(context.GetString(0), context.GetSizeT(1)).c_str());
}

// str = string.mid(str:string, start:number => 0, len?):map
AScript_DeclareFunction(mid)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "start", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "len", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(mid)
{
	return Value(env, Middle(context.GetString(0),
		context.GetInt(1), context.IsNumber(2)? context.GetInt(2) : -1).c_str());
}

// result = string.find(str:string, sub:string, start:number => 0):map
// attributes: list, rev, icase
AScript_DeclareFunction(find)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str",		VTYPE_String);
	DeclareArg(env, "sub",		VTYPE_String);
	DeclareArg(env, "start",	VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_Symbol(rev));
}

AScript_ImplementFunction(find)
{
	return FindString(env, sig, context.GetString(0), context.GetString(1),
										context.GetInt(2), context.GetAttrs());
}

// result = string.replace(str:string, sub:string, new:string, maxreplace?:number):map
AScript_DeclareFunction(replace)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str",			VTYPE_String);
	DeclareArg(env, "sub",			VTYPE_String);
	DeclareArg(env, "replace",		VTYPE_String);
	DeclareArg(env, "maxreplace",	VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
}

AScript_ImplementFunction(replace)
{
	String result = Replace(context.GetString(0),
			context.GetString(1), context.GetString(2),
			context.IsNumber(3)? context.GetInt(3) : -1, context.GetAttrs());
	return Value(env, result.c_str());
}

// str = string.join(strs[]:string, sep:string => "")
AScript_DeclareFunction(join)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "strs", VTYPE_String, OCCUR_Once, true);
	DeclareArg(env, "sep", VTYPE_String, OCCUR_Once, false, new Expr_String(""));
}

AScript_ImplementFunction(join)
{
	const ValueList &valList = context.GetList(0);
	return Value(env, Join(valList, context.GetString(1)).c_str());
}

// string.eachline(str:string, nlines?:number):map:[chop] {block?}
AScript_DeclareFunction(eachline)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "nlines", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(chop));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(eachline)
{
	Object_String *pSelf = context.GetStringObj(0);
	Object_String *pObj = dynamic_cast<Object_String *>(pSelf->IncRef());
	int maxSplit = context.IsNumber(1)? context.GetInt(1) : -1;
	bool includeEOLFlag = !context.IsSet(AScript_Symbol(chop));
	return ReturnIterator(env, sig, context,
				new Object_String::IteratorLine(pObj, maxSplit, includeEOLFlag));
}

// string.split(str:string, sep?:string, maxnumber?:number):map {block?}
AScript_DeclareFunction(split)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "sep", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareArg(env, "maxnumber", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(split)
{
	Object_String *pSelf = context.GetStringObj(0);
	Object_String *pObj = dynamic_cast<Object_String *>(pSelf->IncRef());
	const char *str = pSelf->GetString();
	int maxSplit = context.IsNumber(2)? context.GetInt(2) : -1;
	Iterator *pIterator = NULL;
	if (context.IsString(1)) {
		const char *sep = context.GetString(1);
		bool ignoreCaseFlag = context.IsSet(AScript_Symbol(icase));
		pIterator = new Object_String::IteratorSplit(
										pObj, sep, maxSplit, ignoreCaseFlag);
	} else {
		pIterator = new Object_String::IteratorEach(pObj, maxSplit);
	}
	return ReturnIterator(env, sig, context, pIterator);
}

// Module entry
AScript_ModuleEntry()
{
	// function assignment
	AScript_AssignFunction(len);
	AScript_AssignFunction(capitalize);
	AScript_AssignFunction(lower);
	AScript_AssignFunction(upper);
	AScript_AssignFunction(strip);
	AScript_AssignFunction(align);
	AScript_AssignFunction(left);
	AScript_AssignFunction(right);
	AScript_AssignFunction(mid);
	AScript_AssignFunction(join);
	AScript_AssignFunction(split);
	AScript_AssignFunction(eachline);
	AScript_AssignFunction(find);
	AScript_AssignFunction(replace);
}

AScript_ModuleTerminate()
{
}

AScript_EndModule(string)

AScript_RegisterModule(string)
