//
// Object_File
//

#include "Object_File.h"
#include "Object_FileStat.h"
#include "Object_Bytes.h"

namespace AScript {

AScript_DeclarePrivSymbol(set);
AScript_DeclarePrivSymbol(cur);
AScript_DeclarePrivSymbol(end);

//-----------------------------------------------------------------------------
// Object_File
//-----------------------------------------------------------------------------
bool Object_File::IsFile() const { return true; }

Object *Object_File::Clone() const
{
	return NULL;
	//return new Object_File(*this);
}

bool Object_File::IsWritable(Signal sig)
{
	if (_file.IsWritable()) return true;
	sig.SetError(ERR_IOError, "%s has not been opened for write",
									ToString(sig, false).c_str());
	return false;
}

bool Object_File::IsReadable(Signal sig)
{
	if (_file.IsReadable()) return true;
	sig.SetError(ERR_IOError, "%s has not been opened for read",
									ToString(sig, false).c_str());
	return false;
}

Iterator *Object_File::CreateIterator(Signal sig)
{
	return new IteratorLine(dynamic_cast<Object_File *>(IncRef()), -1, true);
}

String Object_File::ToString(Signal sig, bool exprFlag)
{
	String str;
	str += "<file:";
	str += _file.GetFileName();
	str += ":";
	if (_file.IsReadable()) str += "R";
	if (_file.IsWritable()) str += "W";
	str += ":";
	str += _file.GetEncoding();
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// Object_File::IteratorLine
//-----------------------------------------------------------------------------
Object_File::IteratorLine::~IteratorLine()
{
	Object::Delete(_pObj);
}

bool Object_File::IteratorLine::DoNext(Signal sig, Value &value)
{
	File &file = _pObj->GetFile();
	String str;
	if (_nLines == _nLinesMax) return false;
	int ch = file.GetChar();
	if (ch < 0) return false;
	for ( ; ch >= 0; ch = file.GetChar()) {
		if (ch == '\n') {
			if (_includeEOLFlag) str += '\n';
			break;
		}
		str += ch;
	}
	_nLines++;
	value = Value(*_pObj, str.c_str());
	return true;
}

String Object_File::IteratorLine::ToString(Signal sig) const
{
	return String("<iterator:file#readlines>");
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_File
//-----------------------------------------------------------------------------
// file#close()
AScript_DeclareMethod(File, close)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(File, close)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	const ValueList &valListArg = context.GetArgs();
	File &file = pSelf->GetFile();
	file.Close();
	return Value::Null;
}

AScript_DeclareMethod(File, seek)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "origin", VTYPE_Symbol, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(File, seek)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	int origin = SEEK_SET;
	if (context.GetValue(1).IsSymbol()) {
		const Symbol *pSymbol = context.GetSymbol(1);
		origin =
			(pSymbol->IsIdentical(AScript_PrivSymbol(set)))? SEEK_SET :
			(pSymbol->IsIdentical(AScript_PrivSymbol(cur)))? SEEK_CUR :
			(pSymbol->IsIdentical(AScript_PrivSymbol(end)))? SEEK_END : SEEK_SET;
	}
	int rtn = pSelf->GetFile().Seek(context.GetLong(0), origin); // fseek()
	return Value::Null;
}

// file#setencoding(encoding:string, eol_flag?:boolean)
AScript_DeclareMethod(File, setencoding)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "encoding", VTYPE_String);
	DeclareArg(env, "eol_flag", VTYPE_Boolean, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(File, setencoding)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	const char *encoding = context.GetString(0);
	bool processEOLFlag = context.IsBoolean(1)? context.GetBoolean(1) : true;
	if (!pSelf->GetFile().InstallCodec(encoding, processEOLFlag)) {
		sig.SetError(ERR_CodecError, "unsupported encoding '%s'", encoding);
		return Value::Null;
	}
	return Value::Null;
}

// file#read(len?:number)
AScript_DeclareMethod(File, read)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "len", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(File, read)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsReadable(sig)) return Value::Null;
	File &file = pSelf->GetFile();
	Value result;
	if (context.IsNumber(0)) {
		size_t len = static_cast<size_t>(context.GetNumber(0));
		char *buff = new char [len];
		size_t lenRead = file.Read(buff, len);
		if (lenRead == 0) {
			delete [] buff;
			return Value::Null;
		}
		result.InitAsBytes(env, Bytes(buff, lenRead));
		delete [] buff;
	} else {
		const int lenSeg = 4096;
		Object_Bytes *pObjBytes = result.InitAsBytes(env);
		Bytes &buff = pObjBytes->GetBytes();
		char *buffSeg = new char [lenSeg];
		size_t lenRead = file.Read(buffSeg, lenSeg);
		if (lenRead == 0) {
			delete [] buffSeg;
			return Value::Null;
		}
		do {
			buff.append(buffSeg, lenRead);
		} while ((lenRead = file.Read(buffSeg, lenSeg)) > 0);
		delete [] buffSeg;
	}
	return result;
}

// file#write(buff:bytes):map
AScript_DeclareMethod(File, write)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "buff", VTYPE_Bytes);
}

AScript_ImplementMethod(File, write)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsWritable(sig)) return Value::Null;
	File &file = pSelf->GetFile();
	const Bytes &bytes = context.GetBytes(0);
	file.Write(bytes.c_str(), bytes.size());
	return Value::Null;
}

// file#readline():[chop]
AScript_DeclareMethod(File, readline)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(chop));
}

AScript_ImplementMethod(File, readline)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsReadable(sig)) return Value::Null;
	int cntChars = 4096;	// tentative
	bool includeEOLFlag = !context.IsSet(AScript_Symbol(chop));
	File &file = pSelf->GetFile();
	String str;
	while (cntChars-- > 0) {
		int ch = file.GetChar();
		if (ch < 0) break;
		if (ch == '\n') {
			if (includeEOLFlag) str += '\n';
			return Value(env, str.c_str());
		}
		str += ch;
	}
	if (str.empty()) return Value::Null;
	return Value(env, str.c_str());
}

// file#readlines(nlines?:number):[chop] {block?}
// conrresponding to string#splitlines()
AScript_DeclareMethod(File, readlines)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(chop));
	DeclareArg(env, "nlines", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(File, readlines)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsReadable(sig)) return Value::Null;
	int nLinesMax = context.IsNumber(0)? static_cast<int>(context.GetNumber(0)) : -1;
	bool includeEOLFlag = !context.IsSet(AScript_Symbol(chop));
	return ReturnIterator(env, sig, context, new Object_File::IteratorLine(
		dynamic_cast<Object_File *>(pSelf->IncRef()), nLinesMax, includeEOLFlag));
}

// file#stat()
AScript_DeclareMethod(File, stat)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(File, stat)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	FileStat fileStat;
	if (!pSelf->GetFile().GetFileStatByHandle(sig, fileStat)) {
		return Value::Null;
	}
	Object_FileStat *pObj = new Object_FileStat(env.LookupClass(VTYPE_FileStat), fileStat);
	return Value(pObj, VTYPE_FileStat);
}

// file#print(values*):map:void
AScript_DeclareMethod(File, print)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementMethod(File, print)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsWritable(sig)) return Value::Null;
	File &file = pSelf->GetFile();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		String str(pValue->ToString(sig, false));
		if (sig.IsSignalled()) break;
		file.PutString(str.c_str());
	}
	return Value::Null;
}

// file#println(values*):map:void
AScript_DeclareMethod(File, println)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementMethod(File, println)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsWritable(sig)) return Value::Null;
	File &file = pSelf->GetFile();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		String str(pValue->ToString(sig, false));
		if (sig.IsSignalled()) break;
		file.PutString(str.c_str());
	}
	file.PutString("\n");
	return Value::Null;
}

// file#printf(format, values*):map:void
AScript_DeclareMethod(File, printf)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementMethod(File, printf)
{
	Object_File *pSelf = Object_File::GetSelfObj(context);
	if (!pSelf->IsWritable(sig)) return Value::Null;
	File &file = pSelf->GetFile();
	//File::Formatter(file).Format(sig, context.GetString(0), context.GetList(1));
	file.Printf(sig, context.GetString(0), context.GetList(1));
	return Value::Null;
}

// Assignment
Class_File::Class_File(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_RealizePrivSymbol(set);
	AScript_RealizePrivSymbol(cur);
	AScript_RealizePrivSymbol(end);
	AScript_AssignMethod(File, close);
	AScript_AssignMethod(File, setencoding);
	AScript_AssignMethod(File, seek);
	AScript_AssignMethod(File, read);
	AScript_AssignMethod(File, write);
	AScript_AssignMethod(File, readline);
	AScript_AssignMethod(File, readlines);
	AScript_AssignMethod(File, stat);
	AScript_AssignMethod(File, print);
	AScript_AssignMethod(File, println);
	AScript_AssignMethod(File, printf);
}

Object *Class_File::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object_File((pClass == NULL)? this : pClass);
}

}
