//
// Object_Function
//

#include "Object_Function.h"
#include "Expr.h"
#include "Module.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Function
//-----------------------------------------------------------------------------
bool Object_Function::IsFunction() const { return true; }

Object_Function::~Object_Function()
{
	Function::Delete(_pFunc);
}

Value Object_Function::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	return _pFunc->EvalExpr(env, sig, contextExpr);
}

Value Object_Function::Eval(Environment &env, Signal sig, Context &context) const
{
	return _pFunc->Eval(env, sig, context);
}

Object *Object_Function::Clone() const
{
	return new Object_Function(*this);
}

String Object_Function::GetFullName(Signal sig)
{
	return _pFunc->GetName();
}

String Object_Function::ToString(Signal sig, bool exprFlag)
{
	return _pFunc->ToString();
}

//-----------------------------------------------------------------------------
// Object_WrappedMethod
//-----------------------------------------------------------------------------
bool Object_WrappedMethod::IsWrappedMethod() const { return true; }

Object_WrappedMethod::~Object_WrappedMethod()
{
}

Value Object_WrappedMethod::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	contextExpr.SetSelf(_valueSelf);
	return _pFunc->EvalExpr(env, sig, contextExpr);
}

Value Object_WrappedMethod::Eval(Environment &env, Signal sig, Context &context) const
{
	context.SetSelf(_valueSelf);
	return _pFunc->Eval(env, sig, context);
}

Object *Object_WrappedMethod::Clone() const
{
	return new Object_WrappedMethod(*this);
}

String Object_WrappedMethod::GetFullName(Signal sig)
{
	String str = MakePrefix(sig);
	if (sig.IsSignalled()) return String("");
	str += _pFunc->GetName();
	return str;
}

String Object_WrappedMethod::ToString(Signal sig, bool exprFlag)
{
	String str = MakePrefix(sig);
	if (sig.IsSignalled()) return String("");
	str += _pFunc->ToString();
	return str;
}

String Object_WrappedMethod::MakePrefix(Signal sig) const
{
	String str;
	const ObjectBase *pObj = _valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) return str;
	if (pObj->IsModule()) {
		str += dynamic_cast<const Module *>(pObj)->GetName();
		str += ".";
	} else if (pObj->IsClass()) {
		str += dynamic_cast<const Class *>(pObj)->GetName();
		str += (_pFunc->GetType() == FUNCTYPE_Instance)? "#" : ".";
	} else if (pObj->IsObject()) {
		str += dynamic_cast<const Object *>(pObj)->GetClass()->GetName();
		str += "#";
	}
	return str;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Function
//-----------------------------------------------------------------------------
// function#name()
AScript_DeclareMethod(Function, name)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	SetHelp("Returns a name of the function object as a string.");
}

AScript_ImplementMethod(Function, name)
{
	Object_Function *pSelf = Object_Function::GetSelfObj(context);
	return Value(env, pSelf->GetName().c_str());
}

// function#fullname()
AScript_DeclareMethod(Function, fullname)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	SetHelp(
	"Returns a full name of the function object as a string, which contains a module\n"
	"or class name that it belongs to.");
}

AScript_ImplementMethod(Function, fullname)
{
	Object_Function *pSelf = Object_Function::GetSelfObj(context);
	String str = pSelf->GetFullName(sig);
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, str.c_str());
}

// function#help(help?:string)
AScript_DeclareMethod(Function, help)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "help", VTYPE_String, OCCUR_ZeroOrOnce);
	SetHelp(
	"If argument help is specified, it sets a help string of the function object.\n"
	"Otherwise, it returns a help string already defined.");
}

AScript_ImplementMethod(Function, help)
{
	Object_Function *pSelf = Object_Function::GetSelfObj(context);
	Function *pFunc = pSelf->GetFunction();
	if (context.IsString(0)) {
		pFunc->SetHelp(context.GetString(0));
		return Value::Null;
	} else {
		return Value(env, pFunc->GetHelp());
	}
}

// function#expr()
AScript_DeclareMethod(Function, expr)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	SetHelp(
	"Returns an expression object that is associated with the function object.");
}

AScript_ImplementMethod(Function, expr)
{
	Object *pObj = context.GetSelfObj();
	Value value;
	const Function *pFunc = dynamic_cast<Object_Function *>(pObj)->GetFunction();
	if (!pFunc->IsCustom()) {
		sig.SetError(ERR_ValueError, "function '%s' is a built-in function", pFunc->GetName());
		return Value::Null;
	}
	const FunctionCustom *pFuncCustom = dynamic_cast<const FunctionCustom *>(pFunc);
	value.InitAsExpr(env, pFuncCustom->GetExprBody()->IncRef());
	return value;
}

// Assignment
Class_Function::Class_Function(Environment &env) :
									Class(env.LookupClass(VTYPE_Object))
{
	// don't assign functions here, because the function objects themselves
	// shall be constructed here!!!!!!! instead, they must be assigned in
	// Prepare() funtion below.
}

void Class_Function::Prepare()
{
	AScript_AssignMethod(Function, name);
	AScript_AssignMethod(Function, fullname);
	AScript_AssignMethod(Function, help);
	AScript_AssignMethod(Function, expr);
}

Object *Class_Function::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
