#include "Declaration.h"
#include "Expr.h"
#include "Environment.h"
#include "Object.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Declaration
//-----------------------------------------------------------------------------
Declaration::Declaration(const Declaration &decl) :
	_pSymbol(decl._pSymbol), _valType(decl._valType),
	_listFlag(decl._listFlag), _occurPattern(decl._occurPattern),
	_pExprDefault((decl._pExprDefault == NULL)? NULL : decl._pExprDefault->IncRef())
{
}

Declaration::Declaration(const Symbol *pSymbol, ValueType valType,
	OccurPattern occurPattern, bool listFlag, Expr *pExprDefault) :
	_pSymbol(pSymbol), _valType(valType),
	_listFlag(listFlag), _occurPattern(occurPattern), _pExprDefault(pExprDefault)
{
}

Declaration::~Declaration()
{
	Expr::Delete(_pExprDefault);
}

Declaration *Declaration::Create(Environment &env, Signal sig, const Expr *pExpr)
{
	bool listFlag = false;
	OccurPattern occurPattern = OCCUR_Once;
	ValueType valType = VTYPE_Any;
	Expr *pExprDefault = NULL;
	if (pExpr->IsDictAssign()) {
		const Expr_DictAssign *pExprDictAssign =
								dynamic_cast<const Expr_DictAssign *>(pExpr);
		pExpr = pExprDictAssign->GetLeft();
		pExprDefault = pExprDictAssign->GetRight()->IncRef();
	}
	if (pExpr->IsSuffix()) {
		const Expr_Suffix *pExprSuffix = dynamic_cast<const Expr_Suffix *>(pExpr);
		pExpr = pExprSuffix->GetChild();
		occurPattern = pExprSuffix->GetOccurPattern();
		if (occurPattern == OCCUR_Invalid) {
			sig.SetError(ERR_SyntaxError, "invalid argument expression");
			return NULL;
		}
		if (pExprDefault) {
			sig.SetError(ERR_SyntaxError, "optional arguments cannot take default value");
			return NULL;
		}
	}
	if (pExpr->IsQuote()) {
		pExpr = pExpr->Unquote();
		valType = VTYPE_Quote;
	}
	if (pExpr->IsIndexer()) {
		pExpr = dynamic_cast<const Expr_Indexer *>(pExpr)->GetCar();
		listFlag = true;
	}
	if (!pExpr->IsSymbol()) {
		sig.SetError(ERR_SyntaxError, "invalid argument expression");
		return NULL;
	}
	const Expr_Symbol *pExprSym = dynamic_cast<const Expr_Symbol *>(pExpr);
	const SymbolList &attrFront = pExprSym->GetAttrFront();
	if (!attrFront.empty()) {
		if (valType != VTYPE_Any) {
			sig.SetError(ERR_SyntaxError, "incompatible type declaration");
			return NULL;
		}
		const ValueTypeInfo *pValueTypeInfo = env.LookupValueType(attrFront);
		if (pValueTypeInfo == NULL) {
			sig.SetError(ERR_TypeError, "invalid type name");
			return NULL;
		}
		valType = pValueTypeInfo->GetValueType();
	}
	return new Declaration(pExprSym->GetSymbol(), valType,
									occurPattern, listFlag, pExprDefault);
}

String Declaration::ToString() const
{
	String str;
	if (_valType == VTYPE_Quote) {
		str += "`";
	}
	str += _pSymbol->GetName();
	if (_listFlag) str += "[]";
	str += GetOccurPatternSymbol(_occurPattern)->GetName();
	if (_valType != VTYPE_Invalid &&
					_valType != VTYPE_Any && _valType != VTYPE_Quote) {
		str += ":";
		str += GetValueTypeSymbol(_valType)->GetName();
	}
	if (_pExprDefault != NULL) {
		str += " ";
		str += "=>";
		str += " ";
		str += _pExprDefault->ToString();
	}
	return str;
}

//-----------------------------------------------------------------------------
// DeclarationList
//-----------------------------------------------------------------------------
const DeclarationList DeclarationList::Null;

DeclarationList::~DeclarationList()
{
}

bool DeclarationList::IsVariableLength() const
{
	foreach_const (DeclarationList, ppDecl, *this) {
		if ((*ppDecl)->IsVariableLength()) return true;
	}
	return false;
}

bool DeclarationList::IsApplicable(const ValueList &valList) const
{
	ValueList::const_iterator pValue = valList.begin();
	const_iterator ppDecl = begin();
	for ( ; pValue != valList.end() && ppDecl != end(); pValue++) {
		const Declaration *pDecl = *ppDecl;
		if (!pDecl->IsApplicable(*pValue)) return false;
		if (!pDecl->IsVariableLength()) ppDecl++;
	}
	return true;
}

void DeclarationList::SetAsLoose()
{
	foreach (DeclarationList, ppDecl, *this) {
		Declaration *pDecl = *ppDecl;
		if (pDecl->GetOccurPattern() == OCCUR_Once) {
			pDecl->SetOccurPattern(OCCUR_ZeroOrOnce);
		} else if (pDecl->GetOccurPattern() == OCCUR_OnceOrMore) {
			pDecl->SetOccurPattern(OCCUR_ZeroOrMore);
		}
	}
}

String DeclarationList::ToString() const
{
	String str;
	foreach_const (DeclarationList, ppDecl, *this) {
		if (ppDecl != begin()) str += ", ";
		str += (*ppDecl)->ToString();
	}
	return str;
}

//-----------------------------------------------------------------------------
// DeclarationOwner
//-----------------------------------------------------------------------------
DeclarationOwner::DeclarationOwner(const DeclarationOwner &declOwner)
{
	foreach_const (DeclarationList, ppDecl, declOwner) {
		push_back((*ppDecl)->Clone());
	}
}

DeclarationOwner::~DeclarationOwner()
{
	foreach (DeclarationOwner, ppDecl, *this) {
		Declaration::Delete(*ppDecl);
	}
}

void DeclarationOwner::operator=(const DeclarationOwner &declOwner)
{
	foreach_const (DeclarationList, ppDecl, declOwner) {
		push_back((*ppDecl)->Clone());
	}
}

}
