#include "ascript.h"

#if defined(HAVE_LIBREADLINE)
#include <readline/readline.h>
#include <readline/history.h>
#endif

namespace AScript {

void PrintVersion(FILE *fp);
void PrintHelp(FILE *fp);
void PrintError(File *pConsole, const Signal &sig);

void ReadEvalPrintLoop(Environment &env, Signal sig, Codec *pCodec);
void AcceptKeyChar(Environment &env, Signal sig, File *pConsole,
									Codec *pCodec, Parser &parser, int ch);

//-----------------------------------------------------------------------------
// Main entry
//-----------------------------------------------------------------------------
int Main(int argc, const char *argv[])
{
	Signal sig;
	static const Option::Info optInfoTbl[] = {
		{ "help",			'h', false	},
		{ "interactive",	'i', false	},
		{ "command",		'c', true	},
		{ "quiet",			'q', false	},
		{ "printcmdline",	'P', false	},
		{ "version",		'v', false	},
	};
	Option opt(optInfoTbl, NUMBEROF(optInfoTbl));
	bool rtn = opt.Parse(sig, argc, argv);
	if (!rtn) {
		::fprintf(stderr, "%s\n", sig.GetErrString().c_str());
		return 1;
	}
	if (opt.IsSet("version")) {
		PrintVersion(stderr);
		return 0;
	}
	if (opt.IsSet("help")) {
		PrintVersion(stderr);
		PrintHelp(stderr);
		return 0;
	}
	if (opt.IsSet("printcmdline") && argc > 1) {
		for (int i = 1; i < argc; i++) {
			::fprintf(stderr, (i == 1)? "%s" : " %s", argv[i]);
		}
		::fprintf(stderr, "\n");
	}
	SFMT_init_gen_rand(1234);	// initialize random generator SFMT
	EnvironmentRoot env(sig, argc, argv);
	File *pConsole = env.GetConsole(AScript_Symbol(stdout));
	bool interactiveFlag = true;
	if (opt.IsSet("command")) {
		Expr *pExpr = Parser().ParseString(env, sig, opt.GetString("command", ""));
		if (sig.IsSignalled()) {
			PrintError(pConsole, sig);
			return 1;
		}
		Value result = pExpr->Exec(env, sig);
		if (sig.IsSignalled()) {
			PrintError(pConsole, sig);
			return 1;
		} else if (result.IsValid()) {
			pConsole->Println(result.ToString(sig).c_str());
		}
		interactiveFlag = false;
	}
	Expr *pExprRoot = NULL;
	if (argc >= 2) {
		pExprRoot = Parser().ParseFile(env, sig, argv[1]);
		if (sig.IsSignalled()) {
			PrintError(pConsole, sig);
			return 1;
		}
		pExprRoot->Exec(env, sig);
		if (sig.IsSignalled()) {
			PrintError(pConsole, sig);
			sig.ClearSignal();
		}
		interactiveFlag = false;
	}
	if (interactiveFlag || opt.IsSet("interactive")) {
		PrintVersion(stdout);
		env.SetEchoFlag(true);
		std::auto_ptr<Codec> pCodecReader(new Codec_None());
		CodecFactory *pCodecFactory = CodecFactory::Lookup("cp932");
		if (pCodecFactory != NULL) {
			pCodecReader.reset(pCodecFactory->CreateDecoder(true));
		}
		ReadEvalPrintLoop(env, sig, pCodecReader.get());
	}
	Expr::Delete(pExprRoot);
	return 0;
}

void PrintVersion(FILE *fp)
{
	::fprintf(fp, "AScript %s copyright (c) 2010- Dan-san\n", GetVersion());
}

void PrintHelp(FILE *fp)
{
	::fprintf(fp,
"usage: ascript [option] [file] [arg] ...\n"
"available options:\n"
"-h      print this help\n"
"-i      interactive mode after running script file if specified\n"
"-c cmd  execute program from command line\n"
"-v      print version string\n"
	);
}

void PrintError(File *pConsole, const Signal &sig)
{
	pConsole->Println(sig.GetErrString().c_str());
}

#if defined(HAVE_LIBREADLINE)
void ReadEvalPrintLoop(Environment &env, Signal sig, Codec *pCodec)
{
	File *pConsole = env.GetConsole(AScript_Symbol(stdout));
	Parser parser;
	char *lineBuff = NULL;
	while (lineBuff = readline(env.GetPrompt(parser.IsContinued()))) {
		for (char *p = lineBuff; ; p++) {
			int ch = (*p == '\0')? '\n' : *p;
			AcceptKeyChar(env, sig, pConsole, pCodec, parser, ch);
			if (ch == '\n') break;
		}
		if (lineBuff[0] != '\0') {
			add_history(lineBuff);
		}
		free(lineBuff);
	}
}
#else
void ReadEvalPrintLoop(Environment &env, Signal sig, Codec *pCodec)
{
	File *pConsole = env.GetConsole(AScript_Symbol(stdout));
	Parser parser;
	pConsole->Print(env.GetPrompt(false));
	for (;;) {
		int ch = ::fgetc(stdin);
		AcceptKeyChar(env, sig, pConsole, pCodec, parser, ch);
		if (ch < 0) break;
		if (ch == '\n') {
			pConsole->Print(env.GetPrompt(parser.IsContinued()));
		}
	}
}
#endif

void AcceptKeyChar(Environment &env, Signal sig, File *pConsole,
								Codec *pCodecReader, Parser &parser, int ch)
{
	char chConv = '\0';
	Codec::Result rtn =
			pCodecReader->FeedChar(static_cast<unsigned char>(ch), chConv);
	if (rtn != Codec::RESULT_Complete) return;
	do {
		Expr *pExpr = parser.ParseChar(env, sig, chConv);
		if (sig.IsSignalled()) {
			if (sig.GetSignalType() == SIGTYPE_DetectEncoding) {
				sig.ClearSignal();
			} else {
				PrintError(pConsole, sig);
				sig.ClearSignal();
				parser.Reset();
			}
		} else if (pExpr != NULL) {
			Value result = pExpr->Exec(env, sig);
			if (sig.IsSignalled()) {
				PrintError(pConsole, sig);
				sig.ClearSignal();
			} else if (!env.GetEchoFlag()) {
				// nothing to do
			} else if (result.IsValid()) {
				pConsole->Println(result.ToString(sig).c_str());
			}
			delete pExpr;
		}
	} while (pCodecReader->FollowChar(chConv));
}

}

int main(int argc, const char *argv[])
{
	return AScript::Main(argc, argv);
}
