//-----------------------------------------------------------------------------
// AScript os module
//-----------------------------------------------------------------------------
#include "Module.h"
#include "Object_FileStat.h"
#include "OAL.h"

AScript_BeginModule(os)

//-----------------------------------------------------------------------------
// AScript module functions: os.path
//-----------------------------------------------------------------------------
AScript_BeginSubModule(path)

// os.path.match(pattern:string, filename:string):map
AScript_DeclareFunction(match)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareArg(env, "filename", VTYPE_String);
	DeclareAttr(AScript_Symbol(icase));
}

AScript_ImplementFunction(match)
{
	const char *pattern = context.GetString(0);
	const char *fileName = context.GetString(1);
	bool ignoreCaseFlag = context.IsSet(AScript_Symbol(icase));
	return Value(File::IsMatchName(pattern, fileName, ignoreCaseFlag));
}

// os.path.split(pathname:string):map
AScript_DeclareFunction(split)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(split)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	valList.push_back(Value(env, File::ExtractDirName(context.GetString(0)).c_str()));
	valList.push_back(Value(env, File::ExtractBaseName(context.GetString(0)).c_str()));
	return result;
}

// os.path.basename(pathname:string):map
AScript_DeclareFunction(basename)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(basename)
{
	return Value(env, File::ExtractBaseName(context.GetString(0)).c_str());
}

// os.path.dirname():map
AScript_DeclareFunction(dirname)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(dirname)
{
	return Value(env, File::ExtractDirName(context.GetString(0)).c_str());
}

// os.path.abspath(filename:string):map
AScript_DeclareFunction(absname)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "filename", VTYPE_String);
}

AScript_ImplementFunction(absname)
{
	return Value(env, File::MakeAbsPath(context.GetString(0)).c_str());
}

// os.path.join(path+:string):map
AScript_DeclareFunction(join)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "path", VTYPE_String, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(join)
{
	String str;
	const ValueList &valList = context.GetList(0);
	foreach_const (ValueList, pValue, valList) {
		if (pValue != valList.begin()) str += File::Separator;
		str += pValue->GetString();
	}
	return Value(env, str.c_str());
}

// os.path.exists(pathname:string):map
AScript_DeclareFunction(exists)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(exists)
{
	return Value(File::IsExist(context.GetString(0)));
}

// os.path.splitext(pathname:string):map
AScript_DeclareFunction(splitext)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(splitext)
{
	const char *pathName = context.GetString(0);
	const char *p = File::ExtractExtName(pathName);
	Value result;
	ValueList &valList = result.InitAsList(env);
	size_t lenLeft = p - pathName;
	if (p > pathName && *(p - 1) == '.') lenLeft--;
	valList.push_back(Value(env, pathName, lenLeft));
	valList.push_back(Value(env, p));
	return result;
}

AScript_ModuleEntry()
{
	AScript_AssignFunction(match);
	AScript_AssignFunction(split);
	AScript_AssignFunction(dirname);
	AScript_AssignFunction(basename);
	AScript_AssignFunction(absname);
	AScript_AssignFunction(join);
	AScript_AssignFunction(exists);
	AScript_AssignFunction(splitext);
}

AScript_EndSubModule(path)

//-----------------------------------------------------------------------------
// AScript module functions: os
//-----------------------------------------------------------------------------
// os.listdir(pathname?:string):map:flat {block?}
class Iterator_listdir : public Iterator {
private:
	Environment _env;
	std::auto_ptr<FileLister> _pFileLister;
public:
	inline Iterator_listdir(Environment &env, const char *dirName) :
		Iterator(false), _env(env), _pFileLister(FileLister::CreateInstance(dirName)) {}
	~Iterator_listdir();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

Iterator_listdir::~Iterator_listdir()
{
}

bool Iterator_listdir::DoNext(Signal sig, Value &value)
{
	FileEntry entry;
	if (!_pFileLister->Next(entry)) return false;
	String pathName(_pFileLister->GetDirName());
	if (!pathName.empty()) pathName += File::Separator;
	pathName += entry.GetFileName();
	value = Value(_env, pathName.c_str());
	return true;
}

String Iterator_listdir::ToString(Signal sig) const
{
	return String("<iterator:os.listdir>");
}

AScript_DeclareFunction(listdir)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_On);
	DeclareArg(env, "pathname", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(listdir)
{
	return ReturnIterator(env, sig, context, new Iterator_listdir(env,
						context.IsString(0)? context.GetString(0) : ""));
}

// os.glob(pattern:string):map:flat {block?}
class Iterator_glob : public Iterator {
public:
	typedef std::deque<size_t> DepthDeque;
private:
	Environment _env;
	bool _ignoreCaseFlag;
	std::auto_ptr<FileLister> _pFileLister;
	size_t _depth;
	StringDeque _dirNameQue;
	DepthDeque _depthQue;
	StringList _patternSegs;
public:
	Iterator_glob(Environment &env, const char *pattern, bool ignoreCaseFlag);
	~Iterator_glob();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

Iterator_glob::Iterator_glob(Environment &env,
							const char *pattern, bool ignoreCaseFlag) :
			Iterator(false), _env(env), _ignoreCaseFlag(ignoreCaseFlag),
			_pFileLister(FileLister::CreateInstance("")), _depth(0)
{
	File::Split(pattern, _patternSegs);
}

Iterator_glob::~Iterator_glob()
{
}

bool Iterator_glob::DoNext(Signal sig, Value &value)
{
	FileEntry entry;
	for (;;) {
		while (!_pFileLister->Next(entry)) {
			if (_dirNameQue.empty()) return false;
			String dirName = _dirNameQue.front();
			_depth = _depthQue.front();
			_dirNameQue.pop_front();
			_depthQue.pop_front();
			_pFileLister.reset(FileLister::CreateInstance(dirName.c_str()));
		}
		if (!File::IsMatchName(_patternSegs[_depth].c_str(),
							entry.GetFileName(), _ignoreCaseFlag)) continue;
		String pathName(_pFileLister->GetDirName());
		if (!pathName.empty()) pathName += File::Separator;
		pathName += entry.GetFileName();
		if (_depth + 1 < _patternSegs.size()) {
			if (entry.IsDirectory()) {
				_dirNameQue.push_back(pathName);
				_depthQue.push_back(_depth + 1);
			}
		} else {
			value = Value(_env, pathName.c_str());
			break;
		}
	}
	return true;
}

String Iterator_glob::ToString(Signal sig) const
{
	return String("<iterator:os.glob>");
}

AScript_DeclareFunction(glob)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_On);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
}

AScript_ImplementFunction(glob)
{
	bool ignoreCaseFlag = context.IsSet(AScript_Symbol(icase));
	return ReturnIterator(env, sig, context,
				new Iterator_glob(env, context.GetString(0), ignoreCaseFlag));
}

// os.walk(pathname?:string):map:flat {block?}
class Iterator_walk : public Iterator {
public:
private:
	Environment _env;
	std::auto_ptr<FileLister> _pFileLister;
	StringDeque _dirNameQue;
public:
	inline Iterator_walk(Environment &env, const char *dirName) :
		Iterator(false), _env(env), _pFileLister(FileLister::CreateInstance(dirName)) {}
	~Iterator_walk();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

Iterator_walk::~Iterator_walk()
{
}

bool Iterator_walk::DoNext(Signal sig, Value &value)
{
	FileEntry entry;
	while (!_pFileLister->Next(entry)) {
		if (_dirNameQue.empty()) return false;
		String dirName = _dirNameQue.front();
		_dirNameQue.pop_front();
		_pFileLister.reset(FileLister::CreateInstance(dirName.c_str()));
	}
	String pathName(_pFileLister->GetDirName());
	if (!pathName.empty()) pathName += File::Separator;
	pathName += entry.GetFileName();
	if (entry.IsDirectory()) {
		_dirNameQue.push_back(pathName);
	}
	value = Value(_env, pathName.c_str());
	return true;
}

String Iterator_walk::ToString(Signal sig) const
{
	return String("<iterator:os.walk>");
}

AScript_DeclareFunction(walk)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_On);
	DeclareArg(env, "pathname", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(walk)
{
	return ReturnIterator(env, sig, context, new Iterator_walk(env,
						context.IsString(0)? context.GetString(0) : ""));
}

// os.stat(pathname:string):map
AScript_DeclareFunction(stat)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(stat)
{
	FileStat fileStat;
	if (!File::GetFileStat(sig, context.GetString(0), fileStat)) {
		return Value::Null;
	}
	Object_FileStat *pObj = new Object_FileStat(env.LookupClass(VTYPE_FileStat), fileStat);
	return Value(pObj, VTYPE_FileStat);
}

// os.rename(src:string, dst:string)
AScript_DeclareFunction(rename)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "src", VTYPE_String);
	DeclareArg(env, "dst", VTYPE_String);
}

AScript_ImplementFunction(rename)
{
	OAL::rename(context.GetString(0), context.GetString(1));
	return Value::Null;
}

// os.remove(pathname:string):map
AScript_DeclareFunction(remove)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(remove)
{
	OAL::remove(context.GetString(0));
	return Value::Null;
}

// os.mkdir(pathname:string):map
AScript_DeclareFunction(mkdir)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
}

AScript_ImplementFunction(mkdir)
{
	OAL::mkdir(context.GetString(0), false);
	return Value::Null;
}

// os.exec(pathname:string, args*:string)
AScript_DeclareFunction(exec)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "pathname", VTYPE_String);
	DeclareArg(env, "args", VTYPE_String, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(exec)
{
	const char *pathName = context.GetString(0);
	const char **args = new const char *[context.GetList(1).size() + 2];
	size_t i = 0;
	String baseName = File::ExtractBaseName(pathName);
	args[i++] = baseName.c_str();
	foreach_const (ValueList, pValue, context.GetList(1)) {
		args[i++] = pValue->GetString();
	}
	args[i++] = NULL;
	OAL::exec(context.GetString(0), args);
	delete [] args;
	return Value::Null;
}

// Module entry
AScript_ModuleEntry()
{
	// value assignment
	do {
		char str[2];
		str[0] = File::Separator;
		str[1] = '\0';
		AScript_AssignValue(sep, Value(env, str));
	} while (0);
	// function assignment
	AScript_AssignFunction(listdir);
	AScript_AssignFunction(glob);
	AScript_AssignFunction(walk);
	AScript_AssignFunction(stat);
	AScript_AssignFunction(rename);
	AScript_AssignFunction(remove);
	AScript_AssignFunction(mkdir);
	AScript_AssignFunction(exec);
	// sub-module assignment
	AScript_AssignSubModule(path);
}

AScript_ModuleTerminate()
{
}

AScript_EndModule(os)

AScript_RegisterModule(os)
