//-----------------------------------------------------------------------------
// AScript time module
//-----------------------------------------------------------------------------
#include "Module.h"
#include "Object_DateTime.h"

AScript_BeginModule(time)

AScript_DeclarePrivSymbol(utc);

static DateTime GetCurDateTime(bool utcFlag);

//-----------------------------------------------------------------------------
// AScript module functions: time
//-----------------------------------------------------------------------------
// time.sleep(secs)
AScript_DeclareFunction(sleep)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "secs", VTYPE_Number);
}

#if defined(HAVE_WINDOWS_H)
AScript_ImplementFunction(sleep)
{
	::Sleep(static_cast<DWORD>(context.GetNumber(0) * 1000));
	return Value::Null;
}
#else
AScript_ImplementFunction(sleep)
{
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = static_cast<int>(context.GetNumber(0) * 1000000);
	::select(0, NULL, NULL, NULL, &tv);
	return Value::Null;
}
#endif

// time.clock()
AScript_DeclareFunction(clock)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

#if defined(HAVE_WINDOWS_H)
AScript_ImplementFunction(clock)
{
	LARGE_INTEGER freq, counter;
	if (::QueryPerformanceFrequency(&freq) && ::QueryPerformanceCounter(&counter)) {
		return Value(static_cast<Number>(counter.QuadPart) / freq.QuadPart);
	} else {
		return Value(static_cast<Number>(::GetTickCount()) / 1000);
	}
}
#else
AScript_ImplementFunction(clock)
{
	return Value::Null;
}
#endif

// time.monthdays(year:number, month:number):map
AScript_DeclareFunction(monthdays)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "year", VTYPE_Number);
	DeclareArg(env, "month", VTYPE_Number);
}

AScript_ImplementFunction(monthdays)
{
	int year = context.GetInt(0);
	int month = context.GetInt(1);
	return Value(static_cast<Number>(DateTime::_GetDaysOfMonth(year, month)));
}

// time.weekday(year:number, month:number, day:number):map
AScript_DeclareFunction(weekday)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "year", VTYPE_Number);
	DeclareArg(env, "month", VTYPE_Number);
	DeclareArg(env, "day", VTYPE_Number);
}

AScript_ImplementFunction(weekday)
{
	int year = context.GetInt(0);
	int month = context.GetInt(1);
	int day = context.GetInt(2);
	return Value(static_cast<Number>(DateTime::_GetDayOfWeek(year, month, day)));
}

// time.now():[utc]
AScript_DeclareFunction(now)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_PrivSymbol(utc));
}

AScript_ImplementFunction(now)
{
	return Value(env, GetCurDateTime(context.IsSet(AScript_PrivSymbol(utc))));
}

// time.today():[utc]
AScript_DeclareFunction(today)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_PrivSymbol(utc));
}

AScript_ImplementFunction(today)
{
	DateTime dateTime(GetCurDateTime(context.IsSet(AScript_PrivSymbol(utc))));
	dateTime.ClearTime();
	return Value(env, dateTime);
}

// time.datetime(year:number, month:number, day:number,
//           hour:number => 0, min:number => 0, sec:number => 0, usec:number => 0):map
AScript_DeclareFunction(datetime)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "year", VTYPE_Number);
	DeclareArg(env, "month", VTYPE_Number);
	DeclareArg(env, "day", VTYPE_Number);
	DeclareArg(env, "hour", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "min", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "sec", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "usec", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
}

AScript_ImplementFunction(datetime)
{
	short year = static_cast<short>(context.GetLong(0));
	char month = static_cast<char>(context.GetLong(1));
	char day = static_cast<char>(context.GetLong(2));
	long sec = static_cast<long>(context.GetLong(3) * 3600 +
						context.GetLong(4) * 60 + context.GetLong(5));
	long usec = context.GetLong(6);
	return Value(env, DateTime(year, month, day, sec, usec));
}

// time.time(hour:number, minute:number, sec:number, usec:number => 0):map
AScript_DeclareFunction(time)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "hour", VTYPE_Number);
	DeclareArg(env, "minute", VTYPE_Number);
	DeclareArg(env, "sec", VTYPE_Number);
	DeclareArg(env, "usec", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
}

AScript_ImplementFunction(time)
{
	short year = 2000;
	char month = 1;
	char day = 1;
	long sec = static_cast<long>(context.GetNumber(0) * 3600 +
						context.GetNumber(1) * 60 + context.GetNumber(2));
	long usec = context.GetLong(3);
	return Value(env, DateTime(year, month, day, sec, usec));
}

// time.delta(days:number => 0, secs:number => 0, usecs:number => 0):map
AScript_DeclareFunction(delta)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "days", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "secs", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "usecs", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "msecs", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "mins", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "hours", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	DeclareArg(env, "weeks", VTYPE_Number, OCCUR_Once, false, new Expr_Value(0));
	SetHelp(
	"Returns a timedelta instance with specified values. The instance actually\n"
	"holds properties of days, secs and usecs. Other values are added to them like follow\n"
	"- adds (msecs * 1000) to usecs\n"
	"- adds (mins * 60) to secs\n"
	"- adds (hours * 3600) to secs\n"
	"- adds (weeks * 7) to days\n");
}

AScript_ImplementFunction(delta)
{
	long days = static_cast<long>(context.GetNumber(0));
	long secs = static_cast<long>(context.GetNumber(1));
	long usecs = static_cast<long>(context.GetNumber(2));
	usecs += static_cast<long>(context.GetNumber(3) * 1000);
	secs += static_cast<long>(context.GetNumber(4) * 60);
	secs += static_cast<long>(context.GetNumber(5) * 3600);
	days += static_cast<long>(context.GetNumber(6) * 7);
	return Value(env, TimeDelta(days, secs, usecs));
}

// time.isleap(year:number):map
AScript_DeclareFunction(isleap)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "year", VTYPE_Number);
}

AScript_ImplementFunction(isleap)
{
	return Value(DateTime::_IsLeapYear(context.GetShort(0)));
}

AScript_ModuleEntry()
{
	AScript_RealizePrivSymbol(utc);
	// value assignment
	AScript_AssignValue(Sunday,		Value(0));
	AScript_AssignValue(Monday,		Value(1));
	AScript_AssignValue(Tuesday,	Value(2));
	AScript_AssignValue(Wednesday,	Value(3));
	AScript_AssignValue(Thursday,	Value(4));
	AScript_AssignValue(Friday,		Value(5));
	AScript_AssignValue(Saturday,	Value(6));
	// function assignment
	AScript_AssignFunction(sleep);
	AScript_AssignFunction(clock);
	AScript_AssignFunction(monthdays);
	AScript_AssignFunction(weekday);
	AScript_AssignFunction(now);
	AScript_AssignFunction(today);
	AScript_AssignFunction(datetime);
	AScript_AssignFunction(time);
	AScript_AssignFunction(delta);
	AScript_AssignFunction(isleap);
}

AScript_ModuleTerminate()
{
}

DateTime GetCurDateTime(bool utcFlag)
{
#if defined(HAVE_WINDOWS_H)
	SYSTEMTIME st;
	if (utcFlag) {
		::GetSystemTime(&st);
	} else {
		::GetLocalTime(&st);
	}
	return DateTime(st);
#else
	time_t t;
	::time(&t);
	return DateTime(t, !utcFlag);
#endif
}

AScript_EndModule(time)

AScript_RegisterModule(time)
