#include "AScriptShell.h"
#include "Config.h"

AScript_IncludeModule(scribble)

//-----------------------------------------------------------------------------
// AScriptShell
//-----------------------------------------------------------------------------
AScriptShell::AScriptShell(wxWindow *pParent, wxWindowID id, long style) :
		wxTextCtrl(pParent, id, wxT(""), wxDefaultPosition, wxDefaultSize,
								style | wxWANTS_CHARS | wxTE_MULTILINE),
		_threadRunningFlag(false), _console(this), _env(0, NULL)
{
	do { // import(scribble)
		AScript::AScript_Module(scribble)::Import(_env, _sig);
		if (_sig.IsSignalled()) return;
	} while (0);
	_env.SetEchoFlag(true);
	_env.SetConsole(true, &_console);
	_env.SetConsole(false, &_console);
	_console.Println(AScript::GetOpening());
	_console.Print(_env.GetPrompt(_parser.IsContinued()));
	_pLineHistory = _lineHistory.rend();
	MarkLineTop();
	UpdateFont();
	UpdateColor();
}

AScriptShell::~AScriptShell()
{
}

void AScriptShell::UpdateFont()
{
	wxFont font(wxSystemSettings::GetFont(wxSYS_OEM_FIXED_FONT));
	font.SetPointSize(Config::AScriptShell::ReadFontSize());
	SetFont(font);
}

void AScriptShell::UpdateColor()
{
	SetForegroundColour(wxColour(Config::AScriptShell::ReadFontColor()));
	SetBackgroundColour(wxColour(Config::AScriptShell::ReadBackgroundColor()));
}

void AScriptShell::EvalFile(const wxString &fileName)
{
	RunThread(new EvalFileThread(this, fileName));
}

BEGIN_EVENT_TABLE(AScriptShell, wxTextCtrl)
	EVT_KEY_DOWN(AScriptShell::OnKeyDown)
END_EVENT_TABLE()

void AScriptShell::OnKeyDown(wxKeyEvent &event)
{
	int keyCode = event.GetKeyCode();
	if (keyCode == 'D' && event.ControlDown()) {
		if (IsThreadRunning()) {
			_sig.SetSignal(AScript::SIGTYPE_Terminate, AScript::Value::Null);
		}
		return;
	}
	if (event.AltDown()) {
		event.Skip();
	} else if (IsThreadRunning()) {
		// nothing to do
	} else if (keyCode == WXK_RETURN) {
		wxString str = GetRange(_posLineTop, GetLastPosition());
		*this << "\n";
		if (!str.empty() && (_lineHistory.empty() || _lineHistory.back() != str)) {
			_lineHistory.push_back(str);
		}
		str << "\n";
		RunThread(new EvalLineThread(this, str));
	} else if (keyCode == WXK_ESCAPE) {
		Replace(GetLineTopPoint(), GetLastPosition(), wxT(""));
	} else if (keyCode == WXK_HOME) {
		SetInsertionPoint(GetLineTopPoint());
	} else if (keyCode == WXK_END) {
		SetInsertionPoint(GetLastPosition());
	} else if (keyCode == WXK_BACK) {
		if (GetInsertionPoint() > GetLineTopPoint()) event.Skip();
	} else if (keyCode == WXK_LEFT) {
		if (GetInsertionPoint() > GetLineTopPoint()) event.Skip();
	} else if (keyCode == WXK_RIGHT) {
		event.Skip();
	} else if (keyCode == WXK_UP) {
		if (_lineHistory.empty()) {
			// nothing to do
		} else if (_pLineHistory == _lineHistory.rend()) {
			_pLineHistory = _lineHistory.rbegin();
			Replace(GetLineTopPoint(), GetLastPosition(), *_pLineHistory);
		} else if (_pLineHistory + 1 != _lineHistory.rend()) {
			_pLineHistory++;
			Replace(GetLineTopPoint(), GetLastPosition(), *_pLineHistory);
		}
	} else if (keyCode == WXK_DOWN) {
		if (_pLineHistory == _lineHistory.rend()) {
			// nothing to do
		} else if (_pLineHistory == _lineHistory.rbegin()) {
			_pLineHistory = _lineHistory.rend();
			Replace(GetLineTopPoint(), GetLastPosition(), wxT(""));
		} else {
			_pLineHistory--;
			Replace(GetLineTopPoint(), GetLastPosition(), *_pLineHistory);
		}
	} else {
		event.Skip();
	}
}

void AScriptShell::RunThread(wxThread *pThread)
{
	_threadRunningFlag = true;
	pThread->Create();
	pThread->Run();
}

void AScriptShell::NotifyEndOfThread()
{
	_pLineHistory = _lineHistory.rend();
	_console.Print(_env.GetPrompt(_parser.IsContinued()));
	ShowPosition(GetLastPosition());
	MarkLineTop();
	_threadRunningFlag = false;
}

//-----------------------------------------------------------------------------
// AScriptShell::ConsoleEx
//-----------------------------------------------------------------------------
AScriptShell::ConsoleEx::ConsoleEx(AScriptShell *pAScriptShell) :
											_pAScriptShell(pAScriptShell)
{
	InstallCodec("cp932", true);
}

void AScriptShell::ConsoleEx::PutRawChar(char ch)
{
	_str << ch;
}

void AScriptShell::ConsoleEx::FlushConsole()
{
	*_pAScriptShell << _str;
	_str.clear();
}

//-----------------------------------------------------------------------------
// AScriptShell::EvalLineThread
//-----------------------------------------------------------------------------
AScriptShell::EvalLineThread::EvalLineThread(AScriptShell *pAScriptShell, const wxString &str) :
	wxThread(wxTHREAD_DETACHED), _pAScriptShell(pAScriptShell), _str(str)
{
}

wxThread::ExitCode AScriptShell::EvalLineThread::Entry()
{
	AScript::Environment &env = _pAScriptShell->GetEnvironment();
	AScript::Signal &sig = _pAScriptShell->GetSignal();
	AScript::Parser &parser = _pAScriptShell->GetParser();
	foreach_const (wxString, p, _str) {
		parser.EvalConsoleChar(env, sig, *p);
		if (sig.IsSignalled()) break;
	}
	_pAScriptShell->NotifyEndOfThread();
	return 0;
}

//-----------------------------------------------------------------------------
// AScriptShell::EvalFileThread
//-----------------------------------------------------------------------------
AScriptShell::EvalFileThread::EvalFileThread(AScriptShell *pAScriptShell, const wxString &fileName) :
	wxThread(wxTHREAD_DETACHED), _pAScriptShell(pAScriptShell), _fileName(fileName)
{
}

wxThread::ExitCode AScriptShell::EvalFileThread::Entry()
{
	AScript::Environment &env = _pAScriptShell->GetEnvironment();
	AScript::Signal &sig = _pAScriptShell->GetSignal();
	AScript::Console &console = *env.GetConsole(false);
	console.Println("");
	AScript::Expr *pExpr = AScript::Parser().ParseFile(env, sig, _fileName.c_str());
	if (sig.IsSignalled()) {
		console.PrintSignal(sig);
		sig.ClearSignal();
	} else {
		AScript::Environment envLocal(&env, AScript::ENVTYPE_Local);
		envLocal.SetEchoFlag(false);
		pExpr->Exec(envLocal, sig);
		envLocal.SetEchoFlag(true);
		if (sig.IsSignalled()) {
			console.PrintSignal(sig);
			sig.ClearSignal();
		}
	}
	_pAScriptShell->NotifyEndOfThread();
	return 0;
}
