#ifndef __ITERATOR_H__
#define __ITERATOR_H__

#include "Environment.h"
#include "Signal.h"
#include "Expr.h"
#include "Value.h"
#include "OAL.h"

#define ITERATOR_HELP \
"If block is specified, it executes the block while evaluating the iterator\n" \
"and returns the last evaluated value in the block as its own result.\n" \
"But, if attributes :list, :xlist, :set and :xset are specified, it gathers results\n" \
"of each block execution and returns a list of them. The rule is as follows\n" \
" :list  a list of result values\n" \
" :xlist a list of result values eliminating nil\n" \
" :set   a list of unique values of results\n" \
" :xset  a list of unique values of results eliminating nil\n" \
"If block is not specified, it simply returns the iterator.\n"

namespace AScript {

class Expr_Caller;
class Context;

//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator {
public:
	class DLLEXPORT Share {
	public:
		typedef std::vector<size_t> IndexList;
	private:
		bool _doneFlag;
		size_t _indexMin;
		IndexList _indexList;
		ValueDeque _valueDeque;
	public:
		Share();
		~Share();
		inline bool IsDone() const { return _doneFlag; }
		inline void SetDone() { _doneFlag = true; }
		int Register();
		bool Next(int id, Value &value);
		void Store(int id, const Value &value);
	};
protected:
	int _cntRef;
	bool _infiniteFlag;
	Share *_pShare;
public:
	inline Iterator(bool infiniteFlag) :
			_cntRef(1), _infiniteFlag(infiniteFlag), _pShare(NULL) {}
	inline Iterator(const Iterator &iter) :
			_cntRef(1), _infiniteFlag(iter._infiniteFlag), _pShare(NULL) {}
	inline int DecRef() { _cntRef--; return _cntRef; }
	inline int GetRefCnt() const { return _cntRef; }
	inline static void Delete(Iterator *pIterator) {
		if (pIterator != NULL && pIterator->DecRef() <= 0) delete pIterator;
	}
	inline bool IsInfinite() const { return _infiniteFlag; }
	inline void SetInfiniteFlag(bool infiniteFlag) { _infiniteFlag = infiniteFlag; }
	bool NextShared(int id, Signal sig, Value &value);
	inline bool Next(Signal sig, Value &value) { return NextShared(0, sig, value); }
	inline Iterator *IncRef() { _cntRef++; return this; }
	virtual bool IsSequence() const;
	virtual bool IsSequenceInf() const;
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value) = 0;
	Value ToList(Environment &env, Signal sig);
	Value Eval(Environment &env, Signal sig, Context &context);
	Value Eval(Environment &env, Signal sig,
								Context &context, const Function *pFuncBlock);
	Value Reduce(Environment &env, Signal sig,
								Value valueAccum, const Function *pFuncBlock);
	Value MinMax(Environment &env, Signal sig,
									bool maxFlag, const SymbolSet &attrs);
	Value Sum(Signal sig, size_t &cnt);
	Value Average(Signal sig, size_t &cnt);
	Value Variance(Signal sig, size_t &cnt);
	Value StandardDeviation(Signal sig, size_t &cnt);
	Value And(Signal sig);
	Value Or(Signal sig);
	size_t Count(Environment &env, Signal sig, const Value &criteria);
	Iterator *Filter(Environment &env, Signal sig, const Value &criteria);
	Iterator *While(Environment &env, Signal sig, const Value &criteria);
	Iterator *Since(Environment &env, Signal sig, const Value &criteria);
	bool IsContain(Signal sig, const Value &value);
	virtual String ToString(Signal sig) const;
	static void SetError_InvalidDataTypeOfElement(Signal sig);
	static void SetError_InfiniteNotAllowed(Signal sig);
protected:
	virtual ~Iterator();
	static bool PrepareRepeaterIterators(Environment &env, Signal sig,
						const ValueList &valListArg, ExprList &exprLeftList,
						IteratorOwner &iteratorOwner);
private:
	inline void operator=(const Iterator &iterator) {}
};

//-----------------------------------------------------------------------------
// Iterator_GenericClone
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_GenericClone : public Iterator {
private:
	Iterator *_pIterator;
	int _id;
public:
	inline Iterator_GenericClone(Iterator *pIterator, int id) :
			Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _id(id) {}
	virtual ~Iterator_GenericClone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
class DLLEXPORT IteratorOwner : public std::vector<Iterator *> {
public:
	inline IteratorOwner() {}
	IteratorOwner(const IteratorOwner &iterOwner);
	virtual ~IteratorOwner();
	bool Next(Signal sig, ValueList &valList);
	bool IsInfinite() const;
};

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Constant : public Iterator {
private:
	Value _value;
public:
	inline Iterator_Constant(const Value &value) : Iterator(true), _value(value) {}
	inline Iterator_Constant(const Iterator_Constant &iter) :
										Iterator(true), _value(iter._value) {}
	virtual ~Iterator_Constant();
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_OneShot
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_OneShot : public Iterator {
private:
	Value _value;
	bool _doneFlag;
public:
	inline Iterator_OneShot(const Value &value) :
				Iterator(false), _value(value), _doneFlag(false) {}
	inline Iterator_OneShot(const Iterator_OneShot &iter) :
				Iterator(false), _value(iter._value), _doneFlag(iter._doneFlag) {}
	virtual ~Iterator_OneShot();
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Fill
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Fill : public Iterator {
private:
	int _cnt;
	Value _value;
	int _idx;
public:
	inline Iterator_Fill(int cnt, const Value &value) :
				Iterator(false), _cnt(cnt), _value(value), _idx(0) {}
	inline Iterator_Fill(const Iterator_Fill &iter) :
				Iterator(false), _cnt(iter._cnt), _value(iter._value),
				_idx(iter._idx) {}
	virtual ~Iterator_Fill();
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Rand
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Rand : public Iterator {
private:
	int _cnt;
	int _range;
	int _idx;
public:
	inline Iterator_Rand(int cnt, int range) :
						Iterator(false), _cnt(cnt), _range(range), _idx(0) {}
	virtual ~Iterator_Rand();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Range
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Range : public Iterator {
private:
	Number _num;
	Number _numBegin;
	Number _numEnd;
	Number _numStep;
public:
	inline Iterator_Range(Number numBegin, Number numEnd, Number numStep) :
			Iterator(false), _num(numBegin),
			_numBegin(numBegin), _numEnd(numEnd), _numStep(numStep) {}
	inline Iterator_Range(const Iterator_Range &iter) :
			Iterator(iter), _num(iter._num),
			_numBegin(iter._numBegin), _numEnd(iter._numEnd), _numStep(iter._numStep) {}
	virtual ~Iterator_Range();
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Sequence
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Sequence : public Iterator {
private:
	Number _num;
	Number _numBegin;
	Number _numEnd;
	Number _numStep;
public:
	inline Iterator_Sequence(Number numBegin, Number numEnd, Number numStep) :
			Iterator(false), _num(numBegin),
			_numBegin(numBegin), _numEnd(numEnd), _numStep(numStep) {}
	inline Iterator_Sequence(const Iterator_Sequence &iter) :
			Iterator(iter), _num(iter._num),
			_numBegin(iter._numBegin), _numEnd(iter._numEnd), _numStep(iter._numStep) {}
	virtual ~Iterator_Sequence();
	virtual bool IsSequence() const;
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
	inline Number GetBegin() const { return _numBegin; }
	inline Number GetEnd() const { return _numEnd; }
	inline Number GetStep() const { return _numStep; }
};

//-----------------------------------------------------------------------------
// Iterator_SequenceInf
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_SequenceInf : public Iterator {
private:
	Number _num;
	Number _numBegin;
public:
	inline Iterator_SequenceInf(Number numBegin) :
					Iterator(true), _num(numBegin), _numBegin(numBegin) {}
	inline Iterator_SequenceInf(const Iterator_SequenceInf &iter) :
					Iterator(iter), _num(iter._num), _numBegin(iter._numBegin) {}
	virtual ~Iterator_SequenceInf();
	virtual bool IsSequenceInf() const;
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
	inline Number GetBegin() const { return _numBegin; }
};

//-----------------------------------------------------------------------------
// Iterator_Interval
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Interval : public Iterator {
private:
	Number _numBegin;
	Number _numEnd;
	int _numSamples;
	Number _numDenom;
	int _iFactor;
	int _idx;
public:
	inline Iterator_Interval(Number numBegin, Number numEnd,
								int numSamples, Number numDenom, int iFactor) :
		Iterator(false), _numBegin(numBegin), _numEnd(numEnd), _numSamples(numSamples), 
		_numDenom(numDenom), _iFactor(iFactor), _idx(0) {}
	inline Iterator_Interval(Number numBegin, Number numEnd, int numSamples) :
		Iterator(false), _numBegin(numBegin), _numEnd(numEnd), _numSamples(numSamples), 
		_numDenom(numSamples - 1), _iFactor(0), _idx(0) {}
	inline Iterator_Interval(const Iterator_Interval &iter) :
		Iterator(iter), _numBegin(iter._numBegin), _numEnd(iter._numEnd), _numSamples(iter._numSamples),
		_numDenom(iter._numDenom), _iFactor(iter._iFactor), _idx(iter._idx) {}
	virtual ~Iterator_Interval();
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
protected:
	inline bool _DoNext(Number &num) {
		if (_idx >= _numSamples) return false;
		num = (_numEnd - _numBegin) * _iFactor / _numDenom + _numBegin;
		_iFactor++, _idx++;
		return true;
	}
};

//-----------------------------------------------------------------------------
// Iterator_ExplicitMap
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_ExplicitMap : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_ExplicitMap(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	Iterator_ExplicitMap(const Iterator_ExplicitMap &iter);
	virtual ~Iterator_ExplicitMap();
	virtual Iterator *Clone();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Mapping
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Mapping : public Iterator {
private:
	Environment _env;
	Function *_pFunc;
	Value _valueSelf;
	IteratorOwner _iterOwner;
public:
	Iterator_Mapping(Environment &env, Signal sig,
		Function *pFunc, const Value &valueSelf, const ValueList &valListArg);
	virtual ~Iterator_Mapping();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_FuncBinder
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_FuncBinder : public Iterator {
private:
	Environment _env;
	Function *_pFunc;
	Value _valueSelf;
	Iterator *_pIterator;
public:
	Iterator_FuncBinder(Environment &env, Signal sig,
				Function *pFunc, const Value &valueSelf, Iterator *pIterator);
	virtual ~Iterator_FuncBinder();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_MemberMapping
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_MemberMapping : public Iterator {
private:
	Iterator *_pIterator;
	Expr *_pExpr;
public:
	inline Iterator_MemberMapping(Iterator *pIterator, Expr *pExpr) :
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _pExpr(pExpr) {}
	virtual ~Iterator_MemberMapping();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_MethodMapping
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_MethodMapping : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Expr_Caller *_pExprCaller;
public:
	inline Iterator_MethodMapping(Environment &env, Iterator *pIterator, Expr_Caller *pExprCaller) :
		Iterator(pIterator->IsInfinite()), _env(env), _pIterator(pIterator), _pExprCaller(pExprCaller) {}
	virtual ~Iterator_MethodMapping();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Fork : public Iterator, public OAL::Thread {
private:
	Environment _env;
	Function *_pFunc;
	IteratorOwner _iterOwner;
	ValueList _valListA;
	ValueList _valListB;
	ValueList *_pValListToWrite;
	ValueList *_pValListToRead;
	ValueList::iterator _pValueRead;
	OAL::Semaphore _semaphore;
	bool _doneFlag;
	struct {
		bool blockedFlag;
		OAL::Event event;
	} _readBlock;
	struct {
		bool blockedFlag;
		OAL::Event event;
	} _writeBlock;
public:
	Iterator_Fork(Environment &env, Signal sig,
							Function *pFunc, const ValueList &valListArg);
	virtual ~Iterator_Fork();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
	virtual void Run();
	void SwapList();
	void ForkProcess();
};

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Delay : public Iterator {
private:
	Iterator *_pIterator;
	Number _delay;
public:
	inline Iterator_Delay(Iterator *pIterator, Number delay) : 
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _delay(delay) {}
	virtual ~Iterator_Delay();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Skip : public Iterator {
private:
	Iterator *_pIterator;
	int _nSkip;
public:
	inline Iterator_Skip(Iterator *pIterator, int nSkip) :
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _nSkip(nSkip) {}
	virtual ~Iterator_Skip();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_SkipInvalid
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_SkipInvalid : public Iterator {
private:
	Iterator *_pIterator;
public:
	inline Iterator_SkipInvalid(Iterator *pIterator) :
				Iterator(pIterator->IsInfinite()), _pIterator(pIterator) {}
	virtual ~Iterator_SkipInvalid();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_FilterWithFunc
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_FilterWithFunc : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_FilterWithFunc(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	virtual ~Iterator_FilterWithFunc();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_FilterWithIter
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_FilterWithIter : public Iterator {
private:
	Iterator *_pIterator;
	Iterator *_pIteratorCriteria;
public:
	inline Iterator_FilterWithIter(Iterator *pIterator, Iterator *pIteratorCriteria) :
			Iterator(pIterator->IsInfinite() && pIteratorCriteria->IsInfinite()),
			_pIterator(pIterator), _pIteratorCriteria(pIteratorCriteria) {}
	virtual ~Iterator_FilterWithIter();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_WhileWithFunc
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_WhileWithFunc : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_WhileWithFunc(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	virtual ~Iterator_WhileWithFunc();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_WhileWithIter
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_WhileWithIter : public Iterator {
private:
	Iterator *_pIterator;
	Iterator *_pIteratorCriteria;
public:
	inline Iterator_WhileWithIter(Iterator *pIterator, Iterator *pIteratorCriteria) :
			Iterator(pIterator->IsInfinite() && pIteratorCriteria->IsInfinite()),
			_pIterator(pIterator), _pIteratorCriteria(pIteratorCriteria) {}
	virtual ~Iterator_WhileWithIter();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_SinceWithFunc
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_SinceWithFunc : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_SinceWithFunc(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	virtual ~Iterator_SinceWithFunc();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_SinceWithIter
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_SinceWithIter : public Iterator {
private:
	Iterator *_pIterator;
	Iterator *_pIteratorCriteria;
public:
	inline Iterator_SinceWithIter(Iterator *pIterator, Iterator *pIteratorCriteria) :
			Iterator(pIterator->IsInfinite() && pIteratorCriteria->IsInfinite()),
			_pIterator(pIterator), _pIteratorCriteria(pIteratorCriteria) {}
	virtual ~Iterator_SinceWithIter();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};
\
//-----------------------------------------------------------------------------
// Iterator_Replace
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Replace : public Iterator {
private:
	Iterator *_pIterator;
	Value _value;
	Value _valueReplace;
public:
	inline Iterator_Replace(Iterator *pIterator,
							const Value &value, const Value &valueReplace) :
			Iterator(pIterator->IsInfinite()), _pIterator(pIterator),
			_value(value), _valueReplace(valueReplace) {}
	virtual ~Iterator_Replace();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_ReplaceInvalid
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_ReplaceInvalid : public Iterator {
private:
	Iterator *_pIterator;
	Value _valueReplace;
public:
	inline Iterator_ReplaceInvalid(Iterator *pIterator, const Value &valueReplace) :
			Iterator(pIterator->IsInfinite()), _pIterator(pIterator),
			_valueReplace(valueReplace) {}
	virtual ~Iterator_ReplaceInvalid();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Format
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Format : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	String _format;
public:
	inline Iterator_Format(Environment &env,
							Iterator *pIterator, const char *format) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _format(format) {}
	virtual ~Iterator_Format();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Pack
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Pack : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	String _format;
public:
	inline Iterator_Pack(Environment &env,
							Iterator *pIterator, const char *format) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _format(format) {}
	virtual ~Iterator_Pack();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Zip
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Zip : public Iterator {
private:
	Environment _env;
	IteratorOwner _iterOwner;
public:
	inline Iterator_Zip(Environment &env, const IteratorOwner &iterOwner) :
		Iterator(iterOwner.IsInfinite()), _env(env), _iterOwner(iterOwner) {}
	virtual ~Iterator_Zip();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Align : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
	Value _valueFill;
public:
	inline Iterator_Align(Iterator *pIterator, int cnt, const Value &valueFill) :
		Iterator(false), _pIterator(pIterator), _cnt(cnt), _valueFill(valueFill) {}
	virtual ~Iterator_Align();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Head
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Head : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
public:
	inline Iterator_Head(Iterator *pIterator, int cnt) :
						Iterator(false), _pIterator(pIterator), _cnt(cnt) {}
	virtual ~Iterator_Head();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Fold
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Fold : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	int _cnt;
public:
	inline Iterator_Fold(Environment &env, Iterator *pIterator, int cnt) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _cnt(cnt) {}
	virtual ~Iterator_Fold();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_FoldSeg
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_FoldSeg : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
	Value _valueNext;
public:
	inline Iterator_FoldSeg(Iterator *pIterator, int cnt, const Value &valueNext) :
			Iterator(false), _pIterator(pIterator),
			_cnt(cnt), _valueNext(valueNext) {}
	virtual ~Iterator_FoldSeg();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_Concat
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_Concat : public Iterator {
private:
	IteratorOwner _iterOwner;
	IteratorOwner::iterator _ppIterator;
public:
	inline Iterator_Concat() : Iterator(false), _ppIterator(_iterOwner.begin()) {}
	virtual ~Iterator_Concat();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
	void Add(Iterator *pIterator);
};

//-----------------------------------------------------------------------------
// Iterator_repeat
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_repeat : public Iterator {
private:
	Environment _env;
	Function *_pFuncBlock;
	bool _skipInvalidFlag;
	bool _standaloneFlag;
	Iterator *_pIteratorSub;
	int _cnt;
	int _idx;
public:
	Iterator_repeat(Environment &env, Signal sig, Function *pFuncBlock,
					bool skipInvalidFlag, bool standaloneFlag, int cnt);
	virtual ~Iterator_repeat();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_while
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_while : public Iterator {
private:
	Environment _env;
	Function *_pFuncBlock;
	bool _skipInvalidFlag;
	bool _standaloneFlag;
	Iterator *_pIteratorSub;
	Expr *_pExpr;
	int _idx;
public:
	Iterator_while(Environment &env, Signal sig, Function *pFuncBlock,
					bool skipInvalidFlag, bool standaloneFlag, Expr *pExpr);
	virtual ~Iterator_while();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_for
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_for : public Iterator {
private:
	Environment _env;
	Function *_pFuncBlock;
	bool _skipInvalidFlag;
	bool _standaloneFlag;
	Iterator *_pIteratorSub;
	ExprList _exprLeftList;
	IteratorOwner _iteratorOwner;
	int _idx;
	bool _doneFlag;
public:
	Iterator_for(Environment &env, Signal sig, Function *pFuncBlock,
			bool skipInvalidFlag, bool standaloneFlag, const ValueList &valListArg);
	virtual ~Iterator_for();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// Iterator_cross
//-----------------------------------------------------------------------------
class DLLEXPORT Iterator_cross : public Iterator {
private:
	Environment _env;
	Function *_pFuncBlock;
	bool _skipInvalidFlag;
	bool _standaloneFlag;
	Iterator *_pIteratorSub;
	ExprList _exprLeftList;
	IteratorOwner _iteratorOwner;
	IteratorOwner _iteratorOwnerOrg;
	ValueList _valListArg;
	int _idx;
	bool _doneFlag;
public:
	Iterator_cross(Environment &env, Signal sig, Function *pFuncBlock,
			bool skipInvalidFlag, bool standaloneFlag, const ValueList &valListArg);
	virtual ~Iterator_cross();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
private:
	bool AdvanceIterators(Signal sig);
};

}

#endif
