#include "Module.h"
#include "Expr.h"
#include "Parser.h"
#include "Object_Custom.h"
#include "Object_List.h"
#include "Object_Dict.h"
#include "Object_Struct.h"
#include "Object_Bytes.h"
#include "Object_Iterator.h"
#include "Object_Function.h"
#include "Object_File.h"
#include "Object_Error.h"
#include "Object_Semaphore.h"

extern "C" {
#include "SFMT.h"
}

AScript_BeginModule(__builtins__)

#define REPEATER_HELP \
"It returns the last evaluated value in the block as its own result,\n" \
"but, if one of :list, :xlist, :set, :xset or :iter is specified,\n" \
"it returns a list or evaluated value or an iterator. The rule is as follows.\n" \
"  :list  returns a list of result values\n" \
"  :xlist returns a list of result values eliminating nil\n" \
"  :set   returns a list of unique values of results\n" \
"  :xset  returns a list of unique values of results eliminating nil\n" \
"  :iter  returns an iterator that executes the block\n" \
"  :xiter returns an iterator that executes the block, skipping nil\n"

// declaration of symbols privately used in this module
AScript_DeclarePrivSymbol(open);
AScript_DeclarePrivSymbol(open_l);
AScript_DeclarePrivSymbol(open_r);

//-----------------------------------------------------------------------------
// AScript module functions: __builtins__
//-----------------------------------------------------------------------------
// class(superclass?):[static] {block?}
AScript_DeclareFunctionEx(class_, "class")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "superclass", VTYPE_Function, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(static_));
	SetHelp(
	"Returns a function object that constructs an instance with methods and\n"
	"properties specified in the block. If superclass, which is supposed to\n"
	"be a constructor function, is specified, the new class shall inherits\n"
	"methods and properties of the class associated with it.");
}

AScript_ImplementFunction(class_)
{
	const Expr_Block *pExprBlock = context.GetBlock();
	Class *pClassSuper = env.LookupClass(VTYPE_Object);
	const Value &valueSuper = context.GetValue(0);
	if (valueSuper.IsFunction()) {
		pClassSuper = valueSuper.GetFunction()->GetClassToConstruct();
		if (pClassSuper == NULL) {
			valueSuper.GetFunction()->SetError_NotConstructor(sig);
			return Value::Null;
		}
	}
	Class_Custom *pClassCustom =
					new Class_Custom(pClassSuper, AScript_Symbol(_anonymous_));
	if (pExprBlock != NULL && !pClassCustom->BuildContent(env, sig, pExprBlock)) {
		Class::Delete(pClassCustom);
		return Value::Null;
	}
	ClassPrototype *pFunc = NULL;
	FunctionType funcType = context.IsSet(AScript_Symbol(static_))?
											FUNCTYPE_Class : FUNCTYPE_Function;
	FunctionCustom *pFuncInit = dynamic_cast<FunctionCustom *>(
					pClassCustom->LookupFunction(AScript_Symbol(__init__), false));
	if (pFuncInit == NULL) {
		pFunc = new ClassPrototype(env, AScript_Symbol(_anonymous_),
									new Expr_Block(), pClassCustom, funcType);
		ContextExpr contextExpr(ExprList::Null);
		if (!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
			Function::Delete(pFunc);
			return Value::Null;
		}
	} else {
		pFunc = new ClassPrototype(env, AScript_Symbol(_anonymous_),
					pFuncInit->GetExprBody()->IncRef(), pClassCustom, funcType);
		pFunc->CopyDeclare(*pFuncInit);
	}
	pClassCustom->SetConstructor(pFunc->IncRef());
	Value result;
	result.InitAsFunction(env, pFunc);
	return result;
}

// struct(`args*) {block?}
// if :loose attribute is specified, arguments in the generated function
// will get the following modification.
// - Once attribute will be modified to ZeroOrOnce.
// - OnceOrMore attribute will be modified to ZeroOrMore
AScript_DeclareFunctionEx(struct_, "struct")
{
	DeclareArg(env, "args", VTYPE_Quote, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(loose));
	SetHelp(
	"Returns a function object that constructs a structure instance that\n"
	"contains properties specified by args. It can optionally take block\n"
	"which declares some additional methods for constructed instances.\n"
	"If :loose attribute is speicied, the generated constructor function\n"
	"makes an existence check of arguments in a loose way.");
}

AScript_ImplementFunction(struct_)
{
	const Expr_Block *pExprBlock = context.GetBlock();
	Class *pClassSuper = env.LookupClass(VTYPE_Struct);
	ExprList exprArgs;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		exprArgs.push_back(const_cast<Expr *>(pValue->GetExpr()));
	}
	Value result;
	Class_Custom *pClassCustom =
					new Class_Custom(pClassSuper, AScript_Symbol(_anonymous_));
	if (pExprBlock != NULL && !pClassCustom->BuildContent(env, sig, pExprBlock)) {
		Class::Delete(pClassCustom);
		return Value::Null;
	}
	StructPrototype *pFunc = new StructPrototype(env, pClassCustom);
	pClassCustom->SetConstructor(pFunc->IncRef());
	ContextExpr contextExpr(context.GetAttrs(), SymbolSet::Null,
										NULL, NULL, Value::Null, exprArgs);
	if (!pFunc->CustomDeclare(env, sig, _attrsOpt, contextExpr)) {
		Function::Delete(pFunc);
		return false;
	}
	if (context.IsSet(AScript_Symbol(loose))) {
		pFunc->GetDeclList().SetAsLoose();
	}
	result.InitAsFunction(env, pFunc);
	return result;
}

// module {block}
AScript_DeclareFunction(module)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Creates a module that contains functions and variables defined in the block\n"
	"and returns it as a module object. This can be used to realize a namespace.");
}

AScript_ImplementFunction(module)
{
	Value result;
	Module *pModule = new Module(&env, AScript_Symbol(_anonymous_), NULL);
	result.InitAsModule(pModule);
	context.GetBlock()->Exec(*pModule, sig);
	return result;
}

// import(`module, `alias?):[force] {block?}
AScript_DeclareFunctionEx(import_, "import")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "module", VTYPE_Quote);
	DeclareArg(env, "alias", VTYPE_Quote, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(force));
	SetHelp(
	"Imports a module file stored in directories specified by a variable sys.path.\n"
	"There are three ways of calling this function like follow:\n"
	"  1. import(foo)\n"
	"  2. import(foo, bar)\n"
	"  3. import(foo) {symbol1, symbol2, symbol3}\n"
	"In the first format, it creates a module object named foo.\n"
	"In the second, it names the module object as bar instead of foo.\n"
	"In the third, it doesn't register the module name into the environment,\n"
	"but it looks up symbols specified in the block and registers them.\n"
	"In thie case, if specified symbols conflict with the existing one,\n"
	"it will cause an error. Attribute :force will disable such an error\n"
	"detection and allow overwriting of symbols. You can specify an asterisk\n"
	"character to include all the registered symbols like follows.\n"
	"  import(foo) {*}");
}

AScript_ImplementFunction(import_)
{
	SymbolSet symbolsToMixIn;
	SymbolSet *pSymbolsToMixIn = NULL;
	if (context.IsBlockSpecified()) {
		foreach_const (ExprList, ppExpr, context.GetBlock()->GetExprList()) {
			if (!(*ppExpr)->IsSymbol()) {
				sig.SetError(ERR_SyntaxError,
					"wrong format for an element in import list");
				return Value::Null;
			}
			const Expr_Symbol *pExprSymbol =
							dynamic_cast<const Expr_Symbol *>(*ppExpr);
			symbolsToMixIn.insert(pExprSymbol->GetSymbol());
		}
		pSymbolsToMixIn = &symbolsToMixIn;
	}
	const Symbol *pSymbol = NULL;
	if (!context.IsExpr(1)) {
		// nothing to do
	} else if (!context.GetExpr(1)->IsSymbol()) {
		sig.SetError(ERR_ValueError, "symbol is expected as a module name");
		return Value::Null;
	} else {
		pSymbol =
			dynamic_cast<const Expr_Symbol *>(context.GetExpr(1))->GetSymbol();
	}
	if (!env.ImportModule(sig, context.GetExpr(0), pSymbol,
				pSymbolsToMixIn, context.IsSet(AScript_Symbol(force)))) {
		return Value::Null;
	}
	return Value::Null;
}

// parsefile(filename:string):map
AScript_DeclareFunction(parsefile)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "filename", VTYPE_String);
	SetHelp("Parse a content of a script file and returns an expr object.");
}

AScript_ImplementFunction(parsefile)
{
	File file;
	file.Open(sig, context.GetString(0), "r", NULL);
	if (sig.IsSignalled()) return Value::Null;
	Expr *pExpr = Parser().ParseFile(env, sig, file);
	if (sig.IsSignalled()) return NULL;
	Value result;
	result.InitAsExpr(env, pExpr);
	return result;
}

// parse(code:string):map
AScript_DeclareFunction(parse)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "code", VTYPE_String);
	SetHelp("Parse a string returns an expr object.");
}

AScript_ImplementFunction(parse)
{
	const char *str = context.GetString(0);
	Expr *pExpr = Parser().ParseString(env, sig, str, ::strlen(str));
	if (sig.IsSignalled()) return Value::Null;
	Value result;
	result.InitAsExpr(env, pExpr);
	return result;
}

// eval(expr:expr):map
AScript_DeclareFunction(eval)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "expr", VTYPE_Expr);
	SetHelp("Evaluate an expr object.");
}

AScript_ImplementFunction(eval)
{
	return context.GetExpr(0)->Exec(env, sig);
}

// locals(module?:module)
AScript_DeclareFunction(locals)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "module", VTYPE_Module, OCCUR_ZeroOrOnce);
	SetHelp(
	"Returns an environment object that belongs to a specified module.\n"
	"If module is omitted, it returns an environment object of the current scope.");
}

AScript_ImplementFunction(locals)
{
	Value value;
	if (context.IsModule(0)) {
		value.InitAsEnvironment(*context.GetModule(0));
	} else {
		value.InitAsEnvironment(env);
	}
	return value;
}

// outers()
AScript_DeclareFunction(outers)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	SetHelp("Returns an environment object that accesses to an outer scope.");
}

AScript_ImplementFunction(outers)
{
	Value value;
	Environment envOuter(&env, ENVTYPE_Outer);
	value.InitAsEnvironment(envOuter);
	return value;
}

// try () {block}
AScript_DeclareFunctionSucceedableEx(try_, "try")
{
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Specify a try block of a statement of try-except-else.\n"
	"It catches signals that occur in the block and executes a corresponding\n"
	"except() or else() function that follow after it.");
}

AScript_ImplementFunction(try_)
{
	Value result = context.GetBlock()->Exec(env, sig);
	sig.SuspendError();
	context.RequestSucceeding(this);
	return result;
}

bool AScript_Function(try_)::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// except (errors*:error) {block}
AScript_DeclareFunctionSucceedableEx(except_, "except")
{
	DeclareArg(env, "errors", VTYPE_Error, OCCUR_ZeroOrMore);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Specify an except block of a statement of try-except-else.\n"
	"It can take multiple numbers of arguments of error objects to handle.\n"
	"If there's no error objects specified, it handles all the errors that are\n"
	"not handled in the preceding except() function calls.\n"
	"Block parameter format: |error:error|\n"
	"error is an error object that contains information of the handled error.");
}

AScript_ImplementFunction(except_)
{
	bool handleFlag = false;
	if (!sig.IsErrorSuspended()) {
		// nothing to do
	} else if (context.GetList(0).empty()) {
		handleFlag = true;
	} else {
		foreach_const (ValueList, pValue, context.GetList(0)) {
			if (pValue->GetErrorType() == sig.GetErrorType()) {
				handleFlag = true;
				break;
			}
		}
	}
	if (!handleFlag) {
		context.RequestSucceeding(this);
		return Value::Null;
	}
	Value value;
	value.InitAsError(env, sig.GetErrorType(),
							sig.GetErrString().c_str(), sig.GetValue());
	ValueList valListArg(value);
	sig.ClearSignal(); // clear even the suspended state
	const Function *pFuncBlock = GetBlockFunction(env, sig, context);
	if (sig.IsSignalled()) return Value::Null;
	Environment envBlock(&env, ENVTYPE_Block);
	Context contextSub(valListArg);
	return pFuncBlock->Eval(envBlock, sig, contextSub);
}

bool AScript_Function(except_)::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// finally () {block}
AScript_DeclareFunctionEx(finally_, "finally")
{
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(finally_)
{
	return context.GetBlock()->Exec(env, sig);
}

// if (`cond) {block}
AScript_DeclareFunctionSucceedableEx(if_, "if")
{
	DeclareArg(env, "cond", VTYPE_Quote);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Specify an if block of a statement of if-elsif-else.\n"
	"Evaluate an expr object cond and, if it has a value of true,\n"
	"the block shall be executed.");
}

AScript_ImplementFunction(if_)
{
	Value value = context.GetExpr(0)->Exec(env, sig);
	if (value.GetBoolean()) {
		return context.GetBlock()->Exec(env, sig);
	}
	context.RequestSucceeding(this);
	return Value::Null;
}

bool AScript_Function(if_)::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// elsif (`cond) {block}
AScript_DeclareFunctionSucceedableEx(elsif_, "elsif")
{
	DeclareArg(env, "cond", VTYPE_Quote);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Specify an elsif block of a statement of if-elsif-else.\n"
	"Evaluate an expr object cond and, if it has a value of true,\n"
	"the block shall be executed.");
}

AScript_ImplementFunction(elsif_)
{
	Value value = context.GetExpr(0)->Exec(env, sig);
	if (value.GetBoolean()) {
		return context.GetBlock()->Exec(env, sig);
	}
	context.RequestSucceeding(this);
	return Value::Null;
}

bool AScript_Function(elsif_)::IsSucceedable(const ICallable *pCallable) const
{
	return true;
}

// else () {block}
AScript_DeclareFunctionEx(else_, "else")
{
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Specify an else block of a statement of if-elsif-else or try-except-else.\n");
}

AScript_ImplementFunction(else_)
{
	// this function works as a terminater of if-else and try-except
	if (sig.IsErrorSuspended()) return Value::Null;
	return context.GetBlock()->Exec(env, sig);
}

// repeat (n?:number) {block}
AScript_DeclareFunction(repeat)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Executes the block for n times. If n is omitted, it repeats the block\n"
	"execution forever.\n"
	REPEATER_HELP
	"Block parameter format: |idx:number|");
}

AScript_ImplementFunction(repeat)
{
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	bool standaloneFlag = (context.IsRsltIterator() || context.IsRsltXIterator());
	Iterator *pIterator = new Iterator_repeat(envBlock, sig, pFuncBlock->IncRef(),
			context.IsRsltXIterator(), standaloneFlag,
			context.IsNumber(0)? context.GetInt(0) : -1);
	return DoRepeater(env, sig, context, pIterator);
}

// while (`cond) {block}
AScript_DeclareFunctionEx(while_, "while")
{
	DeclareArg(env, "cond", VTYPE_Quote);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Executes the block while the evaluation result of cond is true.\n"
	REPEATER_HELP
	"Block parameter format: |idx:number|");
}

AScript_ImplementFunction(while_)
{
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	bool standaloneFlag = (context.IsRsltIterator() || context.IsRsltXIterator());
	Iterator *pIterator = new Iterator_while(envBlock, sig, pFuncBlock->IncRef(),
			context.IsRsltXIterator(), standaloneFlag, context.GetExpr(0)->IncRef());
	return DoRepeater(env, sig, context, pIterator);
}

// for (`expr+) {block}
AScript_DeclareFunctionEx(for_, "for")
{
	DeclareArg(env, "expr", VTYPE_Quote, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Executes the block until any of the exprs of \"var in iteratable\" reach at\n"
	"their ends. You can specify one or more such exprs as arguments.\n"
	"Iterators and lists are the most popular iteratables, but even any objects that\n"
	"are cable of generating iterators can be specified as such.\n"
	REPEATER_HELP
	"Block parameter format: |idx:number|");
}

AScript_ImplementFunction(for_)
{
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	bool standaloneFlag = (context.IsRsltIterator() || context.IsRsltXIterator());
	Iterator *pIterator = new Iterator_for(envBlock, sig, pFuncBlock->IncRef(),
				context.IsRsltXIterator(), standaloneFlag, context.GetList(0));
	return DoRepeater(env, sig, context, pIterator);
}

// cross (`expr+) {block}
AScript_DeclareFunction(cross)
{
	DeclareArg(env, "expr", VTYPE_Quote, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_Once);
	SetHelp(
	"Executes the block until it evaluates all the combinations of results from exprs\n"
	"\"var in iteratable.\" You can specify one or more such exprs as arguments and\n"
	"they are counted up from the one on the right side.\n"
	"Iterators and lists are the most popular iteratables, but even any objects that\n"
	"are cable of generating iterators can be specified as such.\n"
	REPEATER_HELP
	"Block parameter format: |idx:number|");
}

AScript_ImplementFunction(cross)
{
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	bool standaloneFlag = (context.IsRsltIterator() || context.IsRsltXIterator());
	Iterator *pIterator = new Iterator_cross(envBlock, sig, pFuncBlock->IncRef(),
				context.IsRsltXIterator(), standaloneFlag, context.GetList(0));
	return DoRepeater(env, sig, context, pIterator);
}

// break(value?):symbol_func
AScript_DeclareFunctionEx(break_, "break")
{
	SetAsSymbolFunc();
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrOnce);
	SetHelp(
	"Exits from an inside of a loop that is formed with functions repeat(), while()\n"
	"for() and cross(). If it takes an argument, that value is treated as a result of\n"
	"the loop function. Otherwise, the result is nil and an argument list\n"
	"can be omitted. If the loop function is specified with one of :list, :xlist, :set,\n"
	":xset, :iter and :xiter, break()'s value is NOT included in the result.");
}

AScript_ImplementFunction(break_)
{
	sig.SetSignal(SIGTYPE_Break, context.GetValue(0));
	return Value::Null;
}

// continue(value?):symbol_func
AScript_DeclareFunctionEx(continue_, "continue")
{
	SetAsSymbolFunc();
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrOnce);
	SetHelp(
	"In a loop that is formed with functions repeat(), while(), for() and cross(),\n"
	"skips the following part of it and gets to the top of its process.\n"
	"If it takes an argument, that value is treated as a result of the loop function.\n"
	"Otherwise, the result is nil and an argument list can be omitted.\n"
	"If the loop function is specified with one of :list, :xlist, :set,\n"
	":xset, :iter and :xiter, continue()'s value is included in the result.");
}

AScript_ImplementFunction(continue_)
{
	sig.SetSignal(SIGTYPE_Continue, context.GetValue(0));
	return Value::Null;
}

// return(value?):symbol_func
AScript_DeclareFunctionEx(return_, "return")
{
	SetAsSymbolFunc();
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrOnce);
	SetHelp(
	"Exits from a function skipping the following part of it.\n"
	"If it takes an argument, that value is treated as a result of the function.\n"
	"Otherwise, the result is nil and an argument list can be omitted.");
}

AScript_ImplementFunction(return_)
{
	sig.SetSignal(SIGTYPE_Return, context.GetValue(0));
	return Value::Null;
}

// raise(error:error, msg:string => "error", value?)
AScript_DeclareFunction(raise)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "error", VTYPE_Error);
	DeclareArg(env, "msg", VTYPE_String, OCCUR_Once, false, false, new Expr_String("error"));
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrOnce);
	SetHelp(
	"Raises an error signal with a specified error object, a message string and\n"
	"an additional value.");
}

AScript_ImplementFunction(raise)
{
	sig.SetError(context.GetErrorType(0), "%s", context.GetString(1));
	sig.SetValue(context.GetValue(2));
	return Value::Null;
}

// zip(values+) {block?}
AScript_DeclareFunction(zip)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_OnceOrMore);
	DeclareBlock(OCCUR_ZeroOrOnce);
	SetHelp(
	"Creates an iterator generating lists that bind given argument values.\n"
	"When the value is a list or an iterator, each item in it would be zipped.\n"
	"If block is specified, it evaluates the iterator with the block and returns\n"
	"that result. If block is not specified, it returns the iterator as a result.");
}

AScript_ImplementFunction(zip)
{
	IteratorOwner iterOwner;
	bool iteratorFlag = false;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (pValue->IsList() || pValue->IsIterator()) {
			iteratorFlag = true;
			break;
		}
	}
	if (!iteratorFlag) {
		Value result;
		ValueList &valList = result.InitAsList(env);
		foreach_const (ValueList, pValue, context.GetList(0)) {
			valList.push_back(*pValue);
		}
		return result;
	}
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Iterator *pIteratorArg = NULL;
		if (pValue->IsList() || pValue->IsIterator()) {
			pIteratorArg = pValue->CreateIterator(sig);
			if (sig.IsSignalled()) return Value::Null;
		} else {
			pIteratorArg = new Iterator_Constant(*pValue);
		}
		iterOwner.push_back(pIteratorArg);
	}
	Iterator *pIterator = new Iterator_Zip(env, iterOwner);
	return ReturnIterator(env, sig, context, pIterator);
}

// iterator(value+)
AScript_DeclareFunction(iterator)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(iterator)
{
	Iterator_Concat *pIterator = new Iterator_Concat();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Iterator *pIteratorArg = NULL;
		if (pValue->IsList() || pValue->IsIterator()) {
			pIteratorArg = pValue->CreateIterator(sig);
			if (sig.IsSignalled()) return Value::Null;
		} else {
			pIteratorArg = new Iterator_OneShot(*pValue);
		}
		pIterator->Add(pIteratorArg);
	}
	return Value(env, pIterator);
}

// list(iter+:iterator)
AScript_DeclareFunction(list)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(list)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Value value;
		Iterator *pIterator = pValue->GetIterator();
		if (pIterator->IsInfinite()) {
			Iterator::SetError_InfiniteNotAllowed(sig);
			return Value::Null;
		}
		while (pIterator->Next(sig, value)) {
			valList.push_back(value);
		}
		if (sig.IsSignalled()) return Value::Null;
	}
	return result;
}

// xlist(iter+:iterator)
AScript_DeclareFunction(xlist)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "iter", VTYPE_Iterator, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(xlist)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		Value value;
		Iterator *pIterator = pValue->GetIterator();
		if (pIterator->IsInfinite()) {
			Iterator::SetError_InfiniteNotAllowed(sig);
			return Value::Null;
		}
		while (pIterator->Next(sig, value)) {
			if (value.IsValid()) valList.push_back(value);
		}
		if (sig.IsSignalled()) return Value::Null;
	}
	return result;
}

// set(iter+:iterator):[and,xor,or]
// xset(iter+:iterator):[and,xor,or]
AScript_DeclareFunctionBegin(set_xset)
	bool _acceptInvalidFlag;
AScript_DeclareFunctionEnd(set_xset)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	_acceptInvalidFlag = (::strcmp(GetName(), "set") == 0);
	DeclareArg(env, "iter", VTYPE_Iterator, OCCUR_OnceOrMore);
	DeclareAttr(AScript_Symbol(or));
	DeclareAttr(AScript_Symbol(and));
	DeclareAttr(AScript_Symbol(xor));
}

AScript_ImplementFunction(set_xset)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	if (context.IsSet(AScript_Symbol(and))) {			// AND combination
		ValueList valList1, valList2;
		ValueList::const_reverse_iterator pValueArg = context.GetList(0).rbegin();
		Value value;
		for (Iterator *pIterator = pValueArg->GetIterator();
											pIterator->Next(sig, value); ) {
			if ((_acceptInvalidFlag || value.IsValid()) &&
											!valList1.IsContain(value)) {
				valList1.push_back(value);
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		pValueArg++;
		ValueList *pValListAnd = &valList1;
		ValueList *pValListWork = &valList2;
		for ( ; pValueArg != context.GetList(0).rend() &&
										!pValListAnd->empty(); pValueArg++) {
			Value value;
			for (Iterator *pIterator = pValueArg->GetIterator();
											pIterator->Next(sig, value); ) {
				if (pValListAnd->IsContain(value) && !pValListWork->IsContain(value)) {
					pValListWork->push_back(value);
				}
			}
			ValueList *pValListTmp = pValListAnd;
			pValListAnd = pValListWork, pValListWork = pValListTmp;
			pValListWork->clear();
		}
		foreach_const (ValueList, pValue, *pValListAnd) {
			valList.push_back(*pValue);
		}
	} else if (context.IsSet(AScript_Symbol(xor))) {	// XOR combination
		ValueList valList1, valList2;
		ValueList *pValListAnd = &valList1;
		ValueList valListOr;
		ValueList::const_iterator pValueArg = context.GetList(0).begin();
		Value value;
		for (Iterator *pIterator = pValueArg->GetIterator();
										pIterator->Next(sig, value); ) {
			if (!valList1.IsContain(value)) valList1.push_back(value);
			if ((_acceptInvalidFlag || value.IsValid()) &&
											!valListOr.IsContain(value)) {
				valListOr.push_back(value);
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		pValueArg++;
		ValueList *pValListWork = &valList2;
		for ( ; pValueArg != context.GetList(0).end() &&
										!pValListAnd->empty(); pValueArg++) {
			Value value;
			for (Iterator *pIterator = pValueArg->GetIterator();
										pIterator->Next(sig, value); ) {
				if (pValListAnd->IsContain(value)) pValListWork->push_back(value);
				if ((_acceptInvalidFlag || value.IsValid()) &&
												!valListOr.IsContain(value)) {
					valListOr.push_back(value);
				}
			}
			if (sig.IsSignalled()) return Value::Null;
			ValueList *pValListTmp = pValListAnd;
			pValListAnd = pValListWork, pValListWork = pValListTmp;
			pValListWork->clear();
		}
		foreach_const (ValueList, pValue, valListOr) {
			if (!pValListAnd->IsContain(*pValue)) {
				valList.push_back(*pValue);
			}
		}
	} else {										// OR combination
		foreach_const (ValueList, pValue, context.GetList(0)) {
			Value value;
			for (Iterator *pIterator = pValue->GetIterator();
												pIterator->Next(sig, value); ) {
				if ((_acceptInvalidFlag || value.IsValid()) &&
												!valList.IsContain(value)) {
					valList.push_back(value);
				}
			}
			if (sig.IsSignalled()) return Value::Null;
		}
	}
	return result;
}

// min(values+):map
AScript_DeclareFunction(min)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_OnceOrMore);
	SetHelp("Returns the minimum value among the given arguments.");
}

AScript_ImplementFunction(min)
{
	const ValueList &valList = context.GetList(0);
	ValueList::const_iterator pValue = valList.begin();
	Value result = *pValue++;
	for ( ; pValue != valList.end(); pValue++) {
		int cmp = Value::Compare(result, *pValue);
		if (cmp > 0) result = *pValue;
	}
	return result;
}

// max(values+):map
AScript_DeclareFunction(max)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_OnceOrMore);
	SetHelp("Returns the maximum value among the given arguments.");
}

AScript_ImplementFunction(max)
{
	const ValueList &valList = context.GetList(0);
	ValueList::const_iterator pValue = valList.begin();
	Value result = *pValue++;
	for ( ; pValue != valList.end(); pValue++) {
		int cmp = Value::Compare(result, *pValue);
		if (cmp < 0) result = *pValue;
	}
	return result;
}

// choose(index:number, values+):map
AScript_DeclareFunction(choose)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "index", VTYPE_Number);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_OnceOrMore);
}

AScript_ImplementFunction(choose)
{
	size_t index = context.GetSizeT(0);
	const ValueList &valList = context.GetList(1);
	if (index >= valList.size()) {
		sig.SetError(ERR_IndexError, "index is out of range");
		return Value::Null;
	}
	return valList[index];
}

// chooseif(flag, value1, vlaue2):map
AScript_DeclareFunction(chooseif)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "flag", VTYPE_Boolean);
	DeclareArg(env, "value1", VTYPE_Any);
	DeclareArg(env, "value2", VTYPE_Any);
}

AScript_ImplementFunction(chooseif)
{
	return context.GetBoolean(0)? context.GetValue(1) : context.GetValue(2);
}

// tostring(value):map
AScript_DeclareFunction(tostring)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
}

AScript_ImplementFunction(tostring)
{
	return Value(env, context.GetValue(0).ToString(sig, false).c_str());
}

// tonumber(value):map:[strict,raise,zero,nil]
AScript_DeclareFunction(tonumber)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
	DeclareAttr(AScript_Symbol(strict));
	DeclareAttr(AScript_Symbol(raise));
	DeclareAttr(AScript_Symbol(zero));
	DeclareAttr(AScript_Symbol(nil));
}

AScript_ImplementFunction(tonumber)
{
	bool allowPartFlag = !context.IsSet(AScript_Symbol(strict));
	bool successFlag;
	Number num = context.GetValue(0).ToNumber(allowPartFlag, successFlag);
	if (successFlag) {
		return Value(num);
	} else if (context.IsSet(AScript_Symbol(raise))) {
		sig.SetError(ERR_ValueError, "failed to convert to a number");
		return Value::Null;
	} else if (context.IsSet(AScript_Symbol(zero))) {
		return Value(0.);
	} else { // context.IsSet(AScript_PrivSymbol(nil)
		return Value::Null;
	}
}

// tosymbol(str:string):map
AScript_DeclareFunction(tosymbol)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(tosymbol)
{
	return Value(Symbol::Add(context.GetString(0)));
}

// rand(num?)
AScript_DeclareFunction(rand)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "num", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(rand)
{
	if (context.IsInvalid(0)) {
		return Value(::genrand_real2());
	}
	unsigned long num = context.GetULong(0);
	Number result = static_cast<unsigned long>(::genrand_real2() * num);
	return Value(result);
}

// int(value):map
AScript_DeclareFunctionEx(int_, "int")
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
	SetHelp("Converts any value into an integer number.");
}

AScript_ImplementFunction(int_)
{
	const Value &value = context.GetValue(0);
	Value result;
	if (value.IsNumber()) {
		result.SetNumber(value.GetLong());
	} else if (value.IsComplex()) {
		result.SetNumber(static_cast<long>(std::abs(value.GetComplex())));
	} else if (value.IsString()) {
		bool successFlag;
		Number num = value.ToNumber(true, successFlag);
		if (!successFlag) {
			sig.SetError(ERR_ValueError, "failed to convert to a number");
			return Value::Null;
		}
		result.SetNumber(static_cast<long>(num));
	} else if (value.IsValid()) {
		SetError_InvalidValType(sig, value);
	}
	return result;
}

// ord(str:string):map
AScript_DeclareFunction(ord)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(ord)
{
	const char *str = context.GetString(0);
	unsigned long num = static_cast<unsigned char>(*str);
	if (IsUTF8First(*str)) {
		str++;
		for ( ; IsUTF8Follower(*str); str++) {
			num = (num << 8) | static_cast<unsigned char>(*str);
		}
	}
	return Value(num);
}

// chr(num:number):map
AScript_DeclareFunction(chr)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "num", VTYPE_Number);
}

AScript_ImplementFunction(chr)
{
	unsigned long num = context.GetULong(0);
	int i = 0;
	char buff[4];
	buff[i++] = static_cast<unsigned char>(num & 0xff);
	num >>= 8;
	for ( ; num != 0; num >>= 8) {
		buff[i++] = static_cast<unsigned char>(num & 0xff);
	}
	String str;
	for ( ; i > 0; i--) {
		str.push_back(buff[i - 1]);
	}
	return Value(env, str.c_str());
}

// hex(num:number, digits?:number):map:[upper]
AScript_DeclareFunction(hex)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "num", VTYPE_Number);
	DeclareArg(env, "digits", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(upper));
}

AScript_ImplementFunction(hex)
{
	int digits = context.IsNumber(1)? context.GetInt(1) : 0;
	bool upperFlag = context.IsSet(AScript_Symbol(upper));
	String str;
	if (digits <= 0) {
		str = Formatter::Format(sig, upperFlag? "%X" : "%x",
						ValueList(context.GetValue(0)));
	} else {
		str = Formatter::Format(sig, upperFlag? "%0*X" : "%0*x",
						ValueList(Value(digits), context.GetValue(0)));
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, str.c_str());
}

// print(value*):map:void
AScript_DeclareFunction(print)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(print)
{
	Console *pConsole = env.GetConsole(false);
	if (pConsole == NULL) return Value::Null;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pConsole->Print(pValue->ToString(sig, false).c_str());
		if (sig.IsSignalled()) break;
	}
	return Value::Null;
}

// println(value*):map:void
AScript_DeclareFunction(println)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(println)
{
	Console *pConsole = env.GetConsole(false);
	if (pConsole == NULL) return Value::Null;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pConsole->Print(pValue->ToString(sig, false).c_str());
		if (sig.IsSignalled()) break;
	}
	pConsole->Print("\n");
	return Value::Null;
}

// printf(format, values*):map:void
AScript_DeclareFunction(printf)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(printf)
{
	Console *pConsole = env.GetConsole(false);
	if (pConsole == NULL) return Value::Null;
	pConsole->Printf(sig, context.GetString(0), context.GetList(1));
	return Value::Null;
}

// dir(obj?)
AScript_DeclareFunction(dir)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "obj", VTYPE_Any, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(dir)
{
	const Environment::Frame *pFrame = &env.GetBottomFrame();
	const Value &value = context.GetValue(0);
	if (value.IsModule()) {
		pFrame = &value.GetModule()->GetTopFrame();
	} else if (value.IsClass()) {
		pFrame = &value.GetClass()->GetTopFrame();
	} else if (value.IsFunction() &&
						value.GetFunction()->GetClassToConstruct() != NULL) {
		const Class *pClass = value.GetFunction()->GetClassToConstruct();
		pFrame = &pClass->GetTopFrame();
	} else if (value.IsObject()) {
		const Object *pObj = value.GetObject();
		Value result;
		ValueList &valList = result.InitAsList(env);
		foreach_const (Environment::FrameList, ppFrame, pObj->GetFrameList()) {
			const Environment::Frame *pFrame = *ppFrame;
			if (!pFrame->IsType(ENVTYPE_Class) && !pFrame->IsType(ENVTYPE_Instance)) {
				break;
			}
			foreach_const (ValueMap, iter, pFrame->GetValueMap()) {
				valList.push_back(Value(env, iter->first->GetName()));
			}
		}
		return result;
	} else if (value.IsValid()) {
		sig.SetError(ERR_TypeError, "invalid argument");
		return Value::Null;
	}
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ValueMap, iter, pFrame->GetValueMap()) {
		valList.push_back(Value(env, iter->first->GetName()));
	}
	return result;
}

// help(func:function):map:void
AScript_DeclareFunction(help)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "func", VTYPE_Function);
	SetHelp("Print a help message for the specified function object.");
}

AScript_ImplementFunction(help)
{
	Object_Function *pFuncObj = context.GetFunctionObj(0);
	Console *pConsole = env.GetConsole(false);
	pConsole->Println(pFuncObj->ToString(sig, true).c_str());
	if (pFuncObj->GetFunction()->IsHelpExist()) {
		const char *lineTop = "  ";
		bool lineTopFlag = true;
		for (const char *p = pFuncObj->GetFunction()->GetHelp(); *p != '\0'; p++) {
			char ch = *p;
			if (lineTopFlag) {
				pConsole->Print(lineTop);
				lineTopFlag = false;
			}
			pConsole->PutChar(ch);
			if (ch == '\n') lineTopFlag = true;
		}
		if (!lineTopFlag) pConsole->PutChar('\n');
	}
	return Value::Null;
}

// isdefined(`symbol)
AScript_DeclareFunction(isdefined)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "symbol", VTYPE_Quote);
}

AScript_ImplementFunction(isdefined)
{
	bool definedFlag = false;
	const Expr *pExpr = context.GetExpr(0);
	if (pExpr->IsSymbol() || pExpr->IsField()) {
		pExpr->Exec(env, sig);
		if (sig.IsSignalled() && !sig.IsError()) return Value::Null;
		definedFlag = !sig.IsError();
		sig.ClearSignal();
	} else {
		sig.SetError(ERR_ValueError, "argument must be a symbol");
		return Value::Null;
	}
	return Value(definedFlag);
}

// istype*(value)
class AScript_Function(istype_) : public Function {
private:
	ValueType _valType;
public:
	AScript_Function(istype_)(Environment &env, const char *name, ValueType valType);
	virtual Value DoEval(Environment &env, Signal sig, Context &context) const;
};
AScript_Function(istype_)::AScript_Function(istype_)(
					Environment &env, const char *name, ValueType valType) :
	Function(env, Symbol::Add(name), FUNCTYPE_Function), _valType(valType)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
	char buff[128];
	::sprintf(buff, "Check if the type of the specified value is %s.",
									GetValueTypeSymbol(_valType)->GetName());
	SetHelp(buff);
}

AScript_ImplementFunction(istype_)
{
	ValueType valType = context.GetValue(0).GetType();
	ValueType valTypeCmp = _valType;
	if (valType == VTYPE_Number && valTypeCmp == VTYPE_Complex) return Value(true);
	return Value(valType == valTypeCmp);
}

// istype(value, type:expr):map
AScript_DeclareFunction(istype)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
	DeclareArg(env, "type", VTYPE_Expr);
}

AScript_ImplementFunction(istype)
{
	SymbolList symbolList;
	if (!context.GetExpr(1)->GetChainedSymbolList(symbolList)) {
		sig.SetError(ERR_TypeError, "invalid type name");
		return Value::Null;
	}
	const ValueTypeInfo *pValueTypeInfo = env.LookupValueType(symbolList);
	if (pValueTypeInfo == NULL) {
		sig.SetError(ERR_ValueError, "invalid type name");
		return Value::Null;
	}
	ValueType valType = context.GetValue(0).GetType();
	ValueType valTypeCmp = pValueTypeInfo->GetValueType();
	if (valType == VTYPE_Number && valTypeCmp == VTYPE_Complex) return Value(true);
	return Value(valType == valTypeCmp);
}

// isinstance(value, type:expr):map
AScript_DeclareFunction(isinstance)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Any);
	DeclareArg(env, "type", VTYPE_Expr);
}

AScript_ImplementFunction(isinstance)
{
	SymbolList symbolList;
	if (!context.GetExpr(1)->GetChainedSymbolList(symbolList)) {
		sig.SetError(ERR_TypeError, "invalid type name");
		return Value::Null;
	}
	const ValueTypeInfo *pValueTypeInfo = env.LookupValueType(symbolList);
	if (pValueTypeInfo == NULL) {
		sig.SetError(ERR_ValueError, "invalid type name");
		return Value::Null;
	}
	return context.GetValue(0).IsInstanceOf(pValueTypeInfo->GetValueType());
}

// classref(type:expr):map
AScript_DeclareFunction(classref)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "type", VTYPE_Expr);
}

AScript_ImplementFunction(classref)
{
	SymbolList symbolList;
	if (!context.GetExpr(0)->GetChainedSymbolList(symbolList)) {
		sig.SetError(ERR_TypeError, "invalid type name");
		return Value::Null;
	}
	const ValueTypeInfo *pValueTypeInfo = env.LookupValueType(symbolList);
	if (pValueTypeInfo == NULL) {
		sig.SetError(ERR_ValueError, "invalid type name");
		return Value::Null;
	}
	if (pValueTypeInfo->GetClass() == NULL) {
		sig.SetError(ERR_ValueError, "not a class type");
		return Value::Null;
	}
	Value result;
	result.InitAsClass(pValueTypeInfo->GetClass()->IncRef());
	return result;
}

// typename(`value)
AScript_DeclareFunctionEx(typename_, "typename")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Quote);
}

AScript_ImplementFunction(typename_)
{
	const Expr *pExpr = context.GetExpr(0);
	const char *typeName = "unknown";
	if (pExpr->IsSymbol() || pExpr->IsField()) {
		Value value = pExpr->Exec(env, sig);
		if (sig.IsSignalled() && !sig.IsError()) return Value::Null;
		typeName = sig.IsError()? "undefined" : value.GetTypeName();
		sig.ClearSignal();
	} else {
		Value value = pExpr->Exec(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		typeName = value.GetTypeName();
	}
	return Value(env, typeName);
}

// undef(`value+):[raise]
AScript_DeclareFunctionEx(undef_, "undef")
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_Quote, OCCUR_OnceOrMore);
	DeclareAttr(AScript_Symbol(raise));
}

AScript_ImplementFunction(undef_)
{
	bool raiseFlag = context.IsSet(AScript_Symbol(raise));
	foreach_const (ValueList, pValueArg, context.GetList(0)) {
		const Expr *pExpr = pValueArg->GetExpr();
		Environment *pEnv = &env;
		const Symbol *pSymbol = NULL;
		if (pExpr->IsSymbol()) {
			pSymbol = dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
		} else {
			SymbolList symbolList;
			if (!pExpr->GetChainedSymbolList(symbolList)) {
				sig.SetError(ERR_ValueError, "invalid symbol name");
				return Value::Null;
			}
			for (SymbolList::iterator ppSymbol = symbolList.begin();
								ppSymbol + 1 != symbolList.end(); ppSymbol++) {
				Value *pValue = pEnv->LookupValue(*ppSymbol, false);
				if (pValue == NULL) {
					if (raiseFlag) {
						sig.SetError(ERR_ValueError, "undefined symbol");
					}
					return Value::Null;
				}
				if (pValue->IsModule()) {
					pEnv = pValue->GetModule();
				} else if (pValue->IsClass()) {
					pEnv = pValue->GetClass();
				} else if (pValue->IsObject()) {
					pEnv = pValue->GetObject();
				} else {
					sig.SetError(ERR_ValueError, "invalid symbol name");
					return Value::Null;
				}
			}
			pSymbol = symbolList.back();
		}
		if (raiseFlag && !pEnv->LookupValue(pSymbol, false)) {
			sig.SetError(ERR_ValueError, "undefined symbol");
			return Value::Null;
		}
		pEnv->RemoveValue(pSymbol);
	}
	return Value::Null;
}

// range(num:number, num_end?:number, step?:number):map {block?}
AScript_DeclareFunction(range)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "num",		VTYPE_Number);
	DeclareArg(env, "num_end",	VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareArg(env, "step",		VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(range)
{
	Number numBegin = 0.;
	Number numEnd = 0.;
	Number numStep = 1.;
	if (context.GetValue(1).IsInvalid()) {
		numEnd = context.GetValue(0).GetNumber();
		if (numBegin > numEnd) numStep = -1.;
	} else if (context.GetValue(2).IsInvalid()) {
		numBegin = context.GetValue(0).GetNumber();
		numEnd = context.GetValue(1).GetNumber();
		if (numBegin > numEnd) numStep = -1.;
	} else {
		numBegin = context.GetValue(0).GetNumber();
		numEnd = context.GetValue(1).GetNumber();
		numStep = context.GetValue(2).GetNumber();
		if (numStep == 0.) {
			sig.SetError(ERR_ValueError, "step cannot be specified as zero");
			return Value::Null;
		}
	}
	return ReturnIterator(env, sig, context,
						new Iterator_Range(numBegin, numEnd, numStep));
}

// interval(a:number, b:number, samples:number):map {block?}
AScript_DeclareFunction(interval)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "a", VTYPE_Number);
	DeclareArg(env, "b", VTYPE_Number);
	DeclareArg(env, "samples", VTYPE_Number);
	DeclareAttr(AScript_PrivSymbol(open));
	DeclareAttr(AScript_PrivSymbol(open_l));
	DeclareAttr(AScript_PrivSymbol(open_r));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(interval)
{
	const ValueList &valListArg = context.GetArgs();
	Number numBegin = valListArg[0].GetNumber();
	Number numEnd = valListArg[1].GetNumber();
	int numSamples = valListArg[2].GetInt();
	if (numSamples <= 1) {
		sig.SetError(ERR_ValueError, "samples must be more than one");
		return Value::Null;
	}
	bool openFlag = context.IsSet(AScript_PrivSymbol(open));
	bool openLeftFlag = context.IsSet(AScript_PrivSymbol(open_l));
	bool openRightFlag = context.IsSet(AScript_PrivSymbol(open_r));
	int iFactor = 0;
	Number numDenom = numSamples - 1;
	if (openFlag || (openLeftFlag && openRightFlag)) {
		numDenom = numSamples + 1;
		iFactor = 1;
	} else if (openLeftFlag) {
		numDenom = numSamples;
		iFactor = 1;
	} else if (openRightFlag) {
		numDenom = numSamples;
		iFactor = 0;
	}
	Iterator *pIterator =
		new Iterator_Interval(numBegin, numEnd, numSamples, numDenom, iFactor);
	return ReturnIterator(env, sig, context, pIterator);
}

// fill(n:number, value?) {block?}
AScript_DeclareFunction(fill)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(fill)
{
	Iterator *pIterator = new Iterator_Fill(context.GetInt(0), context.GetValue(1));
	return ReturnIterator(env, sig, context, pIterator);
}

// rands(n?:number, range?:number) {block?}
AScript_DeclareFunction(rands)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareArg(env, "range", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(rands)
{
	Iterator *pIterator = new Iterator_Rand(
				context.IsNumber(0)? context.GetInt(0) : -1,
				context.IsNumber(1)? context.GetInt(1) : 0);
	return ReturnIterator(env, sig, context, pIterator);
}

// bytes(buff*)
AScript_DeclareFunction(bytes)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "buff", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(bytes)
{
	Value result;
	Object_Bytes *pObjBytes = result.InitAsBytes(env);
	Bytes &bytes = pObjBytes->GetBytes();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (pValue->IsString()) {
			bytes += pValue->GetString();
		} else if (pValue->IsBytes()) {
			bytes += pValue->GetBytes();
		} else {
			sig.SetError(ERR_ValueError, "string or bytes is expected");
			return Value::Null;
		}
	}
	return result;
}

// pack(format:string, value*):map
AScript_DeclareFunction(pack)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrMore);
}

AScript_ImplementFunction(pack)
{
	Value result;
	Object_Bytes *pObjBytes = result.InitAsBytes(env);
	size_t offset = 0;
	pObjBytes->Pack(sig, offset, context.GetString(0), context.GetList(1));
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// @(func?:Function) {block?}
AScript_DeclareFunction(ListInit)
{
	DeclareArg(env, "func", VTYPE_Function, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(ListInit)
{
	const Expr_Block *pExprBlock = context.GetBlock();
	const Value &valueFunc = context.GetValue(0);
	Value result;
	if (pExprBlock == NULL) {
		result.InitAsList(env);
	} else if (valueFunc.IsFunction()) {
		const Function *pFunc = valueFunc.GetFunction();
		size_t cntArgs = pFunc->GetDeclList().size();
		if (cntArgs == 0) {
			sig.SetError(ERR_TypeError, "function '%s' needs no argument", pFunc->GetName());
			return Value::Null;
		}
		Environment envLister(&env, ENVTYPE_Lister);
		Value valueRaw =
				pExprBlock->GetExprList().ExecForList(envLister, sig, false, false);
		if (sig.IsSignalled() || !valueRaw.IsList()) return Value::Null;
		ValueList &valList = result.InitAsList(env);
		foreach_const (ValueList, pValue, valueRaw.GetList()) {
			if (!pValue->IsList()) {
				sig.SetError(ERR_SyntaxError, "invalid format in list initializer");
				return Value::Null;
			}
			Context contextSub(pValue->GetList());
			Value valueElem = pFunc->Eval(env, sig, contextSub);
			valList.push_back(valueElem);
		}
	} else {
		Environment envLister(&env, ENVTYPE_Lister);
		result = pExprBlock->GetExprList().ExecForList(envLister, sig, false, false);
	}
	return result;
}

// func = lambda(`args*) {block}
class ExprVisitor_GatherArgs : public ExprVisitor {
private:
	SymbolSet _symbolSet;
	ExprList &_exprList;
public:
	inline ExprVisitor_GatherArgs(ExprList &exprList) : _exprList(exprList) {}
	virtual bool Visit(const Expr *pExpr);
};

bool ExprVisitor_GatherArgs::Visit(const Expr *pExpr)
{
	if (pExpr->IsCaller()) {
		// avoid searching in a simple lambda inside
		const Expr *pExprCar =
					dynamic_cast<const Expr_Caller *>(pExpr)->GetCar();
		if (pExprCar->IsSymbol()) {
			const Symbol *pSymbol =
					dynamic_cast<const Expr_Symbol *>(pExprCar)->GetSymbol();
			if (::strcmp(pSymbol->GetName(), "@") == 0) return false;
		}
		
	} else if (pExpr->IsSymbol()) {
		const Symbol *pSymbol =
						dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
		if (pSymbol->GetName()[0] == '$' &&
								_symbolSet.find(pSymbol) == _symbolSet.end()) {
			_exprList.push_back(const_cast<Expr *>(pExpr));
			_symbolSet.insert(pSymbol);
		}
	}
	return true;
}

AScript_DeclareFunction(lambda)
{
	DeclareArg(env, "args", VTYPE_Quote, OCCUR_ZeroOrMore);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(lambda)
{
	const Expr *pExprBlock = context.GetBlock();
	ExprList exprArgs;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		exprArgs.push_back(const_cast<Expr *>(pValue->GetExpr()));
	}
	if (exprArgs.empty()) {
		ExprVisitor_GatherArgs visitor(exprArgs);
		pExprBlock->Accept(visitor);
	}
	FunctionCustom *pFunc = new FunctionCustom(env, AScript_Symbol(_anonymous_),
									pExprBlock->IncRef(), FUNCTYPE_Function);
	ContextExpr contextExpr(context.GetAttrs(), SymbolSet::Null,
									NULL, NULL, Value::Null, exprArgs);
	if (!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
		Function::Delete(pFunc);
		return Value::Null;
	}
	Value result;
	result.InitAsFunction(env, pFunc);
	return result;
}

// matrix(nrows:number, ncols:number, value?)
AScript_DeclareFunction(matrix)
{
	DeclareArg(env, "nrows", VTYPE_Number);
	DeclareArg(env, "ncols", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_Any, OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(matrix)
{
	Value result;
	Value valueZero(0.);
	result.InitAsMatrix(env, context.GetInt(0), context.GetInt(1),
					context.IsValid(2)? context.GetValue(2) : valueZero);
	return result;
}

// @@ {block}
AScript_DeclareFunction(MatrixInit)
{
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementFunction(MatrixInit)
{
	Environment envLister(&env, ENVTYPE_Lister);
	const Expr_Block *pExprBlock = context.GetBlock();
	Value valueInit = pExprBlock->GetExprList().ExecForList(envLister, sig, false, false);
	if (sig.IsSignalled()) return Value::Null;
	Value result;
	result.InitAsMatrix(env, sig, valueInit.GetList());
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// dict(elem[]?) {block?}
AScript_DeclareFunction(dict)
{
	DeclareArg(env, "elem", VTYPE_Any, OCCUR_ZeroOrOnce, true);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(dict)
{
	Object_Dict *pObj = new Object_Dict(env.LookupClass(VTYPE_Dict));
	Value result(pObj, VTYPE_Dict);
	ValueDict &valDict = pObj->GetDict();
	if (context.GetValue(0).IsList()) {
		if (!valDict.Store(sig, context.GetList(0))) return Value::Null;
	}
	if (context.IsBlockSpecified()) {
		Environment envLister(&env, ENVTYPE_Lister);
		Value valueList =
			context.GetBlock()->GetExprList().ExecForList(envLister, sig, false, false);
		if (sig.IsSignalled() || !valueList.IsList()) return Value::Null;
		if (!valDict.Store(sig, valueList.GetList())) return Value::Null;
	}
	return result;
}

// open(filename:string, mode:string => "rt", encoding:string => "iso-8859-1"):map {block?}
AScript_DeclareFunction(open)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "filename", VTYPE_String);
	DeclareArg(env, "mode", VTYPE_String, OCCUR_Once, false, false,
												new Expr_String("rt"));
	DeclareArg(env, "encoding", VTYPE_String, OCCUR_Once, false, false,
												new Expr_String("iso-8859-1"));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(open)
{
	const char *fileName = context.GetString(0);
	const char *mode = context.GetString(1);
	const char *encoding = context.GetString(2);
	Value result;
	Object_File *pObj = result.InitAsFile(env);
	pObj->GetFile().Open(sig, fileName, mode, encoding);
	if (sig.IsSignalled()) return Value::Null;
	if (context.IsBlockSpecified()) {
		const Function *pFuncBlock = GetBlockFunction(env, sig, context);
		if (pFuncBlock == NULL) return Value::Null;
		ValueList valListArg(result);
		Context contextSub(valListArg);
		pFuncBlock->Eval(env, sig, contextSub);
		result = Value::Null;	// object is destroyed here
	}
	return result;
}

// semaphore()
AScript_DeclareFunction(semaphore)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(semaphore)
{
	return Value(new Object_Semaphore(env.LookupClass(VTYPE_Semaphore)), VTYPE_Semaphore);
}

// Module entry
AScript_ModuleEntry()
{
	// symbol realization
	AScript_RealizePrivSymbol(open);
	AScript_RealizePrivSymbol(open_l);
	AScript_RealizePrivSymbol(open_r);
	// value assignment
	AScript_AssignValue(__name__,	Value(env, "__main__"));
	AScript_AssignValue(nil,		Value::Null);
	AScript_AssignValueEx("true",	Value(true));
	AScript_AssignValueEx("false",	Value(false));
	AScript_AssignValueEx("@rem",	Value::Null); // dummy for MS-DOS batch
	AScript_AssignValueEx("*",		Value(env, new Iterator_SequenceInf(0)));
	// function assignment
	AScript_AssignFunction(class_);
	AScript_AssignFunction(struct_);
	AScript_AssignFunction(try_);
	AScript_AssignFunction(except_);
	AScript_AssignFunction(finally_);
	AScript_AssignFunction(if_);
	AScript_AssignFunction(elsif_);
	AScript_AssignFunction(else_);
	AScript_AssignFunction(repeat);
	AScript_AssignFunction(while_);
	AScript_AssignFunction(for_);
	AScript_AssignFunction(cross);
	AScript_AssignFunction(break_);
	AScript_AssignFunction(continue_);
	AScript_AssignFunction(return_);
	AScript_AssignFunction(raise);
	AScript_AssignFunction(iterator);
	AScript_AssignFunction(list);
	AScript_AssignFunction(xlist);
	AScript_AssignFunctionEx(set_xset, "set");
	AScript_AssignFunctionEx(set_xset, "xset");
	AScript_AssignFunction(zip);
	AScript_AssignFunction(min);
	AScript_AssignFunction(max);
	AScript_AssignFunction(choose);
	AScript_AssignFunction(chooseif);
	AScript_AssignFunction(module);
	AScript_AssignFunction(import_);
	AScript_AssignFunction(parsefile);
	AScript_AssignFunction(parse);
	AScript_AssignFunction(eval);
	AScript_AssignFunction(locals);
	AScript_AssignFunction(outers);
	AScript_AssignFunction(tostring);
	AScript_AssignFunction(tonumber);
	AScript_AssignFunction(tosymbol);
	AScript_AssignFunction(rand);
	AScript_AssignFunction(int_);
	AScript_AssignFunction(ord);
	AScript_AssignFunction(chr);
	AScript_AssignFunction(hex);
	AScript_AssignFunction(print);
	AScript_AssignFunction(println);
	AScript_AssignFunction(printf);
	AScript_AssignFunction(dir);
	AScript_AssignFunction(help);
	AScript_AssignFunction(isdefined);
	AScript_AssignFunction(typename_);
	AScript_AssignFunction(undef_);
	AScript_AssignFunctionExx(istype_, "issymbol",		VTYPE_Symbol);
	AScript_AssignFunctionExx(istype_, "isboolean",		VTYPE_Boolean);
	AScript_AssignFunctionExx(istype_, "isnumber",		VTYPE_Number);
	AScript_AssignFunctionExx(istype_, "iscomplex",		VTYPE_Complex);
	AScript_AssignFunctionExx(istype_, "isfunction",	VTYPE_Function);
	AScript_AssignFunctionExx(istype_, "isstring",		VTYPE_String);
	AScript_AssignFunctionExx(istype_, "isbytes",		VTYPE_Bytes);
	AScript_AssignFunctionExx(istype_, "islist",		VTYPE_List);
	AScript_AssignFunctionExx(istype_, "ismatrix",		VTYPE_Matrix);
	AScript_AssignFunctionExx(istype_, "isdict",		VTYPE_Dict);
	AScript_AssignFunctionExx(istype_, "isfile",		VTYPE_File);
	AScript_AssignFunctionExx(istype_, "isfilestat",	VTYPE_FileStat);
	AScript_AssignFunctionExx(istype_, "isdatetime",	VTYPE_DateTime);
	AScript_AssignFunctionExx(istype_, "istimedelta",	VTYPE_TimeDelta);
	AScript_AssignFunctionExx(istype_, "isiterator",	VTYPE_Iterator);
	AScript_AssignFunctionExx(istype_, "isexpr",		VTYPE_Expr);
	AScript_AssignFunctionExx(istype_, "isenvironment",	VTYPE_Environment);
	AScript_AssignFunctionExx(istype_, "iserror",		VTYPE_Error);
	AScript_AssignFunctionExx(istype_, "isuri",			VTYPE_URI);
	AScript_AssignFunctionExx(istype_, "issemaphore",	VTYPE_Semaphore);
	AScript_AssignFunctionExx(istype_, "ismodule",		VTYPE_Module);
	AScript_AssignFunctionExx(istype_, "isclass",		VTYPE_Class);
	AScript_AssignFunction(istype);
	AScript_AssignFunction(isinstance);
	AScript_AssignFunction(classref);
	AScript_AssignFunction(range);
	AScript_AssignFunction(interval);
	AScript_AssignFunction(fill);
	AScript_AssignFunction(rands);
	AScript_AssignFunction(bytes);
	AScript_AssignFunction(pack);
	AScript_AssignFunctionEx(ListInit, "@");
	AScript_AssignFunction(lambda);
	AScript_AssignFunctionEx(lambda, "&");
	AScript_AssignFunction(matrix);
	AScript_AssignFunctionEx(MatrixInit, "@@");
	AScript_AssignFunction(dict);
	AScript_AssignFunctionEx(dict, "%");
	AScript_AssignFunction(open);
	AScript_AssignFunction(semaphore);
}

AScript_ModuleTerminate()
{
}

AScript_EndModule(__builtins__)

AScript_RegisterModule(__builtins__)
