//-----------------------------------------------------------------------------
// AScript net module
//-----------------------------------------------------------------------------
#include "Module.h"
#include "Object_URI.h"

AScript_BeginModule(net)

struct ServiceInfo {
	const char *name;
	int port;
	int protocol;
};

AScript_DeclarePrivClass(Socket);

// declaration of symbols privately used in this module
AScript_DeclarePrivSymbol(tcp);
AScript_DeclarePrivSymbol(udp);

//-----------------------------------------------------------------------------
// Object_Socket
//-----------------------------------------------------------------------------
class Object_Socket : public Object {
public:
	inline static Object_Socket *GetSelfObj(Context &context) {
		return dynamic_cast<Object_Socket *>(context.GetSelfObj());
	}
private:
	int _af;
	SOCKET _sock;
public:
	Object_Socket::Object_Socket(Signal sig, int af, int type, int protocol);
	virtual ~Object_Socket();
	bool Connect(const char *strAddr, short port);
	void Close();
	int Send(const char *buff, int len, int flags);
	int Recv(char *buff, int len, int flags);
};

Object_Socket::Object_Socket(Signal sig, int af, int type, int protocol) :
							Object(AScript_PrivClass(Socket)), _af(af)
{
	_sock = ::socket(af, type, protocol);
	if (_sock == INVALID_SOCKET) {
		sig.SetError(ERR_IOError, "failed to create a socket");
	}
}

Object_Socket::~Object_Socket()
{
	Close();
}

bool Object_Socket::Connect(const char *strAddr, short port)
{
	sockaddr_in addr;
	::memset(&addr, 0x00, sizeof(addr));
	addr.sin_family = _af;
	addr.sin_addr.s_addr = ::inet_addr(strAddr);
	addr.sin_port = ::htons(port);
	int rtn = ::connect(_sock, reinterpret_cast<sockaddr *>(&addr), sizeof(addr));
	return rtn != SOCKET_ERROR;
}

void Object_Socket::Close()
{
	::closesocket(_sock);
}

int Object_Socket::Send(const char *buff, int len, int flags)
{
	return ::send(_sock, buff, len, flags);
}

int Object_Socket::Recv(char *buff, int len, int flags)
{
	return ::recv(_sock, buff, len, flags);
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Socket
//-----------------------------------------------------------------------------
// net.socket#close()
AScript_DeclareMethod(Socket, close)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Socket, close)
{
	Object_Socket *pSelf = Object_Socket::GetSelfObj(context);
	pSelf->Close();
	return Value::Null;
}

// net.socket#getaddr()
AScript_DeclareMethod(Socket, getaddr)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Socket, getaddr)
{
	return Value::Null;
}

// net.socket#recv(len:number)
AScript_DeclareMethod(Socket, recv)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "len", VTYPE_Number);
}

AScript_ImplementMethod(Socket, recv)
{
	Object_Socket *pSelf = Object_Socket::GetSelfObj(context);
	int len = context.GetInt(0);
	char *buff = new char [len];
	pSelf->Recv(buff, len, 0);
	Value result;
	result.InitAsBytes(env, buff);
	delete [] buff;
	return result;
}

// net.socket#send(buff:bytes)
AScript_DeclareMethod(Socket, send)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "buff", VTYPE_Bytes);
}

AScript_ImplementMethod(Socket, send)
{
	Object_Socket *pSelf = Object_Socket::GetSelfObj(context);
	const Bytes &bytes = context.GetBytes(0);
	pSelf->Send(bytes.data(), static_cast<int>(bytes.size()), 0);
	return Value::Null;
}

// net.socket#accept()
AScript_DeclareMethod(Socket, accept)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Socket, accept)
{
	int af = AF_INET;
	int type = SOCK_STREAM;
	int protocol = IPPROTO_TCP;
	Object_Socket *pObj = new Object_Socket(sig, af, type, protocol);
	if (sig.IsSignalled()) {
		delete pObj;
		return Value::Null;
	}
	return Value(pObj, AScript_PrivVTYPE(Socket));
}

// implementation of class Socket
AScript_ImplementPrivClass(Socket)
{
	AScript_AssignMethod(Socket, close);
	AScript_AssignMethod(Socket, getaddr);
	AScript_AssignMethod(Socket, recv);
	AScript_AssignMethod(Socket, send);
	AScript_AssignMethod(Socket, accept);
}

//-----------------------------------------------------------------------------
// AScript module functions: net
//-----------------------------------------------------------------------------
// s = net.connect(host:string, port):map:[tcp,udp]
AScript_DeclareFunction(connect)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "host", VTYPE_String);
	DeclareArg(env, "port", VTYPE_Number);
	DeclareAttr(AScript_PrivSymbol(tcp));
	DeclareAttr(AScript_PrivSymbol(udp));
}

AScript_ImplementFunction(connect)
{
	int af = AF_INET;
	int type = SOCK_STREAM;
	int protocol = IPPROTO_TCP;
	Object_Socket *pObj = new Object_Socket(sig, af, type, protocol);
	if (sig.IsSignalled()) {
		delete pObj;
		return Value::Null;
	}
	if (!pObj->Connect(context.GetString(0), context.GetInt(1))) {
		sig.SetError(ERR_IOError, "failed to connect to a host");
		delete pObj;
		return Value::Null;
	}
	return Value(pObj, AScript_PrivVTYPE(Socket));
}

// s = net.server(port)
AScript_DeclareFunction(server)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "port", VTYPE_Any);
}

AScript_ImplementFunction(server)
{
	int af = AF_INET;
	int type = SOCK_STREAM;
	int protocol = IPPROTO_TCP;
	Object_Socket *pObj = new Object_Socket(sig, af, type, protocol);
	if (sig.IsSignalled()) {
		delete pObj;
		return Value::Null;
	}
	return Value(pObj, AScript_PrivVTYPE(Socket));
}

// net.uri(scheme?:string, user?:string, password?:string,
//         host?:string, port?:string, urlpath?:string)
AScript_DeclareFunction(uri)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "scheme", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareArg(env, "user", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareArg(env, "password", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareArg(env, "host", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareArg(env, "port", VTYPE_String, OCCUR_ZeroOrOnce);
	DeclareArg(env, "urlpath", VTYPE_String, OCCUR_ZeroOrOnce);
	SetHelp(
	"Returns a URI instance that has specified field values.");
}

AScript_ImplementFunction(uri)
{
	Object_URI *pObj = new Object_URI(env.LookupClass(VTYPE_URI));
	if (context.IsString(0)) pObj->SetScheme(context.GetString(0));
	if (context.IsString(1)) pObj->SetUser(context.GetString(1));
	if (context.IsString(2)) pObj->SetPassword(context.GetString(2));
	if (context.IsString(3)) pObj->SetHost(context.GetString(3));
	if (context.IsString(4)) pObj->SetPort(context.GetString(4));
	if (context.IsString(5)) pObj->SetUrlPath(context.GetString(5));
	return Value(pObj, VTYPE_URI);
}

// net.parseuri(str:string)
AScript_DeclareFunction(parseuri)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	SetHelp(
	"Returns a URI instance after parsing str according to RFC1738.");
}

AScript_ImplementFunction(parseuri)
{
	Object_URI *pObj = new Object_URI(env.LookupClass(VTYPE_URI));
	if (!pObj->Parse(sig, context.GetString(0))) {
		Object::Delete(pObj);
		return Value::Null;
	}
	return Value(pObj, VTYPE_URI);
}

// Module entry
AScript_ModuleEntry()
{
	WSADATA wsaData;
	::WSAStartup(MAKEWORD(2, 0), &wsaData);
	// symbol realization
	AScript_RealizePrivSymbol(tcp);
	AScript_RealizePrivSymbol(udp);
	// class realization
	AScript_RealizePrivClass(Socket, "socket");
	// function assignment
	AScript_AssignFunction(connect);
	AScript_AssignFunction(server);
	AScript_AssignFunction(uri);
	AScript_AssignFunction(parseuri);
}

AScript_ModuleTerminate()
{
}

#if 0
static const ServiceInfo _serviceInfoTbl[] = {
	{ "echo",			   7, IPPROTO_TCP },
	{ "echo",			   7, IPPROTO_UDP },
	{ "discard",		   9, IPPROTO_TCP },
	{ "discard",		   9, IPPROTO_UDP },
	{ "systat",			  11, IPPROTO_TCP },
	{ "systat",			  11, IPPROTO_TCP },
	{ "daytime",		  13, IPPROTO_TCP },
	{ "daytime",		  13, IPPROTO_UDP },
	{ "qotd",			  17, IPPROTO_TCP },
	{ "qotd",			  17, IPPROTO_UDP },
	{ "chargen",		  19, IPPROTO_TCP },
	{ "chargen",		  19, IPPROTO_UDP },
	{ "ftp",			  21, IPPROTO_TCP },
	{ "telnet",			  23, IPPROTO_TCP },
	{ "smtp",			  25, IPPROTO_TCP },
	{ "time",			  37, IPPROTO_TCP },
	{ "time",			  37, IPPROTO_UDP },
	{ "rlp",			  39, IPPROTO_UDP },
	{ "nameserver",		  42, IPPROTO_TCP },
	{ "nameserver",		  42, IPPROTO_UDP },
	{ "nicname",		  43, IPPROTO_TCP },
	{ "domain",			  53, IPPROTO_TCP },
	{ "domain",			  53, IPPROTO_UDP },
	{ "bootps",			  67, IPPROTO_UDP },
	{ "bootpc",			  68, IPPROTO_UDP },
	{ "tftp",			  69, IPPROTO_UDP },
	{ "gopher",			  70, IPPROTO_TCP },
	{ "finger",			  79, IPPROTO_TCP },
	{ "http",			  80, IPPROTO_TCP },
	{ "kerberos",		  88, IPPROTO_TCP },
	{ "kerberos",		  88, IPPROTO_UDP },
	{ "hostname",		 101, IPPROTO_TCP },
	{ "rtelnet",		 107, IPPROTO_TCP },
	{ "pop2",			 109, IPPROTO_TCP },
	{ "pop3",			 110, IPPROTO_TCP },
	{ "sunrpc",			 111, IPPROTO_TCP },
	{ "sunrpc",			 111, IPPROTO_UDP },
	{ "auth",			 113, IPPROTO_TCP },
	{ "nntp",			 119, IPPROTO_TCP },
	{ "ntp",			 123, IPPROTO_UDP },
	{ "epmap",			 135, IPPROTO_TCP },
	{ "epmap",			 135, IPPROTO_UDP },
	{ "imap",			 143, IPPROTO_TCP },
	{ "snmp",			 161, IPPROTO_UDP },
	{ "snmptrap",		 162, IPPROTO_UDP },
	{ "bgp",			 179, IPPROTO_TCP },
	{ "irc",			 194, IPPROTO_TCP },
	{ "ipx",			 213, IPPROTO_UDP },
	{ "ldap",			 389, IPPROTO_TCP },
	{ "https",			 443, IPPROTO_TCP },
	{ "https",			 443, IPPROTO_UDP },
	{ "kpasswd",		 464, IPPROTO_TCP },
	{ "kpasswd",		 464, IPPROTO_UDP },
	{ "isakmp",			 500, IPPROTO_UDP },
	{ "exec",			 512, IPPROTO_TCP },
	{ "biff",			 512, IPPROTO_UDP },
	{ "login",			 513, IPPROTO_TCP },
	{ "who",			 513, IPPROTO_UDP },
	{ "cmd",			 514, IPPROTO_TCP },
	{ "syslog",			 514, IPPROTO_UDP },
	{ "printer",		 515, IPPROTO_TCP },
	{ "talk",			 517, IPPROTO_UDP },
	{ "ntalk",			 518, IPPROTO_UDP },
	{ "efs",			 520, IPPROTO_TCP },
	{ "router",			 520, IPPROTO_UDP },
	{ "timed",			 525, IPPROTO_UDP },
	{ "tempo",			 526, IPPROTO_TCP },
	{ "courier",		 530, IPPROTO_TCP },
	{ "conference",		 531, IPPROTO_TCP },
	{ "netnews",		 532, IPPROTO_TCP },
	{ "netwall",		 533, IPPROTO_UDP },
	{ "uucp",			 540, IPPROTO_TCP },
	{ "klogin",			 543, IPPROTO_TCP },
	{ "kshell",			 544, IPPROTO_TCP },
	{ "remotefs",		 556, IPPROTO_TCP },
	{ "rmonitor",		 560, IPPROTO_UDP },
	{ "monitor",		 561, IPPROTO_UDP },
	{ "ldaps",			 636, IPPROTO_TCP },
	{ "doom",			 666, IPPROTO_TCP },
	{ "doom",			 666, IPPROTO_UDP },
	{ "kpop",			1109, IPPROTO_TCP },
	{ "phone",			1167, IPPROTO_UDP },
	{ "wins",			1512, IPPROTO_TCP },
	{ "wins",			1512, IPPROTO_UDP },
	{ "ingreslock",		1524, IPPROTO_TCP },
	{ "l2tp",			1701, IPPROTO_UDP },
	{ "pptp",			1723, IPPROTO_TCP },
	{ "radius",			1812, IPPROTO_UDP },
	{ "radacct",		1813, IPPROTO_UDP },
	{ "nfsd",			2049, IPPROTO_UDP },
	{ "knetd",			2053, IPPROTO_TCP },
	{ "man",			9535, IPPROTO_TCP },
	{ "sapdp00",		3200, IPPROTO_TCP },
	{ "sapdp01",		3201, IPPROTO_TCP },
	{ "sapdp02",		3202, IPPROTO_TCP },
	{ "sapdp03",		3203, IPPROTO_TCP },
	{ "sapdp04",		3204, IPPROTO_TCP },
	{ "sapdp05",		3205, IPPROTO_TCP },
	{ "sapdp06",		3206, IPPROTO_TCP },
	{ "sapdp07",		3207, IPPROTO_TCP },
	{ "sapdp08",		3208, IPPROTO_TCP },
	{ "sapdp09",		3209, IPPROTO_TCP },
	{ "sapdp10",		3210, IPPROTO_TCP },
	{ "sapdp11",		3211, IPPROTO_TCP },
	{ "sapdp12",		3212, IPPROTO_TCP },
	{ "sapdp13",		3213, IPPROTO_TCP },
	{ "sapdp14",		3214, IPPROTO_TCP },
	{ "sapdp15",		3215, IPPROTO_TCP },
	{ "sapdp16",		3216, IPPROTO_TCP },
	{ "sapdp17",		3217, IPPROTO_TCP },
	{ "sapdp18",		3218, IPPROTO_TCP },
	{ "sapdp19",		3219, IPPROTO_TCP },
	{ "sapdp20",		3220, IPPROTO_TCP },
	{ "sapdp21",		3221, IPPROTO_TCP },
	{ "sapdp22",		3222, IPPROTO_TCP },
	{ "sapdp23",		3223, IPPROTO_TCP },
	{ "sapdp24",		3224, IPPROTO_TCP },
	{ "sapdp25",		3225, IPPROTO_TCP },
	{ "sapdp26",		3226, IPPROTO_TCP },
	{ "sapdp27",		3227, IPPROTO_TCP },
	{ "sapdp28",		3228, IPPROTO_TCP },
	{ "sapdp29",		3229, IPPROTO_TCP },
	{ "sapdp30",		3230, IPPROTO_TCP },
	{ "sapdp31",		3231, IPPROTO_TCP },
	{ "sapdp32",		3232, IPPROTO_TCP },
	{ "sapdp33",		3233, IPPROTO_TCP },
	{ "sapdp34",		3234, IPPROTO_TCP },
	{ "sapdp35",		3235, IPPROTO_TCP },
	{ "sapdp36",		3236, IPPROTO_TCP },
	{ "sapdp37",		3237, IPPROTO_TCP },
	{ "sapdp38",		3238, IPPROTO_TCP },
	{ "sapdp39",		3239, IPPROTO_TCP },
	{ "sapdp40",		3240, IPPROTO_TCP },
	{ "sapdp41",		3241, IPPROTO_TCP },
	{ "sapdp42",		3242, IPPROTO_TCP },
	{ "sapdp43",		3243, IPPROTO_TCP },
	{ "sapdp44",		3244, IPPROTO_TCP },
	{ "sapdp45",		3245, IPPROTO_TCP },
	{ "sapdp46",		3246, IPPROTO_TCP },
	{ "sapdp47",		3247, IPPROTO_TCP },
	{ "sapdp48",		3248, IPPROTO_TCP },
	{ "sapdp49",		3249, IPPROTO_TCP },
	{ "sapdp50",		3250, IPPROTO_TCP },
	{ "sapdp51",		3251, IPPROTO_TCP },
	{ "sapdp52",		3252, IPPROTO_TCP },
	{ "sapdp53",		3253, IPPROTO_TCP },
	{ "sapdp54",		3254, IPPROTO_TCP },
	{ "sapdp55",		3255, IPPROTO_TCP },
	{ "sapdp56",		3256, IPPROTO_TCP },
	{ "sapdp57",		3257, IPPROTO_TCP },
	{ "sapdp58",		3258, IPPROTO_TCP },
	{ "sapdp59",		3259, IPPROTO_TCP },
	{ "sapdp60",		3260, IPPROTO_TCP },
	{ "sapdp61",		3261, IPPROTO_TCP },
	{ "sapdp62",		3262, IPPROTO_TCP },
	{ "sapdp63",		3263, IPPROTO_TCP },
	{ "sapdp64",		3264, IPPROTO_TCP },
	{ "sapdp65",		3265, IPPROTO_TCP },
	{ "sapdp66",		3266, IPPROTO_TCP },
	{ "sapdp67",		3267, IPPROTO_TCP },
	{ "sapdp68",		3268, IPPROTO_TCP },
	{ "sapdp69",		3269, IPPROTO_TCP },
	{ "sapdp70",		3270, IPPROTO_TCP },
	{ "sapdp71",		3271, IPPROTO_TCP },
	{ "sapdp72",		3272, IPPROTO_TCP },
	{ "sapdp73",		3273, IPPROTO_TCP },
	{ "sapdp74",		3274, IPPROTO_TCP },
	{ "sapdp75",		3275, IPPROTO_TCP },
	{ "sapdp76",		3276, IPPROTO_TCP },
	{ "sapdp77",		3277, IPPROTO_TCP },
	{ "sapdp78",		3278, IPPROTO_TCP },
	{ "sapdp79",		3279, IPPROTO_TCP },
	{ "sapdp80",		3280, IPPROTO_TCP },
	{ "sapdp81",		3281, IPPROTO_TCP },
	{ "sapdp82",		3282, IPPROTO_TCP },
	{ "sapdp83",		3283, IPPROTO_TCP },
	{ "sapdp84",		3284, IPPROTO_TCP },
	{ "sapdp85",		3285, IPPROTO_TCP },
	{ "sapdp86",		3286, IPPROTO_TCP },
	{ "sapdp87",		3287, IPPROTO_TCP },
	{ "sapdp88",		3288, IPPROTO_TCP },
	{ "sapdp89",		3289, IPPROTO_TCP },
	{ "sapdp90",		3290, IPPROTO_TCP },
	{ "sapdp91",		3291, IPPROTO_TCP },
	{ "sapdp92",		3292, IPPROTO_TCP },
	{ "sapdp93",		3293, IPPROTO_TCP },
	{ "sapdp94",		3294, IPPROTO_TCP },
	{ "sapdp95",		3295, IPPROTO_TCP },
	{ "sapdp96",		3296, IPPROTO_TCP },
	{ "sapdp97",		3297, IPPROTO_TCP },
	{ "sapdp98",		3298, IPPROTO_TCP },
	{ "sapdp99",		3299, IPPROTO_TCP },
	{ "sapgw00",		3300, IPPROTO_TCP },
	{ "sapgw01",		3301, IPPROTO_TCP },
	{ "sapgw02",		3302, IPPROTO_TCP },
	{ "sapgw03",		3303, IPPROTO_TCP },
	{ "sapgw04",		3304, IPPROTO_TCP },
	{ "sapgw05",		3305, IPPROTO_TCP },
	{ "sapgw06",		3306, IPPROTO_TCP },
	{ "sapgw07",		3307, IPPROTO_TCP },
	{ "sapgw08",		3308, IPPROTO_TCP },
	{ "sapgw09",		3309, IPPROTO_TCP },
	{ "sapgw10",		3310, IPPROTO_TCP },
	{ "sapgw11",		3311, IPPROTO_TCP },
	{ "sapgw12",		3312, IPPROTO_TCP },
	{ "sapgw13",		3313, IPPROTO_TCP },
	{ "sapgw14",		3314, IPPROTO_TCP },
	{ "sapgw15",		3315, IPPROTO_TCP },
	{ "sapgw16",		3316, IPPROTO_TCP },
	{ "sapgw17",		3317, IPPROTO_TCP },
	{ "sapgw18",		3318, IPPROTO_TCP },
	{ "sapgw19",		3319, IPPROTO_TCP },
	{ "sapgw20",		3320, IPPROTO_TCP },
	{ "sapgw21",		3321, IPPROTO_TCP },
	{ "sapgw22",		3322, IPPROTO_TCP },
	{ "sapgw23",		3323, IPPROTO_TCP },
	{ "sapgw24",		3324, IPPROTO_TCP },
	{ "sapgw25",		3325, IPPROTO_TCP },
	{ "sapgw26",		3326, IPPROTO_TCP },
	{ "sapgw27",		3327, IPPROTO_TCP },
	{ "sapgw28",		3328, IPPROTO_TCP },
	{ "sapgw29",		3329, IPPROTO_TCP },
	{ "sapgw30",		3330, IPPROTO_TCP },
	{ "sapgw31",		3331, IPPROTO_TCP },
	{ "sapgw32",		3332, IPPROTO_TCP },
	{ "sapgw33",		3333, IPPROTO_TCP },
	{ "sapgw34",		3334, IPPROTO_TCP },
	{ "sapgw35",		3335, IPPROTO_TCP },
	{ "sapgw36",		3336, IPPROTO_TCP },
	{ "sapgw37",		3337, IPPROTO_TCP },
	{ "sapgw38",		3338, IPPROTO_TCP },
	{ "sapgw39",		3339, IPPROTO_TCP },
	{ "sapgw40",		3340, IPPROTO_TCP },
	{ "sapgw41",		3341, IPPROTO_TCP },
	{ "sapgw42",		3342, IPPROTO_TCP },
	{ "sapgw43",		3343, IPPROTO_TCP },
	{ "sapgw44",		3344, IPPROTO_TCP },
	{ "sapgw45",		3345, IPPROTO_TCP },
	{ "sapgw46",		3346, IPPROTO_TCP },
	{ "sapgw47",		3347, IPPROTO_TCP },
	{ "sapgw48",		3348, IPPROTO_TCP },
	{ "sapgw49",		3349, IPPROTO_TCP },
	{ "sapgw50",		3350, IPPROTO_TCP },
	{ "sapgw51",		3351, IPPROTO_TCP },
	{ "sapgw52",		3352, IPPROTO_TCP },
	{ "sapgw53",		3353, IPPROTO_TCP },
	{ "sapgw54",		3354, IPPROTO_TCP },
	{ "sapgw55",		3355, IPPROTO_TCP },
	{ "sapgw56",		3356, IPPROTO_TCP },
	{ "sapgw57",		3357, IPPROTO_TCP },
	{ "sapgw58",		3358, IPPROTO_TCP },
	{ "sapgw59",		3359, IPPROTO_TCP },
	{ "sapgw60",		3360, IPPROTO_TCP },
	{ "sapgw61",		3361, IPPROTO_TCP },
	{ "sapgw62",		3362, IPPROTO_TCP },
	{ "sapgw63",		3363, IPPROTO_TCP },
	{ "sapgw64",		3364, IPPROTO_TCP },
	{ "sapgw65",		3365, IPPROTO_TCP },
	{ "sapgw66",		3366, IPPROTO_TCP },
	{ "sapgw67",		3367, IPPROTO_TCP },
	{ "sapgw68",		3368, IPPROTO_TCP },
	{ "sapgw69",		3369, IPPROTO_TCP },
	{ "sapgw70",		3370, IPPROTO_TCP },
	{ "sapgw71",		3371, IPPROTO_TCP },
	{ "sapgw72",		3372, IPPROTO_TCP },
	{ "sapgw73",		3373, IPPROTO_TCP },
	{ "sapgw74",		3374, IPPROTO_TCP },
	{ "sapgw75",		3375, IPPROTO_TCP },
	{ "sapgw76",		3376, IPPROTO_TCP },
	{ "sapgw77",		3377, IPPROTO_TCP },
	{ "sapgw78",		3378, IPPROTO_TCP },
	{ "sapgw79",		3379, IPPROTO_TCP },
	{ "sapgw80",		3380, IPPROTO_TCP },
	{ "sapgw81",		3381, IPPROTO_TCP },
	{ "sapgw82",		3382, IPPROTO_TCP },
	{ "sapgw83",		3383, IPPROTO_TCP },
	{ "sapgw84",		3384, IPPROTO_TCP },
	{ "sapgw85",		3385, IPPROTO_TCP },
	{ "sapgw86",		3386, IPPROTO_TCP },
	{ "sapgw87",		3387, IPPROTO_TCP },
	{ "sapgw88",		3388, IPPROTO_TCP },
	{ "sapgw89",		3389, IPPROTO_TCP },
	{ "sapgw90",		3390, IPPROTO_TCP },
	{ "sapgw91",		3391, IPPROTO_TCP },
	{ "sapgw92",		3392, IPPROTO_TCP },
	{ "sapgw93",		3393, IPPROTO_TCP },
	{ "sapgw94",		3394, IPPROTO_TCP },
	{ "sapgw95",		3395, IPPROTO_TCP },
	{ "sapgw96",		3396, IPPROTO_TCP },
	{ "sapgw97",		3397, IPPROTO_TCP },
	{ "sapgw98",		3398, IPPROTO_TCP },
	{ "sapgw99",		3399, IPPROTO_TCP },
	{ "sapmsD90",		3615, IPPROTO_TCP },
	{ "sapmsP58",		3600, IPPROTO_TCP },
	{ "sapmsP87",		3665, IPPROTO_TCP },
	{ "sapmsQ87",		3621, IPPROTO_TCP },
	{ "sapmsPC1",		3670, IPPROTO_TCP },
	{ "sapmsPBG",		3675, IPPROTO_TCP },
	{ "sapmsPCM",		3685, IPPROTO_TCP },
	{ "sapmsQCM",		3654, IPPROTO_TCP },
	{ "sapmsP27",		3602, IPPROTO_TCP },
	{ "sapmsQ98",		3652, IPPROTO_TCP },
	{ "sapmsPBD",		3655, IPPROTO_TCP },
	{ "sapmsD33",		3600, IPPROTO_TCP },
	{ "sapmsP98",		3650, IPPROTO_TCP },
	{ "sapmsKP1",		3607, IPPROTO_TCP },
	{ "sapmsDBT",		3630, IPPROTO_TCP },
	{ "sapmsQBT",		3635, IPPROTO_TCP },
	{ "sapmsPBT",		3605, IPPROTO_TCP },
	{ "sapmsQGJ",		3650, IPPROTO_TCP },
	{ "sapmsPB7",		3609, IPPROTO_TCP },
	{ "sapmsQB7",		3607, IPPROTO_TCP },
	{ "sapmsE87",		3663, IPPROTO_TCP },
	{ "sapmsP13",		3600, IPPROTO_TCP },
	{ "sapmsP82",		3665, IPPROTO_TCP },
	{ "sapmsP72",		3625, IPPROTO_TCP },
	{ "sapmsSIP",		3680, IPPROTO_TCP },
	{ "sapmsSIQ",		3685, IPPROTO_TCP },
	{ "sapmsQ74",		3602, IPPROTO_TCP },
	{ "sapmsP74",		3604, IPPROTO_TCP },
	{ "sapmsQ38",		3600, IPPROTO_TCP },
	{ "sapmsP38",		3610, IPPROTO_TCP },
	{ "sapmsLBV",		3620, IPPROTO_TCP },
	{ "sapmsPBV",		3600, IPPROTO_TCP },
	{ "sapmsP81",		3600, IPPROTO_TCP },
	{ "sapmsP62",		3680, IPPROTO_TCP },
	{ "sapmsP64",		3600, IPPROTO_TCP },
	{ "sapmsP67",		3600, IPPROTO_TCP },
	{ "sapmsP97",		3605, IPPROTO_TCP },
	{ "sapmsPGJ",		3610, IPPROTO_TCP },
	{ "sapmsPCH",		3615, IPPROTO_TCP },
	{ "sapmsQ97",		3625, IPPROTO_TCP },
	{ "sapmsP90",		3600, IPPROTO_TCP },
	{ "sapmsQBV",		3625, IPPROTO_TCP },
	{ "sapmsPR1",		3602, IPPROTO_TCP },
	{ "sapmsPR2",		3603, IPPROTO_TCP },
	{ "sapmsPQI",		3605, IPPROTO_TCP },
	{ "sapmsQ90",		3620, IPPROTO_TCP },
	{ "sapmsPBF",		3660, IPPROTO_TCP },
	{ "sapmsPB8",		3610, IPPROTO_TCP },
	{ "sapmsPBE",		3612, IPPROTO_TCP },
	{ "sapmsP23",		3600, IPPROTO_TCP },
};
#endif

AScript_EndModule(net)

AScript_RegisterModule(net)
