//
// Object_Audio
//

#include "Object_Audio.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Audio
//-----------------------------------------------------------------------------
Object_Audio::~Object_Audio()
{
	FreeBuffer();
}

Object *Object_Audio::Clone() const
{
	return NULL; //new Object_Audio(*this);
}

bool Object_Audio::CheckEmpty(Signal sig) const
{
	if (!IsValid()) return true;
	sig.SetError(ERR_ValueError, "audio has already been initialized with a buffer");
	return false;
}

bool Object_Audio::CheckValidity(Signal sig) const
{
	if (IsValid()) return true;
	sig.SetError(ERR_ValueError, "audio does not have a buffer");
	return false;
}

void Object_Audio::ReferenceBuffer(Object *pObjRef, void *buff, size_t nLen)
{
	_ownerFlag = false;
	_pObjRef = pObjRef;
	_buff = buff;
	_nLen = nLen;
}

bool Object_Audio::AllocBuffer(Signal sig, size_t nLen)
{
	FreeBuffer();
	_ownerFlag = true;
	_pObjRef = NULL;
	_buff = OAL::AllocLargeMemory(_nChannels * nLen * GetBytesPerData());
	if (_buff == 0) {
		sig.SetError(ERR_MemoryError, "failed to allocate audio buffer");
		return false;
	}
	_nLen = nLen;
	return true;
}

void Object_Audio::FreeBuffer()
{
	if (_ownerFlag) OAL::FreeLargeMemory(_buff);
	Object::Delete(_pObjRef);
	_ownerFlag = false;
	_pObjRef = NULL;
	_buff = NULL;
	_nLen = 0;
}

Object_Audio::Format Object_Audio::SymbolToFormat(Signal sig, const Symbol *pSymbol)
{
	if (pSymbol->IsIdentical(AScript_Symbol(u8))) {
		return FORMAT_U8;
	} else if (pSymbol->IsIdentical(AScript_Symbol(s8))) {
		return FORMAT_S8;
	} else if (pSymbol->IsIdentical(AScript_Symbol(u16le))) {
		return FORMAT_U16LE;
	} else if (pSymbol->IsIdentical(AScript_Symbol(s16le))) {
		return FORMAT_S16LE;
	} else if (pSymbol->IsIdentical(AScript_Symbol(u16be))) {
		return FORMAT_U16BE;
	} else if (pSymbol->IsIdentical(AScript_Symbol(s16be))) {
		return FORMAT_S16BE;
	} else {
		sig.SetError(ERR_ValueError, "unsupported audio format: %s", pSymbol->GetName());
		return FORMAT_None;
	}
}

const Symbol *Object_Audio::FormatToSymbol(Format format)
{
	if (format == FORMAT_U8) {
		return AScript_Symbol(u8);
	} else if (format == FORMAT_S8) {
		return AScript_Symbol(s8);
	} else if (format == FORMAT_U16LE) {
		return AScript_Symbol(u16le);
	} else if (format == FORMAT_S16LE) {
		return AScript_Symbol(s16le);
	} else if (format == FORMAT_U16BE) {
		return AScript_Symbol(u16be);
	} else if (format == FORMAT_S16BE) {
		return AScript_Symbol(s16be);
	} else {
		return AScript_Symbol(nil);
	}
}

Value Object_Audio::EvalGetter(Signal sig, const Symbol *pSymbol, bool &evaluatedFlag)
{
	evaluatedFlag = true;
	if (pSymbol->IsIdentical(AScript_Symbol(format))) {
		return Value(FormatToSymbol(_format));
	} else if (pSymbol->IsIdentical(AScript_Symbol(channels))) {
		return Value(static_cast<unsigned int>(_nChannels));
	} else if (pSymbol->IsIdentical(AScript_Symbol(len))) {
		return Value(static_cast<unsigned int>(_nLen));
	}
	evaluatedFlag = false;
	return Value::Null;
}

Value Object_Audio::EvalSetter(Signal sig, const Symbol *pSymbol,
									const Value &value, bool &evaluatedFlag)
{
	return EvalGetter(sig, pSymbol, evaluatedFlag);
}

String Object_Audio::ToString(Signal sig, bool exprFlag)
{
	String rtn;
	rtn += "<audio:";
	rtn += FormatToSymbol(_format)->GetName();
	rtn += ":";
	do {
		char buff[32];
		::sprintf(buff, "%dch", _nChannels);
		rtn += buff;
	} while (0);
	rtn += ":";
	if (IsValid()) {
		char buff[32];
		::sprintf(buff, "%d", GetLength());
		rtn += buff;
	} else {
		rtn += "invalid";
	}
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Audio
//-----------------------------------------------------------------------------
// audio#allocbuff(len:number, data?:number)
AScript_DeclareMethod(Audio, allocbuff)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "len", VTYPE_Number);
	DeclareArg(env, "data", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Audio, allocbuff)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckEmpty(sig)) return Value::Null;
	pSelf->AllocBuffer(sig, context.GetSizeT(0));
	//if (context.IsNumber(1)) pSelf->Fill(sig, context.GetNumber(1));
	return Value::Null;
}

// audio#put(channel:number, offset:number, data:number)
AScript_DeclareMethod(Audio, put)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "channel", VTYPE_Number);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "data", VTYPE_Number);
}

AScript_ImplementMethod(Audio, put)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	return Value::Null;
}

// audio#get(channel:number, offset:number)
AScript_DeclareMethod(Audio, get)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "channel", VTYPE_Number);
	DeclareArg(env, "offset", VTYPE_Number);
}

AScript_ImplementMethod(Audio, get)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	return Value::Null;
}

// audio#store(channel:number, offset:number, len:number, src)
AScript_DeclareMethod(Audio, store)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "channel", VTYPE_Number);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "len", VTYPE_Number);
	DeclareArg(env, "src", VTYPE_Any);
}

AScript_ImplementMethod(Audio, store)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	return Value::Null;
}

// audio#extract(channel:number, offset:number, len:number, dst)
AScript_DeclareMethod(Audio, extract)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "channel", VTYPE_Number);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "len", VTYPE_Number);
	DeclareArg(env, "dst", VTYPE_Any);
}

AScript_ImplementMethod(Audio, extract)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	return Value::Null;
}

// audio#fill(data:number)
AScript_DeclareMethod(Audio, fill)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "data", VTYPE_Number);
}

AScript_ImplementMethod(Audio, fill)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	return Value::Null;
}

// audio#fillrange(channel:number, offset:number, len:number, data:number)
AScript_DeclareMethod(Audio, fillrange)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "channel", VTYPE_Number);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "len", VTYPE_Number);
	DeclareArg(env, "data", VTYPE_Number);
}

AScript_ImplementMethod(Audio, fillrange)
{
	Object_Audio *pSelf = Object_Audio::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	return Value::Null;
}

// assignment
Class_Audio::Class_Audio(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_AssignMethod(Audio, allocbuff);
	AScript_AssignMethod(Audio, put);
	AScript_AssignMethod(Audio, get);
	AScript_AssignMethod(Audio, store);
	AScript_AssignMethod(Audio, extract);
	AScript_AssignMethod(Audio, fill);
	AScript_AssignMethod(Audio, fillrange);
}

Object *Class_Audio::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
