// OAL: OS Abstraction Layer
#ifndef __OAL_H__
#define __OAL_H__
#include "Common.h"
#include "DateTime.h"
#if defined(HAVE_SEMAPHORE_H)
#include <semaphore.h>
#endif

#if defined(HAVE_WINDOWS_H)
#define ecvt _ecvt
#define fcvt _fcvt
#define gcvt _gcvt
#endif

namespace AScript {
namespace OAL {

DLLDECLARE String getenv(const char *name);
DLLDECLARE void putenv(const char *name, const char *value);
DLLDECLARE void rename(const char *src, const char *dst);
DLLDECLARE void remove(const char *pathName);
DLLDECLARE bool mkdir(const char *pathName, bool recursiveFlag);
DLLDECLARE bool chdir(const char *pathName);
DLLDECLARE String getcwd();
DLLDECLARE void exec(const char *pathName, const char **argv);
DLLDECLARE void Sleep(Number delay);
DLLDECLARE Number GetTickTime();
DLLDECLARE void GetCurDateTime(DateTime &dateTime, bool utcFlag = false);
#if defined(HAVE_WINDOWS_H)
DateTime ToDateTime(const SYSTEMTIME &st);
DateTime ToDateTime(const FILETIME &ft, bool utcFlag = false);
#else
DateTime ToDateTime(const struct tm &tm);
DateTime ToDateTime(time_t t, bool utcFlag = false);
#endif

class DLLDECLARE LargeMemory {
private:
	size_t _bytes;
	char *_buff;
public:
	inline LargeMemory(const LargeMemory &largeMem) :
								_bytes(largeMem._bytes), _buff(NULL) {
		::memcpy(Allocate(_bytes), largeMem.GetPointer(), _bytes);
	}
	LargeMemory(size_t bytes = 0);
	~LargeMemory();
	inline size_t GetSize() const { return _bytes; }
	void *Allocate(size_t bytes);
	void *Resize(size_t bytes, size_t bytesToCopy);
	void Free();
	void *GetPointer(size_t offset = 0) const;
};

class DLLDECLARE Thread {
public:
	void Start();
	virtual void Run() = 0;
};

class DLLDECLARE Semaphore {
private:
#if defined(HAVE_WINDOWS_H)
	HANDLE _hMutex;
#elif defined(HAVE_SEMAPHORE_H)
	sem_t _sem;
#endif
public:
	Semaphore();
	~Semaphore();
	void Wait();
	void Release();
};

class DLLDECLARE Event {
private:
#if defined(HAVE_WINDOWS_H)
	HANDLE _hEvent;
#else
#endif
public:
	Event();
	~Event();
	void Wait();
	void Notify();
};

}
}

#endif
