#
# A.T.Edit -- Another Tcl/Tk based text editor
# Copyright (C) 2004 Shinji Moiino
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##-- main.tcl --
#
### Package extension #########################################################
lappend auto_path [file join $BINDIR BWidget-1.7.0]
package require BWidget

### Global variables. #########################################################
source [file join $BINDIR global.tcl]

### Analyze arguments #########################################################
set FILES {}        ;# list of files specified by command line.
if {$tcl_platform(platform) == "unix"} {
    set CONFDIR [file join $env(HOME) .atedit]
} else {
    set CONFDIR [file join $CONFDIR $tcl_platform(user)]
}
#
# Read user configuration file.
append conf $CONFDIR / $CONFFILE
if [catch {open $conf r} fd] {
    # configuration file not found. Do nothing.
} else {
    while {[gets $fd line] >= 0} {
        if {[llength $line] == 2} {
            set w1 [lindex $line 0]
            set w2 [lindex $line 1]
            switch $w1 {
                "LANG" {
                    set USR_CNF_LANG $w2
                }
                "T_WIDTH" {
                    if {[string is digit $w2] && $w2 > 0} {
                        set USR_CNF_T_WIDTH $w2
                    }
                }
                "T_HEIGHT" {
                    if {[string is digit $w2] && $w2 > 0} {
                        set USR_CNF_T_HEIGHT $w2
                    }
                }
                "T_TAB" {
                    if {[string is digit $w2] && $w2 > 0} {
                        set USR_CNF_T_TAB $w2
                    }
                }
                "T_FONT" {
                    set USR_CNF_T_FONT $w2
                }
                "T_FG" {
                    set USR_CNF_T_FG $w2
                }
                "T_BG" {
                    set USR_CNF_T_BG $w2
                }
                "T_CUR_COLOR" {
                    set USR_CNF_T_CUR_COLOR $w2
                }
                "HIGHLIGHT" {
                    set USR_CNF_HIGHLIGHT $w2
                }
                "EXT" {
                    set USR_CNF_EXT $w2
                }
                "LPR" {
                    set USR_CNF_LPR $w2
                }
                "KING_EJ" {
                    set USR_CNF_KING_EJ $w2
                }
                "KING_JE" {
                    set USR_CNF_KING_JE $w2
                }
                "KING_DICT" {
                    set USR_CNF_KING_DICT $w2
                }
            }
        }
    }
}

# Analyze arguments.
for {set i 0} {$i < $argc} {incr i} {
    set arg [lindex $argv $i]
    switch -- $arg {
        "-dummy" {
            set DUMMY_FG 1
        }

        "-trans" {
            if {$tcl_platform(platform) == "unix"} {
                if {$i < [expr $argc - 2]} {
                    set TRANS_FG 1
                    incr i
                    set TEMP_FILENAME [file join $CONFDIR [lindex $argv $i]]
                    incr i
                    set TEMP_CWD [lindex $argv $i]
                } else {
                    tk_messageBox -type ok -title "ERROR" -icon error \
                         -message "-trans option needs filename to translate."
                    exit
                }
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "Invalid option -trans."
                exit
            }
        }

        "-lang" {
            if {$i < [expr $argc - 1]} {
                incr i
                set arg [lindex $argv $i]
                switch $arg {
                    "ja" {
                        set USR_CNF_LANG "ja"
                    }
                    default {
                        set USR_CNF_LANG "en"
                    }
                }
            }
        }

        "-width" {
            if {$i < [expr $argc - 1]} {
                incr i
                set arg [lindex $argv $i]
                if [catch {set USR_CNF_T_WIDTH [expr $arg]}] {
                    tk_messageBox -type ok -title "ERROR" -icon error \
                          -message "-width option needs INTEGER columns."
                    exit
                }
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "-width option needs columns."
                exit
            }
        }

        "-height" {
            if {$i < [expr $argc - 1]} {
                incr i
                set arg [lindex $argv $i]
                if [catch {set USR_CNF_T_HEIGHT [expr $arg]}] {
                    tk_messageBox -type ok -title "ERROR" -icon error \
                          -message "-height option needs INTEGER columns."
                    exit
                }
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "-height option needs columns."
                exit
            }
        }

        "-tab" {
            if {$i < [expr $argc - 1]} {
                incr i
                set arg [lindex $argv $i]
                if [catch {set USR_CNF_T_TAB [expr $arg]}] {
                    tk_messageBox -type ok -title "ERROR" -icon error \
                          -message "-tab option needs INTEGER columns."
                    exit
                }
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "-tab option needs columns."
                exit
            }
        }

        "-king" {
            if {$tcl_platform(platform) == "unix"} {
                set USE_KING 1
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "Invalid option -king."
                exit
            }
        }

        "-noking" {
            if {$tcl_platform(platform) == "unix"} {
                set USE_KING 0
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "Invalid option -noking."
                exit
            }
        }

        default {
            if {[string index $arg 0] != "-"} {
                  lappend FILES $arg
            } else {
                tk_messageBox -type ok -title "ERROR" -icon error \
                              -message "Invalid option $arg."
                exit
            }
        }
    }
}

### Messages ##################################################################
switch $USR_CNF_LANG {
    "ja" {
        source [file join $BINDIR messages/messages_ja.tcl]
    }
    default {
        source [file join $BINDIR messages/messages_en.tcl]
    }
}

### Tcl/Tk version check. #####################################################
if {$tcl_version < "8.3"} {
    tk_messageBox -type ok -icon error -title $M3001 -message $M4000
    exit
}
if {$tk_version < "8.3"} {
    tk_messageBox -type ok -icon error -title $M3001 -message $M4000
    exit
}

### Check if $CONFDIR exists. #################################################
if ![file isdirectory $CONFDIR] {
    if [file exists $CONFDIR] {
        tk_messageBox -type ok -icon error -title $M3001 \
                      -message "$CONFDIR: $M4016"
        exit
    }
    if [catch {file mkdir $CONFDIR}] {
        tk_messageBox -type ok -icon error -title $M3001 \
                      -message "$CONFDIR: $M4017"
        exit
    }
    switch $tcl_platform(platform) {
        "unix" {
            file attributes $CONFDIR -permissions rwx------
        }
    }
}

### GUI components ############################################################
#
#--- Common -------------------------------------------------------------------
menu       .menu
frame      .base
frame      .toolbar
  ButtonBox  .toolbar.filebar   -spacing 0 -padx 1 -pady 1
  Separator  .toolbar.s1        -orient vertical
  ButtonBox  .toolbar.editbar   -spacing 0 -padx 1 -pady 1
  Separator  .toolbar.s2        -orient vertical
  ButtonBox  .toolbar.printbar  -spacing 0 -padx 1 -pady 1
  Separator  .toolbar.s3        -orient vertical
  ButtonBox  .toolbar.findbar   -spacing 0 -padx 1 -pady 1
  Separator  .toolbar.s4        -orient vertical
  ButtonBox  .toolbar.markbar   -spacing 0 -padx 1 -pady 1
  Separator  .toolbar.s5        -orient vertical

if {$tk_version < "8.4"} {
    text     .base.textarea -width $USR_CNF_T_WIDTH                       \
                            -height $USR_CNF_T_HEIGHT                     \
                            -fg $USR_CNF_T_FG -bg $USR_CNF_T_BG -bd $T_BD \
                            -font $USR_CNF_T_FONT                         \
                            -insertwidth $T_CSR_WIDTH                     \
                            -yscrollcommand {.base.y set}                 \
                            -tabs [expr $USR_CNF_T_TAB * 8]               \
                            -exportselection true
} else {
    text     .base.textarea -width $USR_CNF_T_WIDTH                       \
                            -height $USR_CNF_T_HEIGHT                     \
                            -fg $USR_CNF_T_FG -bg $USR_CNF_T_BG -bd $T_BD \
                            -font $USR_CNF_T_FONT                         \
                            -insertwidth $T_CSR_WIDTH                     \
                            -yscrollcommand {.base.y set}                 \
                            -tabs [expr $USR_CNF_T_TAB * 8]               \
                            -exportselection true                         \
                            -undo true
}
.base.textarea tag configure sel -background $USR_CNF_HIGHLIGHT

scrollbar  .base.y -command {.base.textarea yview} -orient vertical

if {$tcl_platform(platform) != "windows"} {
    bind .base.textarea <Control-v> {
        event generate .base.textarea <<Paste>>
        break
    }
}
bind .base.textarea <<Copy>> {
    tk_textCopy %W
    .base.textarea tag remove sel 1.0 end
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <<Open>> {
    cb_filemenu_open
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <<Save>> {
    cb_filemenu_save
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <<Find>> {
    cb_editmenu_find
    set TEMP_FROM_ANOTHER_EVENT 1
    # 'break' is needed not to excute bind scripts of Text widget.
    break
}
bind .base.textarea <<FindForwards>> {
    editmenu_search_forwards
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <<FindBackwards>> {
    editmenu_search_backwards
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <<Replace>> {
    cb_editmenu_replace
    break
}
bind .base.textarea <<SelectAll>> {
    cb_editmenu_select_all
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <<GotoLine>> {
    cb_editmenu_goto_line
    set TEMP_FROM_ANOTHER_EVENT 1
    break
}
bind .base.textarea <Button-1> {
    display_cursor_pos
}
switch $tcl_platform(platform) {
    windows {
        bind .base.textarea <KeyPress> {
            bind_textarea_key_pressed %A %K
        }
    }
    unix {
        bind .base.textarea <KeyRelease> {
            bind_textarea_key_pressed %A %K
        }
    }
    default {
        bind .base.textarea <KeyRelease> {
            bind_textarea_key_pressed %A %K
        }
    }
}

frame      .sbar
  label    .sbar.s1 -width 11 -justify left -font $CST_SBAR_FONT \
                    -relief sunken -anchor w
  label    .sbar.s2 -justify left -font $CST_SBAR_FONT -anchor w \
                    -relief sunken

menu .base.textarea.popup -tearoff no
.base.textarea.popup add command -label $M0240 -underline $UM0240 \
                                 -command cb_editmenu_paste
.base.textarea.popup add separator
.base.textarea.popup add command -label $M0220 -underline $UM0220 \
                                 -command cb_editmenu_cut
.base.textarea.popup add command -label $M0230 -underline $UM0230 \
                                 -command cb_editmenu_copy
bind .base.textarea <3> { tk_popup .base.textarea.popup %X %Y }

#
#--- King -------------------------------------------------------------------
if {$USE_KING == 1} {
    source [file join $BINDIR king_gui.tcl]
    source [file join $BINDIR king_utils.tcl]
}

### Include other files. ######################################################
source [file join $BINDIR file_utils.tcl]

### Functions. ################################################################
#
#
# @set_text_modified
#
proc set_text_modified {A} {
    global M2001 M9002
    global TEMP_TEXT_MODIFIED TEMP_FILENAME
    global TEMP_FROM_ANOTHER_EVENT

    if {$TEMP_TEXT_MODIFIED == 1} {
        set TEMP_FROM_ANOTHER_EVENT 0
        return
    }
    if {$TEMP_FROM_ANOTHER_EVENT == 1} {
        set TEMP_FROM_ANOTHER_EVENT 0
        return
    }
    
    if {[string compare "" $A] != 0} {
        wm title . "$M9002 $TEMP_FILENAME $M2001"
        set TEMP_TEXT_MODIFIED 1
        set TEMP_FROM_ANOTHER_EVENT 0
    }
    set TEMP_FROM_ANOTHER_EVENT 0
}

#
# @display_cursor_pos
#
proc display_cursor_pos {} {
    after 10 {
        .sbar.s1 configure -text [.base.textarea index insert]
        .sbar.s2 configure -text {}
        color_line
    }
}

#
# @color_line
#
proc color_line {} {
    global USR_CNF_T_CUR_COLOR

    .base.textarea tag delete colorline
    .base.textarea tag add colorline \
                            [.base.textarea index "insert linestart"] \
                            [.base.textarea index "insert lineend"  ]
    .base.textarea tag configure colorline -foreground $USR_CNF_T_CUR_COLOR
}


# @bind_textarea_key_pressed
#
proc bind_textarea_key_pressed {a k} {
    switch $k {
        "XF86_Switch_VT_3" {
            event generate .base.textarea <<FindBackwards>>
        }
        "Escape" {
            return
        }
        "Delete" {
            set_text_modified A
            return
        }
    } 
    set_text_modified $a
    display_cursor_pos
}

### Callback functions. #######################################################
source [file join $BINDIR file_menu.tcl]
source [file join $BINDIR edit_menu.tcl]
source [file join $BINDIR option_menu.tcl]
source [file join $BINDIR help_menu.tcl]
source [file join $BINDIR mark.tcl]

### Functions #################################################################
#
#------------------------------------------------------------------------------
# @make_menu
#------------------------------------------------------------------------------
proc make_menu {} {
    global tk_version
    global M0100 M0200 M0300 M0400 M0500
    global UM0100 UM0200 UM0300 UM0400 UM0500
    global M0110 M0120 M0130 M0140 M0150 M0160 M0170 M0180
    global UM0110 UM0120 UM0130 UM0140 UM0150 UM0160 UM0170
    global M0210 M0220 M0230 M0240 M0250 M0260 M0270 M0280
    global UM0210 UM0220 UM0230 UM0240 UM0250 UM0260 UM0270 UM0280
    global M0310 M0320 M0330
    global UM0310 UM0320 UM0330
    global M0410
    global UM0410
    global M0510
    global UM0510
    global USE_KING

    .menu add cascade -label $M0100 -underline $UM0100 -menu .menu.file
    .menu add cascade -label $M0200 -underline $UM0200 -menu .menu.edit
    if {$USE_KING == 1} {
        .menu add cascade -label $M0300 -underline $UM0300 -menu .menu.trans
    }
    .menu add cascade -label $M0400 -underline $UM0400 -menu .menu.option
    .menu add cascade -label $M0500 -underline $UM0500 -menu .menu.help

    menu .menu.file -tearoff no
    .menu.file add command -label $M0110 -underline $UM0110 \
                           -command {cb_filemenu_new {}}
    .menu.file add command -label $M0120 -underline $UM0120 \
                           -command cb_filemenu_open
    .menu.file add command -label $M0180 -command cb_filemenu_close_and_open
    .menu.file add command -label $M0170 -underline $UM0170 \
                           -command cb_filemenu_open_at_cursor
    .menu.file add separator
    .menu.file add command -label $M0130 -underline $UM0130 \
                           -command cb_filemenu_save
    .menu.file add command -label $M0140 -underline $UM0140 \
                           -command cb_filemenu_save_as
    .menu.file add separator
    .menu.file add command -label $M0150 -underline $UM0150 \
                           -command cb_filemenu_print
    .menu.file add separator
    .menu.file add command -label $M0160 -underline $UM0160 \
                           -command cb_filemenu_quit


    menu .menu.edit -tearoff no
    if {$tk_version > "8.3"} {
        .menu.edit add command -label $M0210 -underline $UM0210 \
                               -command cb_editmenu_undo
        .menu.edit add separator
    }
    .menu.edit add command -label $M0220 -underline $UM0220 \
                           -command cb_editmenu_cut
    .menu.edit add command -label $M0230 -underline $UM0230 \
                           -command cb_editmenu_copy
    .menu.edit add command -label $M0240 -underline $UM0240 \
                           -command cb_editmenu_paste
    .menu.edit add separator
    .menu.edit add command -label $M0250 -underline $UM0250 \
                           -command cb_editmenu_find
    .menu.edit add command -label $M0260 -underline $UM0260 \
                           -command cb_editmenu_replace
    .menu.edit add separator
    .menu.edit add command -label $M0270 -underline $UM0270 \
                           -command cb_editmenu_select_all
    .menu.edit add separator
    .menu.edit add command -label $M0280 -underline $UM0280 \
                           -command cb_editmenu_goto_line

    if {$USE_KING == 1} {
        menu .menu.trans -tearoff no
        .menu.trans add command -label $M0310 -underline $UM0310 \
                           -command {cb_trans "jtoe"}
        .menu.trans add command -label $M0320 -underline $UM0320 \
                           -command {cb_trans "etoj"}
        .menu.trans add separator
        .menu.trans add command -label $M0330 -underline $UM0330 \
                           -command {cb_trans_dict "toolbar"}
    }

    menu .menu.option -tearoff no
    .menu.option add command -label $M0410 -underline $UM0410 \
                             -command cb_optionmenu_config

    menu .menu.help -tearoff no
    .menu.help add command -label $M0510 -underline $UM0510 \
                           -command cb_helpmenu_about
}

#------------------------------------------------------------------------------
# @make_toolbar
#------------------------------------------------------------------------------
proc make_toolbar {} {
    global tk_version BINDIR
    global USE_KING
    global M1001 M1002 M1003 M1004 M1005 M1006 M1007 M1008
    global M1009 M1010 M1011 M1012 M1013

    .toolbar.filebar add -image [Bitmap::get new]                  \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1001   \
                 -command {cb_filemenu_new {}}
    .toolbar.filebar add -image [Bitmap::get open]                 \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1002   \
                 -command cb_filemenu_open
    .toolbar.filebar add -image [Bitmap::get save]                 \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1003   \
                 -command cb_filemenu_save
    if {$tk_version > "8.3"} {
        .toolbar.editbar add -image [Bitmap::get undo]                 \
                     -highlightthickness 0 -takefocus 0 -relief link   \
                     -borderwidth 1 -padx 1 -pady 1 -helptext $M1005   \
                     -command cb_editmenu_undo
    }
    .toolbar.editbar add -image [Bitmap::get cut]                  \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1006   \
                 -command cb_editmenu_cut
    .toolbar.editbar add -image [Bitmap::get copy]                 \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1007   \
                 -command cb_editmenu_copy
    .toolbar.editbar add -image [Bitmap::get paste]                \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1008   \
                 -command cb_editmenu_paste
    .toolbar.printbar add -image [Bitmap::get print]               \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1004   \
                 -command cb_filemenu_print
    .toolbar.findbar add -image [xpm-to-image $BINDIR/icon/find.xpm] \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1009   \
                 -command cb_editmenu_find
    .toolbar.findbar add -image [xpm-to-image $BINDIR/icon/findf.xpm] \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1010   \
                 -command {event generate .base.textarea <<FindForwards>>}
    .toolbar.findbar add -image [xpm-to-image $BINDIR/icon/findb.xpm] \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1011   \
                 -command {event generate .base.textarea <<FindBackwards>>}
    .toolbar.markbar add -image [xpm-to-image $BINDIR/icon/mark.xpm] \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1012   \
                 -command cb_mark_set
    .toolbar.markbar add -image [xpm-to-image $BINDIR/icon/gotomark.xpm] \
                 -highlightthickness 0 -takefocus 0 -relief link   \
                 -borderwidth 1 -padx 1 -pady 1 -helptext $M1013   \
                 -command cb_goto_mark

    pack .toolbar.filebar  -side left -anchor w
    pack .toolbar.s1       -side left -fill y -padx 4 -pady 4
    pack .toolbar.editbar  -side left -anchor w
    pack .toolbar.s2       -side left -fill y -padx 4 -pady 4
    pack .toolbar.printbar -side left -anchor w
    pack .toolbar.s3       -side left -fill y -padx 4 -pady 4
    pack .toolbar.findbar  -side left -anchor w
    pack .toolbar.s4       -side left -fill y -padx 4 -pady 4
    pack .toolbar.markbar  -side left -anchor w
    pack .toolbar.s5       -side left -fill y -padx 4 -pady 4
    if {$USE_KING == 1} { make_kingbar }
    pack .toolbar -side top  -fill x -padx 1
}

#------------------------------------------------------------------------------
# @make_textarea
#------------------------------------------------------------------------------
proc make_textarea {} {
    pack .base -fill both -expand 1 -padx 1 -pady 1
##  pack .base.textarea -fill both -expand 1
    grid .base.textarea -column 0 -row 0 -sticky news
    grid .base.y        -column 1 -row 0 -sticky news
    grid rowconfigure .base 0 -weight 1
    grid columnconfigure .base 0 -weight 1
}

#------------------------------------------------------------------------------
# @init
#------------------------------------------------------------------------------
proc init {} {
    global tk_version
    global FILES CONFDIR
    global TEMP_FILENAME TEMP_UNTITLED TEMP_TEXT_MODIFIED TEMP_CWD
    global DUMMY_FG TRANS_FG
    global M9001 M9002

    # Open files specified as arguments.
    if {$TRANS_FG == 1} {
        set fd [open_file $TEMP_FILENAME 0]
        if {$fd > 0} {
            .base.textarea insert end [read  $fd]
            close $fd
            file delete -force $TEMP_FILENAME
            .base.textarea see 1.0
            .base.textarea mark set insert 1.0
            if {$tk_version > "8.3"} {
                .base.textarea edit reset
            }
        } else {
            exit
        }
        set TEMP_FILENAME $M9001
        set TEMP_UNTITLED 1
        set TEMP_TEXT_MODIFIED 0
    } else {
        if {[llength $FILES] > 0} {
            if {$DUMMY_FG == 1} {
                set cn 1
            } else {
                set cn 0
            }
            foreach fn $FILES {
                if {$fn != ""} {
                    set fd [open_file $fn 0]
                    if {$fd > 0} {
                        if {$cn == 0} {
                            .base.textarea insert end [read  $fd]
                            close $fd
                            .base.textarea see 1.0
                            .base.textarea mark set insert 1.0
                            if {$tk_version > "8.3"} {
                                .base.textarea edit reset
                            }
                            set TEMP_FILENAME $fn
                            set TEMP_UNTITLED 0
                            get_cwd $fn
                            incr cn
                        } else {
                            cb_filemenu_new $fn
                        }
                    } else {
                        if {$cn == 0} {
                            if {$fn != ""} {
                                set TEMP_FILENAME $fn
                                set TEMP_UNTITLED 0
                            } else {
                                set TEMP_FILENAME $M9001
                                set TEMP_UNTITLED 1
                            }
                            get_cwd $fn
                            incr cn
                        } else {
                            cb_filemenu_new $fn
                        }
                    }
                } else {
                    set TEMP_CWD [pwd]
                    set TEMP_FILENAME $M9001
                    set TEMP_UNTITLED 1
                    incr cn
                }
            }
        } else {
            if {$DUMMY_FG == 1} {
                cb_filemenu_new {}
            } else {
                set TEMP_CWD [pwd]
                set TEMP_FILENAME $M9001
                set TEMP_UNTITLED 1
            }
        }
    }
    wm title . "$M9002 $TEMP_FILENAME"

    if {$DUMMY_FG == 1} {
        exit
    }
}

### Main ######################################################################
init
wm protocol . WM_DELETE_WINDOW cb_filemenu_quit
wm iconbitmap . @[file join $BINDIR icon atedit.xbm]
make_menu
. configure -menu .menu
make_toolbar
make_textarea

pack .sbar -fill x -expand 0
pack .sbar.s1 -side left -padx 2
pack .sbar.s2 -side left -fill x -expand 1
display_cursor_pos

focus .base.textarea

