#define MOO_DEBUG 0
#define MOO_DMADELAY (100)

/***************************************************************************

 Wild West C.O.W.boys of Moo Mesa
 Bucky O'Hare
 (c) 1992 Konami
 Driver by R. Belmont and Acho A. Tang based on xexex.c by Olivier Galibert.
 Moo Mesa protection information thanks to ElSemi and OG.

 These are the final Xexex hardware games before the pre-GX/Mystic Warriors
 hardware took over.

Bug Fixes and Outstanding Issues
--------------------------------
Moo:
 - 54338 color blender support. Works fine with Bucky but needs a correct
   enable/disable bit in Moo. (intro gfx missing and fog blocking view)
 - Enemies coming out of the jail cells in the last stage have wrong priority.
   Could be tile priority or the typical "small Z, big pri" sprite masking
   trick currently not supported by K053247_sprites_draw().

Moo (bootleg):
 - No sprites appear, and the game never enables '246 interrupts.  Of course,
   they're using some copy of a '246 on FPGAs, so who knows what's going on...

Bucky:
 - Shadows sometimes have wrong priority. (unsupported priority modes)
 - Gaps between zoomed sprites. (fraction round-off)
 - Rogue sprites keep popping on screen after stage 2. They can usually be
   found near 950000 with sprite code around 5e40 or f400. The GFX viewer
   only shows blanks at 5e40, however. Are they invalid data from bad
   sprite ROMs or markers that aren't supposed to be displayed? These
   artifacts have one thing in common: they all have zero zcode. In fact
   no other sprites in Bucky seems to have zero zcode.

   Update: More garbages seen in later stages with a great variety.
   There's enough indication to assume Bucky simply ignores sprites with
   zero Z. I wonder why nobody reported this.

***************************************************************************/

#include "driver.h"
#include "state.h"
#include "vidhrdw/generic.h"
#include "vidhrdw/konamiic.h"
#include "cpu/m68000/m68000.h"
#include "cpu/z80/z80.h"
#include "machine/eeprom.h"

VIDEO_START(moo);
VIDEO_UPDATE(moo);
void moo_set_alpha(int on);

static int init_eeprom_count, init_nosound_count, game_type;
static data16_t *workram;
static data16_t protram[16];
static data16_t cur_control2;


static struct EEPROM_interface eeprom_interface =
{
	7,			/* address bits */
	8,			/* data bits */
	"011000",		/* read command */
	"011100",		/* write command */
	"0100100000000",	/* erase command */
	"0100000000000",	/* lock command */
	"0100110000000" 	/* unlock command */
};

static NVRAM_HANDLER( moo )
{
	if (read_or_write)
		EEPROM_save(file);
	else
	{
		EEPROM_init(&eeprom_interface);

		if (file)
		{
			init_eeprom_count = 0;
			EEPROM_load(file);
		}
		else
			init_eeprom_count = 10;
	}
}

static READ16_HANDLER( control1_r )
{
	int res;

	/* bit 0 is EEPROM data */
	/* bit 1 is EEPROM ready */
	/* bit 3 is service button */
	/* bits 4-7 are DIP switches */
	res = EEPROM_read_bit() | input_port_1_r(0);

	if (init_eeprom_count)
	{
		init_eeprom_count--;
		res &= 0xf7;
	}

	return res;
}

static READ16_HANDLER( control2_r )
{
	return cur_control2;
}

static WRITE16_HANDLER( control2_w )
{
	/* bit 0  is data */
	/* bit 1  is cs (active low) */
	/* bit 2  is clock (active high) */
	/* bit 5  is enable irq 5 (unconfirmed) */
	/* bit 8  is enable sprite ROM reading */
	/* bit 10 is watchdog */
	/* bit 11 is enable irq 4 (unconfirmed) */

	COMBINE_DATA(&cur_control2);

	EEPROM_write_bit(cur_control2 & 0x01);
	EEPROM_set_cs_line((cur_control2 & 0x02) ? CLEAR_LINE : ASSERT_LINE);
	EEPROM_set_clock_line((cur_control2 & 0x04) ? ASSERT_LINE : CLEAR_LINE);

	if (data & 0x100)
	{
		K053246_set_OBJCHA_line(ASSERT_LINE);
	}
	else
	{
		K053246_set_OBJCHA_line(CLEAR_LINE);
	}
}


static void moo_objdma(int type)
{
	int counter, num_inactive;
	data16_t *src, *dst, zmask;

	K053247_export_config(&dst, (struct GfxElement**)&src, (void**)&src, &counter, &counter);
	src = spriteram16;
	num_inactive = counter = 256;

	zmask = (type) ? 0x00ff : 0xffff;

	do {
		if ((*src & 0x8000) && (*src & zmask))
		{
			memcpy(dst, src, 0x10);
			dst += 8;
			num_inactive--;
		}
		src += 0x80;
	}
	while (--counter);

	if (num_inactive) do { *dst = 0; dst += 8; } while (--num_inactive);
}

static void dmaend_callback(int data)
{
	if (cur_control2 & 0x800)
		cpu_set_irq_line(0, 4, HOLD_LINE);
}

static INTERRUPT_GEN(moo_interrupt)
{
	if (K053246_is_IRQ_enabled())
	{
		moo_objdma(game_type);

		// schedule DMA end interrupt (delay shortened to catch up with V-blank)
		timer_set(TIME_IN_USEC(MOO_DMADELAY), 0, dmaend_callback);
	}

	// trigger V-blank interrupt
	if (cur_control2 & 0x20)
		cpu_set_irq_line(0, 5, HOLD_LINE);
}

static INTERRUPT_GEN(moobl_interrupt)
{
	moo_objdma(game_type);

	// schedule DMA end interrupt (delay shortened to catch up with V-blank)
	timer_set(TIME_IN_USEC(MOO_DMADELAY), 0, dmaend_callback);

	// trigger V-blank interrupt
	cpu_set_irq_line(0, 5, HOLD_LINE);
}

static WRITE16_HANDLER( sound_cmd1_w )
{
	if((data & 0x00ff0000) == 0) {
		data &= 0xff;
		soundlatch_w(0, data);
	}
}

static WRITE16_HANDLER( sound_cmd2_w )
{
	if((data & 0x00ff0000) == 0) {
		soundlatch2_w(0, data & 0xff);
	}
}

static WRITE16_HANDLER( sound_irq_w )
{
	cpu_set_irq_line(1, 0, HOLD_LINE);
}

static READ16_HANDLER( sound_status_r )
{
	int latch = soundlatch3_r(0);

	/* make test pass with sound off.
	   these games are trickier than your usual konami stuff, they expect to
	   read 0xff (meaning the z80 booted properly) then 0x80 (z80 busy) then
	   the self-test result */
	if (!Machine->sample_rate) {
		if (init_nosound_count < 10)
		{
			if (!init_nosound_count)
				latch = 0xff;
			else
				latch = 0x80;
			init_nosound_count++;
		}
		else
		{
			latch = 0x0f;
		}
	}

	return latch;
}

static WRITE_HANDLER( sound_bankswitch_w )
{
	cpu_setbank(2, memory_region(REGION_CPU2) + 0x10000 + (data&0xf)*0x4000);
}


#if 0 // (for reference; do not remove)

/* the interface with the 053247 is weird. The chip can address only 0x1000 bytes */
/* of RAM, but they put 0x10000 there. The CPU can access them all. */
static READ16_HANDLER( K053247_scattered_word_r )
{
	if (offset & 0x0078)
		return spriteram16[offset];
	else
	{
		offset = (offset & 0x0007) | ((offset & 0x7f80) >> 4);
		return K053247_word_r(offset,mem_mask);
	}
}

static WRITE16_HANDLER( K053247_scattered_word_w )
{
	if (offset & 0x0078)
		COMBINE_DATA(spriteram16+offset);
	else
	{
		offset = (offset & 0x0007) | ((offset & 0x7f80) >> 4);

		K053247_word_w(offset,data,mem_mask);
	}
}

#endif


static READ16_HANDLER( player1_r ) 	// players 1 and 3
{
	return input_port_2_r(0) | (input_port_4_r(0)<<8);
}

static READ16_HANDLER( player2_r )	// players 2 and 4
{
	return input_port_3_r(0) | (input_port_5_r(0)<<8);
}

static WRITE16_HANDLER( moo_prot_w )
{
	UINT32 src1, src2, dst, length, a, b, res;

	COMBINE_DATA(&protram[offset]);

	if (offset == 0xc)	// trigger operation
	{
		src1 = (protram[1]&0xff)<<16 | protram[0];
		src2 = (protram[3]&0xff)<<16 | protram[2];
		dst = (protram[5]&0xff)<<16 | protram[4];
		length = protram[0xf];

		while (length)
		{
			a = cpu_readmem24bew_word(src1);
			b = cpu_readmem24bew_word(src2);
			res = a+2*b;

			cpu_writemem24bew_word(dst, res);

			src1 += 2;
			src2 += 2;
			dst += 2;
			length--;
		}
	}
}


static WRITE16_HANDLER( moobl_oki_bank_w )
{
	logerror("%x to OKI bank\n", data);

	OKIM6295_set_bank_base(0, (data & 0x0f)* 0x40000);
}

static MEMORY_READ16_START( readmem )
	{ 0x000000, 0x07ffff, MRA16_ROM },
	{ 0x0c4000, 0x0c4001, K053246_word_r },
	{ 0x0d6014, 0x0d6015, sound_status_r },
	{ 0x0d6000, 0x0d601f, MRA16_RAM },			/* sound regs fall through */
	{ 0x0da000, 0x0da001, player1_r },
	{ 0x0da002, 0x0da003, player2_r },
	{ 0x0dc000, 0x0dc001, input_port_0_word_r },
	{ 0x0dc002, 0x0dc003, control1_r },
	{ 0x0de000, 0x0de001, control2_r },
	{ 0x100000, 0x17ffff, MRA16_ROM },
	{ 0x180000, 0x18ffff, MRA16_RAM },			/* Work RAM */
	{ 0x190000, 0x19ffff, MRA16_RAM },			/* Sprite RAM */
	{ 0x1a0000, 0x1a1fff, K056832_ram_word_r },	/* Graphic planes */
	{ 0x1a2000, 0x1a3fff, K056832_ram_word_r },	/* Graphic planes mirror */
	{ 0x1b0000, 0x1b1fff, K056832_rom_word_r },	/* Passthrough to tile roms */
	{ 0x1c0000, 0x1c1fff, MRA16_RAM },
#if MOO_DEBUG
	{ 0x0c0000, 0x0c003f, K056832_word_r },
	{ 0x0c2000, 0x0c2007, K053246_reg_word_r },
	{ 0x0ca000, 0x0ca01f, K054338_word_r },
	{ 0x0cc000, 0x0cc01f, K053251_lsb_r },
	{ 0x0d0000, 0x0d001f, MRA16_RAM },
	{ 0x0d8000, 0x0d8007, K056832_b_word_r },
#endif
MEMORY_END

static MEMORY_WRITE16_START( writemem )
	{ 0x000000, 0x07ffff, MWA16_ROM },
	{ 0x0c0000, 0x0c003f, K056832_word_w },
	{ 0x0c2000, 0x0c2007, K053246_word_w },
	{ 0x0ca000, 0x0ca01f, K054338_word_w },		/* K054338 alpha blending engine */
	{ 0x0cc000, 0x0cc01f, K053251_lsb_w },
	{ 0x0ce000, 0x0ce01f, moo_prot_w },
	{ 0x0d0000, 0x0d001f, MWA16_RAM },			/* CCU regs (ignored) */
	{ 0x0d4000, 0x0d4001, sound_irq_w },
	{ 0x0d600c, 0x0d600d, sound_cmd1_w },
	{ 0x0d600e, 0x0d600f, sound_cmd2_w },
	{ 0x0d6000, 0x0d601f, MWA16_RAM },			/* sound regs fall through */
	{ 0x0d8000, 0x0d8007, K056832_b_word_w },	/* VSCCS regs */
	{ 0x0de000, 0x0de001, control2_w },
	{ 0x100000, 0x17ffff, MWA16_ROM },
	{ 0x180000, 0x18ffff, MWA16_RAM, &workram },
	{ 0x190000, 0x19ffff, MWA16_RAM, &spriteram16 },
	{ 0x1a0000, 0x1a1fff, K056832_ram_word_w },	/* Graphic planes */
	{ 0x1a2000, 0x1a3fff, K056832_ram_word_w },	/* Graphic planes mirror */
	{ 0x1c0000, 0x1c1fff, paletteram16_xrgb_word_w, &paletteram16 },
MEMORY_END

static MEMORY_READ16_START( readmembl )
	{ 0x000000, 0x07ffff, MRA16_ROM },
	{ 0x0c2f00, 0x0c2f01, MRA16_NOP },              /* heck if I know, but it's polled constantly */
	{ 0x0c4000, 0x0c4001, K053246_word_r },
	{ 0x0d6ffe, 0x0d6fff, OKIM6295_status_0_lsb_r },
	{ 0x0da000, 0x0da001, player1_r },
	{ 0x0da002, 0x0da003, player2_r },
	{ 0x0dc000, 0x0dc001, input_port_0_word_r },
	{ 0x0dc002, 0x0dc003, control1_r },
	{ 0x0de000, 0x0de001, control2_r },
	{ 0x100000, 0x17ffff, MRA16_ROM },
	{ 0x180000, 0x18ffff, MRA16_RAM },              /* Work RAM */
	{ 0x190000, 0x19ffff, MRA16_RAM },              /* Sprite RAM */
	{ 0x1a0000, 0x1a1fff, K056832_ram_word_r },     /* Graphic planes */
	{ 0x1a2000, 0x1a3fff, K056832_ram_word_r },	/* Graphic planes mirror */
	{ 0x1b0000, 0x1b1fff, K056832_rom_word_r },	/* Passthrough to tile roms */
	{ 0x1c0000, 0x1c1fff, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( writemembl )
	{ 0x000000, 0x07ffff, MWA16_ROM },
	{ 0x0c0000, 0x0c003f, K056832_word_w },
	{ 0x0c2000, 0x0c2007, K053246_word_w },
	{ 0x0ca000, 0x0ca01f, K054338_word_w },         /* K054338 alpha blending engine */
	{ 0x0cc000, 0x0cc01f, K053251_lsb_w },
	{ 0x0d0000, 0x0d001f, MWA16_RAM },              /* CCU regs (ignored) */
	{ 0x0d6ffc, 0x0d6ffd, moobl_oki_bank_w },
	{ 0x0d6ffe, 0x0d6fff, OKIM6295_data_0_lsb_w },
	{ 0x0d8000, 0x0d8007, K056832_b_word_w },       /* VSCCS regs */
	{ 0x0de000, 0x0de001, control2_w },
	{ 0x100000, 0x17ffff, MWA16_ROM },
	{ 0x180000, 0x18ffff, MWA16_RAM, &workram },
	{ 0x190000, 0x19ffff, MWA16_RAM, &spriteram16 },
	{ 0x1a0000, 0x1a1fff, K056832_ram_word_w },	/* Graphic planes */
	{ 0x1a2000, 0x1a3fff, K056832_ram_word_w },	/* Graphic planes mirror */
	{ 0x1c0000, 0x1c1fff, paletteram16_xrgb_word_w, &paletteram16 },
MEMORY_END

static MEMORY_READ16_START( buckyreadmem )
	{ 0x000000, 0x07ffff, MRA16_ROM },
	{ 0x080000, 0x08ffff, MRA16_RAM },
	{ 0x090000, 0x09ffff, MRA16_RAM },			/* Sprite RAM */
	{ 0x0a0000, 0x0affff, MRA16_RAM },			/* extra sprite RAM? */
	{ 0x0c4000, 0x0c4001, K053246_word_r },
	{ 0x0d2000, 0x0d20ff, K054000_lsb_r },
	{ 0x0d6014, 0x0d6015, sound_status_r },
	{ 0x0d6000, 0x0d601f, MRA16_RAM },			/* sound regs fall through */
	{ 0x0da000, 0x0da001, player1_r },
	{ 0x0da002, 0x0da003, player2_r },
	{ 0x0dc000, 0x0dc001, input_port_0_word_r },
	{ 0x0dc002, 0x0dc003, control1_r },
	{ 0x0de000, 0x0de001, control2_r },
	{ 0x180000, 0x181fff, K056832_ram_word_r },	/* Graphic planes */
	{ 0x182000, 0x183fff, K056832_ram_word_r },	/* Graphic planes mirror */
	{ 0x184000, 0x187fff, MRA16_RAM },			/* extra tile RAM? */
	{ 0x190000, 0x191fff, K056832_rom_word_r },	/* Passthrough to tile roms */
	{ 0x1b0000, 0x1b3fff, MRA16_RAM },
	{ 0x200000, 0x23ffff, MRA16_ROM },			/* data */
#if MOO_DEBUG
	{ 0x0c0000, 0x0c003f, K056832_word_r },
	{ 0x0c2000, 0x0c2007, K053246_reg_word_r },
	{ 0x0ca000, 0x0ca01f, K054338_word_r },
	{ 0x0cc000, 0x0cc01f, K053251_lsb_r },
	{ 0x0d0000, 0x0d001f, MRA16_RAM },
	{ 0x0d8000, 0x0d8007, K056832_b_word_r },
#endif
MEMORY_END

static MEMORY_WRITE16_START( buckywritemem )
	{ 0x000000, 0x07ffff, MWA16_ROM },
	{ 0x080000, 0x08ffff, MWA16_RAM },
	{ 0x090000, 0x09ffff, MWA16_RAM, &spriteram16 },	/* Sprite RAM */
	{ 0x0a0000, 0x0affff, MWA16_RAM },			/* extra sprite RAM? */
	{ 0x0c0000, 0x0c003f, K056832_word_w },
	{ 0x0c2000, 0x0c2007, K053246_word_w },
	{ 0x0ca000, 0x0ca01f, K054338_word_w },		/* K054338 alpha blending engine */
	{ 0x0cc000, 0x0cc01f, K053251_lsb_w },
	{ 0x0ce000, 0x0ce01f, moo_prot_w },
	{ 0x0d0000, 0x0d001f, MWA16_RAM },			/* CCU regs (ignored) */
	{ 0x0d2000, 0x0d20ff, K054000_lsb_w },
	{ 0x0d4000, 0x0d4001, sound_irq_w },
	{ 0x0d600c, 0x0d600d, sound_cmd1_w },
	{ 0x0d600e, 0x0d600f, sound_cmd2_w },
	{ 0x0d6000, 0x0d601f, MWA16_RAM },			/* sound regs fall through */
	{ 0x0d8000, 0x0d8007, K056832_b_word_w },	/* VSCCS regs */
	{ 0x0de000, 0x0de001, control2_w },
	{ 0x180000, 0x181fff, K056832_ram_word_w },	/* Graphic planes */
	{ 0x182000, 0x183fff, K056832_ram_word_w },	/* Graphic planes mirror */
	{ 0x184000, 0x187fff, MWA16_RAM },			/* extra tile RAM? */
	{ 0x1b0000, 0x1b3fff, paletteram16_xrgb_word_w, &paletteram16 },
	{ 0x200000, 0x23ffff, MWA16_ROM },
MEMORY_END

static MEMORY_READ_START( sound_readmem )
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0xbfff, MRA_BANK2 },
	{ 0xc000, 0xdfff, MRA_RAM },
	{ 0xe000, 0xe22f, K054539_0_r },
	{ 0xec01, 0xec01, YM2151_status_port_0_r },
	{ 0xf002, 0xf002, soundlatch_r },
	{ 0xf003, 0xf003, soundlatch2_r },
MEMORY_END

static MEMORY_WRITE_START( sound_writemem )
	{ 0x0000, 0xbfff, MWA_ROM },
	{ 0xc000, 0xdfff, MWA_RAM },
	{ 0xe000, 0xe22f, K054539_0_w },
	{ 0xec00, 0xec00, YM2151_register_port_0_w },
	{ 0xec01, 0xec01, YM2151_data_port_0_w },
	{ 0xf000, 0xf000, soundlatch3_w },
	{ 0xf800, 0xf800, sound_bankswitch_w },
MEMORY_END


INPUT_PORTS_START( moo )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE4 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* EEPROM data */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_SPECIAL )	/* EEPROM ready (always 1) */
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_DIPNAME( 0x10, 0x00, "Sound Output")
	PORT_DIPSETTING(    0x10, "Mono")
	PORT_DIPSETTING(    0x00, "Stereo")
	PORT_DIPNAME( 0x20, 0x20, "Coin Mechanism")
	PORT_DIPSETTING(    0x20, "Common")
	PORT_DIPSETTING(    0x00, "Independant")
	PORT_DIPNAME( 0xc0, 0x80, "Number of Players")
	PORT_DIPSETTING(    0xc0, "2")
	PORT_DIPSETTING(    0x40, "3")
	PORT_DIPSETTING(    0x80, "4")

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START3 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START4 )
INPUT_PORTS_END

/* Same as 'moo', but additional "Button 3" for all players */
INPUT_PORTS_START( bucky )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE4 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* EEPROM data */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_SPECIAL )	/* EEPROM ready (always 1) */
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_DIPNAME( 0x10, 0x00, "Sound Output")
	PORT_DIPSETTING(    0x10, "Mono")
	PORT_DIPSETTING(    0x00, "Stereo")
	PORT_DIPNAME( 0x20, 0x20, "Coin Mechanism")
	PORT_DIPSETTING(    0x20, "Common")
	PORT_DIPSETTING(    0x00, "Independant")
	PORT_DIPNAME( 0xc0, 0x80, "Number of Players")
	PORT_DIPSETTING(    0xc0, "2")
	PORT_DIPSETTING(    0x40, "3")
	PORT_DIPSETTING(    0x80, "4")

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START3 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER4 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START4 )
INPUT_PORTS_END


static struct YM2151interface ym2151_interface =
{
	1,
	4000000,
	{ YM3012_VOL(50,MIXER_PAN_LEFT,50,MIXER_PAN_RIGHT) },
	{ 0 }
};

static struct K054539interface k054539_interface =
{
	1,			/* 1 chip */
	48000,
	{ REGION_SOUND1 },
	{ { 100, 100 } },
};

static struct OKIM6295interface okim6295_interface =
{
	1,			/* 1 chip */
	{ 8000 },
	{ REGION_SOUND1 },
	{ 100 }
};

static MACHINE_INIT( moo )
{
	init_nosound_count = 0;
}


static MACHINE_DRIVER_START( moo )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", M68000, 16000000)
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_VBLANK_INT(moo_interrupt, 1)

	MDRV_CPU_ADD_TAG("sound", Z80, 8000000)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(sound_readmem,sound_writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(1200) // should give IRQ4 sufficient time to update scroll registers

	MDRV_MACHINE_INIT(moo)

	MDRV_NVRAM_HANDLER(moo)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_RGB_DIRECT | VIDEO_HAS_SHADOWS | VIDEO_HAS_HIGHLIGHTS | VIDEO_UPDATE_AFTER_VBLANK)
	MDRV_SCREEN_SIZE(64*8, 32*8)
	MDRV_VISIBLE_AREA(40, 40+384-1, 16, 16+224-1)

	MDRV_PALETTE_LENGTH(2048)

	MDRV_VIDEO_START(moo)
	MDRV_VIDEO_UPDATE(moo)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2151, ym2151_interface)
	MDRV_SOUND_ADD(K054539, k054539_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( moobl )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", M68000, 16100000)
	MDRV_CPU_MEMORY(readmembl,writemembl)
	MDRV_CPU_VBLANK_INT(moobl_interrupt, 1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(1200) // should give IRQ4 sufficient time to update scroll registers

	MDRV_MACHINE_INIT(moo)
	MDRV_NVRAM_HANDLER(moo)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_RGB_DIRECT | VIDEO_HAS_SHADOWS | VIDEO_HAS_HIGHLIGHTS | VIDEO_UPDATE_AFTER_VBLANK)
	MDRV_SCREEN_SIZE(64*8, 32*8)
	MDRV_VISIBLE_AREA(40, 40+384-1, 16, 16+224-1)

	MDRV_PALETTE_LENGTH(2048)

	MDRV_VIDEO_START(moo)
	MDRV_VIDEO_UPDATE(moo)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( bucky )
	MDRV_IMPORT_FROM(moo)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(buckyreadmem,buckywritemem)

	/* video hardware */
	MDRV_PALETTE_LENGTH(4096)
MACHINE_DRIVER_END



ROM_START( moo )
	ROM_REGION( 0x180000, REGION_CPU1, 0 )
	/* main program */
	ROM_LOAD16_BYTE( "151b01",    0x000000,  0x40000, 0xfb2fa298 )
	ROM_LOAD16_BYTE( "151b02.ea", 0x000001,  0x40000, 0x37b30c01 )

	/* data */
	ROM_LOAD16_BYTE( "151a03", 0x100000,  0x40000, 0xc896d3ea )
	ROM_LOAD16_BYTE( "151a04", 0x100001,  0x40000, 0x3b24706a )

	ROM_REGION( 0x050000, REGION_CPU2, 0 )
	/* Z80 sound program */
	ROM_LOAD( "151a07", 0x000000, 0x040000, 0xcde247fc )
	ROM_RELOAD(         0x010000, 0x040000 )

	ROM_REGION( 0x200000, REGION_GFX1, 0 )
	/* tilemaps */
	ROM_LOAD( "151a05", 0x000000, 0x100000, 0xbc616249 )
	ROM_LOAD( "151a06", 0x100000, 0x100000, 0x38dbcac1 )

	ROM_REGION( 0x800000, REGION_GFX2, 0 )
	/* sprites */
	ROM_LOAD( "151a10", 0x000000, 0x200000, 0x376c64f1 )
	ROM_LOAD( "151a11", 0x200000, 0x200000, 0xe7f49225 )
	ROM_LOAD( "151a12", 0x400000, 0x200000, 0x4978555f )
	ROM_LOAD( "151a13", 0x600000, 0x200000, 0x4771f525 )

	ROM_REGION( 0x200000, REGION_SOUND1, 0 )
	/* K054539 samples */
	ROM_LOAD( "151a08", 0x000000, 0x200000, 0x962251d7 )
ROM_END

ROM_START( mooua )
	ROM_REGION( 0x180000, REGION_CPU1, 0 )
	/* main program */
	ROM_LOAD16_BYTE( "151b01", 0x000000,  0x40000, 0xfb2fa298 )
	ROM_LOAD16_BYTE( "151b02", 0x000001,  0x40000, 0x3d9f4d59 )

	/* data */
	ROM_LOAD16_BYTE( "151a03", 0x100000,  0x40000, 0xc896d3ea )
	ROM_LOAD16_BYTE( "151a04", 0x100001,  0x40000, 0x3b24706a )

	ROM_REGION( 0x050000, REGION_CPU2, 0 )
	/* Z80 sound program */
	ROM_LOAD( "151a07", 0x000000, 0x040000, 0xcde247fc )
	ROM_RELOAD(         0x010000, 0x040000 )

	ROM_REGION( 0x200000, REGION_GFX1, 0 )
	/* tilemaps */
	ROM_LOAD( "151a05", 0x000000, 0x100000, 0xbc616249 )
	ROM_LOAD( "151a06", 0x100000, 0x100000, 0x38dbcac1 )

	ROM_REGION( 0x800000, REGION_GFX2, 0 )
	/* sprites */
	ROM_LOAD( "151a10", 0x000000, 0x200000, 0x376c64f1 )
	ROM_LOAD( "151a11", 0x200000, 0x200000, 0xe7f49225 )
	ROM_LOAD( "151a12", 0x400000, 0x200000, 0x4978555f )
	ROM_LOAD( "151a13", 0x600000, 0x200000, 0x4771f525 )

	ROM_REGION( 0x200000, REGION_SOUND1, 0 )
	/* K054539 samples */
	ROM_LOAD( "151a08", 0x000000, 0x200000, 0x962251d7 )
ROM_END

ROM_START( bucky )
	ROM_REGION( 0x240000, REGION_CPU1, 0 )
	/* main program */
	ROM_LOAD16_BYTE( "173ea.b01", 0x000000,  0x40000, 0x7785ac8a )
	ROM_LOAD16_BYTE( "173ea.b02", 0x000001,  0x40000, 0x9b45f122 )

	/* data */
	ROM_LOAD16_BYTE( "t5", 0x200000,  0x20000, 0xcd724026 )
	ROM_LOAD16_BYTE( "t6", 0x200001,  0x20000, 0x7dd54d6f )

	ROM_REGION( 0x050000, REGION_CPU2, 0 )
	/* Z80 sound program */
	ROM_LOAD("173.a07", 0x000000, 0x40000, 0x4cdaee71 )
	ROM_RELOAD(         0x010000, 0x040000 )

	ROM_REGION( 0x200000, REGION_GFX1, 0 )
	/* tilemaps */
	ROM_LOAD( "173a05.t8",  0x000000, 0x100000, 0xd14333b4 )
	ROM_LOAD( "173a06.t10", 0x100000, 0x100000, 0x6541a34f )

	ROM_REGION( 0x800000, REGION_GFX2, 0 )
	/* sprites */
	ROM_LOAD( "173a10.b8",  0x000000, 0x200000, 0x42fb0a0c )
	ROM_LOAD( "173a11.a8",  0x200000, 0x200000, 0xb0d747c4 )
	ROM_LOAD( "173a12.b10", 0x400000, 0x200000, 0x0fc2ad24 )
	ROM_LOAD( "173a13.a10", 0x600000, 0x200000, 0x4cf85439 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 )
	/* K054539 samples */
	ROM_LOAD("173a08.b6", 0x000000, 0x200000, 0xdcdded95 )
	ROM_LOAD("173a09.a6", 0x200000, 0x200000, 0xc93697c4 )
ROM_END

ROM_START( buckyua )
	ROM_REGION( 0x240000, REGION_CPU1, 0 )
	/* main program */
	ROM_LOAD16_BYTE( "q5", 0x000000,  0x40000, 0xdcaecca0 )
	ROM_LOAD16_BYTE( "q6", 0x000001,  0x40000, 0xe3c856a6 )

	/* data */
	ROM_LOAD16_BYTE( "t5", 0x200000,  0x20000, 0xcd724026 )
	ROM_LOAD16_BYTE( "t6", 0x200001,  0x20000, 0x7dd54d6f )

	ROM_REGION( 0x050000, REGION_CPU2, 0 )
	/* Z80 sound program */
	ROM_LOAD("173.a07", 0x000000, 0x40000, 0x4cdaee71 )
	ROM_RELOAD(         0x010000, 0x040000 )

	ROM_REGION( 0x200000, REGION_GFX1, 0 )
	/* tilemaps */
	ROM_LOAD( "173a05.t8",  0x000000, 0x100000, 0xd14333b4 )
	ROM_LOAD( "173a06.t10", 0x100000, 0x100000, 0x6541a34f )

	ROM_REGION( 0x800000, REGION_GFX2, 0 )
	/* sprites */
	ROM_LOAD( "173a10.b8",  0x000000, 0x200000, 0x42fb0a0c )
	ROM_LOAD( "173a11.a8",  0x200000, 0x200000, 0xb0d747c4 )
	ROM_LOAD( "173a12.b10", 0x400000, 0x200000, 0x0fc2ad24 )
	ROM_LOAD( "173a13.a10", 0x600000, 0x200000, 0x4cf85439 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 )
	/* K054539 samples */
	ROM_LOAD("173a08.b6", 0x000000, 0x200000, 0xdcdded95 )
	ROM_LOAD("173a09.a6", 0x200000, 0x200000, 0xc93697c4 )
ROM_END


static DRIVER_INIT( moo )
{
	konami_rom_deinterleave_2(REGION_GFX1);
	konami_rom_deinterleave_4(REGION_GFX2);

	state_save_register_INT32("Moo", 0, "control2", (INT32 *)&cur_control2, 1);
	state_save_register_UINT16("Moo", 0, "protram", (UINT16 *)protram, 1);

	game_type = (!strcmp(Machine->gamedrv->name, "bucky") || !strcmp(Machine->gamedrv->name, "buckyua"));
}

ROM_START( moobl )
	ROM_REGION( 0x180000, REGION_CPU1, 0 )
	ROM_LOAD16_WORD_SWAP( "moo03.rom", 0x000000, 0x80000, 0xfed6a1cb )
	ROM_LOAD16_WORD_SWAP( "moo04.rom", 0x100000, 0x80000, 0xec45892a )

	ROM_REGION( 0x100000, REGION_USER1, 0 )
	ROM_LOAD16_WORD_SWAP( "moo03.rom", 0x000000, 0x80000, 0xfed6a1cb )
	ROM_LOAD16_WORD_SWAP( "moo04.rom", 0x080000, 0x80000, 0xec45892a )

	ROM_REGION( 0x200000, REGION_GFX1, 0 )
	ROM_LOAD( "moo05.rom", 0x000000, 0x080000, 0x8c045f9c )
	ROM_LOAD( "moo06.rom", 0x080000, 0x080000, 0x1261aa89 )
	ROM_LOAD( "moo07.rom", 0x100000, 0x080000, 0xb9e29f50 )
	ROM_LOAD( "moo08.rom", 0x180000, 0x080000, 0xe6937229 )

	// sprites from bootleg not included in dump, taken from original game
	ROM_REGION( 0x800000, REGION_GFX2, 0 )
	ROM_LOAD( "151a10", 0x000000, 0x200000, 0x376c64f1 )
	ROM_LOAD( "151a11", 0x200000, 0x200000, 0xe7f49225 )
	ROM_LOAD( "151a12", 0x400000, 0x200000, 0x4978555f )
	ROM_LOAD( "151a13", 0x600000, 0x200000, 0x4771f525 )

	ROM_REGION( 0x340000, REGION_SOUND1, 0 )
	ROM_LOAD( "moo01.rom", 0x000000, 0x040000, 0x3311338a )//bank 0 lo & hi
	ROM_CONTINUE(          0x040000+0x30000, 0x010000)//bank 1 hi
	ROM_CONTINUE(          0x080000+0x30000, 0x010000)//bank 2 hi
	ROM_CONTINUE(          0x0c0000+0x30000, 0x010000)//bank 3 hi
	ROM_CONTINUE(          0x100000+0x30000, 0x010000)//bank 4 hi
	ROM_RELOAD(            0x040000, 0x30000 )//bank 1 lo
	ROM_RELOAD(            0x080000, 0x30000 )//bank 2 lo
	ROM_RELOAD(            0x0c0000, 0x30000 )//bank 3 lo
	ROM_RELOAD(            0x100000, 0x30000 )//bank 4 lo
	ROM_RELOAD(            0x140000, 0x30000 )//bank 5 lo
	ROM_RELOAD(            0x180000, 0x30000 )//bank 6 lo
	ROM_RELOAD(            0x1c0000, 0x30000 )//bank 7 lo
	ROM_RELOAD(            0x200000, 0x30000 )//bank 8 lo
	ROM_RELOAD(            0x240000, 0x30000 )//bank 9 lo
	ROM_RELOAD(            0x280000, 0x30000 )//bank a lo
	ROM_RELOAD(            0x2c0000, 0x30000 )//bank b lo
	ROM_RELOAD(            0x300000, 0x30000 )//bank c lo

	ROM_LOAD( "moo02.rom", 0x140000+0x30000, 0x010000, 0x2cf3a7c6 )//bank 5 hi
	ROM_CONTINUE(          0x180000+0x30000, 0x010000)//bank 6 hi
	ROM_CONTINUE(          0x1c0000+0x30000, 0x010000)//bank 7 hi
	ROM_CONTINUE(          0x200000+0x30000, 0x010000)//bank 8 hi
	ROM_CONTINUE(          0x240000+0x30000, 0x010000)//bank 9 hi
	ROM_CONTINUE(          0x280000+0x30000, 0x010000)//bank a hi
	ROM_CONTINUE(          0x2c0000+0x30000, 0x010000)//bank b hi
	ROM_CONTINUE(          0x300000+0x30000, 0x010000)//bank c hi
ROM_END

GAME( 1992, moo,     0,       moo,     moo,     moo,      ROT0, "Konami", "Wild West C.O.W.-Boys of Moo Mesa (World version EA)")
GAME( 1992, mooua,   moo,     moo,     moo,     moo,      ROT0, "Konami", "Wild West C.O.W.-Boys of Moo Mesa (US version UA)")
GAMEX( 1992, moobl,   moo,     moobl,   moo,     moo,      ROT0, "<unknown>", "Wild West C.O.W.-Boys of Moo Mesa (bootleg version AA)", GAME_NOT_WORKING)
GAME( 1992, bucky,   0,       bucky,   bucky,   moo,      ROT0, "Konami", "Bucky O'Hare (World version EA)")
GAME( 1992, buckyua, bucky,   bucky,   bucky,   moo,      ROT0, "Konami", "Bucky O'Hare (US version UA)")
