/***************************************************************************

	Namco PuckMan

    driver by Nicola Salmoria and many others

    Games supported:
		* PuckMan
		* Pac-Man Plus
		* Ms. Pac-Man
		* Crush Roller
		* Ponpoko
		* Eyes
		* Mr. TNT
		* Gorkans
		* Lizard Wizard
		* The Glob
		* Dream Shopper
		* Van Van Car
		* Ali Baba and 40 Thieves
		* Jump Shot
		* Shoot the Bull
		* Big Bucks
		* Driving Force
		* Eight Ball Action
		* Porky

	Known issues:
		* mystery items in Ali Baba don't work correctly because of protection

****************************************************************************

	Pac-Man memory map (preliminary)

	0000-3fff ROM
	4000-43ff Video RAM
	4400-47ff Color RAM
	4c00-4fff RAM
	8000-9fff ROM (Ms Pac-Man and Ponpoko only)
	a000-bfff ROM (Ponpoko only)

	memory mapped ports:

	read:
	5000      IN0
	5040      IN1
	5080      DSW 1
	50c0	  DSW 2 (Ponpoko only)
	see the input_ports definition below for details on the input bits

	write:
	4ff0-4fff 8 pairs of two bytes:
	          the first byte contains the sprite image number (bits 2-7), Y flip (bit 0),
			  X flip (bit 1); the second byte the color
	5000      interrupt enable
	5001      sound enable
	5002      ????
	5003      flip screen
	5004      1 player start lamp
	5005      2 players start lamp
	5006      coin lockout
	5007      coin counter
	5040-5044 sound voice 1 accumulator (nibbles) (used by the sound hardware only)
	5045      sound voice 1 waveform (nibble)
	5046-5049 sound voice 2 accumulator (nibbles) (used by the sound hardware only)
	504a      sound voice 2 waveform (nibble)
	504b-504e sound voice 3 accumulator (nibbles) (used by the sound hardware only)
	504f      sound voice 3 waveform (nibble)
	5050-5054 sound voice 1 frequency (nibbles)
	5055      sound voice 1 volume (nibble)
	5056-5059 sound voice 2 frequency (nibbles)
	505a      sound voice 2 volume (nibble)
	505b-505e sound voice 3 frequency (nibbles)
	505f      sound voice 3 volume (nibble)
	5060-506f Sprite coordinates, x/y pairs for 8 sprites
	50c0      Watchdog reset

	I/O ports:
	OUT on port $0 sets the interrupt vector


****************************************************************************

	Make Trax protection description:

	Make Trax has a "Special" chip that it uses for copy protection.
	The following chart shows when reads and writes may occur:

	AAAAAAAA AAAAAAAA
	11111100 00000000  <- address bits
	54321098 76543210
	xxx1xxxx 01xxxxxx - read data bits 4 and 7
	xxx1xxxx 10xxxxxx - read data bits 6 and 7
	xxx1xxxx 11xxxxxx - read data bits 0 through 5

	xxx1xxxx 00xxx100 - write to Special
	xxx1xxxx 00xxx101 - write to Special
	xxx1xxxx 00xxx110 - write to Special
	xxx1xxxx 00xxx111 - write to Special

	In practical terms, it reads from Special when it reads from
	location $5040-$50FF.  Note that these locations overlap our
	inputs and Dip Switches.  Yuk.

	I don't bother trapping the writes right now, because I don't
	know how to interpret them.  However, comparing against Crush
	Roller gives most of the values necessary on the reads.

	Instead of always reading from $5040, $5080, and $50C0, the Make
	Trax programmers chose to read from a wide variety of locations,
	probably to make debugging easier.  To us, it means that for the most
	part we can just assign a specific value to return for each address and
	we'll be OK.  This falls apart for the following addresses:  $50C0, $508E,
	$5090, and $5080.  These addresses should return multiple values.  The other
	ugly thing happening is in the ROMs at $3AE5.  It keeps checking for
	different values of $50C0 and $5080, and weird things happen if it gets
	the wrong values.  The only way I've found around these is to patch the
	ROMs using the same patches Crush Roller uses.  The only thing to watch
	with this is that changing the ROMs will break the beginning checksum.
	That's why we use the rom opcode decode function to do our patches.

	Incidentally, there are extremely few differences between Crush Roller
	and Make Trax.  About 98% of the differences appear to be either unused
	bytes, the name of the game, or code related to the protection.  I've
	only spotted two or three actual differences in the games, and they all
	seem minor.

	If anybody cares, here's a list of disassembled addresses for every
	read and write to the Special chip (not all of the reads are
	specifically for checking the Special bits, some are for checking
	player inputs and Dip Switches):

	Writes: $0084, $012F, $0178, $023C, $0C4C, $1426, $1802, $1817,
		$280C, $2C2E, $2E22, $3205, $3AB7, $3ACC, $3F3D, $3F40,
		$3F4E, $3F5E
	Reads:  $01C8, $01D2, $0260, $030E, $040E, $0416, $046E, $0474,
		$0560, $0568, $05B0, $05B8, $096D, $0972, $0981, $0C27,
		$0C2C, $0F0A, $10B8, $10BE, $111F, $1127, $1156, $115E,
		$11E3, $11E8, $18B7, $18BC, $18CA, $1973, $197A, $1BE7,
		$1C06, $1C9F, $1CAA, $1D79, $213D, $2142, $2389, $238F,
		$2AAE, $2BF4, $2E0A, $39D5, $39DA, $3AE2, $3AEA, $3EE0,
		$3EE9, $3F07, $3F0D

****************************************************************************

Todo: fix the sets according to this

Puckman is labeled wrong.  Puckman set 1 is
likely a bootleg since the protection is patched out, set 2 would likely
be correct if the roms were split differently.  Nicola had said that he
had a readme that mentioned 2k roms, which is my understanding.
Although the board will accept either, it is likely they were all 2k or
all 4k, not mixed.  Also the set labeled "harder?" is not:

Comparing files npacmod.6j and NAMCOPAC.6J

00000031: AF 25    ;3031 is sub for fail rom check.
00000032: C3 7C ;301c is sub for pass rom check
00000033: 1C E6    ;so it now clears the sum (reg A) and
00000034: 30 F0    ;jumps to pass if it fails rom check.

000007F8: 31 30  c 1981 / c 1980

0000080B: 40 4E  ghost / nickname
0000080C: 47 49
0000080D: 48 43
0000080E: 4F 4B
0000080F: 53 4E
00000810: 54 41
00000811: 40 4D
00000812: 40 45

00000FFF: 00 F1  checksum

Dave Widel

****************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "pacman.h"
#include "pengo.h"
#include "cpu/s2650/s2650.h"


static UINT8 speedcheat = 0;	/* a well known hack allows to make Pac Man run at four times */
								/* his usual speed. When we start the emulation, we check if the */
								/* hack can be applied, and set this flag accordingly. */



/*************************************
 *
 *	Machine init
 *
 *************************************/

MACHINE_INIT( pacman )
{
	unsigned char *RAM = memory_region(REGION_CPU1);

	/* check if the loaded set of ROMs allows the Pac Man speed hack */
	if ((RAM[0x180b] == 0xbe && RAM[0x1ffd] == 0x00) ||
			(RAM[0x180b] == 0x01 && RAM[0x1ffd] == 0xbd))
		speedcheat = 1;
	else
		speedcheat = 0;
}


MACHINE_INIT( pacplus )
{
	unsigned char *RAM = memory_region(REGION_CPU1);

	/* check if the loaded set of ROMs allows the Pac Man speed hack */
	if ((RAM[0x182d] == 0xbe && RAM[0x1ffd] == 0xff) ||
			(RAM[0x182d] == 0x01 && RAM[0x1ffd] == 0xbc))
		speedcheat = 1;
	else
		speedcheat = 0;
}


MACHINE_INIT( mschamp )
{
	data8_t *rom = memory_region(REGION_CPU1) + 0x10000;
	int bankaddr = ((readinputport(3) & 1) * 0x8000);

	cpu_setbank(1,&rom[bankaddr]);
	cpu_setbank(2,&rom[bankaddr+0x4000]);
}

MACHINE_INIT( piranha )
{
	unsigned char *RAM = memory_region(REGION_CPU1);

	/* check if the loaded set of ROMs allows the Pac Man speed hack */
	if ((RAM[0x180b] == 0xbe && RAM[0x1ffd] == 0x00) ||
			(RAM[0x180b] == 0x01 && RAM[0x1ffd] == 0xbd))
		speedcheat = 1;
	else
	speedcheat = 0;
}

/*************************************
 *
 *	Interrupts
 *
 *************************************/

static INTERRUPT_GEN( pacman_interrupt )
{
	unsigned char *RAM = memory_region(REGION_CPU1);

	/* speed up cheat */
	if (speedcheat)
	{
		if (readinputport(4) & 1)	/* check status of the fake dip switch */
		{
			/* activate the cheat */
			RAM[0x180b] = 0x01;
			RAM[0x1ffd] = 0xbd;
		}
		else
		{
			/* remove the cheat */
			RAM[0x180b] = 0xbe;
			RAM[0x1ffd] = 0x00;
		}
	}

	irq0_line_hold();
}


static INTERRUPT_GEN( pacplus_interrupt )
{
	unsigned char *RAM = memory_region(REGION_CPU1);

	/* speed up cheat */
	if (speedcheat)
	{
		if (readinputport(4) & 1)	/* check status of the fake dip switch */
		{
			/* activate the cheat */
			RAM[0x182d] = 0x01;
			RAM[0x1ffd] = 0xbc;
		}
		else
		{
			/* remove the cheat */
			RAM[0x182d] = 0xbe;
			RAM[0x1ffd] = 0xff;
		}
	}

	irq0_line_hold();
}


static INTERRUPT_GEN( mspacman_interrupt )
{
	unsigned char *RAM = memory_region(REGION_CPU1);

	/* speed up cheat */
	if (speedcheat)
	{
		if (readinputport(4) & 1)	/* check status of the fake dip switch */
		{
			/* activate the cheat */
			RAM[0x1180b] = 0x01;
			RAM[0x11ffd] = 0xbd;
		}
		else
		{
			/* remove the cheat */
			RAM[0x1180b] = 0xbe;
			RAM[0x11ffd] = 0x00;
		}
	}

	irq0_line_hold();
}

/*
   The piranha board has a sync bus controler card similar to Midway's pacman. It
   stores the LSB of the interupt vector using port 00 but it alters the byte to prevent
   it from running on normal pacman hardware and vice versa. I wrote a program to print
   out the even numbers and the vectors they convert to.  Thanks to Dave France for
   burning the roms.  The numbers that didn't print here convert to odd numbers.  It's
   slightly possible some numbers listed converted to odd numbers and coincidentally
   printed a valid even number.  Since it only uses 2 vectors($fa,$fc) I didn't complete
   the table or attempt to find the algorythm.

   David Widel
   d_widel@hotmail.com
  out vec  out vec  out vec  out vec
  c0 44    80 04    40 44    00 04
  c2 40    82 00    42 C4    02 84
  c4 C4    84 84    44 C4    04 00
  c6 40    86 00
  c8 4C    88 0C    48 4C    08 0C
  ca 48    8A 08    4A CC    0A 8C
  cc CC    8C 8C    4C 48    0C 08
  ce 48    8E 08
  d0 54    90 14    50 54    10 14
  d2 50    92 10    52 D4    12 94
  d4 D4    94 94    54 50    14 10
  d6 50    96 10
  d8 5C    98 1C    58 5C    18 1C
  da 58    9A 18    5A DC    1A 9C
  dc DC    9C 9C    5C 58    1C 18
  de 58    9E 18
  e0 64    a0 24    60 64    20 24
  e2 60    a2 20    62 E4    22 A4
  e4 E4    a4 A4    64 60    24 20
  e6 60    a6 20
  e8 6C    a8 2C    68 6C    28 2C
  ea 68    aA 28    6A EC    2A AC
  ec EC    aC AC    6C 68    2C 28
  ee 68    aE 28
  f0 74    b0 34    70 74    30 34
  f2 70    b2 30    72 F4    32 84
  f4 F4    b4 B4    74 70    34 30
  f6 70    b6 30
  f8 7C    b8 3C    78 7C    38 3C
  fa 78    bA 38    7A FC    3A BC
  fc FC    bC BC    7C 78    3C 38
  fe 78    bE 38


Naughty Mouse uses the same board as Piranha with a different pal to encrypt the vectors.
Incidentally we don't know the actual name of this game.  Other than the word naughty at
the top of the playfield there's no name.  It shares some character data with the missing
game Woodpecker, they may be related.

I haven't examined the code thoroughly but what I
did look at(sprite buffer), was copied from Pacman.  The addresses for the variables seem
to be the same as well.
*/

static WRITE_HANDLER( piranha_interrupt_vector_w)
{
	if (data==0xfa) data=0x78;
	if (data==0xfc) data=0xfc;
	cpu_irq_line_vector_w( 0, 0, data );
}


static WRITE_HANDLER( nmouse_interrupt_vector_w)
{
	if (data==0xbf) data=0x3c;
	if (data==0xc6) data=0x40;
	if (data==0xfc) data=0xfc;
	cpu_irq_line_vector_w( 0, 0, data );
}


/*************************************
 *
 *	LEDs/coin counters
 *
 *************************************/

static WRITE_HANDLER( pacman_leds_w )
{
	set_led_status(offset,data & 1);
}


static WRITE_HANDLER( pacman_coin_counter_w )
{
	coin_counter_w(offset,data & 1);
}


static WRITE_HANDLER( pacman_coin_lockout_global_w )
{
	coin_lockout_global_w(~data & 0x01);
}



/*************************************
 *
 *	Ali Baba sound
 *
 *************************************/

static WRITE_HANDLER( alibaba_sound_w )
{
	/* since the sound region in Ali Baba is not contiguous, translate the
	   offset into the 0-0x1f range */
 	if (offset < 0x10)
		pengo_sound_w(offset, data);
	else if (offset < 0x20)
		spriteram_2[offset - 0x10] = data;
	else
		pengo_sound_w(offset - 0x10, data);
}


static READ_HANDLER( alibaba_mystery_1_r )
{
	// The return value determines what the mystery item is.  Each bit corresponds
	// to a question mark

	return rand() & 0x0f;
}


static READ_HANDLER( alibaba_mystery_2_r )
{
	static int mystery = 0;

	// The single bit return value determines when the mystery is lit up.
	// This is certainly wrong

	mystery++;
	return (mystery >> 10) & 1;
}



/*************************************
 *
 *	Make Trax input handlers
 *
 *************************************/

static READ_HANDLER( maketrax_special_port2_r )
{
	int data = input_port_2_r(offset);
	int pc = activecpu_get_previouspc();

	if ((pc == 0x1973) || (pc == 0x2389)) return data | 0x40;

	switch (offset)
	{
		case 0x01:
		case 0x04:
			data |= 0x40; break;
		case 0x05:
			data |= 0xc0; break;
		default:
			data &= 0x3f; break;
	}

	return data;
}


static READ_HANDLER( maketrax_special_port3_r )
{
	int pc = activecpu_get_previouspc();

	if (pc == 0x040e) return 0x20;

	if ((pc == 0x115e) || (pc == 0x3ae2)) return 0x00;

	switch (offset)
	{
		case 0x00:
			return 0x1f;
		case 0x09:
			return 0x30;
		case 0x0c:
			return 0x00;
		default:
			return 0x20;
	}
}

static READ_HANDLER( korosuke_special_port2_r )
{
	int data = input_port_2_r(offset);
	int pc = activecpu_get_previouspc();

	if ((pc == 0x196e) || (pc == 0x2387)) return data | 0x40;

	switch (offset)
	{
		case 0x01:
		case 0x04:
			data |= 0x40; break;
		case 0x05:
			data |= 0xc0; break;
		default:
			data &= 0x3f; break;
	}

	return data;
}

static READ_HANDLER( korosuke_special_port3_r )
{
	int pc = activecpu_get_previouspc();

	if (pc == 0x0445) return 0x20;

	if ((pc == 0x115b) || (pc == 0x3ae6)) return 0x00;

	switch (offset)
	{
		case 0x00:
			return 0x1f;
		case 0x09:
			return 0x30;
		case 0x0c:
			return 0x00;
		default:
			return 0x20;
	}
}

/*************************************
 *
 *	Zola kludge
 *
 *************************************/

static READ_HANDLER( mschamp_kludge_r )
{
	static UINT8 counter;
	return counter++;
}

/************************************
 *
 *	Big Bucks questions roms handlers
 *
 ************************************/

static int bigbucks_bank = 0;

static WRITE_HANDLER( bigbucks_bank_w )
{
	bigbucks_bank = data;
}

static READ_HANDLER( bigbucks_question_r )
{

	UINT8 *question = memory_region(REGION_USER1);
	UINT8 ret;

	ret = question[(bigbucks_bank << 16) | (offset ^ 0xffff)];

	return ret;
}

/************************************
 *
 *	S2650 cpu based games
 *
 ************************************/

static INTERRUPT_GEN( s2650_interrupt )
{
	cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0x03);
}

static READ_HANDLER( s2650_mirror_r )
{
	return cpu_readmem16(0x1000 + offset);
}

static WRITE_HANDLER( s2650_mirror_w )
{
	cpu_writemem16(0x1000 + offset, data);
}

static READ_HANDLER( drivfrcp_port1_r )
{
	switch (activecpu_get_pc())
	{
		case 0x0030:
		case 0x0291:
			return 0x01;
	}

    return 0;
}

static READ_HANDLER( _8bpm_port1_r )
{
	switch (activecpu_get_pc())
	{
		case 0x0030:
		case 0x0466:
			return 0x01;
	}

    return 0;
}

static READ_HANDLER( porky_port1_r )
{
	switch (activecpu_get_pc())
	{
		case 0x0034:
			return 0x01;
	}

    return 0;
}


/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x47ff, MRA_RAM },	/* video and color RAM */
	{ 0x4c00, 0x4fff, MRA_RAM },	/* including sprite codes at 4ff0-4fff */
	{ 0x5000, 0x503f, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x507f, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r },	/* DSW1 */
	{ 0x50c0, 0x50ff, input_port_3_r },	/* DSW2 */
	{ 0x8000, 0xbfff, MRA_ROM },	/* Ms. Pac-Man / Ponpoko only */
MEMORY_END


static MEMORY_WRITE_START( writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x43ff, videoram_w, &videoram, &videoram_size },
	{ 0x4400, 0x47ff, colorram_w, &colorram },
	{ 0x4c00, 0x4fef, MWA_RAM },
	{ 0x4ff0, 0x4fff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x5000, 0x5000, interrupt_enable_w },
	{ 0x5001, 0x5001, pengo_sound_enable_w },
	{ 0x5002, 0x5002, MWA_NOP },
	{ 0x5003, 0x5003, pengo_flipscreen_w },
 	{ 0x5004, 0x5005, pacman_leds_w },
// 	{ 0x5006, 0x5006, pacman_coin_lockout_global_w },	this breaks many games
 	{ 0x5007, 0x5007, pacman_coin_counter_w },
	{ 0x5040, 0x505f, pengo_sound_w, &pengo_soundregs },
	{ 0x5060, 0x506f, MWA_RAM, &spriteram_2 },
	{ 0x50c0, 0x50c0, watchdog_reset_w },
	{ 0x8000, 0xbfff, MWA_ROM },	/* Ms. Pac-Man / Ponpoko only */
	{ 0xc000, 0xc3ff, videoram_w }, /* mirror address for video ram, */
	{ 0xc400, 0xc7ef, colorram_w }, /* used to display HIGH SCORE and CREDITS */
	{ 0xffff, 0xffff, MWA_NOP },	/* Eyes writes to this location to simplify code */
MEMORY_END


static MEMORY_READ_START( mschamp_readmem )
	{ 0x0000, 0x3fff, MRA_BANK1 },		/* By Sil: Zola/Ms. Champ */
	{ 0x4000, 0x47ff, MRA_RAM },		/* video and color RAM */
	{ 0x4c00, 0x4fff, MRA_RAM },		/* including sprite codes at 4ff0-4fff */
	{ 0x5000, 0x503f, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x507f, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r },	/* DSW */
	{ 0x8000, 0x9fff, MRA_BANK2 },		/* By Sil: Zola/Ms. Champ */
MEMORY_END


static MEMORY_READ_START( mspacman_readmem )
	{ 0x0000, 0x3fff, MRA_BANK1 },
	{ 0x4000, 0x47ff, MRA_RAM },	/* video and color RAM */
	{ 0x4c00, 0x4fff, MRA_RAM },	/* including sprite codes at 4ff0-4fff */
	{ 0x5000, 0x503f, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x507f, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r },	/* DSW1 */
	{ 0x50c0, 0x50ff, input_port_3_r },	/* DSW2 */
	{ 0x8000, 0xbfff, MRA_BANK1 },
MEMORY_END


static MEMORY_WRITE_START( mspacman_writemem )
	{ 0x0000, 0x3fff, MWA_BANK1 },
	{ 0x4000, 0x43ff, videoram_w, &videoram, &videoram_size },
	{ 0x4400, 0x47ff, colorram_w, &colorram },
	{ 0x4c00, 0x4fef, MWA_RAM },
	{ 0x4ff0, 0x4fff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x5000, 0x5000, interrupt_enable_w },
	{ 0x5001, 0x5001, pengo_sound_enable_w },
	{ 0x5002, 0x5002, MWA_NOP },
	{ 0x5003, 0x5003, pengo_flipscreen_w },
 	{ 0x5004, 0x5005, pacman_leds_w },
	{ 0x5006, 0x5006, mspacman_activate_rom },	/* Not actually, just handy */
// 	{ 0x5006, 0x5006, pacman_coin_lockout_global_w },	this breaks many games
 	{ 0x5007, 0x5007, pacman_coin_counter_w },
	{ 0x5040, 0x505f, pengo_sound_w, &pengo_soundregs },
	{ 0x5060, 0x506f, MWA_RAM, &spriteram_2 },
	{ 0x50c0, 0x50c0, watchdog_reset_w },
	{ 0x8000, 0xbfff, MWA_BANK1 },	/* Ms. Pac-Man / Ponpoko only */
	{ 0xc000, 0xc3ff, videoram_w }, /* mirror address for video ram, */
	{ 0xc400, 0xc7ef, colorram_w }, /* used to display HIGH SCORE and CREDITS */
	{ 0xffff, 0xffff, MWA_NOP },	/* Eyes writes to this location to simplify code */
MEMORY_END


static MEMORY_READ_START( alibaba_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x47ff, MRA_RAM },	/* video and color RAM */
	{ 0x4c00, 0x4fff, MRA_RAM },	/* including sprite codes at 4ef0-4eff */
	{ 0x5000, 0x503f, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x507f, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r },	/* DSW1 */
	{ 0x50c0, 0x50c0, alibaba_mystery_1_r },
	{ 0x50c1, 0x50c1, alibaba_mystery_2_r },
	{ 0x8000, 0x8fff, MRA_ROM },
	{ 0x9000, 0x93ff, MRA_RAM },
	{ 0xa000, 0xa7ff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( alibaba_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x43ff, videoram_w, &videoram, &videoram_size },
	{ 0x4400, 0x47ff, colorram_w, &colorram },
	{ 0x4ef0, 0x4eff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x4c00, 0x4fff, MWA_RAM },
	{ 0x5000, 0x5000, watchdog_reset_w },
 	{ 0x5004, 0x5005, pacman_leds_w },
 	{ 0x5006, 0x5006, pacman_coin_lockout_global_w },
 	{ 0x5007, 0x5007, pacman_coin_counter_w },
	{ 0x5040, 0x506f, alibaba_sound_w, &pengo_soundregs },  /* the sound region is not contiguous */
	{ 0x5060, 0x506f, MWA_RAM, &spriteram_2 }, /* actually at 5050-505f, here to point to free RAM */
	{ 0x50c0, 0x50c0, pengo_sound_enable_w },
	{ 0x50c1, 0x50c1, pengo_flipscreen_w },
	{ 0x50c2, 0x50c2, interrupt_enable_w },
	{ 0x8000, 0x8fff, MWA_ROM },
	{ 0x9000, 0x93ff, MWA_RAM },
	{ 0xa000, 0xa7ff, MWA_ROM },
	{ 0xc000, 0xc3ff, videoram_w }, /* mirror address for video ram, */
	{ 0xc400, 0xc7ef, colorram_w }, /* used to display HIGH SCORE and CREDITS */
MEMORY_END


static MEMORY_READ_START( theglobp_readmem )
	{ 0x0000, 0x3fff, MRA_BANK1 },
	{ 0x4000, 0x47ff, MRA_RAM },	/* video and color RAM */
	{ 0x4c00, 0x4fff, MRA_RAM },	/* including sprite codes at 4ff0-4fff */
	{ 0x5000, 0x503f, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x507f, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r },	/* DSW1 */
	{ 0x50c0, 0x50ff, input_port_3_r },	/* DSW2 */
MEMORY_END


static MEMORY_READ_START( vanvan_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x47ff, MRA_RAM },	/* video and color RAM */
	{ 0x4800, 0x4fff, MRA_RAM },	/* including sprite codes at 4ff0-4fff */
	{ 0x5000, 0x5000, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x5040, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x5080, input_port_2_r },	/* DSW1 */
	{ 0x50c0, 0x50c0, input_port_3_r },	/* DSW2 */
	{ 0x8000, 0x8fff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( vanvan_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x43ff, videoram_w, &videoram, &videoram_size },
	{ 0x4400, 0x47ff, colorram_w, &colorram },
	{ 0x4800, 0x4fef, MWA_RAM },
	{ 0x4ff0, 0x4fff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x5000, 0x5000, interrupt_enable_w },
	{ 0x5001, 0x5001, vanvan_bgcolor_w },
	{ 0x5003, 0x5003, pengo_flipscreen_w },
	{ 0x5005, 0x5006, MWA_NOP },	/* always written together with 5001 */
 	{ 0x5007, 0x5007, pacman_coin_counter_w },
	{ 0x5060, 0x506f, MWA_RAM, &spriteram_2 },
	{ 0x5080, 0x5080, MWA_NOP },	/* ??? toggled before reading 5000 */
	{ 0x50c0, 0x50c0, watchdog_reset_w },
	{ 0x8000, 0x8fff, MWA_ROM },
	{ 0xb800, 0xb87f, MWA_NOP },	/* probably a leftover from development: the Sanritsu version */
									/* writes the color lookup table here, while the Karateko version */
									/* writes garbage. */
MEMORY_END


static MEMORY_READ_START( acitya_readmem )
	{ 0x0000, 0x3fff, MRA_BANK1 },
	{ 0x4000, 0x47ff, MRA_RAM }, /* video and color RAM */
	{ 0x4c00, 0x4fff, MRA_RAM }, /* including sprite codes at 4ff0-4fff */
	{ 0x5000, 0x503f, input_port_0_r }, /* IN0 */
	{ 0x5040, 0x507f, input_port_1_r }, /* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r }, /* DSW1 */
	{ 0x50c0, 0x50ff, input_port_3_r }, /* DSW2 */
	{ 0x8000, 0xbfff, MRA_ROM }, /* Ms. Pac-Man / Ponpoko only */
MEMORY_END


static MEMORY_WRITE_START( bigbucks_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x43ff, videoram_w, &videoram, &videoram_size },
	{ 0x4400, 0x47ff, colorram_w, &colorram },
	{ 0x4c00, 0x4fbf, MWA_RAM },
	{ 0x5000, 0x5000, interrupt_enable_w },
	{ 0x5001, 0x5001, pengo_sound_enable_w },
	{ 0x5003, 0x5003, pengo_flipscreen_w },
	{ 0x5007, 0x5007, MWA_NOP }, //?
	{ 0x5040, 0x505f, pengo_sound_w, &pengo_soundregs },
	{ 0x50c0, 0x50c0, watchdog_reset_w },
	{ 0x5100, 0x5100, MWA_NOP }, //?
	{ 0x6000, 0x6000, bigbucks_bank_w },
	{ 0x8000, 0x9fff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( s2650games_readmem )
	{ 0x0000, 0x0fff, MRA_ROM },
	{ 0x1500, 0x1500, input_port_0_r },
	{ 0x1540, 0x1540, input_port_1_r },
	{ 0x1580, 0x1580, input_port_2_r },
	{ 0x1c00, 0x1fef, MRA_RAM },
	{ 0x2000, 0x2fff, MRA_ROM },
	{ 0x3000, 0x3fff, s2650_mirror_r },
	{ 0x4000, 0x4fff, MRA_ROM },
	{ 0x5000, 0x5fff, s2650_mirror_r },
	{ 0x6000, 0x6fff, MRA_ROM },
	{ 0x7000, 0x7fff, s2650_mirror_r },
MEMORY_END

static MEMORY_WRITE_START( s2650games_writemem )
	{ 0x0000, 0x0fff, MWA_ROM },
	{ 0x1000, 0x13ff, s2650games_colorram_w },
	{ 0x1400, 0x141f, s2650games_scroll_w },
	{ 0x1420, 0x148f, MWA_RAM },
	{ 0x1490, 0x149f, MWA_RAM, &sprite_bank },
	{ 0x14a0, 0x14bf, s2650games_tilesbank_w, &tiles_bankram },
	{ 0x14c0, 0x14ff, MWA_RAM },
	{ 0x1500, 0x1502, MWA_NOP },
	{ 0x1503, 0x1503, s2650games_flipscreen_w },
	{ 0x1504, 0x1506, MWA_NOP },
	{ 0x1507, 0x1507, pacman_coin_counter_w },
	{ 0x1508, 0x155f, MWA_RAM },
	{ 0x1560, 0x156f, MWA_RAM, &spriteram_2 },
	{ 0x1570, 0x157f, MWA_RAM },
	{ 0x1586, 0x1587, MWA_NOP },
	{ 0x15c0, 0x15c0, watchdog_reset_w },
	{ 0x15c7, 0x15c7, MWA_RAM },
	{ 0x1800, 0x1bff, s2650games_videoram_w, &videoram },
	{ 0x1c00, 0x1fef, MWA_RAM },
	{ 0x1ff0, 0x1fff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x2000, 0x2fff, MWA_ROM },
	{ 0x3000, 0x3fff, s2650_mirror_w },
	{ 0x4000, 0x4fff, MWA_ROM },
	{ 0x5000, 0x5fff, s2650_mirror_w },
	{ 0x6000, 0x6fff, MWA_ROM },
	{ 0x7000, 0x7fff, s2650_mirror_w },
MEMORY_END


/*************************************
 *
 *	Main CPU port handlers
 *
 *************************************/

static PORT_WRITE_START( writeport )
	{ 0x00, 0x00, interrupt_vector_w },	/* Pac-Man only */
PORT_END

static PORT_WRITE_START( vanvan_writeport )
	{ 0x01, 0x01, SN76496_0_w },
	{ 0x02, 0x02, SN76496_1_w },
PORT_END

static PORT_WRITE_START( dremshpr_writeport )
	{ 0x06, 0x06, AY8910_write_port_0_w },
	{ 0x07, 0x07, AY8910_control_port_0_w },
PORT_END

static PORT_WRITE_START( piranha_writeport )
	{ 0x00, 0x00, piranha_interrupt_vector_w },
PORT_END

static PORT_WRITE_START( nmouse_writeport )
	{ 0x00, 0x00, nmouse_interrupt_vector_w },
PORT_END

static PORT_READ_START( theglobp_readport )
	{ 0x00, 0xff, theglobp_decrypt_rom },	/* Switch protection logic */
PORT_END

static PORT_READ_START( acitya_readport )
	{ 0x00, 0xff, acitya_decrypt_rom }, /* Switch protection logic */
PORT_END

static PORT_READ_START( mschamp_readport )
	{ 0x00,0x00, mschamp_kludge_r },
PORT_END

static PORT_READ_START( bigbucks_readport )
	{ 0x0000, 0xffff, bigbucks_question_r },
PORT_END

static PORT_READ_START( drivfrcp_readport )
	{ 0x00, 0x00, MRA_NOP },
	{ 0x01, 0x01, drivfrcp_port1_r },
	{ S2650_SENSE_PORT, S2650_SENSE_PORT, input_port_3_r },
PORT_END

static PORT_READ_START( _8bpm_readport )
	{ 0x00, 0x00, MRA_NOP },
	{ 0x01, 0x01, _8bpm_port1_r },
	{ 0xe0, 0xe0, MRA_NOP },
	{ S2650_SENSE_PORT, S2650_SENSE_PORT, input_port_3_r },
PORT_END

static PORT_READ_START( porky_readport )
	{ 0x01, 0x01, porky_port1_r },
	{ S2650_SENSE_PORT, S2650_SENSE_PORT, input_port_3_r },
PORT_END

static PORT_WRITE_START( s2650games_writeport )
	{ S2650_DATA_PORT, S2650_DATA_PORT, SN76496_0_w },
PORT_END


/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( pacman )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BITX(    0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_DIPNAME(0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(   0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(   0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "5" )
	PORT_DIPNAME( 0x30, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x10, "15000" )
	PORT_DIPSETTING(    0x20, "20000" )
	PORT_DIPSETTING(    0x30, "None" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_DIPNAME( 0x80, 0x80, "Ghost Names" )
	PORT_DIPSETTING(    0x80, "Normal" )
	PORT_DIPSETTING(    0x00, "Alternate" )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* FAKE */
	/* This fake input port is used to get the status of the fire button */
	/* and activate the speedup cheat if it is. */
	PORT_BITX(    0x01, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Speedup Cheat", KEYCODE_LCONTROL, JOYCODE_1_BUTTON1 )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
INPUT_PORTS_END


/* Ms. Pac-Man input ports are identical to Pac-Man, the only difference is */
/* the missing Ghost Names dip switch. */
INPUT_PORTS_START( mspacman )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BITX(    0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "5" )
	PORT_DIPNAME( 0x30, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x10, "15000" )
	PORT_DIPSETTING(    0x20, "20000" )
	PORT_DIPSETTING(    0x30, "None" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* FAKE */
	/* This fake input port is used to get the status of the fire button */
	/* and activate the speedup cheat if it is. */
	PORT_BITX(    0x01, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Speedup Cheat", KEYCODE_LCONTROL, JOYCODE_1_BUTTON1 )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
INPUT_PORTS_END


/* Same as 'mspacman', but no fake input port */
INPUT_PORTS_START( mspacpls )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BITX(    0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Also invincibility when playing
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )	// Also speed-up when playing
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "5" )
	PORT_DIPNAME( 0x30, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x10, "15000" )
	PORT_DIPSETTING(    0x20, "20000" )
	PORT_DIPSETTING(    0x30, "None" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( mschamp )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BITX(    0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN3 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "5" )
	PORT_DIPNAME( 0x30, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x10, "15000" )
	PORT_DIPSETTING(    0x20, "20000" )
	PORT_DIPSETTING(    0x30, "None" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* DSW 2 */
	PORT_DIPNAME( 0x01, 0x01, "Game" )
	PORT_DIPSETTING(    0x01, "Champion Edition" )
	PORT_DIPSETTING(    0x00, "Super Zola Pac Gal" )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( maketrax )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Cocktail ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection */

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPSETTING(    0x0c, "6" )
	PORT_DIPNAME( 0x10, 0x10, "First Pattern" )
	PORT_DIPSETTING(    0x10, "Easy" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_DIPNAME( 0x20, 0x20, "Teleport Holes" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
 	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection */

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( korosuke )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Cocktail ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection */

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPSETTING(    0x0c, "6" )
	PORT_DIPNAME( 0x10, 0x10, "First Pattern" )
	PORT_DIPSETTING(    0x10, "Easy" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_DIPNAME( 0x20, 0x20, "Teleport Holes" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
 	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection */

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( mbrush )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Cocktail ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection in Make Trax */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection in Make Trax */

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "4" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection in Make Trax */

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( paintrlr )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Cocktail ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection in Make Trax */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection in Make Trax */

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPSETTING(    0x0c, "6" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )  /* Protection in Make Trax */

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( ponpoko )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	/* The 2nd player controls are used even in upright mode */
	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x01, "10000" )
	PORT_DIPSETTING(    0x02, "30000" )
	PORT_DIPSETTING(    0x03, "50000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, "0" )
	PORT_DIPSETTING(    0x04, "1" )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x0c, "3" )
	PORT_DIPNAME( 0x30, 0x20, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPSETTING(    0x10, "3" )
	PORT_DIPSETTING(    0x20, "4" )
	PORT_DIPSETTING(    0x30, "5" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_DIPNAME( 0x0f, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, "A 3/1 B 3/1" )
	PORT_DIPSETTING(    0x0e, "A 3/1 B 1/2" )
	PORT_DIPSETTING(    0x0f, "A 3/1 B 1/4" )
	PORT_DIPSETTING(    0x02, "A 2/1 B 2/1" )
	PORT_DIPSETTING(    0x0d, "A 2/1 B 1/1" )
	PORT_DIPSETTING(    0x07, "A 2/1 B 1/3" )
	PORT_DIPSETTING(    0x0b, "A 2/1 B 1/5" )
	PORT_DIPSETTING(    0x0c, "A 2/1 B 1/6" )
	PORT_DIPSETTING(    0x01, "A 1/1 B 1/1" )
	PORT_DIPSETTING(    0x06, "A 1/1 B 4/5" )
	PORT_DIPSETTING(    0x05, "A 1/1 B 2/3" )
	PORT_DIPSETTING(    0x0a, "A 1/1 B 1/3" )
	PORT_DIPSETTING(    0x08, "A 1/1 B 1/5" )
	PORT_DIPSETTING(    0x09, "A 1/1 B 1/6" )
	PORT_DIPSETTING(    0x03, "A 1/2 B 1/2" )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )  /* Most likely unused */
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )  /* Most likely unused */
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )  /* Most likely unused */
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( eyes )
	PORT_START  /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x0c, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x30, "50000" )
	PORT_DIPSETTING(    0x20, "75000" )
	PORT_DIPSETTING(    0x10, "100000" )
	PORT_DIPSETTING(    0x00, "125000" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )  /* Not accessed */
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( mrtnt )
	PORT_START  /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x0c, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x30, "75000" )
	PORT_DIPSETTING(    0x20, "100000" )
	PORT_DIPSETTING(    0x10, "125000" )
	PORT_DIPSETTING(    0x00, "150000" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( lizwiz )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x0c, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x30, "75000" )
	PORT_DIPSETTING(    0x20, "100000" )
	PORT_DIPSETTING(    0x10, "125000" )
	PORT_DIPSETTING(    0x00, "150000" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( theglobp )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "3" )
	PORT_DIPSETTING(    0x02, "4" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "6" )
	PORT_DIPNAME( 0x1c, 0x1c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x1c, "Easiest" )
	PORT_DIPSETTING(    0x18, "Very Easy" )
	PORT_DIPSETTING(    0x14, "Easy" )
	PORT_DIPSETTING(    0x10, "Normal" )
	PORT_DIPSETTING(    0x0c, "Difficult" )
	PORT_DIPSETTING(    0x08, "Very Difficult" )
	PORT_DIPSETTING(    0x04, "Very Hard" )
	PORT_DIPSETTING(    0x00, "Hardest" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( vanvan )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x08, "20k and 100k" )
	PORT_DIPSETTING(    0x04, "40k and 140k" )
	PORT_DIPSETTING(    0x00, "70k and 200k" )
	PORT_DIPSETTING(    0x0c, "None" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x30, "3" )
	PORT_DIPSETTING(    0x20, "4" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPSETTING(    0x00, "6" )
	PORT_DIPNAME( 0xc0, 0xc0, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_3C ) )

	/* When all DSW2 are ON, there is no sprite collision detection */
	PORT_START	/* DSW 2 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_BITX(    0x02, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Invulnerability", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )		// Missile effect
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )		// Killer car is destroyed
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )			// Killer car is not destroyed
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( vanvank )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x08, "20k and 100k" )
	PORT_DIPSETTING(    0x04, "40k and 140k" )
	PORT_DIPSETTING(    0x00, "70k and 200k" )
	PORT_DIPSETTING(    0x0c, "None" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x30, "3" )
	PORT_DIPSETTING(    0x20, "4" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPSETTING(    0x00, "6" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_3C ) )

	/* When all DSW2 are ON, there is no sprite collision detection */
	PORT_START	/* DSW 2 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_BITX(    0x02, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Invulnerability", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )		// Missile effect
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )		// Killer car is destroyed
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )			// Killer car is not destroyed
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( dremshpr )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x08, "30000" )
	PORT_DIPSETTING(    0x04, "50000" )
	PORT_DIPSETTING(    0x00, "70000" )
	PORT_DIPSETTING(    0x0c, "None" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x30, "3" )
	PORT_DIPSETTING(    0x20, "4" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPSETTING(    0x00, "6" )
	PORT_DIPNAME( 0xc0, 0xc0, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_3C ) )

	PORT_START	/* DSW 2 */
  //PORT_BITX(    0x01, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Invulnerability", IP_KEY_NONE, IP_JOY_NONE )
  //PORT_DIPSETTING(    0x00, DEF_STR( Off ) )		/* turning this on crashes puts the */
  //PORT_DIPSETTING(    0x01, DEF_STR( On ) )       /* emulated machine in an infinite loop once in a while */
//	PORT_DIPNAME( 0xff, 0x00, DEF_STR( Unused ) )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( alibaba )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BITX(0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "5" )
	PORT_DIPNAME( 0x30, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x10, "15000" )
	PORT_DIPSETTING(    0x20, "20000" )
	PORT_DIPSETTING(    0x30, "None" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( jumpshot )
	PORT_START /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2  )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START /* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, "Time"  )
//	PORT_DIPSETTING(    0x00,  "2 Minutes"  )
	PORT_DIPSETTING(    0x02,  "2 Minutes" )
	PORT_DIPSETTING(    0x03,  "3 Minutes" )
	PORT_DIPSETTING(    0x01,  "4 Minutes"  )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, "2 Players Game" )
	PORT_DIPSETTING(    0x20, "1 Credit" )
	PORT_DIPSETTING(    0x00, "2 Credits" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START /* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END



INPUT_PORTS_START( shootbul )
	PORT_START /* IN0 */
	PORT_ANALOG( 0x0f, 0x0f, IPT_TRACKBALL_X , 50, 25, 0, 0)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN3 )

	PORT_START /* IN1 */
	PORT_ANALOG( 0x0f, 0x0f, IPT_TRACKBALL_Y | IPF_REVERSE, 50, 25, 0, 0)
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START /* DSW 1 */
	PORT_DIPNAME( 0x07, 0x07, "Time"  )
	PORT_DIPSETTING(    0x01, "Short")
	PORT_DIPSETTING(    0x07, "Average" )
	PORT_DIPSETTING(    0x03, "Long" )
	PORT_DIPSETTING(    0x05, "Longer" )
	PORT_DIPSETTING(    0x06, "Longest" )
	PORT_DIPNAME( 0x08, 0x08, "Title Page Sounds"  )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START /* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

// New Atlantic City Action / Board Walk Casino Inputs //
// Annoyingly enough, you can't get into service mode on bwcasino if the
// cocktail mode is set. To test player 2's inputs, select Upright Mode on
// the dipswitches, and enter test mode. Now select cocktail mode and you
// can test everything. Wierd.

INPUT_PORTS_START( bwcasino )
	PORT_START /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 | IPF_COCKTAIL )

	PORT_START /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON5 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON6 )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )

	PORT_START /* DSW 1 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x1e, 0x1e, "Hands per Game" )
	PORT_DIPSETTING(    0x1e, "3" )
	PORT_DIPSETTING(    0x1c, "4" )
	PORT_DIPSETTING(    0x1a, "5" )
	PORT_DIPSETTING(    0x18, "6" )
	PORT_DIPSETTING(    0x16, "7" )
	PORT_DIPSETTING(    0x14, "8" )
	PORT_DIPSETTING(    0x12, "9" )
	PORT_DIPSETTING(    0x10, "10" )
	PORT_DIPSETTING(    0x0e, "11" )
	PORT_DIPSETTING(    0x0c, "12" )
	PORT_DIPSETTING(    0x0a, "13" )
	PORT_DIPSETTING(    0x08, "14" )
	PORT_DIPSETTING(    0x06, "15" )
	PORT_DIPSETTING(    0x04, "16" )
	PORT_DIPSETTING(    0x02, "17" )
	PORT_DIPSETTING(    0x00, "18" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START /* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

// ATLANTIC CITY ACTION (acitya)
// Unlike "Boardwalk Casino", "Atlantic City Action" does not appear to
// have a cocktail mode, and uses service button connected differently to
// "Boardwalk"
INPUT_PORTS_START( acitya )
	PORT_START /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON5 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON6 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START /* DSW 1 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x1e, 0x1e, "Hands per Game" )
	PORT_DIPSETTING(    0x1e, "3" )
	PORT_DIPSETTING(    0x1c, "4" )
	PORT_DIPSETTING(    0x1a, "5" )
	PORT_DIPSETTING(    0x18, "6" )
	PORT_DIPSETTING(    0x16, "7" )
	PORT_DIPSETTING(    0x14, "8" )
	PORT_DIPSETTING(    0x12, "9" )
	PORT_DIPSETTING(    0x10, "10" )
	PORT_DIPSETTING(    0x0e, "11" )
	PORT_DIPSETTING(    0x0c, "12" )
	PORT_DIPSETTING(    0x0a, "13" )
	PORT_DIPSETTING(    0x08, "14" )
	PORT_DIPSETTING(    0x06, "15" )
	PORT_DIPSETTING(    0x04, "16" )
	PORT_DIPSETTING(    0x02, "17" )
	PORT_DIPSETTING(    0x00, "18" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START /* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( nmouse )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BITX(    0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_DIPNAME(0x80, 0x80, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(   0x80, DEF_STR( Upright ) )
	PORT_DIPSETTING(   0x00, DEF_STR( Cocktail ) )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "5" )
	PORT_DIPNAME( 0x30, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "5000" )
	PORT_DIPSETTING(    0x10, "10000" )
	PORT_DIPSETTING(    0x20, "15000" )
	PORT_DIPSETTING(    0x30, "None" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x00, "Hard" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

INPUT_PORTS_END


INPUT_PORTS_START( bigbucks )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP	  | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_DIPNAME( 0x10, 0x10, "Enable Category Adult Affairs" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* DSW 1 */
	PORT_DIPNAME( 0x01, 0x00, "Time to bet / answer" )
	PORT_DIPSETTING(    0x00, "15 sec. / 10 sec." )
	PORT_DIPSETTING(    0x01, "20 sec. / 15 sec." )
	PORT_DIPNAME( 0x02, 0x00, "Continue if player busts" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Show correct answer" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START	/* DSW 2 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( drivfrcp )
	PORT_START      /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START      /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START /* Sense */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_VBLANK )
INPUT_PORTS_END


INPUT_PORTS_START( 8bpm )
	PORT_START      /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START      /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT_NAME( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1, "Start 1 / P1 Button 1" )
	PORT_BIT_NAME( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2, "Start 2 / P1 Button 1" )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START /* Sense */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_VBLANK )
INPUT_PORTS_END


INPUT_PORTS_START( porky )
	PORT_START      /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START      /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START /* Sense */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_VBLANK )
INPUT_PORTS_END

/*************************************
 *
 *	Graphics layouts
 *
 *************************************/

static struct GfxLayout tilelayout =
{
	8,8,	/* 8*8 characters */
	RGN_FRAC(1,1),
    2,  /* 2 bits per pixel */
    { 0, 4 },   /* the two bitplanes for 4 pixels are packed into one byte */
    { 8*8+0, 8*8+1, 8*8+2, 8*8+3, 0, 1, 2, 3 }, /* bits are packed in groups of four */
    { 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8 },
    16*8    /* every char takes 16 bytes */
};


static struct GfxLayout spritelayout =
{
	16,16,	/* 16*16 sprites */
	RGN_FRAC(1,1),
	2,	/* 2 bits per pixel */
	{ 0, 4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3, 0, 1, 2, 3 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8,
			32*8, 33*8, 34*8, 35*8, 36*8, 37*8, 38*8, 39*8 },
	64*8	/* every sprite takes 64 bytes */
};


static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &tilelayout,   0, 32 },
	{ REGION_GFX2, 0, &spritelayout, 0, 32 },
	{ -1 } /* end of array */
};


static struct GfxDecodeInfo mschampgfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x0000, &tilelayout,   0, 32 },
	{ REGION_GFX1, 0x1000, &spritelayout, 0, 32 },
	{ -1 } /* end of array */
};


static struct GfxDecodeInfo s2650games_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x0000, &tilelayout,      0, 32 },
    { REGION_GFX1, 0x0000, &spritelayout,    0, 32 },
    { -1 } /* end of array */
};



/*************************************
 *
 *	Sound interfaces
 *
 *************************************/

static struct namco_interface namco_interface =
{
	3072000/32,	/* sample rate */
	3,			/* number of voices */
	100,		/* playback volume */
	REGION_SOUND1	/* memory region */
};


static struct SN76496interface sn76496_interface =
{
	2,
	{ 1789750, 1789750 },	/* 1.78975 MHz ? */
	{ 75, 75 }
};


static struct AY8910interface dremshpr_ay8910_interface =
{
	1,	/* 1 chip */
	14318000/8,	/* 1.78975 MHz ??? */
	{ 50 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 }
};


static struct SN76496interface sn76489_interface =
{
	1,	/* 1 chip */
	{ 18432000/6/2/3 },	/* ? MHz */
	{ 100 }
};


/*************************************
 *
 *	Machine drivers
 *
 *************************************/

static MACHINE_DRIVER_START( pacman )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", Z80, 18432000/6)
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_PORTS(0,writeport)
	MDRV_CPU_VBLANK_INT(pacman_interrupt,1)

	MDRV_FRAMES_PER_SECOND( (18432000.0 / 2) / (576 * 264) )
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	MDRV_MACHINE_INIT(pacman)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(36*8, 28*8)
	MDRV_VISIBLE_AREA(0*8, 36*8-1, 0*8, 28*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(16)
	MDRV_COLORTABLE_LENGTH(4*32)

	MDRV_PALETTE_INIT(pacman)
	MDRV_VIDEO_START(pacman)
	MDRV_VIDEO_UPDATE(pengo)

	/* sound hardware */
	MDRV_SOUND_ADD_TAG("namco", NAMCO, namco_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( pacplus )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_VBLANK_INT(pacplus_interrupt,1)

	MDRV_MACHINE_INIT(pacplus)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mspacman )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mspacman_readmem,mspacman_writemem)
	MDRV_CPU_VBLANK_INT(mspacman_interrupt,1)

	MDRV_MACHINE_INIT(mspacman)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mspacpls )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_VBLANK_INT(irq0_line_hold,1)

	MDRV_MACHINE_INIT(NULL)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mschamp )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mschamp_readmem,writemem)
	MDRV_CPU_PORTS(mschamp_readport,writeport)
	MDRV_CPU_VBLANK_INT(irq0_line_hold,1)

	MDRV_MACHINE_INIT(mschamp)

	/* video hardware */
	MDRV_GFXDECODE(mschampgfxdecodeinfo)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( theglobp )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(theglobp_readmem,writemem)
	MDRV_CPU_PORTS(theglobp_readport,writeport)

	MDRV_MACHINE_INIT(theglobp)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( vanvan )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(vanvan_readmem,vanvan_writemem)
	MDRV_CPU_PORTS(0,vanvan_writeport)
	MDRV_CPU_VBLANK_INT(nmi_line_pulse,1)

	MDRV_MACHINE_INIT(NULL)

	/* video hardware */
	MDRV_VISIBLE_AREA(2*8, 34*8-1, 0*8, 28*8-1)
	MDRV_VIDEO_UPDATE(vanvan)

	/* sound hardware */
	MDRV_SOUND_REPLACE("namco", SN76496, sn76496_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( dremshpr )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(0,dremshpr_writeport)
	MDRV_CPU_VBLANK_INT(nmi_line_pulse,1)

	MDRV_MACHINE_INIT(NULL)

	/* sound hardware */
	MDRV_SOUND_REPLACE("namco", AY8910, dremshpr_ay8910_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( alibaba )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(alibaba_readmem,alibaba_writemem)
	MDRV_CPU_VBLANK_INT(irq0_line_hold,1)

	MDRV_MACHINE_INIT(NULL)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( piranha )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_PORTS(0,piranha_writeport)

	MDRV_MACHINE_INIT(piranha)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( nmouse )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_PORTS(0,nmouse_writeport)

	MDRV_MACHINE_INIT(NULL)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( acitya )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(acitya_readmem,writemem)
	MDRV_CPU_PORTS(acitya_readport,writeport)

	MDRV_MACHINE_INIT(acitya)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( bigbucks )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_FLAGS(CPU_16BIT_PORT)
	MDRV_CPU_MEMORY(readmem,bigbucks_writemem)
	MDRV_CPU_PORTS(bigbucks_readport,0)
	MDRV_CPU_VBLANK_INT(irq0_line_hold,20)

	MDRV_MACHINE_INIT(NULL)

	MDRV_VISIBLE_AREA(0*8, 36*8-1, 0*8-1, 28*8-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( s2650games )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(pacman)

	MDRV_CPU_REMOVE("main")
	MDRV_CPU_ADD_TAG("main", S2650, 18432000/6/2/3)	/* ??? */
	MDRV_CPU_MEMORY(s2650games_readmem,s2650games_writemem)
	MDRV_CPU_VBLANK_INT(s2650_interrupt,1)

	MDRV_SCREEN_SIZE(32*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 32*8-1, 2*8, 30*8-1)
	MDRV_GFXDECODE(s2650games_gfxdecodeinfo)

	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	MDRV_VIDEO_START(s2650games)
	MDRV_VIDEO_UPDATE(s2650games)

	/* sound hardware */
	MDRV_SOUND_REPLACE("namco", SN76496, sn76489_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( drivfrcp )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(s2650games)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(drivfrcp_readport,s2650games_writeport)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( 8bpm )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(s2650games)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(_8bpm_readport,s2650games_writeport)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( porky )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(s2650games)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(porky_readport,s2650games_writeport)
MACHINE_DRIVER_END


/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( puckman )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "namcopac.6e",  0x0000, 0x1000, 0xfee263b3 )
	ROM_LOAD( "namcopac.6f",  0x1000, 0x1000, 0x39d1fc83 )
	ROM_LOAD( "namcopac.6h",  0x2000, 0x1000, 0x02083b03 )
	ROM_LOAD( "namcopac.6j",  0x3000, 0x1000, 0x7a36fe55 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5e",    0x0000, 0x1000, 0x0c944964 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( puckmod )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "namcopac.6e",  0x0000, 0x1000, 0xfee263b3 )
	ROM_LOAD( "namcopac.6f",  0x1000, 0x1000, 0x39d1fc83 )
	ROM_LOAD( "namcopac.6h",  0x2000, 0x1000, 0x02083b03 )
	ROM_LOAD( "npacmod.6j",   0x3000, 0x1000, 0x7d98d5f5 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5e",    0x0000, 0x1000, 0x0c944964 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( puckmana )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "pacman.6e",    0x0000, 0x1000, 0xc1e6ab10 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "pacman.6h",    0x2000, 0x1000, 0xbcdd1beb )
	ROM_LOAD( "prg7",         0x3000, 0x0800, 0xb6289b26 )
	ROM_LOAD( "prg8",         0x3800, 0x0800, 0x17a88c13 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "chg1",         0x0000, 0x0800, 0x2066a0b7 )
	ROM_LOAD( "chg2",         0x0800, 0x0800, 0x3591b89d )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( pacman )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "pacman.6e",    0x0000, 0x1000, 0xc1e6ab10 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "pacman.6h",    0x2000, 0x1000, 0xbcdd1beb )
	ROM_LOAD( "pacman.6j",    0x3000, 0x1000, 0x817d94e3 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5e",    0x0000, 0x1000, 0x0c944964 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( pacmod )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "pacmanh.6e",   0x0000, 0x1000, 0x3b2ec270 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "pacmanh.6h",   0x2000, 0x1000, 0x18811780 )
	ROM_LOAD( "pacmanh.6j",   0x3000, 0x1000, 0x5c96a733 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacmanh.5e",   0x0000, 0x1000, 0x299fb17a )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( hangly )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "hangly.6e",    0x0000, 0x1000, 0x5fe8610a )
	ROM_LOAD( "hangly.6f",    0x1000, 0x1000, 0x73726586 )
	ROM_LOAD( "hangly.6h",    0x2000, 0x1000, 0x4e7ef99f )
	ROM_LOAD( "hangly.6j",    0x3000, 0x1000, 0x7f4147e6 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5e",    0x0000, 0x1000, 0x0c944964 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( hangly2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "hangly.6e",    0x0000, 0x1000, 0x5fe8610a )
	ROM_LOAD( "hangly2.6f",   0x1000, 0x0800, 0x5ba228bb )
	ROM_LOAD( "hangly2.6m",   0x1800, 0x0800, 0xbaf5461e )
	ROM_LOAD( "hangly.6h",    0x2000, 0x1000, 0x4e7ef99f )
	ROM_LOAD( "hangly2.6j",   0x3000, 0x0800, 0x51305374 )
	ROM_LOAD( "hangly2.6p",   0x3800, 0x0800, 0x427c9d4d )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacmanh.5e",   0x0000, 0x1000, 0x299fb17a )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END

ROM_START( hangly3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "hm1.6e",   0x0000, 0x0800, 0x9d027c4a )
	ROM_LOAD( "hm5.6k",	  0x0800, 0x0800, 0x194c7189 )
	ROM_LOAD( "hangly2.6f",   0x1000, 0x0800, 0x5ba228bb ) // hm2.6f
	ROM_LOAD( "hangly2.6m",   0x1800, 0x0800, 0xbaf5461e ) // hm6.6m
	ROM_LOAD( "hm3.6h",   0x2000, 0x0800, 0x08419c4a )
	ROM_LOAD( "hm7.6n",   0x2800, 0x0800, 0xab74b51f )
	ROM_LOAD( "hm4.6j",   0x3000, 0x0800, 0x5039b082 )
	ROM_LOAD( "hm8.6p",   0x3800, 0x0800, 0x931770d7 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "hm9.5e", 	0x0000, 0x0800, 0x5f4be3cc )
	ROM_LOAD( "hm11.5h",    0x0800, 0x0800, 0x3591b89d )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "hm10.5f", 	0x0000, 0x0800, 0x9e39323a )
	ROM_LOAD( "hm12.5j", 	0x0800, 0x0800, 0x1b1d9096 )


	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( newpuckx )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "puckman.6e",   0x0000, 0x1000, 0xa8ae23c5 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "puckman.6h",   0x2000, 0x1000, 0x197443f8 )
	ROM_LOAD( "puckman.6j",   0x3000, 0x1000, 0x2e64a3ba )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5e",    0x0000, 0x1000, 0x0c944964 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5f",    0x0000, 0x1000, 0x958fedf9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( pacheart )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "1.6e",         0x0000, 0x0800, 0xd844b679 )
	ROM_LOAD( "pacheart.pg2", 0x0800, 0x0800, 0xb9152a38 )
	ROM_LOAD( "2.6f",         0x1000, 0x0800, 0x7d177853 )
	ROM_LOAD( "pacheart.pg4", 0x1800, 0x0800, 0x842d6574 )
	ROM_LOAD( "3.6h",         0x2000, 0x0800, 0x9045a44c )
	ROM_LOAD( "7.6n",         0x2800, 0x0800, 0x888f3c3e )
	ROM_LOAD( "pacheart.pg7", 0x3000, 0x0800, 0xf5265c10 )
	ROM_LOAD( "pacheart.pg8", 0x3800, 0x0800, 0x1a21a381 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacheart.ch1", 0x0000, 0x0800, 0xc62bbabf )
	ROM_LOAD( "chg2",         0x0800, 0x0800, 0x3591b89d )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacheart.ch3", 0x0000, 0x0800, 0xca8c184c )
	ROM_LOAD( "pacheart.ch4", 0x0800, 0x0800, 0x1b1d9096 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )  /* timing - not used */
ROM_END


ROM_START( joyman )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )     /* 64k for code */
	ROM_LOAD( "1.6e",         0x0000, 0x0800, 0xd844b679 )
	ROM_LOAD( "5.6k",         0x0800, 0x0800, 0xab9c8f29 )
	ROM_LOAD( "2.6f",         0x1000, 0x0800, 0x7d177853 )
	ROM_LOAD( "6.6m",         0x1800, 0x0800, 0xb3c8d32e )
	ROM_LOAD( "3.6h",         0x2000, 0x0800, 0x9045a44c )
	ROM_LOAD( "7.6n",         0x2800, 0x0800, 0x888f3c3e )
	ROM_LOAD( "4.6j",         0x3000, 0x0800, 0x00b553f8 )
	ROM_LOAD( "8.6p",         0x3800, 0x0800, 0x5d5ce992 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "9.5e",  0x0000, 0x0800, 0x39b557bc )
	ROM_LOAD( "11.5h", 0x0800, 0x0800, 0x33e0289e )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "10.5f", 0x0000, 0x0800, 0x338771a6 )
	ROM_LOAD( "12.5j", 0x0800, 0x0800, 0xf4f0add5 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )  /* timing - not used */
ROM_END


ROM_START( piranha )
	ROM_REGION( 0x10000, REGION_CPU1,0 )     /* 64k for code */
	ROM_LOAD( "pir1.bin", 0x0000, 0x0800, 0x69a3e6ea )
	ROM_LOAD( "pir5.bin", 0x0800, 0x0800, 0x245e753f )
	ROM_LOAD( "pir2.bin", 0x1000, 0x0800, 0x62cb6954 )
	ROM_LOAD( "pir6.bin", 0x1800, 0x0800, 0xcb0700bc )
	ROM_LOAD( "pir3.bin", 0x2000, 0x0800, 0x843fbfe5 )
	ROM_LOAD( "pir7.bin", 0x2800, 0x0800, 0x73084d5e )
	ROM_LOAD( "pir4.bin", 0x3000, 0x0800, 0x4cdf6704 )
	ROM_LOAD( "pir8.bin", 0x3800, 0x0800, 0xb86fedb3 )

	ROM_REGION( 0x1000, REGION_GFX1 , ROMREGION_DISPOSE)
	ROM_LOAD( "pir9.bin", 0x0000, 0x0800, 0x0f19eb28 )
	ROM_LOAD( "pir11.bin", 0x0800, 0x0800, 0x5f8bdabe )

	ROM_REGION( 0x1000, REGION_GFX2 , ROMREGION_DISPOSE )
	ROM_LOAD( "pir10.bin", 0x0000, 0x0800, 0xd19399fb )
	ROM_LOAD( "pir12.bin", 0x0800, 0x0800, 0xcfb4403d )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f", 0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "piranha.4a", 0x0020, 0x0100, 0x08c9447b )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m", 0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m", 0x0100, 0x0100, 0x77245b66 )  /* timing - not used */
ROM_END

ROM_START( piranhao )
	ROM_REGION( 0x10000, REGION_CPU1,0 )     /* 64k for code */
	ROM_LOAD( "p1.bin", 0x0000, 0x0800, 0xc6ce1bfc )
	ROM_LOAD( "p5.bin", 0x0800, 0x0800, 0xa2655a33 )
	ROM_LOAD( "pir2.bin", 0x1000, 0x0800, 0x62cb6954 )
	ROM_LOAD( "pir6.bin", 0x1800, 0x0800, 0xcb0700bc )
	ROM_LOAD( "pir3.bin", 0x2000, 0x0800, 0x843fbfe5 )
	ROM_LOAD( "pir7.bin", 0x2800, 0x0800, 0x73084d5e )
	ROM_LOAD( "p4.bin", 0x3000, 0x0800, 0x9363a4d1 )
	ROM_LOAD( "p8.bin", 0x3800, 0x0800, 0x2769979c )

	ROM_REGION( 0x1000, REGION_GFX1 , ROMREGION_DISPOSE )
	ROM_LOAD( "p9.bin", 0x0000, 0x0800, 0x94eb7563 )
	ROM_LOAD( "p11.bin", 0x0800, 0x0800, 0xa3606973 )

	ROM_REGION( 0x1000, REGION_GFX2 , ROMREGION_DISPOSE )
	ROM_LOAD( "p10.bin", 0x0000, 0x0800, 0x84165a2c )
	ROM_LOAD( "p12.bin", 0x0800, 0x0800, 0x2699ba9e )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f", 0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "piranha.4a", 0x0020, 0x0100, 0x08c9447b )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m", 0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m", 0x0100, 0x0100, 0x77245b66 )  /* timing - not used */
ROM_END

ROM_START( piranhah )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "pr1.cpu",      0x0000, 0x1000, 0xbc5ad024 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "pr3.cpu",      0x2000, 0x1000, 0x473c379d )
	ROM_LOAD( "pr4.cpu",      0x3000, 0x1000, 0x63fbf895 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pr5.cpu",      0x0000, 0x0800, 0x3fc4030c )
	ROM_LOAD( "pr7.cpu",      0x0800, 0x0800, 0x30b9a010 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pr6.cpu",      0x0000, 0x0800, 0xf3e9c9d5 )
	ROM_LOAD( "pr8.cpu",      0x0800, 0x0800, 0x133d720d )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( pacplus )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "pacplus.6e",   0x0000, 0x1000, 0xd611ef68 )
	ROM_LOAD( "pacplus.6f",   0x1000, 0x1000, 0xc7207556 )
	ROM_LOAD( "pacplus.6h",   0x2000, 0x1000, 0xae379430 )
	ROM_LOAD( "pacplus.6j",   0x3000, 0x1000, 0x5a6dff7b )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pacplus.5e",   0x0000, 0x1000, 0x022c35da )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacplus.5f",   0x0000, 0x1000, 0x4de65cdd )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "pacplus.7f",   0x0000, 0x0020, 0x063dd53a )
	ROM_LOAD( "pacplus.4a",   0x0020, 0x0100, 0xe271a166 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mspacman )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 64k for code+64k for decrypted code */
	ROM_LOAD( "pacman.6e",    0x0000, 0x1000, 0xc1e6ab10 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "pacman.6h",    0x2000, 0x1000, 0xbcdd1beb )
	ROM_LOAD( "pacman.6j",    0x3000, 0x1000, 0x817d94e3 )
	ROM_LOAD( "u5",           0x8000, 0x0800, 0xf45fbbcd )
	ROM_LOAD( "u6",           0x9000, 0x1000, 0xa90e7000 )
	ROM_LOAD( "u7",           0xb000, 0x1000, 0xc82cd714 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x1000, 0x5c281d01 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f",           0x0000, 0x1000, 0x615af909 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mspacmab )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "boot1",        0x0000, 0x1000, 0xd16b31b7 )
	ROM_LOAD( "boot2",        0x1000, 0x1000, 0x0d32de5e )
	ROM_LOAD( "boot3",        0x2000, 0x1000, 0x1821ee0b )
	ROM_LOAD( "boot4",        0x3000, 0x1000, 0x165a9dd8 )
	ROM_LOAD( "boot5",        0x8000, 0x1000, 0x8c3e6de6 )
	ROM_LOAD( "boot6",        0x9000, 0x1000, 0x368cb165 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x1000, 0x5c281d01 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f",           0x0000, 0x1000, 0x615af909 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mspacmat )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 64k for code+64k for decrypted code */
	ROM_LOAD( "pacman.6e",    0x0000, 0x1000, 0xc1e6ab10 )
	ROM_LOAD( "pacman.6f",    0x1000, 0x1000, 0x1a6fb2d4 )
	ROM_LOAD( "pacman.6h",    0x2000, 0x1000, 0xbcdd1beb )
	ROM_LOAD( "pacman.6j",    0x3000, 0x1000, 0x817d94e3 )
	ROM_LOAD( "u5",           0x8000, 0x0800, 0xf45fbbcd )
	ROM_LOAD( "u6pacatk",     0x9000, 0x1000, 0xf6d83f4d )
	ROM_LOAD( "u7",           0xb000, 0x1000, 0xc82cd714 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x1000, 0x5c281d01 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f",           0x0000, 0x1000, 0x615af909 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mspacpls )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "boot1",        0x0000, 0x1000, 0xd16b31b7 )
	ROM_LOAD( "mspacatk.2",   0x1000, 0x1000, 0x0af09d31 )
	ROM_LOAD( "boot3",        0x2000, 0x1000, 0x1821ee0b )
	ROM_LOAD( "boot4",        0x3000, 0x1000, 0x165a9dd8 )
	ROM_LOAD( "mspacatk.5",   0x8000, 0x1000, 0xe6e06954 )
	ROM_LOAD( "mspacatk.6",   0x9000, 0x1000, 0x3b5db308 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x1000, 0x5c281d01 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f",           0x0000, 0x1000, 0x615af909 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( pacgal )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "boot1",        0x0000, 0x1000, 0xd16b31b7 )
	ROM_LOAD( "boot2",        0x1000, 0x1000, 0x0d32de5e )
	ROM_LOAD( "pacman.7fh",   0x2000, 0x1000, 0x513f4d5c )
	ROM_LOAD( "pacman.7hj",   0x3000, 0x1000, 0x70694c8e )
	ROM_LOAD( "boot5",        0x8000, 0x1000, 0x8c3e6de6 )
	ROM_LOAD( "boot6",        0x9000, 0x1000, 0x368cb165 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x1000, 0x5c281d01 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pacman.5ef",   0x0000, 0x0800, 0x65a3ee71 )
	ROM_LOAD( "pacman.5hj",   0x0800, 0x0800, 0x50c7477d )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s129.4a",    0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mschamp )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )
	ROM_LOAD( "pm4.bin", 0x10000, 0x10000, 0x7d6b6303 )	/* banked */

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pm5.bin", 0x0000, 0x0800, 0x7fe6b9e2 )
	ROM_CONTINUE(        0x1000, 0x0800 )
	ROM_CONTINUE(        0x0800, 0x0800 )
	ROM_CONTINUE(        0x1800, 0x0800 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )
ROM_END


ROM_START( crush )
	ROM_REGION( 2*0x10000, REGION_CPU1, 0 )	/* 64k for code + 64k for opcode copy to hack protection */
	ROM_LOAD( "crushkrl.6e",  0x0000, 0x1000, 0xa8dd8f54 )
	ROM_LOAD( "crushkrl.6f",  0x1000, 0x1000, 0x91387299 )
	ROM_LOAD( "crushkrl.6h",  0x2000, 0x1000, 0xd4455f27 )
	ROM_LOAD( "crushkrl.6j",  0x3000, 0x1000, 0xd59fc251 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "maketrax.5e",  0x0000, 0x1000, 0x91bad2da )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "maketrax.5f",  0x0000, 0x1000, 0xaea79f55 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( crush2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "tp1",          0x0000, 0x0800, 0xf276592e )
	ROM_LOAD( "tp5a",         0x0800, 0x0800, 0x3d302abe )
	ROM_LOAD( "tp2",          0x1000, 0x0800, 0x25f42e70 )
	ROM_LOAD( "tp6",          0x1800, 0x0800, 0x98279cbe )
	ROM_LOAD( "tp3",          0x2000, 0x0800, 0x8377b4cb )
	ROM_LOAD( "tp7",          0x2800, 0x0800, 0xd8e76c8c )
	ROM_LOAD( "tp4",          0x3000, 0x0800, 0x90b28fa3 )
	ROM_LOAD( "tp8",          0x3800, 0x0800, 0x10854e1b )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "tpa",          0x0000, 0x0800, 0xc7617198 )
	ROM_LOAD( "tpc",          0x0800, 0x0800, 0xe129d76a )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "tpb",          0x0000, 0x0800, 0xd1899f05 )
	ROM_LOAD( "tpd",          0x0800, 0x0800, 0xd35d1caf )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( crush3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "unkmol.4e",    0x0000, 0x0800, 0x49150ddf )
	ROM_LOAD( "unkmol.6e",    0x0800, 0x0800, 0x21f47e17 )
	ROM_LOAD( "unkmol.4f",    0x1000, 0x0800, 0x9b6dd592 )
	ROM_LOAD( "unkmol.6f",    0x1800, 0x0800, 0x755c1452 )
	ROM_LOAD( "unkmol.4h",    0x2000, 0x0800, 0xed30a312 )
	ROM_LOAD( "unkmol.6h",    0x2800, 0x0800, 0xfe4bb0eb )
	ROM_LOAD( "unkmol.4j",    0x3000, 0x0800, 0x072b91c9 )
	ROM_LOAD( "unkmol.6j",    0x3800, 0x0800, 0x66fba07d )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "unkmol.5e",    0x0000, 0x0800, 0x338880a0 )
	ROM_LOAD( "unkmol.5h",    0x0800, 0x0800, 0x4ce9c81f )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "unkmol.5f",    0x0000, 0x0800, 0x752e3780 )
	ROM_LOAD( "unkmol.5j",    0x0800, 0x0800, 0x6e00d2ac )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( maketrax )
	ROM_REGION( 2*0x10000, REGION_CPU1, 0 )	/* 64k for code + 64k for opcode copy to hack protection */
	ROM_LOAD( "maketrax.6e",  0x0000, 0x1000, 0x0150fb4a )
	ROM_LOAD( "maketrax.6f",  0x1000, 0x1000, 0x77531691 )
	ROM_LOAD( "maketrax.6h",  0x2000, 0x1000, 0xa2cdc51e )
	ROM_LOAD( "maketrax.6j",  0x3000, 0x1000, 0x0b4b5e0a )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "maketrax.5e",  0x0000, 0x1000, 0x91bad2da )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "maketrax.5f",  0x0000, 0x1000, 0xaea79f55 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( maketrxb )
	ROM_REGION( 2*0x10000, REGION_CPU1, 0 )	/* 64k for code + 64k for opcode copy to hack protection */
	ROM_LOAD( "maketrax.6e",  0x0000, 0x1000, 0x0150fb4a )
	ROM_LOAD( "maketrax.6f",  0x1000, 0x1000, 0x77531691 )
	ROM_LOAD( "maketrxb.6h",  0x2000, 0x1000, 0x6ad342c9 )
	ROM_LOAD( "maketrxb.6j",  0x3000, 0x1000, 0xbe27f729 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "maketrax.5e",  0x0000, 0x1000, 0x91bad2da )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "maketrax.5f",  0x0000, 0x1000, 0xaea79f55 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END

ROM_START( korosuke )
	ROM_REGION( 2*0x10000, REGION_CPU1, 0 )	/* 64k for code + 64k for opcode copy to hack protection */
	ROM_LOAD( "kr.6e",        0x0000, 0x1000, 0x69f6e2da )
	ROM_LOAD( "kr.6f",        0x1000, 0x1000, 0xabf34d23 )
	ROM_LOAD( "kr.6h",        0x2000, 0x1000, 0x76a2e2e2 )
	ROM_LOAD( "kr.6j",        0x3000, 0x1000, 0x33e0e3bb )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "kr.5e",        0x0000, 0x1000, 0xe0380be8 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "kr.5f",        0x0000, 0x1000, 0x63fec9ee )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mbrush )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "mbrush.6e",    0x0000, 0x1000, 0x750fbff7 )
	ROM_LOAD( "mbrush.6f",    0x1000, 0x1000, 0x27eb4299 )
	ROM_LOAD( "mbrush.6h",    0x2000, 0x1000, 0xd297108e )
	ROM_LOAD( "mbrush.6j",    0x3000, 0x1000, 0x6fd719d0 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "tpa",          0x0000, 0x0800, 0xc7617198 )
	ROM_LOAD( "mbrush.5h",    0x0800, 0x0800, 0xc15b6967 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "mbrush.5f",    0x0000, 0x0800, 0xd5bc5cb8 )  /* copyright sign was removed */
	ROM_LOAD( "tpd",          0x0800, 0x0800, 0xd35d1caf )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( paintrlr )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "paintrlr.1",   0x0000, 0x0800, 0x556d20b5 )
	ROM_LOAD( "paintrlr.5",   0x0800, 0x0800, 0x4598a965 )
	ROM_LOAD( "paintrlr.2",   0x1000, 0x0800, 0x2da29c81 )
	ROM_LOAD( "paintrlr.6",   0x1800, 0x0800, 0x1f561c54 )
	ROM_LOAD( "paintrlr.3",   0x2000, 0x0800, 0xe695b785 )
	ROM_LOAD( "paintrlr.7",   0x2800, 0x0800, 0x00e6eec0 )
	ROM_LOAD( "paintrlr.4",   0x3000, 0x0800, 0x0fd5884b )
	ROM_LOAD( "paintrlr.8",   0x3800, 0x0800, 0x4900114a )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "tpa",          0x0000, 0x0800, 0xc7617198 )
	ROM_LOAD( "mbrush.5h",    0x0800, 0x0800, 0xc15b6967 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "mbrush.5f",    0x0000, 0x0800, 0xd5bc5cb8 )  /* copyright sign was removed */
	ROM_LOAD( "tpd",          0x0800, 0x0800, 0xd35d1caf )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "2s140.4a",     0x0020, 0x0100, 0x63efb927 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( ponpoko )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "ppokoj1.bin",  0x0000, 0x1000, 0xffa3c004 )
	ROM_LOAD( "ppokoj2.bin",  0x1000, 0x1000, 0x4a496866 )
	ROM_LOAD( "ppokoj3.bin",  0x2000, 0x1000, 0x17da6ca3 )
	ROM_LOAD( "ppokoj4.bin",  0x3000, 0x1000, 0x9d39a565 )
	ROM_LOAD( "ppoko5.bin",   0x8000, 0x1000, 0x54ca3d7d )
	ROM_LOAD( "ppoko6.bin",   0x9000, 0x1000, 0x3055c7e0 )
	ROM_LOAD( "ppoko7.bin",   0xa000, 0x1000, 0x3cbe47ca )
	ROM_LOAD( "ppokoj8.bin",  0xb000, 0x1000, 0x04b63fc6 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "ppoko9.bin",   0x0000, 0x1000, 0xb73e1a06 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "ppoko10.bin",  0x0000, 0x1000, 0x62069b5d )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( ponpokov )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "ppoko1.bin",   0x0000, 0x1000, 0x49077667 )
	ROM_LOAD( "ppoko2.bin",   0x1000, 0x1000, 0x5101781a )
	ROM_LOAD( "ppoko3.bin",   0x2000, 0x1000, 0xd790ed22 )
	ROM_LOAD( "ppoko4.bin",   0x3000, 0x1000, 0x4e449069 )
	ROM_LOAD( "ppoko5.bin",   0x8000, 0x1000, 0x54ca3d7d )
	ROM_LOAD( "ppoko6.bin",   0x9000, 0x1000, 0x3055c7e0 )
	ROM_LOAD( "ppoko7.bin",   0xa000, 0x1000, 0x3cbe47ca )
	ROM_LOAD( "ppoko8.bin",   0xb000, 0x1000, 0xb39be27d )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "ppoko9.bin",   0x0000, 0x1000, 0xb73e1a06 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "ppoko10.bin",  0x0000, 0x1000, 0x62069b5d )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( eyes )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "d7",           0x0000, 0x1000, 0x3b09ac89 )
	ROM_LOAD( "e7",           0x1000, 0x1000, 0x97096855 )
	ROM_LOAD( "f7",           0x2000, 0x1000, 0x731e294e )
	ROM_LOAD( "h7",           0x3000, 0x1000, 0x22f7a719 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "d5",           0x0000, 0x1000, 0xd6af0030 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "e5",           0x0000, 0x1000, 0xa42b5201 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s129.4a",    0x0020, 0x0100, 0xd8d78829 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( eyes2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "g38201.7d",    0x0000, 0x1000, 0x2cda7185 )
	ROM_LOAD( "g38202.7e",    0x1000, 0x1000, 0xb9fe4f59 )
	ROM_LOAD( "g38203.7f",    0x2000, 0x1000, 0xd618ba66 )
	ROM_LOAD( "g38204.7h",    0x3000, 0x1000, 0xcf038276 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "g38205.5d",    0x0000, 0x1000, 0x03b1b4c7 )  /* this one has a (c) sign */

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "e5",           0x0000, 0x1000, 0xa42b5201 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s129.4a",    0x0020, 0x0100, 0xd8d78829 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( mrtnt )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "tnt.1",        0x0000, 0x1000, 0x0e836586 )
	ROM_LOAD( "tnt.2",        0x1000, 0x1000, 0x779c4c5b )
	ROM_LOAD( "tnt.3",        0x2000, 0x1000, 0xad6fc688 )
	ROM_LOAD( "tnt.4",        0x3000, 0x1000, 0xd77557b3 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "tnt.5",        0x0000, 0x1000, 0x3038cc0e )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "tnt.6",        0x0000, 0x1000, 0x97634d8b )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "mrtnt08.bin",  0x0000, 0x0020, 0 )
	ROM_LOAD( "mrtnt04.bin",  0x0020, 0x0100, 0 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m"  ,  0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END

ROM_START( gorkans )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "gorkans8.rom",        0x0000, 0x0800, 0x55100b18 )
	ROM_LOAD( "gorkans4.rom",        0x0800, 0x0800, 0xb5c604bf )
 	ROM_LOAD( "gorkans7.rom",        0x1000, 0x0800, 0xb8c6def4 )
	ROM_LOAD( "gorkans3.rom",        0x1800, 0x0800, 0x4602c840 )
	ROM_LOAD( "gorkans6.rom",        0x2000, 0x0800, 0x21412a62 )
	ROM_LOAD( "gorkans2.rom",        0x2800, 0x0800, 0xa013310b )
	ROM_LOAD( "gorkans5.rom",        0x3000, 0x0800, 0x122969b2 )
	ROM_LOAD( "gorkans1.rom",        0x3800, 0x0800, 0xf2524b11 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "gorkgfx4.rom",        0x0000, 0x0800, 0x39cd0dbc )
	ROM_LOAD( "gorkgfx2.rom",        0x0800, 0x0800, 0x33d52535 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "gorkgfx3.rom",        0x0000, 0x0800, 0x4b6b7970 )
	ROM_LOAD( "gorkgfx1.rom",        0x0800, 0x0800, 0xed70bb3c )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "gorkprom.4",   0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "gorkprom.1",   0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "gorkprom.3",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "gorkprom.2"  ,  0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END

ROM_START( eggor )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "1.bin",        0x0000, 0x0800, 0x818ed154 )
	ROM_LOAD( "5.bin",        0x0800, 0x0800, 0xa4b21d93 )
	ROM_LOAD( "2.bin",        0x1000, 0x0800, 0x5d7a23ed )
	ROM_LOAD( "6.bin",        0x1800, 0x0800, 0xe9dbca8d )
	ROM_LOAD( "3.bin",        0x2000, 0x0800, 0x4318ab85 )
	ROM_LOAD( "7.bin",        0x2800, 0x0800, 0x03214d7f )
	ROM_LOAD( "4.bin",        0x3000, 0x0800, 0xdc805be4 )
	ROM_LOAD( "8.bin",        0x3800, 0x0800, 0xf9ae204b )


	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "9.bin",        0x0000, 0x0800, 0x96ad8626 )
	ROM_LOAD( "11.bin",       0x0800, 0x0800, 0xcc324017 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "10.bin",        0x0000, 0x0800, 0x7c97f513 )
	ROM_LOAD( "12.bin",        0x0800, 0x0800, 0x2e930602 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	/* the board was stripped of its proms, these are the standard ones from Pacman, they look reasonable
	   but without another board its impossible to say if they are actually good */
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END

ROM_START( lizwiz )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "6e.cpu",       0x0000, 0x1000, 0x32bc1990 )
	ROM_LOAD( "6f.cpu",       0x1000, 0x1000, 0xef24b414 )
	ROM_LOAD( "6h.cpu",       0x2000, 0x1000, 0x30bed83d )
	ROM_LOAD( "6j.cpu",       0x3000, 0x1000, 0xdd09baeb )
	ROM_LOAD( "wiza",         0x8000, 0x1000, 0xf6dea3a6 )
	ROM_LOAD( "wizb",         0x9000, 0x1000, 0xf27fb5a8 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e.cpu",       0x0000, 0x1000, 0x45059e73 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f.cpu",       0x0000, 0x1000, 0xd2469717 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "7f.cpu",       0x0000, 0x0020, 0x7549a947 )
	ROM_LOAD( "4a.cpu",       0x0020, 0x0100, 0x5fdca536 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m"  ,  0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( theglobp )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "glob.u2",      0x0000, 0x2000, 0x829d0bea )
	ROM_LOAD( "glob.u3",      0x2000, 0x2000, 0x31de6628 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "glob.5e",      0x0000, 0x1000, 0x53688260 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "glob.5f",      0x0000, 0x1000, 0x051f59c7 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "glob.7f",      0x0000, 0x0020, 0x1f617527 )
	ROM_LOAD( "glob.4a",      0x0020, 0x0100, 0x28faa769 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m"  ,  0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( beastf )
	ROM_REGION( 0x20000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "bf-u2.bin",    0x0000, 0x2000, 0x3afc517b )
	ROM_LOAD( "bf-u3.bin",    0x2000, 0x2000, 0x8dbd76d0 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "beastf.5e",    0x0000, 0x1000, 0x5654dc34 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "beastf.5f",    0x0000, 0x1000, 0x1b30ca61 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "glob.7f",      0x0000, 0x0020, 0x1f617527 )
	ROM_LOAD( "glob.4a",      0x0020, 0x0100, 0x28faa769 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m"  ,  0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( vanvan )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "van-1.50",     0x0000, 0x1000, 0xcf1b2df0 )
	ROM_LOAD( "van-2.51",     0x1000, 0x1000, 0xdf58e1cb )
	ROM_LOAD( "van-3.52",     0x2000, 0x1000, 0x15571e24 )
	ROM_LOAD( "van-4.53",     0x3000, 0x1000, 0xb724cbe0 )
	ROM_LOAD( "van-5.39",     0x8000, 0x1000, 0xdb67414c )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "van-20.18",    0x0000, 0x1000, 0x60efbe66 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "van-21.19",    0x0000, 0x1000, 0x5dd53723 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "6331-1.6",     0x0000, 0x0020, 0xce1d9503 )
	ROM_LOAD( "6301-1.37",    0x0020, 0x0100, 0x4b803d9f )
ROM_END

ROM_START( vanvank )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "van1.bin",	  0x0000, 0x1000, 0x00f48295 )
	ROM_LOAD( "van-2.51",     0x1000, 0x1000, 0xdf58e1cb )
	ROM_LOAD( "van-3.52",     0x2000, 0x1000, 0x15571e24 )
	ROM_LOAD( "van4.bin",     0x3000, 0x1000, 0xf8b37ed5 )
	ROM_LOAD( "van5.bin",     0x8000, 0x1000, 0xb8c1e089 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "van-20.18",    0x0000, 0x1000, 0x60efbe66 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "van-21.19",    0x0000, 0x1000, 0x5dd53723 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "6331-1.6",     0x0000, 0x0020, 0xce1d9503 )
	ROM_LOAD( "6301-1.37",    0x0020, 0x0100, 0x4b803d9f )
ROM_END

ROM_START( dremshpr )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "red_1.50",	  0x0000, 0x1000, 0x830c6361 )
	ROM_LOAD( "red_2.51",     0x1000, 0x1000, 0xd22551cc )
	ROM_LOAD( "red_3.52",     0x2000, 0x1000, 0x0713a34a )
	ROM_LOAD( "red_4.53",     0x3000, 0x1000, 0xf38bcaaa )
	ROM_LOAD( "red_5.39",     0x8000, 0x1000, 0x6a382267 )
	ROM_LOAD( "red_6.40",     0x9000, 0x1000, 0x4cf8b121 )
	ROM_LOAD( "red_7.41",     0xa000, 0x1000, 0xbd4fc4ba )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "red-20.18",    0x0000, 0x1000, 0x2d6698dc )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "red-21.19",    0x0000, 0x1000, 0x38c9ce9b )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "6331-1.6",     0x0000, 0x0020, 0xce1d9503 )
	ROM_LOAD( "6301-1.37",    0x0020, 0x0100, 0x39d6fb5c )
ROM_END


ROM_START( alibaba )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "6e",           0x0000, 0x1000, 0x38d701aa )
	ROM_LOAD( "6f",           0x1000, 0x1000, 0x3d0e35f3 )
	ROM_LOAD( "6h",           0x2000, 0x1000, 0x823bee89 )
	ROM_LOAD( "6k",           0x3000, 0x1000, 0x474d032f )
	ROM_LOAD( "6l",           0x8000, 0x1000, 0x5ab315c1 )
	ROM_LOAD( "6m",           0xa000, 0x0800, 0x438d0357 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x0800, 0x85bcb8f8 )
	ROM_LOAD( "5h",           0x0800, 0x0800, 0x38e50862 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f",           0x0000, 0x0800, 0xb5715c86 )
	ROM_LOAD( "5k",           0x0800, 0x0800, 0x713086b3 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "alibaba.7f",   0x0000, 0x0020, 0 )  /* missing */
	ROM_LOAD( "alibaba.4a",   0x0020, 0x0100, 0 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( jumpshot )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "6e",           0x0000, 0x1000, 0xf00def9a )
	ROM_LOAD( "6f",           0x1000, 0x1000, 0xf70deae2 )
	ROM_LOAD( "6h",           0x2000, 0x1000, 0x894d6f68 )
	ROM_LOAD( "6j",           0x3000, 0x1000, 0xf15a108a )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e",           0x0000, 0x1000, 0xd9fa90f5 )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "5f",           0x0000, 0x1000, 0x2ec711c1 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "prom.7f",      0x0000, 0x0020, 0x872b42f3 )
	ROM_LOAD( "prom.4a",      0x0020, 0x0100, 0x0399f39f )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */
ROM_END


ROM_START( shootbul )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* 64k for code */
	ROM_LOAD( "sb6e.cpu",     0x0000, 0x1000, 0x25daa5e9 )
	ROM_LOAD( "sb6f.cpu",     0x1000, 0x1000, 0x92144044 )
	ROM_LOAD( "sb6h.cpu",     0x2000, 0x1000, 0x43b7f99d )
	ROM_LOAD( "sb6j.cpu",     0x3000, 0x1000, 0xbc4d3bbf )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "sb5e.cpu",     0x0000, 0x1000, 0x07c6c5aa )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "sb5f.cpu",     0x0000, 0x1000, 0xeaec6837 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "7f.rom",       0x0000, 0x0020, 0xec578b98 )
	ROM_LOAD( "4a.rom",       0x0020, 0x0100, 0x81a6b30f )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 ) /* timing - not used */
ROM_END


ROM_START( acitya )
	ROM_REGION( 0x20000, REGION_CPU1, 0 ) /* 64k for code */
	ROM_LOAD( "aca_u2.bin",   0x0000, 0x2000, 0x261c2fdc )
	ROM_LOAD( "aca_u3.bin",   0x2000, 0x2000, 0x05fab4ca )


	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "aca_5e.bin",   0x0000, 0x1000, 0x7f2dd2c9 )

 	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "aca_5e.bin",   0x0000, 0x1000, 0x7f2dd2c9 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "aca_7f.bin",   0x0000, 0x0020, 0x133bb744 )
	ROM_LOAD( "aca_4a.bin",   0x0020, 0x0100, 0x8e29208f )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m"  ,  0x0100, 0x0100, 0x77245b66 ) /* timing - not used */
ROM_END

ROM_START( bwcasino )
	ROM_REGION( 0x20000, REGION_CPU1, 0 ) /* 64k for code */
	ROM_LOAD( "bwc_u2.bin",   0x0000, 0x2000, 0xe2eea868 )
	ROM_LOAD( "bwc_u3.bin",   0x2000, 0x2000, 0xa935571e )


	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "bwc_5e.bin",   0x0000, 0x1000, 0xe334c01e )

	ROM_REGION( 0x1000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "bwc_5e.bin",   0x0000, 0x1000, 0xe334c01e )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "aca_7f.bin",   0x0000, 0x0020, 0x133bb744 )
	ROM_LOAD( "aca_4a.bin",   0x0020, 0x0100, 0x8e29208f )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m"  ,  0x0100, 0x0100, 0x77245b66 ) /* timing - not used */
ROM_END


ROM_START( newpuc2 )
	ROM_REGION( 0x10000, REGION_CPU1,0 )     /* 64k for code */
	ROM_LOAD( "6e.cpu", 0x0000, 0x0800, 0x69496a98 )
	ROM_LOAD( "6k.cpu", 0x0800, 0x0800, 0x158fc01c )
	ROM_LOAD( "6f.cpu", 0x1000, 0x0800, 0x7d177853 )
	ROM_LOAD( "6m.cpu", 0x1800, 0x0800, 0x70810ccf )
	ROM_LOAD( "6h.cpu", 0x2000, 0x0800, 0x81719de8 )
	ROM_LOAD( "6n.cpu", 0x2800, 0x0800, 0x3f250c58 )
	ROM_LOAD( "6j.cpu", 0x3000, 0x0800, 0xe6675736 )
	ROM_LOAD( "6p.cpu", 0x3800, 0x0800, 0x1f81e765 )

	ROM_REGION( 0x1000, REGION_GFX1 , ROMREGION_DISPOSE)
	ROM_LOAD( "5e.cpu", 0x0000, 0x0800, 0x2066a0b7 )
	ROM_LOAD( "5h.cpu", 0x0800, 0x0800, 0x777c70d3 )

	ROM_REGION( 0x1000, REGION_GFX2 , ROMREGION_DISPOSE )
	ROM_LOAD( "5f.cpu", 0x0000, 0x0800, 0xca8c184c )
	ROM_LOAD( "5j.cpu", 0x0800, 0x0800, 0x7dc75a81 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f", 0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m", 0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m", 0x0100, 0x0100, 0x77245b66 )   /*timing - not used */
ROM_END


ROM_START( newpuc2b )
	ROM_REGION( 0x10000, REGION_CPU1,0 )     /* 64k for code */
	ROM_LOAD( "np2b1.bin", 0x0000, 0x0800, 0x9d027c4a )
	ROM_LOAD( "6k.cpu", 0x0800, 0x0800, 0x158fc01c )
	ROM_LOAD( "6f.cpu", 0x1000, 0x0800, 0x7d177853 )
	ROM_LOAD( "6m.cpu", 0x1800, 0x0800, 0x70810ccf )
	ROM_LOAD( "np2b3.bin", 0x2000, 0x0800, 0xf5e4b2b1 )
	ROM_LOAD( "6n.cpu", 0x2800, 0x0800, 0x3f250c58 )
	ROM_LOAD( "np2b4.bin", 0x3000, 0x0800, 0xf068e009 )
	ROM_LOAD( "np2b8.bin", 0x3800, 0x0800, 0x1fadcc2f )

	ROM_REGION( 0x1000, REGION_GFX1 , ROMREGION_DISPOSE)
	ROM_LOAD( "5e.cpu", 0x0000, 0x0800, 0x2066a0b7 )
	ROM_LOAD( "5h.cpu", 0x0800, 0x0800, 0x777c70d3 )

	ROM_REGION( 0x1000, REGION_GFX2 , ROMREGION_DISPOSE )
	ROM_LOAD( "5f.cpu", 0x0000, 0x0800, 0xca8c184c )
	ROM_LOAD( "5j.cpu", 0x0800, 0x0800, 0x7dc75a81 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f", 0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 ) /* sound PROMs */
	ROM_LOAD( "82s126.1m", 0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m", 0x0100, 0x0100, 0x77245b66 )   /*timing - not used */
ROM_END

ROM_START( nmouse )
	ROM_REGION( 0x10000, REGION_CPU1,0 )     /* 64k for code */
	ROM_LOAD( "naumouse.d7", 0x0000, 0x0800, 0xe447ecfa )
	ROM_LOAD( "naumouse.d6", 0x0800, 0x0800, 0x2e6f13d9 )
	ROM_LOAD( "naumouse.e7", 0x1000, 0x0800, 0x44a80f97 )
	ROM_LOAD( "naumouse.e6", 0x1800, 0x0800, 0x9c7a46bd )
	ROM_LOAD( "naumouse.h7", 0x2000, 0x0800, 0x5bc94c5d )
	ROM_LOAD( "naumouse.h6", 0x2800, 0x0800, 0x1af29e22 )
	ROM_LOAD( "naumouse.j7", 0x3000, 0x0800, 0xcc3be185 )
	ROM_LOAD( "naumouse.j6", 0x3800, 0x0800, 0x66b3e5dc )

	ROM_REGION( 0x1000, REGION_GFX1 , ROMREGION_DISPOSE)
	ROM_LOAD( "naumouse.d5", 0x0000, 0x0800, 0x2ea7cc3f )
	ROM_LOAD( "naumouse.h5", 0x0800, 0x0800, 0x0511fcea )

	ROM_REGION( 0x1000, REGION_GFX2 , ROMREGION_DISPOSE )
	ROM_LOAD( "naumouse.e5", 0x0000, 0x0800, 0xf5a627cd )
	ROM_LOAD( "naumouse.j5", 0x0800, 0x0800, 0x65f2580e )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f", 0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "naumouse.a4", 0x0020, 0x0100, 0xd8772167 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m", 0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m", 0x0100, 0x0100, 0x77245b66 )   /*timing - not used */
ROM_END

ROM_START( nmouseb )
	ROM_REGION( 0x10000, REGION_CPU1,0 )     /* 64k for code */
	ROM_LOAD( "naumouse.d7", 0x0000, 0x0800, 0xe447ecfa )
	ROM_LOAD( "naumouse.d6", 0x0800, 0x0800, 0x2e6f13d9 )
	ROM_LOAD( "naumouse.e7", 0x1000, 0x0800, 0x44a80f97 )
	ROM_LOAD( "naumouse.e6", 0x1800, 0x0800, 0x9c7a46bd )
	ROM_LOAD( "snatch2.bin", 0x2000, 0x0800, 0x405aa389 )
	ROM_LOAD( "snatch6.bin", 0x2800, 0x0800, 0xf58e7df4 )
	ROM_LOAD( "snatch3.bin", 0x3000, 0x0800, 0x06fb18ec )
	ROM_LOAD( "snatch7.bin", 0x3800, 0x0800, 0xd187b82b )

	ROM_REGION( 0x1000, REGION_GFX1 , ROMREGION_DISPOSE)
	ROM_LOAD( "naumouse.d5", 0x0000, 0x0800, 0x2ea7cc3f )
	ROM_LOAD( "naumouse.h5", 0x0800, 0x0800, 0x0511fcea )

	ROM_REGION( 0x1000, REGION_GFX2 , ROMREGION_DISPOSE )
	ROM_LOAD( "naumouse.e5", 0x0000, 0x0800, 0xf5a627cd )
	ROM_LOAD( "snatch11.bin", 0x0800, 0x0800, 0x330230a5 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f", 0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "naumouse.a4", 0x0020, 0x0100, 0xd8772167 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m", 0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m", 0x0100, 0x0100, 0x77245b66 )   /*timing - not used */
ROM_END


ROM_START( bigbucks )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "p.rom",        0x0000, 0x4000, 0xeea6c1c9 )
	ROM_LOAD( "m.rom",        0x8000, 0x2000, 0xbb8f7363 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "5e.cpu",       0x0000, 0x1000, 0x18442c37 )

	ROM_REGION( 0x1000, REGION_GFX2, 0 )
	/* Not Used */

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.7f",    0x0000, 0x0020, 0x2fc650bd )
	ROM_LOAD( "82s126.4a",    0x0020, 0x0100, 0x3eb3a8e4 )

	ROM_REGION( 0x0200, REGION_SOUND1, 0 )	/* sound PROMs */
	ROM_LOAD( "82s126.1m",    0x0000, 0x0100, 0xa9cc86bf )
	ROM_LOAD( "82s126.3m",    0x0100, 0x0100, 0x77245b66 )	/* timing - not used */

	ROM_REGION( 0x60000, REGION_USER1, 0 )	/* Question ROMs */
	ROM_LOAD( "rom1.rom",     0x00000, 0x8000, 0x90b7785f )
	ROM_LOAD( "rom2.rom",     0x08000, 0x8000, 0x60172d77 )
	ROM_LOAD( "rom3.rom",     0x10000, 0x8000, 0xa2207320 )
	ROM_LOAD( "rom4.rom",     0x18000, 0x8000, 0x5a74c1f9 )
	ROM_LOAD( "rom5.rom",     0x20000, 0x8000, 0x93bc1080 )
	ROM_LOAD( "rom6.rom",     0x28000, 0x8000, 0xeea2423f )
	ROM_LOAD( "rom7.rom",     0x30000, 0x8000, 0x96694055 )
	ROM_LOAD( "rom8.rom",     0x38000, 0x8000, 0xe68ebf8e )
	ROM_LOAD( "rom9.rom",     0x40000, 0x8000, 0xfd20921d )
	ROM_LOAD( "rom10.rom",    0x48000, 0x8000, 0x5091b951 )
	ROM_LOAD( "rom11.rom",    0x50000, 0x8000, 0x705128db )
	ROM_LOAD( "rom12.rom",    0x58000, 0x8000, 0x74c776e7 )
ROM_END


ROM_START( drivfrcp )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )	/* 32k for code */
	ROM_LOAD( "drivforc.1",   0x0000, 0x1000, 0x10b59d27 )
	ROM_CONTINUE(			  0x2000, 0x1000 )
	ROM_CONTINUE(			  0x4000, 0x1000 )
	ROM_CONTINUE(			  0x6000, 0x1000 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "drivforc.2",   0x0000, 0x1000, 0x56331cb5 )
	ROM_CONTINUE(			  0x2000, 0x1000 )
	ROM_CONTINUE(			  0x1000, 0x1000 )
	ROM_CONTINUE(			  0x3000, 0x1000 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "drivforc.pr1", 0x0000, 0x0020, 0x045aa47f )
	ROM_LOAD( "drivforc.pr2", 0x0020, 0x0100, 0x9e6d2f1d )
ROM_END


ROM_START( 8bpm )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )	/* 32k for code */
	ROM_LOAD( "8bpmp.bin",    0x0000, 0x1000, 0xb4f7eba7 )
	ROM_CONTINUE(			  0x2000, 0x1000 )
	ROM_CONTINUE(			  0x4000, 0x1000 )
	ROM_CONTINUE(			  0x6000, 0x1000 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "8bpmc.bin",    0x0000, 0x1000, 0x1c894a6d )
	ROM_CONTINUE(			  0x2000, 0x1000 )
	ROM_CONTINUE(			  0x1000, 0x1000 )
	ROM_CONTINUE(			  0x3000, 0x1000 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "8bpm.pr1", 0x0000, 0x0020, 0 )
	ROM_LOAD( "8bpm.pr2", 0x0020, 0x0100, 0 )
	/* used only to display something until we have the dumps */
	ROM_LOAD( "drivforc.pr1", 0x0000, 0x0020, 0x045aa47f )
	ROM_LOAD( "drivforc.pr2", 0x0020, 0x0100, 0x9e6d2f1d )
ROM_END


ROM_START( porky )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )	/* 32k for code */
	ROM_LOAD( "pp",			  0x0000, 0x1000, 0x00592624 )
	ROM_CONTINUE(			  0x2000, 0x1000 )
	ROM_CONTINUE(			  0x4000, 0x1000 )
	ROM_CONTINUE(			  0x6000, 0x1000 )

	/* what is it used for ? */
	ROM_REGION( 0x4000, REGION_USER1, 0 )
	ROM_LOAD( "ps",			  0x0000, 0x4000, 0x2efb9861 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pc",			  0x0000, 0x1000, 0xa20e3d39 )
	ROM_CONTINUE(			  0x2000, 0x1000 )
	ROM_CONTINUE(			  0x1000, 0x1000 )
	ROM_CONTINUE(			  0x3000, 0x1000 )

	ROM_REGION( 0x0120, REGION_PROMS, 0 )
	ROM_LOAD( "7f",			  0x0000, 0x0020, 0x98bce7cc )
	ROM_LOAD( "4a",			  0x0020, 0x0100, 0x30fe0266 )
ROM_END

/*************************************
 *
 *	Driver initialization
 *
 *************************************/

static void maketrax_rom_decode(void)
{
	unsigned char *rom = memory_region(REGION_CPU1);
	int diff = memory_region_length(REGION_CPU1) / 2;


	/* patch protection using a copy of the opcodes so ROM checksum */
	/* tests will not fail */
	memory_set_opcode_base(0,rom+diff);

	memcpy(rom+diff,rom,diff);

	rom[0x0415 + diff] = 0xc9;
	rom[0x1978 + diff] = 0x18;
	rom[0x238e + diff] = 0xc9;
	rom[0x3ae5 + diff] = 0xe6;
	rom[0x3ae7 + diff] = 0x00;
	rom[0x3ae8 + diff] = 0xc9;
	rom[0x3aed + diff] = 0x86;
	rom[0x3aee + diff] = 0xc0;
	rom[0x3aef + diff] = 0xb0;
}

static DRIVER_INIT( maketrax )
{
	/* set up protection handlers */
	install_mem_read_handler(0, 0x5080, 0x50bf, maketrax_special_port2_r);
	install_mem_read_handler(0, 0x50c0, 0x50ff, maketrax_special_port3_r);

	maketrax_rom_decode();
}

static void korosuke_rom_decode(void)
{
	unsigned char *rom = memory_region(REGION_CPU1);
	int diff = memory_region_length(REGION_CPU1) / 2;


	/* patch protection using a copy of the opcodes so ROM checksum */
	/* tests will not fail */
	memory_set_opcode_base(0,rom+diff);

	memcpy(rom+diff,rom,diff);

	rom[0x044c + diff] = 0xc9;
	rom[0x1973 + diff] = 0x18;
	rom[0x238c + diff] = 0xc9;
	rom[0x3ae9 + diff] = 0xe6;	// not changed
	rom[0x3aeb + diff] = 0x00;
	rom[0x3aec + diff] = 0xc9;
	rom[0x3af1 + diff] = 0x86;
	rom[0x3af2 + diff] = 0xc0;
	rom[0x3af3 + diff] = 0xb0;
}

static DRIVER_INIT( korosuke )
{
	/* set up protection handlers */
	install_mem_read_handler(0, 0x5080, 0x5080, korosuke_special_port2_r);
	install_mem_read_handler(0, 0x50c0, 0x50ff, korosuke_special_port3_r);

	korosuke_rom_decode();
}

static DRIVER_INIT( ponpoko )
{
	int i, j;
	unsigned char *RAM, temp;

	/* The gfx data is swapped wrt the other Pac-Man hardware games. */
	/* Here we revert it to the usual format. */

	/* Characters */
	RAM = memory_region(REGION_GFX1);
	for (i = 0;i < memory_region_length(REGION_GFX1);i += 0x10)
	{
		for (j = 0; j < 8; j++)
		{
			temp          = RAM[i+j+0x08];
			RAM[i+j+0x08] = RAM[i+j+0x00];
			RAM[i+j+0x00] = temp;
		}
	}

	/* Sprites */
	RAM = memory_region(REGION_GFX2);
	for (i = 0;i < memory_region_length(REGION_GFX2);i += 0x20)
	{
		for (j = 0; j < 8; j++)
		{
			temp          = RAM[i+j+0x18];
			RAM[i+j+0x18] = RAM[i+j+0x10];
			RAM[i+j+0x10] = RAM[i+j+0x08];
			RAM[i+j+0x08] = RAM[i+j+0x00];
			RAM[i+j+0x00] = temp;
		}
	}
}

static void eyes_decode(unsigned char *data)
{
	int j;
	unsigned char swapbuffer[8];

	for (j = 0; j < 8; j++)
	{
		swapbuffer[j] = data[(j >> 2) + (j & 2) + ((j & 1) << 2)];
	}

	for (j = 0; j < 8; j++)
	{
		char ch = swapbuffer[j];

		data[j] = (ch & 0x80) | ((ch & 0x10) << 2) |
					 (ch & 0x20) | ((ch & 0x40) >> 2) | (ch & 0x0f);
	}
}

static DRIVER_INIT( eyes )
{
	int i;
	unsigned char *RAM;

	/* CPU ROMs */

	/* Data lines D3 and D5 swapped */
	RAM = memory_region(REGION_CPU1);
	for (i = 0; i < 0x4000; i++)
	{
		RAM[i] =  (RAM[i] & 0xc0) | ((RAM[i] & 0x08) << 2) |
				  (RAM[i] & 0x10) | ((RAM[i] & 0x20) >> 2) | (RAM[i] & 0x07);
	}


	/* Graphics ROMs */

	/* Data lines D4 and D6 and address lines A0 and A2 are swapped */
	RAM = memory_region(REGION_GFX1);
	for (i = 0;i < memory_region_length(REGION_GFX1);i += 8)
		eyes_decode(&RAM[i]);
	RAM = memory_region(REGION_GFX2);
	for (i = 0;i < memory_region_length(REGION_GFX2);i += 8)
		eyes_decode(&RAM[i]);
}


static DRIVER_INIT( pacplus )
{
	pacplus_decode();
}

static DRIVER_INIT( jumpshot )
{
	jumpshot_decode();
}

static DRIVER_INIT( 8bpm )
{
	UINT8 *RAM = memory_region(REGION_CPU1);
	int i;

	/* Data lines D0 and D6 swapped */
	for( i = 0; i < 0x8000; i++ )
	{
		RAM[i] = BITSWAP8(RAM[i],7,0,5,4,3,2,1,6);
	}
}

static DRIVER_INIT( porky )
{
	UINT8 *RAM = memory_region(REGION_CPU1);
	int i;

	/* Data lines D0 and D4 swapped */
	for(i = 0; i < 0x8000; i++)
	{
		RAM[i] = BITSWAP8(RAM[i],7,6,5,0,3,2,1,4);
	}

}


/*************************************
 *
 *	Game drivers
 *
 *************************************/

/*          rom       parent    machine   inp       init */
GAME( 1980, puckman,  0,        pacman,   pacman,   0,        ROT90,  "Namco", "PuckMan (Japan set 1)" )
GAME( 1980, puckmana, puckman,  pacman,   pacman,   0,        ROT90,  "Namco", "PuckMan (Japan set 2)" )
GAME( 1980, pacman,   puckman,  pacman,   pacman,   0,        ROT90,  "[Namco] (Midway license)", "Pac-Man (Midway)" )
GAME( 1981, puckmod,  puckman,  pacman,   pacman,   0,        ROT90,  "Namco", "PuckMan (harder?)" )
GAME( 1981, pacmod,   puckman,  pacman,   pacman,   0,        ROT90,  "[Namco] (Midway license)", "Pac-Man (Midway, harder)" )
GAME( 1981, hangly,   puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Hangly-Man (set 1)" )
GAME( 1981, hangly2,  puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Hangly-Man (set 2)" )
GAME( 1981, hangly3,  puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Hangly-Man (set 3)" )
GAME( 1980, newpuckx, puckman,  pacman,   pacman,   0,        ROT90,  "hack", "New Puck-X" )
GAME( 1981, pacheart, puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Pac-Man (Hearts)" )
GAME( 1982, joyman,   puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Joyman" )
GAME( 1980, newpuc2,  puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Newpuc2" )
GAME( 1980, newpuc2b, puckman,  pacman,   pacman,   0,        ROT90,  "hack", "Newpuc2 (set 2)" )
GAME( 1981, piranha,  puckman,  piranha,  mspacman, eyes,     ROT90,  "GL (US Billiards License)", "Piranha" )
GAME( 1981, piranhao, puckman,  piranha,  mspacman, eyes,     ROT90,  "GL (US Billiards License)", "Piranha (older)" )
GAME( 1981, piranhah, puckman,  pacman,   mspacman, 0,        ROT90,  "hack", "Piranha (hack)" )
GAME( 1981, nmouse,   0	     ,  nmouse ,  nmouse,   eyes,     ROT90,  "Amenip (Palcom Queen River)", "Naughty Mouse (set 1)" )
GAME( 1981, nmouseb,  nmouse ,  nmouse ,  nmouse,   eyes,     ROT90,  "Amenip Nova Games Ltd.", "Naughty Mouse (set 2)" )
GAME( 1982, pacplus,  0,        pacplus,  pacman,   pacplus,  ROT90,  "[Namco] (Midway license)", "Pac-Man Plus" )
GAME( 1981, mspacman, 0,        mspacman, mspacman, 0,        ROT90,  "Midway", "Ms. Pac-Man" )
GAME( 1981, mspacmab, mspacman, pacman,   mspacman, 0,        ROT90,  "bootleg", "Ms. Pac-Man (bootleg)" )
GAME( 1981, mspacmat, mspacman, mspacman, mspacman, 0,        ROT90,  "hack", "Ms. Pac Attack" )
GAME( 1981, mspacpls, mspacman, mspacpls, mspacpls, 0,        ROT90,  "hack", "Ms. Pac-Man Plus" )
GAME( 1981, pacgal,   mspacman, pacman,   mspacman, 0,        ROT90,  "hack", "Pac-Gal" )
GAME( 1995, mschamp,  mspacman, mschamp,  mschamp,  0,        ROT90,  "hack", "Ms. Pacman Champion Edition / Super Zola Pac Gal" )
GAME( 1981, crush,    0,        pacman,   maketrax, maketrax, ROT90,  "Kural Samno Electric", "Crush Roller (Kural Samno)" )
GAME( 1981, crush2,   crush,    pacman,   maketrax, 0,        ROT90,  "Kural Esco Electric", "Crush Roller (Kural Esco - bootleg?)" )
GAME( 1981, crush3,   crush,    pacman,   maketrax, eyes,     ROT90,  "Kural Electric", "Crush Roller (Kural - bootleg?)" )
GAME( 1981, maketrax, crush,    pacman,   maketrax, maketrax, ROT270, "[Kural] (Williams license)", "Make Trax (set 1)" )
GAME( 1981, maketrxb, crush,    pacman,   maketrax, maketrax, ROT270, "[Kural] (Williams license)", "Make Trax (set 2)" )
GAME( 1981, korosuke, crush,    pacman,   korosuke, korosuke, ROT90,  "Kural Electric", "Korosuke Roller" )
GAME( 1981, mbrush,   crush,    pacman,   mbrush,   0,        ROT90,  "bootleg", "Magic Brush" )
GAME( 1981, paintrlr, crush,    pacman,   paintrlr, 0,        ROT90,  "bootleg", "Paint Roller" )
GAME( 1982, ponpoko,  0,        pacman,   ponpoko,  ponpoko,  ROT0,   "Sigma Enterprises Inc.", "Ponpoko" )
GAME( 1982, ponpokov, ponpoko,  pacman,   ponpoko,  ponpoko,  ROT0,   "Sigma Enterprises Inc. (Venture Line license)", "Ponpoko (Venture Line)" )
GAME( 1982, eyes,     0,        pacman,   eyes,     eyes,     ROT90,  "Digitrex Techstar (Rock-ola license)", "Eyes (Digitrex Techstar)" )
GAME( 1982, eyes2,    eyes,     pacman,   eyes,     eyes,     ROT90,  "Techstar (Rock-ola license)", "Eyes (Techstar)" )
GAMEX(1983, mrtnt,    0,        pacman,   mrtnt,    eyes,     ROT90,  "Telko", "Mr. TNT", GAME_WRONG_COLORS )
GAME( 1983, gorkans,  mrtnt,    pacman,   mrtnt,    0,        ROT90,  "Techstar", "Gorkans" )
GAMEX(1983, eggor,    0,        pacman,   mrtnt,    eyes,     ROT90,  "Telko", "Eggor", GAME_WRONG_COLORS | GAME_IMPERFECT_SOUND  )
GAME( 1985, lizwiz,   0,        pacman,   lizwiz,   0,        ROT90,  "Techstar (Sunn license)", "Lizard Wizard" )
GAME( 1983, theglobp, suprglob, theglobp, theglobp, 0,        ROT90,  "Epos Corporation", "The Glob (Pac-Man hardware)" )
GAME( 1984, beastf,   suprglob, theglobp, theglobp, 0,        ROT90,  "Epos Corporation", "Beastie Feastie" )
GAME( 1983, bwcasino, 0,        acitya,   bwcasino, 0,        ROT90,  "Epos Corporation", "Boardwalk Casino" )
GAME( 1982, dremshpr, 0,        dremshpr, dremshpr, 0,        ROT270, "Sanritsu", "Dream Shopper" )
GAME( 1983, acitya,   bwcasino, acitya,   acitya,   0,        ROT90,  "Epos Corporation", "Atlantic City Action" )
GAME( 1983, vanvan,   0,        vanvan,   vanvan,   0,        ROT270, "Sanritsu", "Van-Van Car" )
GAME( 1983, vanvank,  vanvan,   vanvan,   vanvank,  0,        ROT270, "Karateco", "Van-Van Car (Karateco)" )
GAMEX(1982, alibaba,  0,        alibaba,  alibaba,  0,        ROT90,  "Sega", "Ali Baba and 40 Thieves", GAME_WRONG_COLORS | GAME_UNEMULATED_PROTECTION )
GAME( 1985, jumpshot, 0,        pacman,   jumpshot, jumpshot, ROT90,  "Bally Midway", "Jump Shot" )
GAME( 1985, shootbul, 0,        pacman,   shootbul, jumpshot, ROT90,  "Bally Midway", "Shoot the Bull" )
GAME( 1986, bigbucks, 0,		bigbucks, bigbucks, 0,        ROT90,  "Dynasoft Inc.", "Big Bucks" )
GAME( 1984, drivfrcp, 0,        drivfrcp, drivfrcp, 0,        ROT90,  "Shinkai Inc. (Magic Eletronics Inc. licence)", "Driving Force (Pac-Man conversion)" )
GAMEX(1985, 8bpm,	  8ballact,	8bpm,	  8bpm,		8bpm,     ROT90,  "Seatongrove Ltd (Magic Eletronics USA licence)", "Eight Ball Action (Pac-Man conversion)", GAME_WRONG_COLORS )
GAMEX(1985, porky,	  0,        porky,	  porky,	porky,    ROT90,  "Shinkai Inc. (Magic Eletronics Inc. licence)", "Porky", GAME_NO_SOUND )
