/***************************************************************************

	Taito Super Rider driver
	
	driver by Aaron Giles
	
	Games supported:
		* Super Rider

	Known issues:
		* clocks on sound chips and CPU not verified yet
		* the board seems to contain a discrete sound portion
		
	Questions:
		* there appears to be a second color bank for the tilemaps, where
		  is it used, and how is it activated (if at all)?
		* what are the writes to $08DB and $08E8 for?
		  (guess: a discrete sound effect)
  
****************************************************************************

	PCB Layout
	----------

	Top board

	REF. SR-8327A-B
	|----------------------------------------------------------------|
	|                 SR-11                                          |
	|                                                                |
	|          DIPSW  SR-10                                          |
	|                                                                |
	|                                                                |
	|                                                                |
	|                 NE555                                          |
	|                                     NEC-D780C                  |
	|  AY-3-8910                                                     |
	|                                                                |
	|  AY-3-8910                                                     |
	|                 SR-09           HM6116   SR-06    SR-03        |
	|  NEC-D780C                                                     |
	|                                 SR-08    SR-05    SR-02        |
	|                                                                |
	|                       ?.000MHz  SR-07    SR-04    SR-01        |
	|----------------------------------------------------------------|
	      

	Bottom board

	REF. SR-8327B-B
	|----------------------------------------------------------------|
	|                                                                |
	|    SR-12                                                       |
	|                                                                |
	|    SR-13                                                       |
	|                                                                |
	|    SR-14                                                       |
	|                                        SR-15                   |
	|                                                                |
	|                                        SR-16                   |
	|                                                                |
	|                                        SR-17                   |
	|                                                                |
	|                                       M58725P                  |
	|                                                                |
	|                                                                |
	|    18.432MHz                                                   |
	|----------------------------------------------------------------|
	
	
	Epoxy module (exact layout unknown)
	
	REF. ???
	|-------------------------------|
	|                               |
	|      1        2        3      |
	|                               |
	|-------------------------------|

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "suprridr.h"

static UINT8 nmi_enable;
static UINT8 sound_data;



/*************************************
 *
 *	Interrupt generation
 *
 *************************************/

static WRITE_HANDLER( nmi_enable_w )
{
	nmi_enable = data;
}


static INTERRUPT_GEN( main_nmi_gen )
{
	if (nmi_enable)
		cpu_set_irq_line(0, IRQ_LINE_NMI, PULSE_LINE);
}



/*************************************
 *
 *	Sound CPU communication
 *
 *************************************/

static void delayed_sound_w(int param)
{
	sound_data = param;
	cpu_set_irq_line(1, 0, ASSERT_LINE);
}


static WRITE_HANDLER( sound_data_w )
{
	timer_set(TIME_NOW, data, delayed_sound_w);
}


static READ_HANDLER( sound_data_r )
{
	return sound_data;
}


static WRITE_HANDLER( sound_irq_ack_w )
{
	cpu_set_irq_line(1, 0, CLEAR_LINE);
}



/*************************************
 *
 *	Misc handlers
 *
 *************************************/

static WRITE_HANDLER( coin_lock_w )
{
	/* cleared when 9 credits are hit, but never reset! */
/*	coin_lockout_global_w(~data & 1); */
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem )
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0x87ff, MRA_RAM },
	{ 0x8800, 0x8bff, MRA_RAM },
	{ 0x9000, 0x97ff, MRA_RAM },
	{ 0x9800, 0x983f, MRA_RAM },
	{ 0x9840, 0x987f, MRA_RAM },
	{ 0x9880, 0x9bff, MRA_RAM },
	{ 0xa000, 0xa000, input_port_0_r },
	{ 0xa800, 0xa800, input_port_1_r },
	{ 0xb000, 0xb000, input_port_2_r },
	{ 0xc000, 0xefff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( writemem )
	{ 0x0000, 0x7fff, MWA_ROM },
	{ 0x8000, 0x87ff, MWA_RAM },
	{ 0x8800, 0x8bff, suprridr_bgram_w, &suprridr_bgram },
	{ 0x9000, 0x97ff, suprridr_fgram_w, &suprridr_fgram },
	{ 0x9800, 0x983f, MWA_RAM },
	{ 0x9840, 0x987f, MWA_RAM, &spriteram },
	{ 0x9880, 0x9bff, MWA_RAM },
	{ 0xb000, 0xb000, nmi_enable_w },
	{ 0xb002, 0xb003, coin_lock_w },
	{ 0xb006, 0xb006, suprridr_flipx_w },
	{ 0xb007, 0xb007, suprridr_flipy_w },
	{ 0xb800, 0xb800, sound_data_w },
	{ 0xc801, 0xc801, suprridr_fgdisable_w },
	{ 0xc802, 0xc802, suprridr_fgscrolly_w },
	{ 0xc804, 0xc804, suprridr_bgscrolly_w },
	{ 0xc000, 0xefff, MWA_ROM },
MEMORY_END

static PORT_READ_START( readport )
	{ 0x00, 0x00, watchdog_reset_r },
PORT_END

/*************************************
 *
 *	Sound CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( sound_readmem )
	{ 0x0000, 0x0fff, MRA_ROM },
	{ 0x3800, 0x3bff, MRA_RAM },
MEMORY_END

static MEMORY_WRITE_START( sound_writemem )
	{ 0x0000, 0x0fff, MWA_ROM },
	{ 0x3800, 0x3bff, MWA_RAM },
MEMORY_END

static PORT_READ_START( sound_readport )
	{ 0x8d, 0x8d, AY8910_read_port_0_r },
	{ 0x8f, 0x8f, AY8910_read_port_1_r },
PORT_END

static PORT_WRITE_START( sound_writeport )
	{ 0x00, 0x00, sound_irq_ack_w },
	{ 0x8c, 0x8c, AY8910_control_port_0_w },
	{ 0x8d, 0x8d, AY8910_write_port_0_w },
	{ 0x8e, 0x8e, AY8910_control_port_1_w },
	{ 0x8f, 0x8f, AY8910_write_port_1_w },
PORT_END


/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( suprridr )
	PORT_START		/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x0c, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START		/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_SERVICE1 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START		/* DSW0 */
	PORT_DIPNAME( 0x07, 0x01, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 1C_7C ) )
	PORT_DIPNAME( 0x18, 0x00, "Timer Speed" )
	PORT_DIPSETTING(	0x18, "Slow" )
	PORT_DIPSETTING(	0x10, "Medium" )
	PORT_DIPSETTING(	0x08, "Fast" )
	PORT_DIPSETTING(	0x00, "Fastest" )
	PORT_DIPNAME( 0x20, 0x00, "Bonus" )
	PORT_DIPSETTING(	0x00, "200k" )
	PORT_DIPSETTING(	0x20, "400k" )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(	0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(	0x40, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x80, 0x00, "Invulnerability?" )
	PORT_DIPSETTING(	0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x80, DEF_STR( On ) )
INPUT_PORTS_END



/*************************************
 *
 *	Graphics layouts
 *
 *************************************/

static struct GfxLayout charlayout =
{
	8,8,
	RGN_FRAC(1,2),
	4,
	{ RGN_FRAC(1,2),RGN_FRAC(1,2)+4, 0,4 },
	{ STEP4(0,1), STEP4(8,1) },
	{ STEP8(0,16) },
	8*8*2
};


static struct GfxLayout spritelayout =
{
	16,16,
	RGN_FRAC(1,3),
	3,
	{ RGN_FRAC(0,3), RGN_FRAC(1,3), RGN_FRAC(2,3) },
	{ STEP8(0,1), STEP8(8*8,1) },
	{ STEP8(0,8), STEP8(8*8*2,8) },
	16*16
};


static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &charlayout,    0, 2 },
	{ REGION_GFX2, 0, &charlayout,   32, 2 },
	{ REGION_GFX3, 0, &spritelayout, 64, 2 },
	{ -1 }
};



/*************************************
 *
 *	Sound interfaces
 *
 *************************************/

static struct AY8910interface ay8910_interface =
{
	2,
	10000000/8,		/* just a guess */
	{ 25, 25 },
	{ 0, sound_data_r },
	{ 0 },
	{ 0 },
	{ 0 }
};



/*************************************
 *
 *	Machine driver
 *
 *************************************/

static MACHINE_DRIVER_START( suprridr )

	/* basic machine hardware */
	MDRV_CPU_ADD(Z80, 10000000/4)		/* just a guess */
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_PORTS(readport,0)
	MDRV_CPU_VBLANK_INT(main_nmi_gen,1)
	
	MDRV_CPU_ADD(Z80, 10000000/4)		/* just a guess */
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(sound_readmem,sound_writemem)
	MDRV_CPU_PORTS(sound_readport,sound_writeport)
	
	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)
	
	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(32*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 32*8-1, 2*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(96)

	MDRV_PALETTE_INIT(suprridr)
	MDRV_VIDEO_START(suprridr)
	MDRV_VIDEO_UPDATE(suprridr)
	
	/* sound hardware */
	MDRV_SOUND_ADD(AY8910, ay8910_interface)
MACHINE_DRIVER_END



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( suprridr )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "sr8",    0x0000, 0x1000, 0x4a1f0a6c )
	ROM_LOAD( "sr7",    0x1000, 0x1000, 0x523ee717 )
	ROM_LOAD( "sr4",    0x2000, 0x1000, 0x300370ae )
	ROM_LOAD( "sr5",    0x3000, 0x1000, 0xc5bca683 )
	ROM_LOAD( "sr6",    0x4000, 0x1000, 0x563bab28 )
	ROM_LOAD( "sr3",    0x5000, 0x1000, 0x4b9d2ec5 )
	ROM_LOAD( "sr2",    0x6000, 0x1000, 0x6fe18e1d )
	ROM_LOAD( "sr1",    0x7000, 0x1000, 0xf2ae64b3 )
	ROM_LOAD( "1",      0xc000, 0x1000, 0xcaf12fa2 )
	ROM_LOAD( "2",      0xd000, 0x1000, 0x2b3c638e )
	ROM_LOAD( "3",      0xe000, 0x1000, 0x2abdb5f4 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sr9",    0x0000, 0x1000, 0x1c5dba78 )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "sr10",   0x0000, 0x1000, 0xa57ac8d0 )
	ROM_LOAD( "sr11",   0x1000, 0x1000, 0xaa7ec7b2 )

	ROM_REGION( 0x2000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "sr15",   0x0000, 0x1000, 0x744f3405 )
	ROM_LOAD( "sr16",   0x1000, 0x1000, 0x3e1a876b )

	ROM_REGION( 0x3000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "sr12",   0x0000, 0x1000, 0x81494fe8 )
	ROM_LOAD( "sr13",   0x1000, 0x1000, 0x63e94648 )
	ROM_LOAD( "sr14",   0x2000, 0x1000, 0x277a70af )

	ROM_REGION( 0x0060, REGION_PROMS, ROMREGION_DISPOSE )
	ROM_LOAD( "clr.1b", 0x0000, 0x0020, 0x87a79fe8 )
	ROM_LOAD( "clr.9c", 0x0020, 0x0020, 0x10d63240 )
	ROM_LOAD( "clr.8a", 0x0040, 0x0020, 0x917eabcd )
ROM_END



/*************************************
 *
 *	Game drivers
 *
 *************************************/

GAMEX( 1983, suprridr, 0, suprridr, suprridr, 0, ROT90, "Venture Line (Taito Corporation license)", "Super Rider", GAME_IMPERFECT_SOUND )
