/*
 * Copyright 2010 Project atnd4j
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.hkzo.atnd4j;

import static org.hkzo.atnd4j.UserAgent.USER_AGENT;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

import net.arnx.jsonic.JSON;
import net.arnx.jsonic.JSONException;

/**
 * ATND4Jメインクラスです。
 * <p>
 * ATND APIに対応する以下のメソッドが用意されています。<br>
 * <ul>
 * <li>イベントサーチAPI （http://api.atnd.org/events/）<br>
 *  {@link #getEvents(String)}</li>
 * <li>出欠確認API （http://api.atnd.org/events/users/）<br>
 *  {@link #getEventsUsers(String)} </li>
 * </ul>
 * 引数の query string は format 指定を除く　ATND　の検索クエリ文字列をそのまま指定します。<br>
 * </p> 
 * @see <a href="http://api.atnd.org/">http://api.atnd.org/</a>
 */
public class Atnd {

	static final protected String API_EVENTS = "http://api.atnd.org/events/?format=json&";
	static final protected String API_EVENTS_USERS = "http://api.atnd.org/events/users/?format=json&";

	private int connectTimeout = 8000;	// 8sec
	private int readTimeout = 8000;		// 8sec
	
	/**
	 * @param api
	 * @param query
	 * @return EventsResult
	 * @throws IllegalArgumentException
	 * @throws SocketTimeoutException
	 * @throws IOException
	 */
	protected EventsResult callApi(String api, String query) throws IllegalArgumentException, SocketTimeoutException, IOException {
		Logger logger = Logger.getLogger(this.getClass().getName());
		boolean isDebug = logger.isLoggable(Level.FINE);

		if (api == null || api.equals("")) {
			throw new IllegalArgumentException("api param is null.");
		}
		if (query == null) {
			throw new IllegalArgumentException("query param is null.");
		}

		EventsResult res = null;
		HttpURLConnection con = null;
		InputStream ins = null;
		try {
			if (isDebug) {
				logger.fine("Request URL=[" + api + query + "]");
			}

			con = getURLConnection(api + query);
			con.setRequestMethod("GET");
			con.setRequestProperty("User-Agent", USER_AGENT);
			con.setConnectTimeout(connectTimeout);
			con.setReadTimeout(readTimeout);
			if (isDebug) {
				logger.fine("set ConnectTimeout=" + con.getConnectTimeout() + ", ReadTimeout=" + con.getReadTimeout());
			}

			con.connect();
			if (con.getResponseCode() == HttpURLConnection.HTTP_OK) {
				ins = con.getInputStream();
				res = JSON.decode(ins, EventsResult.class);
			} else {
				logger.warning("connect error: ResponseCode=" + con.getResponseCode() + ", URL=[" + api + query + "]");
			}
		}
		catch (JSONException e) {
			logger.warning("JSONException: ErrorCode=" + e.getErrorCode() + ", MSG=" + e.getMessage());
		}
		finally {
			try {
				if (ins != null) {
					ins.close();
				}
				if (con != null) {
					con.disconnect();
				}
			} catch (IOException e1) {
				// no action
			}
		}
		
		if (res == null) {
			// 0件オブジェクトを返却
			res = new EventsResult();
		}
		return res;
	}
	
	/**
	 * @param url
	 * @return HttpURLConnection
	 * @throws IOException
	 */
	protected HttpURLConnection getURLConnection(String url) throws IOException {
		URL u = new URL(url);
		return (HttpURLConnection)u.openConnection();
	}
	
	/**
	 * イベントサーチAPIです。<br>
	 * <br>
	 * <strong>検索クエリ文字列はメソッド内で一切加工していません。適切にエンコードしてください。<br>
	 * また、format は指定しないでください。</strong> 
	 * 
	 * @param query 検索クエリ文字列
	 * @return EventsResult
	 * @throws IllegalArgumentException
	 * @throws SocketTimeoutException
	 * @throws IOException
	 */
	public EventsResult getEvents(String query) throws IllegalArgumentException, SocketTimeoutException, IOException {
		return callApi(API_EVENTS, query);
	}

	/**
	 * 出欠確認APIです。<br>
	 * <br>
	 * イベント情報はATND APIの仕様に書かれている項目のみ使用可能であることに注意してください。<br>
	 * （例）description, owner_id などは返却されません。空文字となります。<br>
	 * <br>
	 * <strong>検索クエリ文字列はメソッド内で一切加工していません。適切にエンコードしてください。<br>
	 * また、format は指定しないでください。</strong> 
	 * 
	 * @param query 検索クエリ文字列
	 * @return EventsResult
	 * @throws IllegalArgumentException
	 * @throws SocketTimeoutException
	 * @throws IOException
	 */
	public EventsResult getEventsUsers(String query) throws IllegalArgumentException, SocketTimeoutException, IOException {
		return callApi(API_EVENTS_USERS, query);
	}

	/**
	 * @return the connectTimeout
	 */
	public int getConnectTimeout() {
		return connectTimeout;
	}

	/**
	 * @param connectTimeout the connectTimeout to set
	 */
	public void setConnectTimeout(int connectTimeout) {
		this.connectTimeout = connectTimeout;
	}

	/**
	 * @return the readTimeout
	 */
	public int getReadTimeout() {
		return readTimeout;
	}

	/**
	 * @param readTimeout the readTimeout to set
	 */
	public void setReadTimeout(int readTimeout) {
		this.readTimeout = readTimeout;
	}
}
