/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import javax.net.ssl.HttpsURLConnection;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * [%helper%] I/O Helper.
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
public final class Device {

    /**
     * HTTP URL を識別する - 'http[s]://<host>/<path>[#encoding]`
     */
    private static final String PREFIX_HTTP_URL = "http:";
    private static final String PREFIX_HTTPS_URL = "https:";

    /**
     * Don't let anyone instantiate this class.
     */
    private Device() {
        super();
    }

    /**
     * 文字コードを返す.
     *
     * @param file - 'http://<host>/<path>[#reference(encoding)]`
     *             - 'file:///<path>[#<reference(encoding)>]`
     *             - '<path>`
     *                                     REMIND Pathに`#`を含む場合は、`#encoding`は必須.
     * @param mode HTTP or FILE を識別する.
     */
    static String getCharset(final String file, char... mode) {
        String charset = // フラグメント (参照)
                (file.contains("://") && 0 <= file.indexOf('#')) ?
                        file.replaceFirst("^.*#", "") : // REMIND 最長一致
                        "";
        if (charset.isEmpty()) { // 存在しない場合
            if (0 < mode.length)
                return IoConstants.CHARSET_DEFAULT_NAME; // FILE
            return IoConstants.CHARSET_UTF8_NAME; // HTTP
        }
        return charset;
    }

    /**
     * Path を返す.
     *
     * @param file - 'file:///<path>[#<reference(encoding)>]`
     *             - '<path>`
     *             REMIND Pathに`#`を含む場合は、`#encoding`は必須.
     */
    private static String getPath(final String file) {
        return (file.contains("://")) ?
                file.replaceAll("^(file:///)|(#[-\\w]+)$",
                        "") :
                file;
    }

    /**
     * 指定されたストリームを開く.
     *
     * @param file ファイル名
     */
    public static Reader openInput(String file) throws IOException {
        if (Io.STDIN.equals(file) || "-".equals(file)) { // STDIN
            return new TextReader(System.in);

        } else if (Io.STDNUL.equals(file)) { // NULL
            return new TextReader(NullInputStream.THIS);
        }

        if (file.startsWith(PREFIX_HTTP_URL)) { // HTTP
            final URL url = new URL(file);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.connect();
            int responseCode = conn.getResponseCode();
            if (responseCode != HttpURLConnection.HTTP_OK) {
                System.err.println("HTTP responseCode: " + responseCode);
            }
            InputStream in = conn.getInputStream();
            return new InputStreamReader(in, getCharset(file));
        }

        if (file.startsWith(PREFIX_HTTPS_URL)) { // HTTPS
            final URL url = new URL(file);
            HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
            conn.connect();
            int responseCode = conn.getResponseCode();
            if (responseCode != HttpsURLConnection.HTTP_OK) {
                System.err.println("HTTPS responseCode: " + responseCode);
            }
            InputStream in = conn.getInputStream();
            return new InputStreamReader(in, getCharset(file));
        }

        final String path = getPath(file); // FILE (URLを剥がす)
        if (!new File(path).exists()) {
            // REMIND メッセージにファイル名が含まれないため、ここで表示する.
            throw new FileNotFoundException("'" + path + "'");
        }
        InputStream in = new FileInputStream(path);
        return new InputStreamReader(in, getCharset(file, 'r')); // REMIND 文字セット
    }

    /**
     * 指定されたストリームを開く.
     *
     * @param rid  リダイレクト指定子
     * @param file ファイル名
     */
    public static Writer openOutput(String rid, String file)
            throws IOException {
        if (Io.STDOUT.equals(file) || "-".equals(file)) { // STDOUT
            return new StreamWriter(System.out); // OS. Default encoding

        } else if (Io.STDERR.equals(file)) { // STDERR
            return new StreamWriter(System.err); // OS. Default encoding

        } else if (Io.STDNUL.equals(file)) { // NULL
            return new StreamWriter(NullOutputStream.THIS);
        }

        final String path = getPath(file); // FILE (URLを剥がす)
        final FileOutputStream out = openOutputStreamImpl(rid, path);
        return new StreamWriter(out, getCharset(file, 'w')); // REMIND 文字セット
    }

    /**
     * ファイル出力ストリームを開く.
     *
     * @param rid  リダイレクト指定子
     * @param file ファイル名
     */
    private static FileOutputStream openOutputStreamImpl(
            String rid, String file) throws IOException {
        final File dir = new File(file).getParentFile();
        if ((null != dir) && !dir.exists()) {
            if (!dir.mkdirs()) { // 親ディレクトリが存在しなければ作成する
                throw new IOException("mkdirs: " + dir);
            }
        }
        if (">".equals(rid)) { // > output
            return new FileOutputStream(file);
        }
        final RandomAccessFile rao = new RandomAccessFile(file, "rw"); // >> append
        try {
            rao.seek(rao.length()); // >> append
            return new FileOutputStream(rao.getFD());

        } catch (IOException e) {
            try {
                rao.close();
            } catch (IOException ex) { // エラーを無視し本来のエラーを投げる
            }
            throw e;
        }
    }
}