/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;

/**
 * Parser Helper.
 *
 * @author kunio himei.
 */
abstract class PerseHelper extends TypeHelper {
    // ------------------------------------------------------------------------
    // Helper.
    // ------------------------------------------------------------------------

    //* $0.
    private static final Node.Arr RECORD_ZERO =
            new Node.Arr("$", new Object[]{Term.NUMBER_ZERO}); // $0

    //* has to be at least one.
    private static final LexRegx
            rxENDLIST = new LexRegx("^[<|>;)}\n]");

    /**
     * クラス名オブジェクトを返す (未定義オブジェクトの場合はエラー).
     */
    static Term.BOXING className(Object e) {
        final Term.BOXING x = classNameOrNull(e);
        if (null != x) {
            return x;
        }
        throw new IllegalStateException("unmatch: want Class Name");
    }

    /**
     * クラス名に対応するクラスオブジェクトを返す.
     */
    @Nullable
    static Term.BOXING classNameOrNull(Object e) {
        if (e instanceof Term.YyValue) {
            // BOXING: { SimplePackageName '.' } SimpleClassName
            return new Term.BOXING(((Term.YyValue) e).value.toString());
        } else if (e instanceof Node.YyVariable) {
            // NAME: SimpleClassName
            return new Term.BOXING(((Node.YyVariable) e).name);
        }
        return null;
    }

    /**
     * 真値判定 (Boolean|0|1,REGEXP)オブジェクトを返す.
     */
    static Object genCompare(Object e) {
        final Object x = Type.unWrapList(e);
        final int typ = Type.getNodeType(e) & Flags.T14X512M;
        if (Flags.T02BOOL == typ) {
            return x;
        } else if (Flags.T08REGX == typ) {
            return new Node.Comp("~", RECORD_ZERO, x);
        } else {
            return new Node.B00l(Node.BOOL_OP, x);
        }
    }

    /**
     * 区切リスト expression, ... を返す.
     */
    final Object[] commalist() {
        final Collection<Object> buf = new ArrayList<>();
        if (!rxENDLIST.find(super.tok)) { // has to be at least one
            buf.add(expression());
            while (",".equals(super.tok)) {
                eat(super.tok);
                buf.add(expression());
            }
        }
        return buf.toArray();
    }

    /**
     * 式を返す.
     */
    abstract Object expression();

    /**
     * 省略可能な 括弧リスト (paren list) | comma list を返す.
     */
    final Object[] optparenlist() { //
        return "(".equals(super.tok) ? parenlist() : commalist();
    }

    /**
     * 括弧リスト(comma list) を返す.
     */
    final Object[] parenlist() {
        eat("(");
        final Object[] e = commalist();
        eat(")");
        return e;
    }
}