/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.reflect;

import org.jetbrains.annotations.Nullable;
import plus.util.NumHelper;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;

/**
 * Reflection LibVar (singleton).
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
final class RefHelper {

    /**
     * [%enclosure%] プリミティブとラッパオブジェクトの対応表.
     */
    static final Map<Class<?>, Class<?>> PRIM_TYPES = new HashMap<>();
    /**
     * empty Constructor Array.
     */
    private static final Constructor<?>[] EMPTY_CONSTRUCTOR_ARRAY = {};

    /**
     * empty Method Array.
     */
    private static final Method[] EMPTY_METHOD_ARRAY = {};

    /**
     * パラメータ一抽出メソドのリスト.
     */
    private static final Comparable[] EXTRACTS = {
            /* 数値から数値以外(文字列など)へのキャストを除外. */
            new Comparable() {
                /**
                 * Check Parameter to Argument.
                 * TRUE: exclude
                 */
                @Override
                public boolean exclude(Class<?> prm, Class<?> arg,
                                       Object val) {
                    return ((val instanceof Number) && (null == NUMBER_TYPES.get(prm)));
                }
            },
            /* 数値の暗黙型変換が不可なものを除外. */
            new Comparable() {
                /**
                 * Check Parameter to Argument.
                 * TRUE: exclude
                 */
                @Override
                public boolean exclude(Class<?> prm, Class<?> arg,
                                       Object val) {
                    boolean bc = false; // Byte, Double, Float, Integer, Long, Short,
                    // Boolean, Character
                    if ((val instanceof Number)
                            && (prm.isPrimitive() || (null != NUMBER_TYPES.get(prm)))) {
                        if (((Integer.TYPE == prm) || (Integer.class == prm))
                                && ((Long.class == arg) || (Double.class == arg) || (Float.class == arg))) {
                            bc = true;
                        } else if (((Long.TYPE == prm) || (Long.class == prm))
                                && ((Double.class == arg) || (Float.class == arg))) {
                            bc = true;
                        } else if (((Short.TYPE == prm) || (Short.class == prm))
                                && (Byte.class != arg)) {
                            bc = true;
                        }
                    }
                    return bc;
                }
            },
            /* 同一タイプでない数値を除外. */
            new Comparable() {
                /**
                 * Check Parameter to Argument.
                 * TRUE: exclude
                 */
                @Override
                public boolean exclude(Class<?> prm, Class<?> arg,
                                       Object val) {
                    // Byte, Double, Float, Integer, Long, Short
                    return (val instanceof Number) && (arg != NUMBER_TYPES.get(prm));
                }
            },
            /* プリミティブ型と同一タイプでないものを除外. */
            new Comparable() {
                /**
                 * Check Parameter to Argument.
                 * TRUE: exclude
                 */
                @Override
                public boolean exclude(Class<?> prm, Class<?> arg,
                                       Object val) {
                    return (prm.isPrimitive()) ? (arg != PRIM_TYPES.get(prm))
                            : (prm != arg); // NOPMD: == this surely will never happen
                }
            },
            /* 等しくないものを除外. */
            null};

    /**
     * デフォルトの初期容量 (13/0.75f+1).
     */
    private static final int NUMBER_CAPACITY = 19;

    /**
     * [%enclosure%] 数値タイプオブジェクト.
     */
    static final Map<Class<?>, Class<?>> NUMBER_TYPES = new HashMap<>(
            NUMBER_CAPACITY);

    static {
        // Primitive -> wrapper Class
        PRIM_TYPES.put(Boolean.TYPE, Boolean.class);
        PRIM_TYPES.put(Byte.TYPE, Byte.class);
        PRIM_TYPES.put(Character.TYPE, Character.class);
        PRIM_TYPES.put(Double.TYPE, Double.class);
        PRIM_TYPES.put(Float.TYPE, Float.class);
        PRIM_TYPES.put(Integer.TYPE, Integer.class);
        PRIM_TYPES.put(Long.TYPE, Long.class);
        PRIM_TYPES.put(Short.TYPE, Short.class);

        // Primitive -> wrapper Class
        NUMBER_TYPES.put(Byte.TYPE, Byte.class);
        NUMBER_TYPES.put(Double.TYPE, Double.class);
        NUMBER_TYPES.put(Float.TYPE, Float.class);
        NUMBER_TYPES.put(Integer.TYPE, Integer.class);
        NUMBER_TYPES.put(Long.TYPE, Long.class);
        NUMBER_TYPES.put(Short.TYPE, Short.class);
        // wrapper Class -> Primitive
        NUMBER_TYPES.put(Byte.class, Byte.TYPE);
        NUMBER_TYPES.put(Double.class, Double.TYPE);
        NUMBER_TYPES.put(Float.class, Float.TYPE);
        NUMBER_TYPES.put(Integer.class, Integer.TYPE);
        NUMBER_TYPES.put(Long.class, Long.TYPE);
        NUMBER_TYPES.put(Short.class, Short.TYPE);
        // Number Class
        NUMBER_TYPES.put(Number.class, Number.class);
        // NUMBER_TYPES.put(AtomicInteger.class, Number.class);
        // NUMBER_TYPES.put(AtomicLong.class, Number.class);
        // NUMBER_TYPES.put(BigDecimal.class, Number.class);
        // NUMBER_TYPES.put(BigInteger.class, Number.class);
    }

    /**
     * Don't let anyone instantiate this class.
     */
    private RefHelper() {
        super();
    }

    /**
     * 引数をキャスト.
     */
    static Object cast(final Class<?> prmType,
                       Class<?> argType, Object arg) {
        Object a = arg;
        if ((prmType != argType) && (null != prmType)) { // NOPMD: == this surely will never happen
            if (prmType.isPrimitive()) { // Byte, Double, Float, Integer, Long,
                // Short
                double d = NumHelper.doubleValue(arg); // Boolean, Character
                if (prmType == Double.TYPE) {
                    a = d;
                } else if (prmType == Integer.TYPE) {
                    a = (int) d;
                } else if (prmType == Boolean.TYPE) {
                    a = 0 != d;
                } else if (prmType == Long.TYPE) {
                    a = (long) d;
                } else if (prmType == Short.TYPE) {
                    a = (short) d;
                } else if (prmType == Float.TYPE) {
                    a = (float) d;
                } else if (prmType == Byte.TYPE) {
                    a = (byte) d;
                } else if (prmType == Character.TYPE) {
                    a = (char) d;
                }
            } else if (((CharSequence.class == prmType) || (String.class == prmType))
                    && !(arg instanceof String)) {
                a = arg.toString();
            }
        }
        return a;
    }

    /**
     * 引数配列をキャスト.
     */
    static Object[] cast(Class<?>[] prmTypes, Class<?>[] argTypes,
                         Object[] args) {
        int prmlen = prmTypes.length;
        int i = args.length;
        Object[] arr = new Object[i];
        while (0 <= --i) {
            Class<?> prm = (prmlen > i) ? prmTypes[i] //
                    : ((1 <= prmlen) ? prmTypes[prmlen - 1] : null);
            arr[i] = cast(prm, argTypes[i], args[i]);
        }
        return arr;
    }

    /**
     * パラメータ一覧を抽出してパラメータ一覧返す.
     */
    private static List<Class<?>[]> extract(Object[] args,
                                            Class<?>[] argTypes, List<Class<?>[]> parmTypes) {
        List<Class<?>[]> buf = parmTypes;
        for (Comparable comp : EXTRACTS) {
            List<Class<?>[]> xxx = new ArrayList<>();
            for (Class<?>[] parms : buf) {
                int prmlen = parms.length;
                boolean hasNext = true;
                for (int i = 0, len = argTypes.length; hasNext && (len > i); i++) {
                    Class<?> arg = argTypes[i];
                    Class<?> prm = (prmlen > i) ? parms[i]
                            : ((1 <= prmlen) ? parms[prmlen - 1] : null);
                    if ((prm != arg) // NOPMD: == this surely will never happen
                            && ((null == comp) || comp.exclude(prm, arg,
                            args[i]))) {
                        hasNext = false;
                    }
                }
                if (hasNext) { // 全て一致した
                    xxx.add(parms);
                }
            }
            if (1 == xxx.size()) {
                return xxx;
            }
            buf = (xxx.isEmpty()) ? buf : xxx;
        }
        return buf;
    }

    /**
     * コンストラクタを返す.
     */
    @Nullable
    static Constructor<?> getConstructor(Class<?> clazz,
                                         Object[] args, Class<?>[] argTypes) {
        Constructor<?>[] x = getConstructors(clazz, args.length);
        if (2 <= x.length) { // 1件に絞り込めない
            Map<Class<?>[], Constructor<?>> map = new HashMap<>();
            List<Class<?>[]> buf = new ArrayList<>();
            for (Constructor<?> m : x) {
                Class<?>[] prmTypes = m.getParameterTypes();
                if (isAssignable(args, argTypes, prmTypes)) {
                    map.put(prmTypes, m); // 引数型定義との一致を確認して絞込み
                    buf.add(prmTypes);
                }
            }
            if (1 < buf.size()) {
                List<Class<?>[]> tmp = new ArrayList<>();
                for (Class<?>[] prmTypes : buf) {
                    if (isInstance(args, prmTypes)) {
                        tmp.add(prmTypes); // 代入互換関係にあるかどうかで絞込み
                    }
                }
                if (!tmp.isEmpty()) {
                    buf = tmp;
                }
                if (1 < buf.size()) {
                    buf = extract(args, argTypes, buf);
                }
            }
            if (1 != buf.size()) { // 絞り込めない！
                info(mkString("WARNING: newInstance: " + clazz, argTypes));
                info(mkString(buf.toArray()));
                // REMIND ここで0件になる場合は、引数キャスト(castArgs)で対応すること.
            }
            return buf.isEmpty() ? x[0] : map.get(buf.get(0)); // とりあえず先頭の要素を返す
        }
        return (1 == x.length) ? x[0] : null;
    }

    /**
     * 引数の数が一致するコンストラクタを抽出してコンストラクタ一覧を返す.
     */
    private static Constructor<?>[] getConstructors(Class<?> clazz,
                                                    int argslen) {
        String name = clazz.getName();
        String key = argslen + "," + name;
        Constructor<?>[] x = Cache.CONSTRUCTOR.get(key);
        if (null == x) {
            List<Constructor<?>> buf = new ArrayList<>();
            for (Constructor<?> m : clazz.getConstructors()) {
                Class<?>[] prm = m.getParameterTypes();
                int prmlen = prm.length;
                if ((prmlen == argslen)
                        || (m.isVarArgs() && ((prmlen - 1) <= argslen))) {
                    buf.add(m); // 名前と引数の数が一致するものを抽出
                    if (0 == prmlen) {
                        break;
                    }
                }
            }
            if (!buf.isEmpty()) {
                x = buf.toArray(new Constructor<?>[0]);
                Cache.CONSTRUCTOR.put(key, x);
            }
        }
        return (null == x) ? EMPTY_CONSTRUCTOR_ARRAY : x;
    }

    /**
     * フィールドを返す.
     */
    @Nullable
    static Field getField(Class<?> clazz, String name) {
        String key = getFieldKey(clazz, name);
        Field x = Cache.FIELD.get(key);
        if (null == x) { // このオブジェクトによって宣言されたすべてのフィールド
            // 1) このオブジェクトによって宣言されたすべてのフィールド(継承は含まない)
            // 2) このオブジェクトによって宣言されたpublicフィールド(継承を含む)
            for (Field[] g : new Field[][]{clazz.getDeclaredFields(),
                    clazz.getFields()}) {
                for (Field m : g) {
                    if (m.getName().equals(name)) { // 名前が一致するフィールドを抽出
                        Cache.FIELD.put(key, m);
                        return m;
                    }
                }
            }
        }
        return x;
    }

    /**
     * フィールドキーを返す.
     */
    static String getFieldKey(Class<?> clazz, String name) {
        return clazz.getName() + ',' + name;
    }

    /**
     * メソドを返す.
     */
    @Nullable
    static Method getMethod(Class<?> clazz, String name,
                            Object[] args, Class<?>[] argTypes) {
        Method[] x = getMethods(clazz, name, args.length); // 名前と引数の数が一致
        if (2 <= x.length) {
            Map<Class<?>[], Method> map = new HashMap<>();
            List<Class<?>[]> buf = new ArrayList<>();
            for (Method m : x) {
                Class<?>[] prm = m.getParameterTypes();
                if (isAssignable(args, argTypes, prm)) {
                    map.put(prm, m); // 引数型定義との一致を確認して絞込み
                    buf.add(prm);
                }
            }
            if (1 < buf.size()) {
                List<Class<?>[]> tmp = new ArrayList<>();
                for (Class<?>[] prmTypes : buf) {
                    if (isInstance(args, prmTypes)) {
                        tmp.add(prmTypes); // 代入互換関係にあるかどうかで絞込み
                    }
                }
                if (!tmp.isEmpty()) {
                    buf = tmp;
                }
                if (1 < buf.size()) {
                    buf = extract(args, argTypes, buf);
                }
            }
            if (1 != buf.size()) { // 絞り込めない (要仕様検討!)
                info(mkString("WARNING: Invoke: " + clazz + '.' + name,
                        argTypes));
                info(mkString(buf.toArray()));
            } // REMIND ここで0件になる場合は、引数をキャスト(castArgs)で対応すること.
            return buf.isEmpty() ? x[0] : map.get(buf.get(0)); // とりあえず先頭の要素を返す.
        }
        return (1 == x.length) ? x[0] : null;
    }

    /**
     * 名前と引数の数が一致するメソドを抽出してメソド一覧を返す.
     */
    static Method[] getMethods(Class<?> clazz, String name,
                               int argslen) {
        String key = argslen + "," + clazz.getName() + '#' + name;
        Method[] x = Cache.METHOD.get(key);
        if (null == x) {
            Map<String, Method> map = new HashMap<>();
            // 1) このオブジェクトによって宣言されたすべてのメソッド(継承メソッドは含まない)
            // 2) このオブジェクトによって宣言されたpublicメソッド(継承メソッドを含む)
            loop:
            for (Method[] g : new Method[][]{
                    clazz.getDeclaredMethods(), clazz.getMethods()}) {
                for (Method m : g) {
                    if (m.getName().equals(name)) { // 名前が一致するメソッドを抽出
                        Class<?>[] prm = m.getParameterTypes();
                        int prmlen = prm.length;
                        boolean hasVarArgs = ((prmlen - 1) <= argslen)
                                && m.isVarArgs(); // 可変長引数
                        if ((prmlen == argslen) || hasVarArgs) {
                            // 名前と引数の数が一致するメソドを
                            // 親クラスの定義を自クラスの定義で上書きして抽出
                            String ps = Arrays.toString(m
                                    .getParameterTypes());
                            map.put(name + '`' + ps, m);
                            if (0 == prmlen) {
                                break loop;
                            }
                        }
                    }
                }
            }
            List<Method> buf = new ArrayList<>(map.values());
            if (!buf.isEmpty()) {
                x = buf.toArray(new Method[0]);
                Cache.METHOD.put(key, x);
            }
        }
        return (null == x) ? EMPTY_METHOD_ARRAY : x;
    }

    /**
     *
     */
    static void info(String msg) {
        System.err.println("INFO: " + msg);
    }

    /**
     * この Class が表すオブジェクトと引数の型が等しいあるいはスーパークラスか?.
     */
    private static boolean isAssignable(Object[] args,
                                        Class<?>[] argTypes, Class<?>[] prmTypes) {
        int prmlen = prmTypes.length;
        for (int i = 0, len = args.length; len > i; i++) {
            Class<?> arg = argTypes[i];
            Class<?> prm = (prmlen > i) ? prmTypes[i] : ((1 <= prmlen)
                    ? prmTypes[prmlen - 1] : null);
            // 引数の型が、パラメータと等しいあるいはスーパークラスまたはスーパーインタフェースであるかどうか
            if ((null == prm) || (prm.isPrimitive() && ((null == args[i]) //
                    || ((prm != arg) // NOPMD: == this surely will never happen
                    && !prm.isAssignableFrom(arg) && !(args[i] instanceof Number))))) {
                // byte, short, int, long, float, double, boolean, char
                return false;
            }
        }
        return true; // 全て一致した
    }

    /**
     * この Class が表すオブジェクトと代入互換の関係にあるかどうか?.
     */
    private static boolean isInstance(Object[] args,
                                      Class<?>[] prmTypes) {
        int prmlen = prmTypes.length;
        for (int i = 0, len = args.length; len > i; i++) {
            Class<?> prm = (prmlen > i) ? prmTypes[i] : ((1 <= prmlen)
                    ? prmTypes[prmlen - 1] : null);
            if ((null == prm) || !prm.isInstance(args[i])) {
                return false;
            }
        }
        return true; // 全て一致した
    }

    /**
     * Displays all elements of the Array.
     */
    private static String mkString(Object[] args) {
        return mkString("\t", "\n\t", "", args);
    }

    /**
     * Displays all elements of the Array in a string using start, end, and
     * separator strings.
     */
    static String mkString(String prefix, Object[] args) {
        return mkString(prefix + '[', ", ", "]", args);
    }

    /**
     * Displays all elements of the Array in a string using start, end, and
     * separator strings.
     */
    private static String mkString(String start, String sep,
                                   String end, Object[] args) {
        StringBuilder sb = new StringBuilder(start);
        boolean is1st = true;
        for (Object x : args) {
            if (is1st) {
                is1st = false;
            } else {
                sb.append(sep);
            }
            if (x instanceof Object[]) {
                sb.append(Arrays.toString((Object[]) x));
            } else {
                sb.append(x);
            }
        }
        sb.append(end);
        return sb.toString();
    }

    /**
     * Comparable.
     */
    private interface Comparable {
        /**
         * Check Parameter to Argument.
         * TRUE: exclude
         */
        boolean exclude(Class<?> prm, Class<?> arg, Object val);
    }
}