/**
 * Copyright (C) 2023 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
use crate::Path;
use std::fs::File;
use std::io::BufReader;
use std::io::{BufWriter, Read, Write};
// use std::io::prelude::*;
use std::sync::mpsc;
use std::thread;

use crate::iomod::get_meta_len;
use crate::iomod::path_to_unix;

// pub const BUFNUM: u32 = 1024 * 2; // overflowed its stack
const BUFNUM: u32 = 1024;
const BUFSIZE: usize = (1024 * BUFNUM) as usize;
/**
 * copy maxbuf from to -> length
 */
pub fn copymax<P: AsRef<Path>>(from: P, to: P) -> u64 {
    let f: &Path = from.as_ref();
    let t: &Path = to.as_ref();
    let mut fr = File::open(f).expect(&path_to_unix(f));
    let mut fw = File::create(t).expect(&path_to_unix(t));
    let mut buf: [u8; BUFSIZE] = [0_u8; BUFSIZE];
    let mut length: usize = 0;
    loop {
        let readsize: usize = fr.read(&mut buf).unwrap();
        if readsize == 0 {
            break;
        }
        length += fw.write(&buf[..readsize]).unwrap();
    }
    let _ = fw.flush();
    let fromsize: usize = get_meta_len(f) as usize;
    assert_eq!(fromsize, length);
    length as u64
}

const CHBUFNUM: u32 = 7; // overflowed its stack
const CHBUFSIZE: usize = (1024 * CHBUFNUM) as usize;
const IOBUFSIZE: usize = 1024 * 1024;
/**
 * copy channel from | to -> length
 */
pub fn copych<P: AsRef<Path>>(from: P, to: P) -> u64 {
    let f: &Path = from.as_ref();
    let t: &Path = to.as_ref();
    let fr = File::open(f).expect(&path_to_unix(f));
    let fw = File::create(t).expect(&path_to_unix(t));
    let mut reader = BufReader::with_capacity(IOBUFSIZE, fr);
    let mut writer = BufWriter::with_capacity(IOBUFSIZE, fw);
    let (tx, rx) = mpsc::channel();
    thread::spawn(move || {
        let mut ch = CH {
            buf: [0_u8; CHBUFSIZE], // stack に確保される
            length: 0,
        };
        loop {
            // let readsize: usize = fr.read(&mut ch.buf).unwrap();
            let readsize: usize = reader.read(&mut ch.buf).unwrap();
            if readsize == 0 {
                break;
            }
            ch.length = readsize;
            // tx.send(ch.clone()).unwrap();
            tx.send(ch).unwrap();
        }
    });
    let mut length: usize = 0;
    for received in rx {
        let receivesize: usize = received.length;
        let slice: &[u8] = received._get_buf();
        // let rcv: &[u8] = &received.buf[..receivesize];
        writer.write(&slice).unwrap();
        // fw.write(&slice).unwrap();
        length += receivesize;
    }
    let _ = writer.flush();
    let fromsize: usize = get_meta_len(f) as usize;
    assert_eq!(fromsize, length, "(original:result) ");
    length as u64
}

#[derive(Debug, Clone, Copy)] // channel buffer
struct CH {
    buf: [u8; CHBUFSIZE],
    length: usize,
}
impl CH {
    pub fn _get_buf(&self) -> &[u8] {
        &self.buf[..self.length]
    }
}

// pub const CHBUFNUM: u32 = 1;
// const CHBUFSIZE: usize = (1024 * CHBUFNUM) as usize;
/*
 * copy channel from | to -> length
 */
/*
pub fn _copycherr<P: AsRef<Path>>(from: P, to: P) -> u64 {
    let f: &Path = from.as_ref();
    let t: &Path = to.as_ref();
    let mut fr = File::open(f).expect(&path_to_unix(f));
    let mut fw = File::create(t).expect(&path_to_unix(t));
    let (tx, rx) = mpsc::channel();
    thread::spawn(move || loop {
        let mut buf: [u8; CHBUFSIZE] = [0_u8; CHBUFSIZE];
        let readsize: usize = fr.read(&mut buf).unwrap();
        if readsize == 0 {
            break;
        }
        let slice: &[u8] = &buf[..readsize]; // ※
        tx.send(slice).unwrap();
    });
    let mut length: usize = 0;
    for received in rx {
        fw.write(&received).unwrap();
        length += received.len();
    }
    length as u64
}
 */
