/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import java.util.regex.Pattern;

/**
 * cutSpace - Parser (構文解析器)
 *
 * @author kunio himei.
 */
class Parser {

    static Template TEMPLATE = new Template();
    private static Space SPACE = new Space();

    // Tag to start a new line
    private static final Pattern SECTION_TAG = Pattern.compile(
            "</?(html|head|script|style|body|main|header|footer)",
            Pattern.CASE_INSENSITIVE);

    private static final Pattern HAS_BR_TAG = Pattern.compile(
            "<br[^>]*?>[\\s]*", Pattern.CASE_INSENSITIVE);

    static final StringBuilder out = new StringBuilder(1024 * 16); //dev/stdout
    static final StringBuilder err = new StringBuilder(1024 * 16); //dev/stderr
    static String expand = ""; // -e(expand) [Script, Css, Html] - Execution options
    static boolean deprecated; // -x(deprecated)
    static int debug; // -d[debug level] -1, 0...9
    private static boolean hasPRE; // <pre>

    static void execute(String input) {
        boolean d1 = 1 <= debug;
        if (0 != debug)
            err.append(Main.lineNumber).append("\t[").append(input).append("]\n");

        T2TPL t2 = Modules.joinInputLine(input);
        if (null == t2) return;
        String s = t2.source();
        String template = t2.template();

        if (d1 && 0 <= template.indexOf(Template.DEL)) // Print Template
            err.append(" T\t[").append(template).append("]\n");

        if (hasPRE) {
            if (template.contains("</pre>")) // </pre>終了. (</pre> end)
                s = Space.cutSpace(s);
            else if (!HAS_BR_TAG.matcher(s).find()) // Don't have <br>
                s = Tools.rTrim(s) + "<br>";
        } else {
            s = SPACE.space(s, deprecated); // "<pre"開始. ("<pre" started)
        }
        if (!s.isEmpty()) { // Output processing
            if (0 == debug) {
                Modules.appendLine(err, s, ' ');
                Modules.appendNewLine(err);
            } else { // debug
                err.append(' ').append(Lang.top()).append(' ');
                Modules.appendLine(err, s, Template.RS).append('\n');
            }
            if (!Tools.endsWith(out, '\n') &&
                    SECTION_TAG.matcher(template).find()) { // Output to new line
                Modules.appendNewLine(out);
            }
            Modules.appendLine(out, s, ' ');
            if (Lang.isExpand(expand) || Template.HTML_LANGUAGE_END_TAG.matcher(template).find())
                Modules.appendNewLine(out);
        }
        if (template.contains("</script>")) endLanguage();
        if (template.contains("</style>")) endLanguage();
        if (template.contains("<pre")) hasPRE = true;
        if (template.contains("</pre>")) hasPRE = false;
        if (t2.hasMark()) Lang.reset(); // ※ Lang.mark->reset
    }

    // End of language.
    private static void endLanguage() {
        TEMPLATE = new Template();
        SPACE = new Space();
        hasPRE = false;
    }
}
//  Parser.err.append("[").append(x).append("]").append('\n'); // TODO