/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;

import java.util.regex.Pattern;

import static java.util.regex.Pattern.CASE_INSENSITIVE;

/**
 * cutSpace - String, Comment parser.
 *
 * @author kunio himei.
 */
class Template {

    private static final Pattern HTML_LANGUAGE_TAG = Pattern.compile(
            "</?(script|style)[^>]*?>", CASE_INSENSITIVE);
    static final Pattern HTML_LANGUAGE_END_TAG = Pattern.compile(
            "</(script|style)>", Pattern.CASE_INSENSITIVE);

    // REMIND <LANG..[>]$ で番兵を利用できないためこの仕様になっています.
    //  This is the specification because the sentinel cannot be used with [>].
    private static final Pattern LANG_INTEGRATE_TAG = Pattern.compile( // Language tag
            "</?(?:script|style)[^>]*?$", Pattern.CASE_INSENSITIVE);
    private static final Pattern LANG_SCRIPT_START = Pattern.compile(
            "<script[^>]*?$", Pattern.CASE_INSENSITIVE);
    private static final Pattern LANG_CSS_START = Pattern.compile(
            "<style[^>]*?$", Pattern.CASE_INSENSITIVE);

    private static final Pattern REGEX_ESCAPE = Pattern.compile("[\\\\]/");
    private static final Pattern REGEX_JUDGEMENT = Pattern.compile(
            "^((?:[\\s]*/[*].*?[*]/)*" + "[\\s]*/)[^*].*?/"); // [/*Comment*/] /regex/

    // REMIND /REGEX'/'は、割り算と競合するため別途処理する.
    //  /REGEX'/' is processed separately because it conflicts with division.
    private static final String QUOTES_SCRIPT = "\"'`"; // Quotation marks
    private static final String QUOTES_HTML = "\"'";

    private static final String CMT_SCRIPT_START = "/*"; // Comment
    private static final String CMT_SCRIPT_END = "*/";
    private static final String CMT_HTML_START = "<!--";
    private static final String CMT_HTML_END = "-->";
    private String commentStart; // Start commenting
    private String commentEnd; // End of comment
    private String quotes; // Quotation marks in the target language

    static final char NUL = '\0'; // Empty string
    static final char DEL = '\u007F'; // Deleted character (※ Marked character)
    static final char RS = '\u001E'; // Record separator (line break '\n')
    @SuppressWarnings("unused")
    static final char FS = '\u001C'; // Field separator (Modules#joinOutputLine)
    @SuppressWarnings("unused")
    private static final char FF = '\f'; // '\u000C': Form Feed (regex blank)
    @SuppressWarnings("unused")
    private static final char VT = '\u000B'; // Vertical tab (regex blank)
    @SuppressWarnings("unused")
    private static final char DLE = '\u0010'; //
    @SuppressWarnings("unused")
    private static final String ESC = "\u001B"; // '\/' Escape string (regex)
    private static final String DEL01 = "\u007F";
    private static final String DEL02 = "\u007F\u007F";
    private static final String NL = "\\n"; // New line string (Template literals)

    private final StringBuilder sb = new StringBuilder(1024); // Source
    private final StringBuilder sx = new StringBuilder(1024); // Template

    // CSS and Script are always ON, html is ON in the tag <>.
    private boolean inAction; // html analysis
    private boolean isSCRIPT;
    // private boolean isCSS;
    private boolean isHTML;
    private boolean hasComment;
    private boolean hasBackQuote; // `Template literal`
    private boolean hasQuote;
    private char quote;
    private boolean hasBracket;
    private int bracket;

    // Language information settings
    private void changeLanguage(Lang lang) {
        Lang top = Lang.top();
        Lang.push(lang);
        isSCRIPT = Lang.is(Lang.SCRIPT);
        isHTML = !(isSCRIPT || Lang.is(Lang.CSS));
        if (isHTML) {
            quotes = QUOTES_HTML;
            commentStart = CMT_HTML_START;
            commentEnd = CMT_HTML_END;
            if (top != lang) inAction = false;  // Reset due to language change
        } else {
            quotes = QUOTES_SCRIPT;
            commentStart = CMT_SCRIPT_START;
            commentEnd = CMT_SCRIPT_END;
            inAction = true; // Always ON
            if (top != lang) {
                hasBracket = false;
                bracket = 0;
            }
        }
    }

    /**
     * Mark string constants.
     * strings.
     * - "string" 'string' /regex/
     * - `backQuote` // → 'Template literal'
     * Remove comment.
     * - CSS, Script: ホワイトスペースが置ける場所ならばどこにでも書ける。(空白' 'で置き換える)
     * - html: タグ以外で場所を問わずどこにでも書ける。(空文字''で置き換える)
     * Append semicolon. (Script)
     * 注意! コメント及び html のタグ外は、不完全な文字定数が書ける。
     * caution! Incomplete character constants can be written outside
     *
     * @param input text.
     * @return Source text, Marked Template (lower case text)
     */
    @NotNull
    T2TPL parse(@NotNull String input) {
        int langEndTagPosition = -1; // Position of </(script|style)>
        changeLanguage(Lang.top());
        for (int i = 0; i < input.length(); i++) {
            char t, c = t = input.charAt(i);

            if (!hasQuote && !hasComment) { // Idling process

                // NOTE When the language changes.
                //  1行に言語終了タグが現れた場合の対策. e.g. foo</script>
                //  What to do if the language end tag appears on one line.
                //  → {interpretation(HTML) foo</script>}
                //  → {interpretation(SCRIPT) foo[\n]}{(HTML) </script>} ※ Split line.
                if ('>' == c &&
                        Tools.match(LANG_INTEGRATE_TAG, sx, 0)) {
                    if (LANG_SCRIPT_START.matcher(sx).find()) changeLanguage(Lang.SCRIPT);
                    else if (LANG_CSS_START.matcher(sx).find()) changeLanguage(Lang.CSS);
                    else {
                        String w = sx.substring(0, Tools.RSTART).trim();
                        if (!w.isEmpty() && 0 > langEndTagPosition) // foo</script>
                            langEndTagPosition = Tools.RSTART; // Adopt the first match
                        changeLanguage(Lang.HTML);
                    }

                } else if (i == input.indexOf(commentStart, i)) { // ※ Start commenting
                    hasComment = true;
                    i += commentStart.length() - 1;
                    continue;
                }

                if (isSCRIPT) {
                    if (i == input.indexOf("//", i)) // ※ Line comment
                        break;

                    if ('{' == c) {
                        hasBracket = true;
                        bracket += 1;
                    } else if ('}' == c) {
                        if (0 > --bracket) {
                            Parser.err.append(Main.lineNumber)
                                    .append(" ERROR: {}} are not paired.")
                                    .append(" ('}' Has been removed)\n");
                            continue;
                        }
                    }

                    // Start of REGEX: var=/regex/, match(/regex/), foo(bar,/regex/)
                    if (0 <= "=(,".indexOf(c) &&
                            i < input.indexOf('/', i)) { // Including /
                        String w = Tools.replaceAll( // Escape \/ (2 letters)
                                REGEX_ESCAPE, input.substring(i + 1), DEL02);
                        if (Tools.match(REGEX_JUDGEMENT, w, 1)) {
                            sb.append(c);
                            sx.append(c);
                            i += Tools.RLENGTH; // Because 'i' points to target -1
                            hasQuote = true;
                            quote = '/';
                            sb.append('/');
                            sx.append(DEL);
                            continue;
                        }
                    }
                } else if (isHTML) {
                    // CSS, Script は常に ON、html は <tag> 内で ON。
                    // CSS and Script are always ON, html is ON in the <tag>.
                    if ('<' == c) inAction = true;
                    else if ('>' == c) inAction = false;
                }
            }
            if (hasComment) {
                if (i == input.indexOf(commentEnd, i)) { // ※ End of comment
                    if (!isHTML) { // Other than HTML
                        Tools.rTrim(sb, sx);
                        sb.append(' ');
                        sx.append(' ');
                    }
                    hasComment = false;
                    i += commentEnd.length() - 1;
                }
                continue;
            }

            if (inAction && 0 <= quotes.indexOf(c)) { // Start and end of "string"
                if (!hasQuote) { // Start of string constant
                    quote = c;
                    if ('`' == c) {
                        c = '"'; // Change ` to "
                        hasBackQuote = true;
                    }
                    hasQuote = true;

                } else if (quote == c) { // End of string constant
                    hasQuote = false;
                    if ('`' == c) {
                        c = '"'; // Change ` to "
                        hasBackQuote = false;
                    }

                } else if ('"' == c && hasBackQuote) { // Escape in hasBackQuote
                    sb.append('\\'); // ` " ` ->  " \" "
                    sx.append(DEL);
                }
                t = DEL;

            } else if (hasQuote) { // String processing
                t = DEL;
                if ('/' == c && '/' == quote) { // end of REGEX
                    hasQuote = false;

                } else if ('\\' == c) { // Escape character
                    // '\\nl' は、行結合で処理済みのため Indexエラーは、発生しない.
                    // Index error does not occur because　'\\nl'　has been processed by row join.
                    c = Tools.charAt(input, ++i);
//                    c = Tools.charAt("", ++i); // REMIND (TEST)
                    if ('`' != c && !hasBackQuote) {
                        sb.append('\\');
                        sx.append(DEL);
                    } // Remove the escape in hasBackQuote \` -> `
/*
                } else if ('"' == c && hasBackQuote) { // Escape in hasBackQuote
                    sb.append('\\');
                    sx.append(DEL);
*/
                }
            }
            sb.append(c);
            sx.append(t);
        }

        Tools.rTrim(sb, sx); // Required to delete whitespace.
        T2TPL t2 = new T2TPL(sb.toString(), sx.toString(),
                null, null, 0);
        String sep1 = "", sep2 = "";
        int request = 0;
        int newLength = 0;

        if (hasBackQuote) { // Join Script, `Template literal..
            sep1 = NL; // \\n (2 letters)
            sep2 = DEL02;
            request = T2TPL.join;

        } else if (isHTML && (hasQuote || inAction)) { // Join html, ".. or <..
            sep1 = " ";
            sep2 = hasQuote ? DEL01 : " ";
            request = T2TPL.join;

        } else if (0 < langEndTagPosition) { // Language changes </script>, Split
            t2 = split(langEndTagPosition);
            newLength = sb.length();
            Lang.mark(); // ※ Go back the language to the past.
            request = T2TPL.mark;  // ※ Reset request (Lang.mark->reset)

        } else if (hasBracket) { // {}
            if (0 >= bracket ||
                    HTML_LANGUAGE_END_TAG.matcher(sx).find()) { // Bracket completed
                bracket = 0;
                hasBracket = false;
//              request = T2TPL.none;
            } else {
                request = T2TPL.join; // } Wait
            }
            t2 = semicolon(t2);

        } else if (!isHTML) {
            // The input may be an empty string.
            char c = sx.isEmpty() ? NUL : Tools.charAtLast(sx); // REMIND ${NUL}
//            char c = Tools.charAtLast(sx); // REMIND (TEST)

            if (isSCRIPT) {
                if (c == NUL || 0 <= JOIN_SCRIPT_LINES.indexOf(c)) {
                    request = T2TPL.join; // Join if ${NUL} or '=,..'

                } else {
                    t2 = semicolon(t2);
//                  request = T2TPL.none;
                }
            } else if (c == NUL || 0 > EXCLUDE_JOIN_CSS_LINES.indexOf(c)) {
                request = T2TPL.join; // Join if ${NUL} or other than '}>'
            }
        }
        sb.setLength(newLength);
        sx.setLength(newLength);
        return new T2TPL(t2.source(), t2.template().toLowerCase(),
                sep1, sep2, request);
    }

    // Row join: Termination code
    // REMIND テンプレートバッファ(sx)を参照しているため/regex/と割り算とのバッティングは無い.
    //  There is no batting between /regex/ and division because
    //  it refers to the template buffer. (sx)
    static final String SCRIPT_OPERATOR = "=.,:?!|&^~+-*/%"; // Ref. Space#SCRIPT_OPT
    private static final String JOIN_SCRIPT_LINES = SCRIPT_OPERATOR + "([{"; // 結合する
    private static final String EXCLUDE_JOIN_CSS_LINES = "}>"; // 結合しない

    /*
     * Append semicolon. (Script)
     */
    private static final String EXCLUDE_SEMICOLON = ";{}>"; // セミコロンを付加しない

    @NotNull
    private T2TPL semicolon(T2TPL t2) {
        if (!sx.isEmpty()) {
            char c = Tools.charAtLast(sx); // The input is not empty
            if (0 > EXCLUDE_SEMICOLON.indexOf(c) && // Exclude a semicolon
                    !HTML_LANGUAGE_TAG.matcher(sx).find()) { // It should appear alone
                sb.append(';');
                sx.append(';');
                return new T2TPL(sb.toString(), sx.toString(),
                        null, null, T2TPL.none);
            }
        }
        return t2;
    }

    /**
     * Split line.
     *
     * @param position Split position. 1.. length()
     */
    @NotNull
    private T2TPL split(int position) {
        String tpl = Tools.rTrim(sx.substring(0, position)); // This data
        String src = sb.substring(0, tpl.length());
        sb.delete(0, position); // Next data
        sx.delete(0, position);
        return new T2TPL(src, tpl, null, null, T2TPL.none);
    }
}

/**
 * Tuple return value - タプル戻り値.
 *
 * @param source   sb: text
 * @param template sx: text
 * @param sep1     Source separator of concatenation.
 * @param sep2     Template Separator of concatenation.
 * @param request  Has a request (Lang.mark->reset) Ref. Parser#execute.
 */
record T2TPL(String source, String template,
             String sep1, String sep2, int request) {

    boolean hasJoin() {
        return join == request;
    }

    boolean hasMark() {
        return mark == request;
    }

    // request
    static final int none = 1;
    static final int join = 2; // join request Ref. Modules#joinInputLine.
    static final int mark = 3; // reset request (Lang.mark->reset) Ref. Parser#execute.
}