/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Writer;
import java.nio.charset.Charset;

/**
 * Writer -> OutputStream. - 2022.4
 * <p>
 * 標準 out/err のエンコードが切り替わらないため作成.
 * Created because the standard out / err encoding does not switch.
 *
 * @author kunio himei.
 */
public class StreamWriter extends Writer {

    private final String charsetName;
    private final OutputStream out;

    @SuppressWarnings("SameParameterValue")
    StreamWriter(OutputStream out) { // NullOutputStream
        this(out, Charset.defaultCharset().name()); // NOTE Java18 UTF-8
    }

    StreamWriter(OutputStream out, String charsetName) { // FileOutputStream
        this.charsetName = charsetName;
        this.out = out;
    }

    StreamWriter(PrintStream out) { // PrintStream (STDOUT, STDERR)
        this.charsetName = null; // REMIND Leave it to PrintStream.
        this.out = out;
    }

    /**
     * ※abstract - Closes the stream, flushing it first - APP.で使用 (Used in APP)
     */
    @Override
    public void close() throws IOException {
        out.close();
    }

    /**
     * ※abstract - Flushes the stream - APP.で使用 (Used in APP)
     */
    @Override
    public void flush() throws IOException {
        out.flush();
    }

    /**
     * ※abstract - Writes a portion of an array of characters.
     */
    @Override
    public void write(@NotNull char[] buf, int off, int len) throws IOException {
        this.write(String.valueOf(buf, off, len)); // --> str
    }

    /**
     * Writes a string - APP.で使用 (Used in APP)
     */
    @Override
    public void write(@NotNull String str) throws IOException {
        if (out instanceof PrintStream ps) ps.print(str);
        else out.write(str.getBytes(charsetName));
    }

    // REMIND unused, Write a part of the string.
    @Override
    public void write(@NotNull String str, int off, int len) throws IOException {
        this.write(str.substring(off, off + len)); // --> str
    }

    // REMIND unused, Write an array of characters.
    @Override
    public void write(@NotNull char[] buf) throws IOException {
        this.write(new String(buf)); // --> str
    }

    // REMIND unused, Write a single character.
    @Override
    public void write(int c) throws IOException {
        this.write(String.valueOf((char) c)); // --> str
    }
}