/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.misc;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.LinkedList;
import java.util.List;

/**
 * 複数のファイルを順次開くReaderです。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class FilesReader extends Reader {

	private Reader reader = null;
	private List<String> files;
	private boolean resource = false;

	//
	FilesReader(boolean resource, List<String> files) {
		this.files    = new LinkedList<String>(files);
		this.resource = resource;
	}

	/**
	 * 物理ファイルからファイルを開きます。
	 * 
	 * @param files 開くファイルのファイル名を格納するリスト
	 * @return インスタンス
	 */
	public static FilesReader newFiles(List<String> files) {
		return new FilesReader(true, files);
	}

	/**
	 * リソースファイルからファイルを開きます。
	 * 
	 * @param files 開くファイルのファイル名を格納するリスト
	 * @return インスタンス
	 */
	public static FilesReader newResorces(List<String> files) {
		return new FilesReader(false, files);
	}

	boolean setreader() throws IOException {
		if(reader != null) {
			return true;
		} else if(files == null) {
			return false;
		} else if(files.size() == 0) {
			files = null;
			return false;
		} else if(resource) {
			reader = new InputStreamReader(
					new FileInputStream(files.remove(0)));
		} else {
			reader = new InputStreamReader(
					FilesReader.class.getResourceAsStream(
							files.remove(0)));
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#close()
	 */
	@Override
	public void close() throws IOException {
		if(reader != null)  reader.close();
		reader = null;
		files  = null;
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#mark(int)
	 */
	@Override
	public void mark(int readAheadLimit) throws IOException {
		throw new IOException("mark not supported");
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#markSupported()
	 */
	@Override
	public boolean markSupported() {
		return false;
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#read()
	 */
	@Override
	public int read() throws IOException {
		int c;

		if(setreader()) {
			while((c = reader.read()) < 0) {
				reader.close();
				reader = null;
				if(!setreader())  break;
			}
			return c;
		} else {
			return -1;
		}
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#read(char[])
	 */
	@Override
	public int read(char[] cbuf) throws IOException {
		return read(cbuf, 0, cbuf.length);
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#ready()
	 */
	@Override
	public boolean ready() throws IOException {
		return reader.ready();
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#reset()
	 */
	@Override
	public void reset() throws IOException {
		throw new IOException("reset not supported");
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#skip(long)
	 */
	@Override
	public long skip(long n) throws IOException {
		long l = n, m;

		if(setreader()) {
			while((m = reader.skip(l)) < l) {
				reader.close();
				reader = null;
				if(setreader()) {
					l -= m;
				} else {
					return n - l;
				}
			}
			return n;
		} else {
			return -1;
		}
	}

	/* (non-Javadoc)
	 * @see java.io.Reader#read(char[], int, int)
	 */
	@Override
	public int read(char[] cbuf, int off, int len) throws IOException {
		int o = off, m, l = len;

		if(setreader()) {
			while((m = reader.read(cbuf, o, l)) < l) {
				reader.close();
				reader = null;
				if(setreader()) {
					o += m;
					l -= m;
				} else {
					return o + m;
				}
			}
			return len;
		} else {
			return -1;
		}
	}

}
