/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh.file;

import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShSignal;
import net.morilib.sh.ShTree;

public abstract class AbstractShEnvironment implements ShEnvironment {

	private static final String P_LCS =
		"([A-Za-z]+)(_([A-Za-z]+)(\\.([\\-A-Za-z_0-9]+))?)?";
	private static final Pattern P_LC = Pattern.compile(P_LCS);
	private static final String PS = ":";

	List<String> path;

	protected void setpath() {
		String s;

		if((s = find("PATH")) != null) {
			s = s.replaceFirst("^" + PS + "+", "");
			s = s.replaceFirst(PS + "+$", "");
			path = Arrays.asList(s.split(PS + "+"));
		}
	}

	public List<String> getPath() {
		return path;
	}

	public void bind(String name, ShTree function) {
		// do nothing
	}

	public void put(String name, ShTree function) {
		// do nothing
	}

	public ShTree findFunction(String name) {
		return null;
	}

	public boolean isEnvironment() {
		return true;
	}

	public boolean isSet(String name) {
		return false;
	}

	public void set(String name) {
	}

	public void reset(String name) {
	}

	public void set(String name, boolean value) {
	}

	public boolean isSystem(String name) {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#getTrap(net.morilib.sh.ShSignal)
	 */
	public String getTrap(ShSignal signal) {
		// OS signals cannot trap by Java VM
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#setTrap(net.morilib.sh.ShSignal, java.lang.String)
	 */
	public void setTrap(ShSignal signal, String cmd) {
		// OS signals cannot trap by Java VM
	}

	public Charset getCharset() {
		String s;

		if((s = getLocale().getVariant()) == null || s.equals("")) {
			return Charset.defaultCharset();
		} else if(s.equalsIgnoreCase("utf8")) {
			return Charset.forName("UTF-8");
		} else if(s.equalsIgnoreCase("eucjp")) {
			return Charset.forName("EUC-JP");
		} else if(s.equalsIgnoreCase("sjis")) {
			return Charset.forName("Windows-31J");
		} else {
			try {
				return Charset.forName(s);
			} catch(IllegalCharsetNameException e) {
				return Charset.defaultCharset();
			} catch(UnsupportedCharsetException e) {
				return Charset.defaultCharset();
			}
		}
	}

	public Locale getLocale() {
		String s, a1, a2, a3;
		Matcher m;

		if((s = find("LANG")) == null || s.equals("") ||
				!(m = P_LC.matcher(s)).matches()) {
			return Locale.getDefault();
		} else if((a1 = m.group(1)).equals("C")) {
			return Locale.US;
		} else if((a2 = m.group(3)) == null) {
			return new Locale(a1);
		} else if((a3 = m.group(5)) == null) {
			return new Locale(a1, a2);
		} else {
			return new Locale(a1, a2, a3);
		}
	}

}
