/*
 * Copyright (C) 2009 Beatcraft.Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * This implements a sensors hardware library for BC9 gadget.
 * the following code should be built as a shared library that will be
 * placed into /system/lib/hw/sensors.default.so
 *
 * it will be loaded by the code in hardware/libhardware/hardware.c
 * which is itself called from com_android_server_SensorService.cpp
 *
 * @author Yoshinari Takaoka <takaoka@beatcraft.com>
 */

#include "sensors_bc9.h"

//   one radian
static float one_rad = 180 / M_PI;


/* return the current time in nanoseconds */
static int64_t data__now_ns(void)
{
    struct timespec  ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);

    return (int64_t)ts.tv_sec * 1000000000 + ts.tv_nsec;
}


static int sensor_bc9_accelerometer_open()
{
    int fd, ret;
    int gsel = BC9_ACCELEROMETER_GSEL_1500;
    D("%s: called", __FUNCTION__);

    fd = open("/dev/accelerometer", O_RDWR);
    if (fd == -1) {
        E("%s: failed to open /dev/accelerometer -> %s",
           __FUNCTION__, strerror(errno));
	return -1;
    }

    //    initialize accelerometer
    struct bc9_accelerometer_conf conf;
    memset(&conf, 0x0, sizeof(struct bc9_accelerometer_conf));
    ret = ioctl(fd, BC9_ACCELEROMETER_GETCONF, &conf);
    if (ret == -1) {
        E("failed to init accelerometer -> %s",
          strerror(errno)
        );
	return -1;
    }
    conf.enable = 1;
    conf.gselect = gsel;
    ret = ioctl(fd, BC9_ACCELEROMETER_SETCONF, &conf);
    if (ret == -1) {
        E("failed to set configuration accelerometer -> %s",
          strerror(errno)
        );
	return -1;
    }

    return fd;
}

static int sensor_bc9_compass_open()
{
    int fd, ret;
    struct hmc5843_config conf;

    fd = open("/dev/compass", O_RDWR);
    if (fd == -1) {
        E("failed to open compass -> %s",
          strerror(errno)
        );
	return -1;
    }

retry_getconf:
    ret = ioctl(fd, HMC5843_IOCTL_GETCONF, &conf);
    if (ret == -1) {
        E("failed to GET configuration compass -> %s",
          strerror(errno)
        );
        goto retry_getconf;
    }

retry_setconf:
    conf.mode = HMC5843_MD_CONTINUE;
    ret = ioctl(fd, HMC5843_IOCTL_SETCONF, &conf);
    if (ret == -1) {
        E("failed to SET configuration compass -> %s",
          strerror(errno)
        );
        goto retry_setconf;
    }

    return fd;
}

static int setup_sensor_control_fd(int *ctl_fds)
{
    int ret = socketpair(AF_LOCAL, SOCK_STREAM, 0, ctl_fds);
    if (ret < 0) {
        E("failed to open control fd -> %s",
          strerror(errno)
        );
	return -1;
    }

    return 0;
}

static float calc_azimuth(float cx, float cy, int reverse_direction)
{
    float azimuth = 0;

    if (cx > 0 && cy > 0) {  // 0 < azimuth < 90 
       azimuth = fabs((atanf(cy/cx) * one_rad));
    }
    if (cx < 0 && cy > 0) {  // 90 < azimuth < 180 
       azimuth = 180 - fabs((atanf(cy/cx) * one_rad));
    }
    if (cx < 0 && cy < 0) {  // 180 < azimuth < 180 
       azimuth = 180 + fabs((atanf(cy/cx) * one_rad));
    }
    if (cx > 0 && cy < 0) {  // 270 < azimuth < 360 
       azimuth = 360 - fabs((atanf(cy/cx) * one_rad));
    }
    if (cx == 0 && cy == 0) {
        azimuth = 0;
    }
    if (cx == 0 && cy > 0) {
        azimuth = 90;
    }
    if (cx < 0 && cy == 0) {
        azimuth = 180;
    }
    if (cx == 0 && cy < 0) {
        azimuth = 270;
    }
    if (cx > 0 && cy == 0) {
        azimuth = 360;
    } 

    if (reverse_direction) {
        if (azimuth >= 0 && azimuth <= 180) {
            azimuth += 180;
        } else if (azimuth > 180 && azimuth <= 360) {
            azimuth -= 180;
        }
    }

    return azimuth;
}

static float calc_pitch(float ay, float az)
{
    float pitch = 0.0f;

    //   TODO: implement convert method.

    return pitch;
}

static float calc_roll(float ay)
{
    float roll = 0.0f;

    //   TODO: implement convert method.

    return roll;
}

static void* start_read_thread(void *arg)
{
    SensorControl* ctl = (void*)arg;
    struct bc9_accelerometer_data accel;
    struct hmc5843_data compass;
    SensorValue values;
    int ret;
    float ax, ay, az;
    float hx, hy;
    float cx, cy, cz;
    float azimuth, roll, pitch;
    float one_rad = 180 / M_PI;

    ctl->already_thread_started = 1;

    while (1) {

        if (ctl->force_stop) {
            break;
        }

        if (ctl->active_sensors & SENSORS_ACCELERATION) {

            memset(&accel, 0x0, sizeof(struct bc9_accelerometer_data));
            memset(&values, 0x0, sizeof(SensorValue));

            ret = ioctl(ctl->accel_fd, BC9_ACCELEROMETER_GETDATA, &accel);
            if (ret == -1) {
                E("%s: failed to get accelerometer data -> %s",
                   __FUNCTION__, strerror(errno));
                continue;
            }

            values.type = ID_ACCELERATION;
            values.data.sensor = ID_ACCELERATION;
            values.data.acceleration.x = ((float)accel.x / 1000.0f) * GRAVITY_EARTH;
            values.data.acceleration.y = ((float)accel.y / 1000.0f) * GRAVITY_EARTH;
            values.data.acceleration.z = ((float)accel.z / 1000.0f) * GRAVITY_EARTH;
            values.data.time = data__now_ns();

            ret = send(ctl->fd, &values, sizeof(SensorValue), 0);
            if (ret != sizeof(SensorValue)) {
                E("accelarator value send failed! -> %s",
                  strerror(errno)
                );
            }
        }
        if (ctl->active_sensors & SENSORS_MAGNETIC_FIELD) {

            memset(&compass, 0x0, sizeof(struct hmc5843_data));
            memset(&values, 0x0, sizeof(SensorValue));

retry_compass:
            ret = ioctl(ctl->compass_fd, HMC5843_IOCTL_GETDATA, &compass);
            if (ret == -1) {
                E("%s: failed to get compass data -> %s",
                   __FUNCTION__, strerror(errno));
                goto retry_compass;
            }

            values.type = ID_MAGNETIC_FIELD;
            values.data.sensor = ID_MAGNETIC_FIELD;

            //
            //   convert milli-gauss to micro-tesla
            //   1T = 10^4 gauss 1G = 10^-4 T
            //
            values.data.magnetic.x = ((float)compass.x / 10.0f);
            values.data.magnetic.y = ((float)compass.y / 10.0f);
            values.data.magnetic.z = ((float)compass.z / 10.0f);
            values.data.time = data__now_ns();

            ret = send(ctl->fd, &values, sizeof(SensorValue), 0);
            if (ret != sizeof(SensorValue)) {
                E("compass value send failed! -> %s",
                  strerror(errno)
                );
            }

        }
        if (ctl->active_sensors & SENSORS_ORIENTATION) {

            memset(&accel, 0x0, sizeof(struct bc9_accelerometer_data));
            memset(&compass, 0x0, sizeof(struct hmc5843_data));
            memset(&values, 0x0, sizeof(SensorValue));

            ret = ioctl(ctl->accel_fd, BC9_ACCELEROMETER_GETDATA, &accel);
            if (ret == -1) {
                E("%s: failed to get accelerometer data -> %s",
                   __FUNCTION__, strerror(errno));
                continue;
            }

            D("got accelerometer data!! ( %d, %d, %d )",
               accel.x, accel.y, accel.z
            );

retry_compass_o:
            ret = ioctl(ctl->compass_fd, HMC5843_IOCTL_GETDATA, &compass);
            if (ret == -1) {
                E("%s: failed to get compass data -> %s",
                   __FUNCTION__, strerror(errno));
                goto retry_compass_o;
            }

            D("got compass data!! ( %d, %d, %d )",
               compass.x, compass.y, compass.z
            );

            values.type = ID_ORIENTATION;
            values.data.sensor = ID_ORIENTATION;
            values.data.time = data__now_ns();

            //  SI unit (g) 
            ax = ((float)accel.x/1000.0f);
            ay = ((float)accel.y/1000.0f);
            az = ((float)accel.z/1000.0f);

            //  micro-tesla 
            cx = ((float)compass.x / 10.0f);
            cy = ((float)compass.y / 10.0f);
            cz = ((float)compass.z / 10.0f);

            //
            //  pitch: Rotation around X axis
            //  (-180<=pitch<=180), with positive values when
            //  the z-axis moves toward the y-axis.
            //
            values.data.orientation.pitch = calc_pitch(ay, az);

            //
            //  roll: Rotation around Y axis
            //  (-90<=roll<=90), with positive values when
            //  the z-axis moves AWAY from the x-axis.
            //
            values.data.orientation.roll  = calc_roll(ax);

            //
            //  azimuth: angle between the magnetic north direction
            //  and the Y axis, around the Z axis (0<=azimuth<360).
            //  0=North, 90=East, 180=South, 270=West
            //
            values.data.orientation.azimuth = calc_azimuth(cx, cy, 1);

            ret = send(ctl->fd, &values, sizeof(SensorValue), 0);
            if (ret != sizeof(SensorValue)) {
                E("orientation value send failed! -> %s",
                  strerror(errno)
                );
            }
        }

        usleep(15000);  // 0.015 msec
    }

    ctl->already_thread_started = 0;

    return (void *)ctl; 
}


///////////////////////////////////////////
//
//   Sensor Control related routine.
//
///////////////////////////////////////////

/**
 * this function must return a file descriptor that will be used to read
 * the sensors data (it is passed to data__data_open() below
 *
 * this function is called from SensorService#getDataChannel and
 * return's fd which will be parcelled by android framework.
 */
static int
control__open_data_source(struct sensors_control_device_t *dev)
{
    SensorControl*  ctl = (void*)dev;
    int fds[2] = {0, 0};
    int result;
    D("%s: called! ctl->fd value: %d", __FUNCTION__, ctl->fd);

    //
    //   this value will be parceled by android framework.
    //   in SensorService, thus, dupped and closed.
    //
    //   @see ~/mydroid/frameworks/services/ ... SenSorService.java 
    //
    
    D("opening ctl->fd in open_data_source ...");
  
    close(ctl->fd);

    result = setup_sensor_control_fd(fds);
    if (result == -1) {
        return -1;
    }
    D("fd0 -> %d", fds[0]);
    D("fd1 -> %d", fds[1]);
    ctl->fd = fds[0];

    if (ctl->accel_fd == -1) {
        ctl->accel_fd = sensor_bc9_accelerometer_open();
        if (ctl->accel_fd == -1) {
            return -1;
        }
    }

    if (ctl->compass_fd == -1) {
        ctl->compass_fd = sensor_bc9_compass_open();
        if (ctl->compass_fd == -1) {
            return -1;
        }
    }       

    D("%s: fd=%d", __FUNCTION__, ctl->fd);

    //   start read thread
    if (ctl->already_thread_started != 1) {
        D("data read thread starting...");
        pthread_create(&ctl->read_thread, NULL, start_read_thread, ctl);
    }

    return fds[1];
}

static int
control__activate(struct sensors_control_device_t *dev,
                  int handle,
                  int enabled)
{
    SensorControl*  ctl = (void*)dev;
    uint32_t        mask, sensors, active, new_sensors, changed;
    char            command[128];
    int             ret;

    D("%s: called", __FUNCTION__);
    D("%s: handle=%s (%d) enabled=%d", __FUNCTION__,
        _sensorIdToName(handle), handle, enabled);

    if (!ID_CHECK(handle)) {
        E("%s: bad handle ID", __FUNCTION__);
        return -1;
    }

    mask    = (1<<handle);  
    sensors = enabled ? mask : 0;

    active      = ctl->active_sensors;
    new_sensors = (active & ~mask) | (sensors & mask);
    changed     = active ^ new_sensors;

    D("active -> %d, new_sensors -> %d changed -> %d",
      active, new_sensors, changed); 

    if (!changed) {
        D("sensor state did not changed");
        return 0;
    }

    ctl->active_sensors = new_sensors;

    return 0;
}

static int
control__set_delay(struct sensors_control_device_t *dev, int32_t ms)
{
    SensorControl*  ctl = (void*)dev;
    char            command[128];

    D("%s: called", __FUNCTION__);
    D("%s: dev=%p delay-ms=%d", __FUNCTION__, dev, ms);
    D("set-delay:%d", ms);

    ctl->event_delay = ms; 

    return 0;
}

static int
control__wake(struct sensors_control_device_t *dev)
{
    SensorControl*  ctl = (void*)dev;
    D("%s: called", __FUNCTION__);
    D("%s: dev=%p", __FUNCTION__, dev);

    /* TODO: this function is used to force-stop the blocking
     * read() in data__poll. we should implement such behavior.
     */

    return 0;
}


static int
control__close(struct hw_device_t *dev) 
{
    D("%s: called", __FUNCTION__);

    SensorControl* ctl = (void*)dev;

    ctl->force_stop = 1;
    pthread_join(ctl->read_thread, NULL);

    close(ctl->fd);
    close(ctl->accel_fd);
    close(ctl->compass_fd);

    ctl->fd = -1;
    ctl->accel_fd = -1;
    ctl->compass_fd = -1;

    free(ctl);
    return 0;
}

///////////////////////////////////////////
//
//   Sensor data related routine.
//
///////////////////////////////////////////

static int
data__data_open(struct sensors_data_device_t *dev, int fd)
{
    SensorData*  data = (void*)dev;
    D("%s: called -> dev=%p fd=%d", __FUNCTION__, dev, fd);

    data->events_fd = dup(fd);
    if (data->events_fd == -1) {
        E("%s: dup failed -> %d", __FUNCTION__, fd);
    }
    D("%s: dupped fd -> %d", __FUNCTION__, data->events_fd);

    return 0;
}

static int
data__data_close(struct sensors_data_device_t *dev)
{
    D("%s called! : dev=%p", __FUNCTION__, dev);
    SensorData*  data = (void*)dev;

    if (data->events_fd > 0) {
        D("closing fd ... -> %d", data->events_fd);
        close(data->events_fd);
        data->events_fd = -1;
    }
    return 0;
}

static int
data__poll(struct sensors_data_device_t *dev, sensors_data_t* values)
{
    SensorData* data = (void*)dev;
    int fd = data->events_fd;
    int ret = 0;
    ssize_t recvsize = 0;
    SensorValue value;

    D("%s called!", __FUNCTION__);

    memset(values, 0x0, sizeof(sensors_data_t));

    recvsize = recv(fd, &value, sizeof(SensorValue), 0);
    if (recvsize != sizeof(SensorValue)) {
        E("Could not receive Sensor data!");
        return -1;
    }

    D("sensor data received!");
    D("type: %d", value.type);
    D("time: %lld", value.data.time);

    values->sensor = value.type;
    values->time = value.data.time;

    switch (value.type) {
    case ID_ACCELERATION:

        values->acceleration.x = value.data.acceleration.x;
        values->acceleration.y = value.data.acceleration.y;
        values->acceleration.z = value.data.acceleration.z;

        D("accel x: %.5f", values->acceleration.x);
        D("accel y: %.5f", values->acceleration.y);
        D("accel z: %.5f", values->acceleration.z);

        return ID_ACCELERATION;

    case ID_MAGNETIC_FIELD:

        values->magnetic.x = value.data.magnetic.x;
        values->magnetic.y = value.data.magnetic.y;
        values->magnetic.z = value.data.magnetic.z;

        D("compass x: %.5f", values->magnetic.x);
        D("compass y: %.5f", values->magnetic.y);
        D("compass z: %.5f", values->magnetic.z);

        return ID_MAGNETIC_FIELD;

    case ID_ORIENTATION:

        values->orientation.pitch = value.data.orientation.pitch;
        values->orientation.roll = value.data.orientation.roll;
        values->orientation.azimuth = value.data.orientation.azimuth;

        D("orientation azimuth: %.5f", values->orientation.azimuth);
        D("orientation pitch: %.5f", values->orientation.pitch);
        D("orientation roll: %.5f", values->orientation.roll);

        return ID_ORIENTATION;

    default:
        E("Unknown sensor type.");
        return -1;
    }
}

static int
data__close(struct hw_device_t *dev) 
{
    D("%s: called", __FUNCTION__);
    SensorData* data = (SensorData*)dev;
    if (data) {
        D("%s: freeing SensorData structure... ", __FUNCTION__);
        if (data->events_fd > 0) {
            D("(device close) about to close fd=%d", data->events_fd);
            close(data->events_fd);
        }
        free(data);
    }
    return 0;
}


/** MODULE REGISTRATION SUPPORT
 **
 ** This is required so that hardware/libhardware/hardware.c
 ** will dlopen() this library appropriately.
 **/

/*
 * the following is the list of all supported sensors.
 * this table is used to build sSensorList declared below
 * according to which hardware sensors are reported as
 * available from the emulator (see get_sensors_list below)
 *
 * note: numerical values for maxRange/resolution/power were
 *       taken from the reference AK8976A implementation
 */
static const struct sensor_t sSensorListInit[] = {
        {
          .name       = "BC9 Accelerometer",
          .vendor     = "Beatcraft.inc",
          .version    = 1,
          .handle     = ID_ACCELERATION,
          .type       = SENSOR_TYPE_ACCELEROMETER,
          .maxRange   = 2.8f,
          .resolution = 1.0f/4032.0f,
          .power      = 3.0f,
          .reserved   = {}
        },

        {
          .name       = "BC9 Magnetic field sensor",
          .vendor     = "Beatcraft.inc",
          .version    = 1,
          .handle     = ID_MAGNETIC_FIELD,
          .type       = SENSOR_TYPE_MAGNETIC_FIELD,
          .maxRange   = 10000.0f,
          .resolution = 4.0f,
          .power      = 0.5f,
          .reserved   = {}
        },

        { .name       = "BC9 Orientation sensor",
          .vendor     = "Beatcraft.inc",
          .version    = 1,
          .handle     = ID_ORIENTATION,
          .type       = SENSOR_TYPE_ORIENTATION,
          .maxRange   = 360.0f,
          .resolution = 1.0f,
          .power      = 9.7f,
          .reserved   = {}
        },

#if 0
        //
        //    These sensors are not supported yet.
        //

        { .name       = "BC9 Temperature sensor",
          .vendor     = "Beatcraft.inc",
          .version    = 1,
          .handle     = ID_TEMPERATURE,
          .type       = SENSOR_TYPE_TEMPERATURE,
          .maxRange   = 80.0f,
          .resolution = 1.0f,
          .power      = 0.0f,
          .reserved   = {}
        },
#endif
};

static struct sensor_t  sSensorList[MAX_NUM_SENSORS];

static uint32_t sensors__get_sensors_list(struct sensors_module_t* module,
        struct sensor_t const** list) 
{
    *list = sSensorListInit;
    return MAX_NUM_SENSORS;
}

/**
 *   this function initializes internal Sensor structure,
 *   and called by SensorService.java#constructor -> jni
 *   (_sensors_control_init())
 */
static int
open_sensors(const struct hw_module_t* module,
             const char*               name,
             struct hw_device_t*      *device)
{
    int  status = -EINVAL;

    D("%s: name=%s", __FUNCTION__, name);

    if (!strcmp(name, SENSORS_HARDWARE_CONTROL))
    {
        SensorControl *dev = malloc(sizeof(*dev));

        memset(dev, 0, sizeof(*dev));

        //    sensor_control_device_t 
        dev->device.common.tag       = HARDWARE_DEVICE_TAG;
        dev->device.common.version   = 0;
        dev->device.common.module    = (struct hw_module_t*) module;
        dev->device.common.close     = control__close;
        dev->device.open_data_source = control__open_data_source;
        dev->device.activate         = control__activate;
        dev->device.set_delay        = control__set_delay;
        dev->device.wake             = control__wake;
        dev->fd                      = -1;
        dev->accel_fd                = -1;
        dev->compass_fd              = -1;
        dev->event_delay             = 0;
        dev->already_thread_started  = 0;
        dev->force_stop              = 0;

        *device = &dev->device.common;
        status  = 0;
    } else if (!strcmp(name, SENSORS_HARDWARE_DATA)) {

        SensorData *dev = malloc(sizeof(*dev));

        memset(dev, 0, sizeof(*dev));

        dev->device.common.tag     = HARDWARE_DEVICE_TAG;
        dev->device.common.version = 0;
        dev->device.common.module  = (struct hw_module_t*) module;
        dev->device.common.close   = data__close;
        dev->device.data_open      = data__data_open;
        dev->device.data_close     = data__data_close;
        dev->device.poll           = data__poll;
        dev->events_fd             = -1;

        *device = &dev->device.common;
        status  = 0;
    }
    return status;
}


static struct hw_module_methods_t sensors_module_methods = {
    .open = open_sensors
};

const struct sensors_module_t HAL_MODULE_INFO_SYM = {
    //   struct hw_module_t
    .common = { 
        .tag = HARDWARE_MODULE_TAG,
        .version_major = 1,
        .version_minor = 0,
        .id = SENSORS_HARDWARE_MODULE_ID,
        .name = "BC9 SENSORS Module",
        .author = "Beatcraft.Inc",
        .methods = &sensors_module_methods,
    },
    .get_sensors_list = sensors__get_sensors_list
};
