package jp.sf.beanbinder;

import java.awt.Component;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Pattern;

import jp.sf.beanbinder.exception.ValidateException;
import jp.sf.beanbinder.strategy.ComponentStrategy;
import jp.sf.beanbinder.strategy.swing.Selectable;
import jp.sf.beanbinder.strategy.swing.SingleSelectable;
import jp.sf.beanbinder.util.BeanProcessBridge;
import net.sf.cglib.proxy.Enhancer;

/**
 * オブジェクトとUIを結びつけます。
 * @author Tomohiro Umeda
 */
public final class BeanBinder {
    protected Map bindMap = new HashMap();//Map<String, Bind>
    private Map objectMap = new HashMap();//Map<String, Object>
    private BindStrategyDispatcher bindStrategyDisplatcher;
    
    /**
     * BeanBinderの実装を選択します。
     * @param bindStrategy
     */
    BeanBinder(BindStrategyDispatcher bindStrategy) {
        this.bindStrategyDisplatcher = bindStrategy;
    }

    /**
     * オブジェクトとコンポーネントのバインド
     * @param bindKey : バインド情報の登録名
     * @param bind : バインド情報
     */
    public Bind bind(String bindKey, Bind bind) {
        this.bindMap.put(bindKey, bind);
        bind.setName(bindKey);
        return bind;
    }
    /**
     * オブジェクトとコンポーネントのバインド
     * @param bindKey : バインド情報の登録名
     * @param objectKey : バインド対象のオブジェクトを示すキー
     * @param component : バインドするコンポーネント
     */
    public Bind bind(String bindKey, String objectKey, Component component) {
        Bind bind = new Bind(objectKey, component);
        return this.bind(bindKey, bind);
    }

    /**
     * プロキシオブジェクトのインスタンスを生成<br>生成と同時にBeanBinderに指定したキーを用いて登録します。
     * @param objectKey : キー
     * @param type : 誕生させたいインスタンスのクラス
     */
    public Object createProxyObject(String objectKey, Class type) {
        try {
            Enhancer enhancer = new Enhancer();
            enhancer.setSuperclass(type);
            enhancer.setCallback(new ProxyInterceptor(this, objectKey));
          
            Object newInstance = enhancer.create();
            this.setObject(objectKey, newInstance);
            return newInstance;
        }catch(Exception e) {
            e.printStackTrace();
        }
        return null;
    }
    /**
     * バインドするオブジェクトの変更
     * @param bindName : オブジェクトを示すキー
     * @param object : オブジェクト
     */
    public void setObject(String objectKey, Object object) {
        this.objectMap.put(objectKey, object);
    }
    /**
     * 登録済みオブジェクトの取得
     * @param objectKey : キー
     * @return : 登録済みオブジェクト
     */
    public Object getObject(String objectKey) {
        String[] props = objectKey.split("[.]");
        Object bean = this.objectMap.get(props[0]);
        if( props.length <= 1 )return bean;
        
        String propertyPath="";
        for(int i=1; i<props.length; i++) {
            propertyPath += props[i];
            if( i != props.length-1 )propertyPath += ".";
        }
        
        try {
            return BeanProcessBridge.getPropertyValue(bean, propertyPath);
        }catch(Exception e) {
            return null;
        }
    }
    /**
     * 登録済みオブジェクトの取得
     * @param objectKey : オブジェクトを示すキー
     * @param property : プロパティ名
     * @return : 登録済みオブジェクト
     */
    public Object getObject(String objectKey, String property) {
        try {
            return BeanProcessBridge.getPropertyValue(this.getObject(objectKey), property);
        }catch(Exception e) {
            return null;
        }
    }
    /**
     * 複数選択可能なコンポーネントへ指定した<br>
     * オブジェクト位置の選択を行います。
     * @param bindKey : 複数選択可能なコンポーネントを示すキー
     * @param objects : 複数選択の基準となるオブジェクト
     */
    public void setSelectedValue(String bindKey, Object[] objects) {
        Bind bind = (Bind)this.bindMap.get(bindKey);
        ComponentStrategy strategy = this.bindStrategyDisplatcher.getStrategy(bind.getComponent());
        
        if( strategy instanceof Selectable ) {
            ((Selectable)strategy).selectBean(bind, bind.getComponent(), objects);
        }else {
            throw new RuntimeException("this component is unsupport");
        }
    }
    /**
     * 選択可能なコンポーネントへ指定した<br>
     * オブジェクト位置の選択を行います。
     * @param bindKey : 選択可能なコンポーネントを示すキー
     * @param object : 選択の基準となるオブジェクト
     */
    public void setSelectedValue(String bindKey, Object object) {
        this.setSelectedValue(bindKey, new Object[]{object});
    }
    /**
     * 複数選択可能なコンポーネントから選択位置にあるオブジェクトを返す
     * @param bindKey : 複数選択可能なコンポーネントを示すキー
     * @return : 選択済みのオブジェクト
     */
    public Object[] getSelectedValues(String bindKey) {
        Bind bind = (Bind)this.bindMap.get(bindKey);
        if( bind == null )throw new RuntimeException("指定したキー名は未登録です。:"+bindKey);
        
        ComponentStrategy strategy = this.bindStrategyDisplatcher.getStrategy(bind.getComponent());
        if( strategy instanceof Selectable ) {
            return ((Selectable)strategy).getSelectedBean(bind, bind.getComponent());
        }else {
            throw new RuntimeException("このコンポーネントには使えません。:"+bind.getComponent());
        }
    }
    /**
     * 選択可能なコンポーネントから選択位置にあるオブジェクトを返す
     * @param bindKey : 選択可能なコンポーネントを示すキー
     * @return : 選択済みのオブジェクト
     */
    public Object getSelectedValue(String bindKey) {
        if( this.getSelectedValues(bindKey) == null )return null;
        return this.getSelectedValues(bindKey)[0];
    }
    /**
     * バインド情報から入力された値を取得する
     * @param bindKey : バインド情報を示すキー
     * @return : コンポーネントに入力された値
     */
    public Object getComponentValue(String bindKey) {
        Bind bind = (Bind)this.bindMap.get(bindKey);
        ComponentStrategy strategy = this.bindStrategyDisplatcher.getStrategy(bind.getComponent());
        return strategy.getValue(bind, bind.getComponent());
    }
    /**
     * @param bindKey : キー
     * @return Component
     */
    protected Object getComponent(String bindKey) {
        return ((Bind)this.bindMap.get(bindKey)).getComponent();
    }
    /**
     * バインド情報の取得
     * @param bindKey : バインド情報を示すキー
     */
    public Bind getBind(String bindKey) {
        return (Bind)this.bindMap.get(bindKey);
    }
    /**
     * バインド情報の取得
     * @param component : 取得したいバインドと関連のあるコンポーネント
     */
    public Bind getBindByComponent(Object component) {
        Iterator itr = this.bindMap.values().iterator();//Iterator<Bind>
        while(itr.hasNext()) {
            Bind bind = (Bind)itr.next();
            if( component.equals(bind.getComponent()) ) {
                return bind;
            }
        }
        return null;
    }
    /**
     * コンポーネントの描画内容の更新
     * @param bindKey : コンポーネントを示すキー又はそれらにマッチする正規表現 
     */
    public void updateComponent(String bindKey) {
        Pattern pattern = Pattern.compile(bindKey);
        
        Iterator itr = this.bindMap.keySet().iterator();
        while(itr.hasNext()) {
            String key = (String)itr.next();
            if( pattern.matcher(key).matches() ) {
                this._updateComponent(key);
            }
        }        
    }

    /**
     * コンポーネントの描画内容の更新
     * @param group : 更新するコンポーネントのグループ名 
     */
    public void updateComponentByGroup(String group) {
        Pattern pattern = Pattern.compile(group);
        
        Iterator itr = this.bindMap.keySet().iterator();
        while(itr.hasNext()) {
            String key = (String)itr.next();
            Bind bind = (Bind)this.bindMap.get(key);
            if( pattern.matcher(bind.getGroup()).matches() ) {
                this._updateComponent(key);
            }
        }        
    }
    private void _updateComponent(String name) {
        Bind bind = (Bind)this.bindMap.get(name);
        if( bind == null || !bind.isEnable() )return;

        Object bean = this.getObject(bind.getDataSource());
        
        if( bean == null ) {
            ComponentStrategy strategy = this.bindStrategyDisplatcher.getStrategy(bind.getComponent());
            strategy.updateComponent(bind, bind.getComponent());
            return;
        }
        
        //コレクション以外のオブジェクトは表示内容を示すプロパティで更新
        if( !(bean instanceof Collection) ) {
            try {
                bean = BeanProcessBridge.getPropertyValue(bean, bind.getProperty(0).getDisplayProperty());
            }catch(Exception e) {
                throw new RuntimeException(e);
            }
        }
        
        //コンポーネントの再描画
        ComponentStrategy strategy = this.bindStrategyDisplatcher.getStrategy(bind.getComponent());
        try {
            strategy.updateDisplay(bind, bind.getComponent(), bean);
        }catch(Exception e) {e.printStackTrace();}
        
        //選択が必要な場合
        if( strategy instanceof SingleSelectable ) {
            try {
                bean = this.getObject(bind.getTarget());
                ((SingleSelectable)strategy).selectBean(bind, bind.getComponent(), new Object[]{bean});
            }catch(Exception e) {e.printStackTrace();}
        }
    }
    /**
     * 全てのオブジェクトを更新
     */
    public void updateAllObjects() throws ValidateException {
        this.updateObject(".*");
    }
    /**
     * コンポーネントによるオブジェクトの更新、指定したオブジェクトと関連のある全ての部品から更新します。
     * @param objectKey : 更新されるオブジェクトを示すキー又はそれらにマッチする正規表現
     */
    public void updateObject(String objectKey) throws ValidateException {
        Pattern pattern = Pattern.compile(objectKey);
        Iterator itr = this.bindMap.values().iterator();
        
        while(itr.hasNext()) {
            Bind bind = (Bind)itr.next();
            if( !bind.isEnable() )continue;
            
            if( bind.getTarget() != null && pattern.matcher(bind.getTarget()).matches() ) {
                Object bean = this.getObject(bind.getTarget());
                ComponentStrategy strategy = this.bindStrategyDisplatcher.getStrategy(bind.getComponent());
                strategy.validate(bind, bind.getComponent());
                
                strategy.updateBean(bean, bind, bind.getComponent());
            }
        }
    }

    public BindStrategyDispatcher getBindStrategyDisplatcher() {
        return bindStrategyDisplatcher;
    }
    public void setBindStrategyDisplatcher(BindStrategyDispatcher bindStrategy) {
        this.bindStrategyDisplatcher = bindStrategy;
    }
}
