package jp.sf.beanbinder;

import java.awt.Component;
import java.text.Format;

import jp.sf.beanbinder.editor.CellEditor;
import jp.sf.beanbinder.validator.Validator;

public final class BindFactory {
    private BindFactory(){}
    
    /**
     * 基本的なコンポーネントの為のバインド情報を作成
     * @param objectKey : バインド対象のオブジェクトを示すキー
     * @param component : バインド対象のコンポーネント
     * @param property : 表示する内容と編集内容をどのプロパティに反映させるか
     * @return
     */
    public final static Bind createBind(String objectKey, Component component, String property) {
        return createBind(objectKey, component, property, null);
    }
    public final static Bind createBind(String objectKey, Component component, String property, Validator validator) {
        Bind bind = new Bind(objectKey, component);
        BindProperty prop = new BindProperty( property, new Ref(property) );
        prop.setValidator(validator);
        bind.addProperty(prop);
        return bind;
    }
    
    /**
     * テキストコンポーネントの為のバインド情報を作成
     * @param objectKey : バインド対象のオブジェクトを示すキー
     * @param component : バインド対象のコンポーネント
     * @param property : 表示する内容と編集内容をどのプロパティに反映させるか
     * @param imeMode : jp.sf.beanbinder.IMEMode
     * @param format : java.text.Format
     * @return
     */
    public final static Bind createBindForText(String objectKey, Component component, String property, IMEMode imeMode, Format format) {
        return createBindForText(objectKey, component, property, imeMode, format, null);
    }
    public final static Bind createBindForText(String objectKey, Component component, String property, IMEMode imeMode, Format format, Validator validator) {
        Bind bind = new Bind(objectKey, component);
        BindProperty prop = new BindProperty( property, new Ref(property) );
        prop.setImeMode( (imeMode==null?IMEMode.DEFAULT:imeMode) );
        prop.setFormat(format);
        prop.setValidator(validator);
        bind.addProperty(prop);
        return bind;
    }
    /**
     * コンボボックスコンポーネントの為のバインド情報を作成
     * @param objectKey : バインド対象のオブジェクトを示すキー
     * @param component : バインド対象のコンポーネント
     * @param targetProp : 選択した内容をどのプロパティに反映させるか 
     * @param targetIdentity : どのオブジェクトが選択されたかを判断する為の情報
     * @param dataSourceKey : データソース（Collection）を示すキー
     * @param displayProp : 表示する内容を示すプロパティ名（データソースから参照）
     * @param dataSrcProp : 選択した時にどのプロパティを取得するか
     * @param dataSrcIdentity : どのオブジェクトが選択されたかを判断する為の情報
     */
    public final static Bind createBindForCombo(
            String objectKey, Component component,
            String targetProp, String targetIdentity,
            String dataSourceKey, String displayProp, String dataSrcProp, String dataSrcIdentity ) {
        
        Bind bind = new Bind(objectKey, component);
        bind.setDataSource(dataSourceKey);
        
        BindProperty prop = new BindProperty( displayProp,
                new Ref(targetProp, targetIdentity),
                new Ref(dataSrcProp, dataSrcIdentity)
        );
        bind.addProperty(prop);
        return bind;
    }
    /**
     * セルエディタコンボボックスの為のバインド情報を作成
     * @param component : バインド対象のコンポーネント
     * @param dataSourceKey : データソース（Collection）を示すキー
     * @param displayProp : 表示する内容を示すプロパティ名（データソースから参照）
     * @param dataSrcProp : 選択した時にどのプロパティを取得するか
     */
    public final static Bind createBindForComboCellEditor(Component component, String dataSourceKey, String displayProp, String dataSrcProp) {
        Bind bind = new Bind(null, component);
        bind.setDataSource(dataSourceKey);
        BindProperty prop = new BindProperty( displayProp, new Ref(null,null), new Ref(dataSrcProp, null) );
        bind.addProperty(prop);
        
        return bind;
    }
    /**
     * リストコンポーネントの為のバインド情報を作成
     * @param objectKey : バインド対象のオブジェクトを示すキー
     * @param component : バインド対象のコンポーネント
     * @param targetProp : 選択した内容をどのプロパティに反映させるか 
     * @param targetIdentity : どのオブジェクトが選択されたかを判断する為の情報
     * @param dataSourceKey : データソース（Collection）を示すキー
     * @param displayProp : 表示する内容を示すプロパティ名（データソースから参照）
     * @param dataSrcProp : 選択した時にどのプロパティを取得するか
     * @param dataSrcIdentity : どのオブジェクトが選択されたかを判断する為の情報
     * @return
     */
    public final static Bind createBindForList(
            String objectKey, Component component,
            String targetProp, String targetIdentity,
            String dataSourceKey, String displayProp, String dataSrcProp, String dataSrcIdentity) {
        
        return BindFactory.createBindForCombo(objectKey, component, targetProp, targetIdentity, dataSourceKey, displayProp, dataSrcProp, dataSrcIdentity);
    }
    /**
     * テーブルコンポーネントの為のバインド情報を作成
     * @param objectKey : バインド対象のオブジェクトを示すキー
     * @param component : バインド対象のコンポーネント
     * @param columns : カラムを表現するBindPropertyの配列
     * @return
     */
    public final static Bind createBindForTable(String objectKey, Component component, BindProperty[] columns) {
        Bind bind = new Bind(objectKey, component);
        for(int i=0; i<columns.length; i++) {
            BindProperty prop = columns[i];
            bind.addProperty(prop);
        }
        return bind;
    }
    /**
     * テーブルコンポーネントの為のカラム情報を作成
     * @param text : 列名 
     * @param property : 表示内容と編集内容を示すプロパティ名
     * @param imeMode : IMEMode
     * @param format : java.text.Format
     * @param readOnly : 読取り専用列かどうか
     * @return
     */
    public final static BindProperty createColumn(String text, String property, IMEMode imeMode, Format format, boolean readOnly) {
        return createColumn(text, property, imeMode, format, null, readOnly);
    }
    public final static BindProperty createColumn(String text, String property, IMEMode imeMode, Format format, Validator validator, boolean readOnly) {
        BindProperty prop = new BindProperty( property, new Ref(property) );
        prop.getColumnProperty().setText(text);
        prop.setEditable(!readOnly);
        prop.setValidator(validator);
        return prop;
    }
    /**
     * テーブルコンポーネントの為のカラム情報を作成
     * @param text : 列名 
     * @param property : 表示内容と編集内容を示すプロパティ名
     * @param readOnly : 読取り専用列かどうか
     * @return
     */
    public final static BindProperty createColumn(String text, String property, boolean readOnly) {
        return createColumn(text, property, IMEMode.DEFAULT, null, readOnly);
    }
    /**
     * テーブルコンポーネントの為のカラム情報を作成
     * @param text : 列名 
     * @param displayProp : 表示する内容を示すプロパティ名
     * @param targetProp : コンポーネントの入力内容をどのプロパティに反映させるか
     * @param cellEditor : セルエディタ
     * @return
     */
    public final static BindProperty createColumn(String text, String displayProp, String targetProp, CellEditor cellEditor) {
        BindProperty prop = new BindProperty( displayProp, new Ref(targetProp) );
        prop.getColumnProperty().setText(text);
        prop.setEditable(true);
        prop.getColumnProperty().setCellEditor(cellEditor);
        return prop;
    }
    /**
     * テーブルコンポーネントの為のカラム情報を作成
     * @param text : 列名 
     * @param displayProp : 表示する内容を示すプロパティ名
     * @param targetProp : コンポーネントの入力内容をどのプロパティに反映させるか
     * @param format : java.text.Format
     * @param cellEditor : セルエディタ
     * @return
     */
    public final static BindProperty createColumn(String text, String displayProp, String targetProp, Format format, CellEditor cellEditor) {
        BindProperty prop = new BindProperty( displayProp, new Ref(targetProp) );
        prop.getColumnProperty().setText(text);
        prop.setEditable(true);
        prop.setFormat(format);
        prop.getColumnProperty().setCellEditor(cellEditor);
        return prop;
    }
    /**
     * テーブルコンポーネントの為のカラム情報を作成
     * @param text : 列名 
     * @param displayProp : 表示する内容を示すプロパティ名
     * @param targetProp : コンポーネントの入力内容をどのプロパティに反映させるか
     * @param cellEditor : セルエディタ
     * @param cellRenderer : セルレンダラ
     * @return
     */
    public final static BindProperty createColumn(String text, String displayProp, String targetProp, CellEditor cellEditor, CellRenderer cellRenderer) {
        BindProperty prop = BindFactory.createColumn(text, displayProp, targetProp, cellEditor);
        prop.getColumnProperty().setCellRenderer(cellRenderer);
        return prop;
    }
    /**
     * テーブルコンポーネントの為のカラム情報を作成
     * @param text : 列名 
     * @param displayProp : 表示する内容を示すプロパティ名
     * @param targetProp : コンポーネントの入力内容をどのプロパティに反映させるか
     * @param format : java.text.Format
     * @param cellEditor : セルエディタ
     * @param cellRenderer : セルレンダラ
     * @return
     */
    public final static BindProperty createColumn(String text, String displayProp, String targetProp, Format format, CellEditor cellEditor, CellRenderer cellRenderer) {
        BindProperty prop = BindFactory.createColumn(text, displayProp, targetProp, format, cellEditor);
        prop.getColumnProperty().setCellRenderer(cellRenderer);
        return prop;
    }
}
