package jp.sf.beanbinder;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.text.Format;
import java.util.ArrayList;
import java.util.List;

import jp.sf.beanbinder.editor.CellEditor;
import jp.sf.beanbinder.listener.EditListener;
import jp.sf.beanbinder.validator.Validator;

public final class BindingBridge {
    private Object target;
    private Field[] fields;
    private BeanBinder binder;
    
    public BindingBridge(BeanBinder binder, Object target) {
        this.target = target;
        this.binder = binder;
        
        //フィールドの取得
        List results = new ArrayList(); 
        parseFields( results, target.getClass() );
        this.fields = (Field[])results.toArray(new Field[]{});
        AccessibleObject.setAccessible(fields, true);
    }
    private void parseFields(List fields, Class currentClass) {
        if( currentClass == null )return;
        Field[] fs = currentClass.getDeclaredFields();
        for(int i=0; i<fs.length; i++) {
            Field f = fs[i];
            fields.add(f);
        }
        parseFields(fields, currentClass.getSuperclass());
    }
    public Field getField(Object target, String field) throws Exception {
        for(int i=0; i<fields.length; i++) {
            Field f = fields[i];
            if( field.equals(f.getName()) )return f;
        }
        
        throw new RuntimeException("指定したフィールドが見つかりません。:"+field);
    }    
    public Object getFieldValue(Object target, String field) throws Exception {
        if( field == null || field.trim().length() <= 0 )return null;
        Field f = this.getField(target, field);
        return f.get(target);
    }
    public Object getConstValue(String fieldName) throws Exception {
        if( fieldName == null || fieldName.length() <= 0 )return null;
        String[] split = fieldName.split("[.]");
        String path="";
        for(int i=1; i<split.length; i++) {
            path += split[i];
            if( i != split.length-1 )path += ".";
        }
        
        Package pack = BeanBinder.class.getPackage();
        try {
            Field field = Class.forName(pack.getName()+"."+split[0]).getField(path);
            return field.get(null);
        }catch(Exception e) {
            throw new Exception("指定したフィールドが見つかりません。:"+fieldName);
        }
    }
    
    
    public void bindBasic(String fieldName, String objectId, String prop, String group,
            IMEMode imeMode, Format format, AlignType horizon, AlignType vertical, Validator validator) throws Exception {
        Object component = this.getFieldValue(target, fieldName);
        
        Bind bind = new Bind(objectId, component);
        bind.setGroup(group);
        
        BindProperty bp = new BindProperty( prop, new Ref(prop) );
        bp.setImeMode(imeMode);
        bp.setFormat( (format==null?new DefaultFormat():format) );
        bp.setHorizon( (horizon==null?AlignType.LEFT:horizon) );
        bp.setVertical( (vertical==null?AlignType.TOP:vertical) );
        bp.setValidator(validator);
        
        bind.addProperty(bp);
        binder.bind(fieldName, bind);
    }
    public void bindSelect(String fieldName, String group,
            String destId, String destValueProp, String destIdProp,
            String srcId, String srcDisplayProp, String srcValueProp, String srcIdProp,
            Format format, AlignType horizon, AlignType vertical, Validator validator ) throws Exception {
        
        Object component = this.getFieldValue(target, fieldName);
        Bind bind = new Bind(destId, component);
        bind.setDataSource(srcId);
        bind.setGroup(group);
        
        BindProperty bp = new BindProperty( srcDisplayProp,
                new Ref(destValueProp, destIdProp),
                new Ref(srcValueProp, srcIdProp)
        );
        bp.setFormat( (format==null?new DefaultFormat():format) );
        bp.setHorizon( (horizon==null?AlignType.LEFT:horizon) );
        bp.setVertical( (vertical==null?AlignType.TOP:vertical) );
        bp.setValidator(validator);
        
        bind.addProperty(bp);
        binder.bind(fieldName, bind);
    }
    public void bindCustom(String fieldName, String objectId,
            String displayProp, String idProp, String group,
            IMEMode imeMode, Format format,
            AlignType horizon, AlignType vertical, Validator validator) throws Exception {
        
        Object component = this.getFieldValue(target, fieldName);
        
        Bind bind = new Bind(objectId, component);
        bind.setGroup(group);
        
        BindProperty bp = new BindProperty( displayProp, new Ref(objectId, idProp) );
        bp.setImeMode(imeMode);
        bp.setFormat( (format==null?new DefaultFormat():format) );
        bp.setHorizon( (horizon==null?AlignType.LEFT:horizon) );
        bp.setVertical( (vertical==null?AlignType.TOP:vertical) );
        bp.setValidator(validator);
        
        bind.addProperty(bp);
        binder.bind(fieldName, bind);
    }
    public void bindTable(String fieldName, String objectId, String group, int rowHeight, boolean sortable, EditListener editListener, BindProperty[] columns) throws Exception {
        Object component = this.getFieldValue(target, fieldName);
        
        Bind bind = new Bind(objectId, component);
        bind.setSortable(sortable);
        bind.setEditListener(editListener);        
        
        for(int i=0; i<columns.length; i++) {
            BindProperty col = columns[i];
            if( rowHeight <= 0 )rowHeight = 20;
            col.setHeight(rowHeight);
            bind.addProperty(col);
        }
        binder.bind(fieldName, bind);
    }
    public BindProperty bindColumn(String displayProp, String valueProp, String idProp, String text,
            boolean readOnly, String prop, int width,
            IMEMode imeMode, Format format, AlignType horizon, AlignType vertical, Validator validator,
            CellEditor cellEditor, CellRenderer cellRenderer ) throws Exception {
        
        BindProperty col = new BindProperty( displayProp, new Ref(valueProp, idProp) );
        if( prop != null && prop.length() > 0 ) {
            col.getRef().setValueProp(prop);
            col.setDisplayProperty(prop);
        }
        
        col.getColumnProperty().setText(text);
        col.setEditable(!readOnly);
        col.setWidth(width);
        
        col.setImeMode(imeMode);
        col.setFormat( (format==null?new DefaultFormat():format) );
        col.setHorizon( (horizon==null?AlignType.LEFT:horizon) );
        col.setVertical( (vertical==null?AlignType.TOP:vertical) );
        col.setValidator(validator);
        col.getColumnProperty().setCellEditor(cellEditor);
        col.getColumnProperty().setCellRenderer(cellRenderer);
        
        return col;
    }
}
