package jp.sf.beanbinder.xml;

import java.io.File;
import java.text.Format;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import jp.sf.beanbinder.AlignType;
import jp.sf.beanbinder.BeanBinder;
import jp.sf.beanbinder.BindProperty;
import jp.sf.beanbinder.BindingBridge;
import jp.sf.beanbinder.CellRenderer;
import jp.sf.beanbinder.IMEMode;
import jp.sf.beanbinder.MappingParser;
import jp.sf.beanbinder.editor.CellEditor;
import jp.sf.beanbinder.listener.EditListener;
import jp.sf.beanbinder.validator.Validator;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public final class XMLParser implements MappingParser {
    private File xmlFile;
    private Object target;
    private BindingBridge bridge;
    private BeanBinder binder;
    
    public XMLParser(BeanBinder binder, Object target, File xmlFile) {
        this.xmlFile = xmlFile;
        this.target = target;
        this.binder = binder;
    }
    public void parse() {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(this.xmlFile);
            Element root = doc.getDocumentElement();
            if( !root.getNodeName().equals("bind") ) {
                throw new Exception("マッピング情報は必ず<bind>タグ内に記述します。:"+this.xmlFile.getAbsolutePath());
            }
                        
            this.bridge = new BindingBridge(this.binder, this.target);

            this.parse(this.target, root);
        }catch(Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        }
    }
    private void parse(Object target, Element root) throws Exception {
        NodeList basicList = root.getElementsByTagName("basic");
        NodeList selectList = root.getElementsByTagName("select");
        NodeList customList = root.getElementsByTagName("custom");
        NodeList tableList = root.getElementsByTagName("table");
        
        for(int i=0; i<basicList.getLength(); i++)this.basicTag(this.target, (Element)basicList.item(i));
        for(int i=0; i<selectList.getLength(); i++)this.selectTag(this.target, (Element)selectList.item(i));
        for(int i=0; i<tableList.getLength(); i++)this.tableTag(this.target, (Element)tableList.item(i));
    }
    private String getTagProp(Element tag, String name) {
        Node node = tag.getElementsByTagName(name).item(0);
        if( node == null ) {
            return tag.getAttribute(name);
        }
        return node.getTextContent();
    }
    private int getTagPropInt(Element tag, String name) {
        String value = this.getTagProp(tag, name);
        return ( (value==null||value.length()<=0)?0:Integer.parseInt(value) );
    }
    private boolean getTagPropBoolean(Element tag, String name) {
        String value = this.getTagProp(tag, name);
        return ( (value==null||value.length()<=0)?false:Boolean.parseBoolean(value) );
    }
    
    private IMEMode getIMEMode(Element tag) throws Exception {
        String _imeMode = getTagProp(tag, "imeMode");
        IMEMode imeMode = (IMEMode)this.bridge.getConstValue(_imeMode);
        if( imeMode == null )imeMode = IMEMode.DEFAULT;
        return imeMode;
    }
    private Format getFormat(Element tag) throws Exception {
        String _format = getTagProp(tag, "formatField");
        Format format = (Format)this.bridge.getFieldValue(target, _format);
        if( format == null )format = new jp.sf.beanbinder.DefaultFormat();
        return format;
    }
    private AlignType getHorizon(Element tag) throws Exception {
        String _horizon = getTagProp(tag, "horizon");
        AlignType horizon = (AlignType)this.bridge.getConstValue(_horizon);
        if( horizon == null )horizon = AlignType.LEFT;
        return horizon;
    }
    private AlignType getVertical(Element tag) throws Exception {
        String _vertical = getTagProp(tag, "vertical");
        AlignType vertical = (AlignType)this.bridge.getConstValue(_vertical);
        if( vertical == null )vertical = AlignType.TOP;
        return vertical;
    }
    private Validator getValidator(Element tag) throws Exception {
        String _validator = getTagProp(tag, "validatorField");
        Validator validator = (Validator)this.bridge.getFieldValue(target, _validator);
        return validator;
    }
    private CellEditor getCellEditor(Element tag) throws Exception {
        String _cellEditor = getTagProp(tag, "cellEditorField");
        CellEditor cellEditor = (CellEditor)this.bridge.getFieldValue(target, _cellEditor);
        return cellEditor;
    }
    private CellRenderer getCellRenderer(Element tag) throws Exception {
        String _cellRenderer = getTagProp(tag, "cellRendererField");
        CellRenderer cellRenderer = (CellRenderer)this.bridge.getFieldValue(target, _cellRenderer);
        return cellRenderer;
    }
    private EditListener getEditListener(Element tag) throws Exception {
        String _editListener = getTagProp(tag, "editListenerField");
        EditListener editListener = (EditListener)this.bridge.getFieldValue(target, _editListener);
        return editListener;
    }
    
    //<basic>
    private void basicTag(Object target, Element tag) throws Exception {
        String fieldName = getTagProp(tag, "field");
        if( fieldName == null )throw new Exception("<field>要素の存在しないタグが発見されました。:<basic>");
        
        String objectKey = getTagProp(tag, "objectKey");
        String prop = getTagProp(tag, "prop");
        String group = getTagProp(tag, "group");
        
        this.bridge.bindBasic(fieldName, objectKey, prop, group,
                this.getIMEMode(tag),
                this.getFormat(tag),
                this.getHorizon(tag),
                this.getVertical(tag),
                this.getValidator(tag)
        );
    }
    
    //<select>
    private void selectTag(Object target, Element tag) throws Exception {
        String fieldName = getTagProp(tag, "field");
        if( fieldName == null )throw new Exception("<field>要素の存在しないタグが発見されました。:<select>");
        
        String destKey = getTagProp(tag, "destKey");
        String destValueProp = getTagProp(tag, "destValueProp");
        String destIdProp = getTagProp(tag, "destIdProp");
        String srcKey = getTagProp(tag, "srcKey");
        String srcDisplayProp = getTagProp(tag, "srcDisplayProp");
        String srcValueProp = getTagProp(tag, "srcValueProp");
        String srcIdProp = getTagProp(tag, "srcIdProp");
        String group = getTagProp(tag, "group");
        
        this.bridge.bindSelect(fieldName, group,
                destKey, destValueProp, destIdProp,
                srcKey, srcDisplayProp, srcValueProp, srcIdProp,
                this.getFormat(tag),
                this.getHorizon(tag),
                this.getVertical(tag),
                this.getValidator(tag)
        );
    }
    
    //<custom>
    private void customTag(Object target, Element tag) throws Exception {
        String fieldName = getTagProp(tag, "field");
        if( fieldName == null )throw new Exception("<field>要素の存在しないタグが発見されました。:<custom>");
        
        String objectKey = getTagProp(tag, "objectKey");
        String displayProp = getTagProp(tag, "displayProp");
        String idProp = getTagProp(tag, "idProp");
        String group = getTagProp(tag, "group");
        
        this.bridge.bindCustom(fieldName, objectKey, displayProp, idProp, group,
                this.getIMEMode(tag),
                this.getFormat(tag),
                this.getHorizon(tag),
                this.getVertical(tag),
                this.getValidator(tag)
        );
    }
    
    //<table>
    List columns = new ArrayList();//List<BindProperty>
    private void tableTag(Object target, Element tag) throws Exception {        
        NodeList nodeList = tag.getElementsByTagName("column");
        for(int i=0; i<nodeList.getLength(); i++) {
            Element columnTag = (Element)nodeList.item(i);
            
            String displayProp = getTagProp(columnTag, "displayProp");
            String valueProp = getTagProp(columnTag, "valueProp");
            String idProp = getTagProp(columnTag, "idProp");
            String text = getTagProp(columnTag, "text");
            boolean readOnly = this.getTagPropBoolean(columnTag, "readOnly");
            String prop = getTagProp(columnTag, "prop");
            int width = this.getTagPropInt(columnTag, "width");
            
            IMEMode imeMode = this.getIMEMode(columnTag);
            Format format = this.getFormat(columnTag);
            AlignType horizon = this.getHorizon(columnTag);
            AlignType vertical = this.getVertical(columnTag);
            Validator validator = this.getValidator(columnTag);
            CellEditor cellEditor = this.getCellEditor(columnTag);
            CellRenderer cellRenderer = this.getCellRenderer(columnTag);
            
            BindProperty col = this.bridge.bindColumn(displayProp, valueProp, idProp, text, 
                    readOnly, prop, width, imeMode, format, 
                    horizon, vertical, validator, cellEditor, cellRenderer
            );
            columns.add(col);
        }
        
        String fieldName = getTagProp(tag, "field");
        if( fieldName == null )throw new Exception("<field>要素の存在しないタグが発見されました。:<table>");
        
        String objectKey = getTagProp(tag, "objectKey");
        String group = getTagProp(tag, "group");
        int rowHeight = this.getTagPropInt(tag, "height");
        boolean sortable = Boolean.parseBoolean( getTagProp(tag, "sortable") );
        EditListener editListener = this.getEditListener(tag);
        
        this.bridge.bindTable(fieldName, objectKey, group, rowHeight, 
                sortable, editListener, 
                (BindProperty[])columns.toArray(new BindProperty[]{})
        );
    }

}

