/*!
  \file
  \brief VAڑ (Linux)

  \author Satofumi KAMIMURA

  $Id$
*/

#include "SerialCtrl.h"
#include "RingBuffer.h"
#include <sys/poll.h>
#include <termios.h>
#include <fcntl.h>
#include <errno.h>
#include <string>


/*!
  \brief SerialCtrl ̓NX
*/
struct SerialCtrl::pImpl {
  std::string error_message;
  int fd;
  struct termios sio;           /*!< ʐM */
  struct pollfd nfds;           /*!< ^CAEg */
  RingBuffer<char> ring_buffer;

  pImpl(void) : error_message("not connected."), fd(-1) {
  }

  ~pImpl(void) {
    disconnect();
  }

  void disconnect(void) {
    if (fd >= 0) {
      close(fd);
      fd = -1;
    }
    ring_buffer.clear();
    error_message = "disconnected.";
  }

  int changeBaudrate(long baudrate) {
    long baudrate_value = -1;
    switch (baudrate) {

    case 9600:
      baudrate_value = B9600;
      break;

    case 19200:
      baudrate_value = B19200;
      break;

    case 38400:
      baudrate_value = B38400;
      break;

    case 57600:
      baudrate_value = B57600;
      break;

    case 115200:
      baudrate_value = B115200;
      break;

    default:
      char* message = new char [80 + 12];
      sprintf(message, "Invalid baudrate value: %ld", baudrate);
      error_message = message;
      delete [] message;
      return -1;
      break;
    }

    cfsetospeed(&sio, baudrate_value);
    cfsetispeed(&sio, baudrate_value);
    tcsetattr(fd, TCSADRAIN, &sio);

    return 0;
  }

  void update(int timeout, int size = 0) {

    enum { BufferSize = 1024 };
    int require_size = (size > 0) ? size : BufferSize;
    int filled = 0;
    while (filled < require_size) {
      if (poll(&nfds, 1, (size == 0) ? 0 : timeout) <= 0) {
	break;			// timeout
      }
      char data[BufferSize];
      int read_n = require_size - filled;
      int n = read(fd, data, (read_n > BufferSize) ? BufferSize : read_n);
      if (n <= 0) {
	return;
      }
      ring_buffer.put(data, n);
      filled += n;
    }
  }
};


SerialCtrl::SerialCtrl(void) : pimpl(new pImpl) {
}


SerialCtrl::~SerialCtrl(void) {
}


const char* SerialCtrl::what(void) {
  return pimpl->error_message.c_str();
}


int SerialCtrl::connect(const char* device, long baudrate) {

  // foCXɐڑ
  int fd = open(device, O_RDWR);
  if (fd < 0) {
    enum { BufferSize = 80 };
    char buffer[BufferSize];
    char* p = strerror_r(errno, buffer, BufferSize);
    pimpl->error_message = std::string(device) + " : " + p;
    return fd;
  }
  pimpl->fd = fd;

  // VAݒ
  tcgetattr(fd, &pimpl->sio);
  pimpl->sio.c_iflag = IGNPAR;
  pimpl->sio.c_oflag = 0;
  pimpl->sio.c_cflag = CS8 | CREAD | CLOCAL;
  pimpl->sio.c_lflag = 0;

  pimpl->sio.c_cc[VMIN] = 0;
  pimpl->sio.c_cc[VTIME] = 0;
  tcsetattr(fd, TCSAFLUSH, &pimpl->sio);
  pimpl->ring_buffer.clear();

  // ^CAEgݒ
  pimpl->nfds.fd = fd;
  pimpl->nfds.events = POLLIN | POLLPRI | POLLERR | POLLHUP | POLLNVAL;
  pimpl->nfds.revents = 0;

  // {[[gݒ
  int ret = pimpl->changeBaudrate(baudrate);
  if (ret < 0) {
    return ret;
  }

  pimpl->error_message = "no error";
  return 0;
}


void SerialCtrl::disconnect(void) {
  pimpl->disconnect();
}


bool SerialCtrl::isConnected(void) {
  return (pimpl->fd < 0) ? false : true;
}


int SerialCtrl::changeBaudrate(long baudrate) {
  return pimpl->changeBaudrate(baudrate);
}


int SerialCtrl::send(const char* data, int size) {
  if (isConnected() == false) {
    return 0;
  }
  return write(pimpl->fd, data, size);
}


int SerialCtrl::recv(char* data, int size, int timeout) {
  if ((isConnected() == false) || (size <= 0)) {
    return 0;
  }

  // vMf[^΁AԂ
  int filled = SerialCtrl::size();
  if (filled >= size) {
    pimpl->ring_buffer.get(data, size);
    return size;
  }

  // Młf[^Ԃ
  int left = size - filled;
  pimpl->update(timeout, left);

  // size() ĂԂ update(0) Ă΂邽
  filled = pimpl->ring_buffer.size();
  pimpl->ring_buffer.get(data, filled);

  return filled;
}


int SerialCtrl::size(void) {
  pimpl->update(0);
  return pimpl->ring_buffer.size();
}


void SerialCtrl::clear(void) {
  pimpl->update(0);
  pimpl->ring_buffer.clear();
}
