/*!
  \file
  \brief VAڑ (Windows)

  \author Satofumi KAMIMURA
  \author Tomoaki YOSHIDA

  $Id$
*/

#include "SerialCtrl.h"
#include "RingBuffer.h"
#include <errno.h>
#include <string>
#include <windows.h>


/*!
  \brief SerialCtrl ̓NX
*/
struct SerialCtrl::pImpl {
  std::string error_message;
  RingBuffer<char> ring_buffer;
  HANDLE hComm;

  pImpl(void) : error_message("Not connected."), hComm(INVALID_HANDLE_VALUE) {
  }

  ~pImpl(void) {
    disconnect();
  }

  void disconnect(void) {
    if (hComm != INVALID_HANDLE_VALUE) {
      CloseHandle(hComm);
      hComm = INVALID_HANDLE_VALUE;
    }
  }

  int changeBaudrate(long baudrate) {

    DCB dcb;
    GetCommState(hComm, &dcb);
    dcb.BaudRate = baudrate;
    dcb.ByteSize = 8;
    dcb.Parity = NOPARITY;
    dcb.fParity = FALSE;
    dcb.StopBits = ONESTOPBIT;
    SetCommState(hComm, &dcb);

    return 0;
  }

  void update(int timeout, int size = 0) {
    enum { BufferSize = 1024 };
    int require_size = (size > 0) ? size : BufferSize;

    if (timeout > 0) {
      COMMTIMEOUTS pcto;
      GetCommTimeouts(hComm, &pcto);
      pcto.ReadIntervalTimeout = 0;
      pcto.ReadTotalTimeoutConstant = timeout;
      pcto.ReadTotalTimeoutMultiplier = 0;
      SetCommTimeouts(hComm, &pcto);

    } else {
      DWORD dwErrors;
      COMSTAT ComStat;
      ClearCommError(hComm, &dwErrors, &ComStat);
      require_size = ((int)ComStat.cbInQue > require_size)
	? require_size : ComStat.cbInQue;
    }
    char data[BufferSize];
    int read_n = (require_size > BufferSize) ? BufferSize : require_size;
    DWORD n;
    ReadFile(hComm, data, read_n, &n, NULL);
    ring_buffer.put(data, n);
  }
};


SerialCtrl::SerialCtrl(void) : pimpl(new pImpl) {
}


SerialCtrl::~SerialCtrl(void) {
}


const char* SerialCtrl::what(void) {
  return pimpl->error_message.c_str();
}


int SerialCtrl::connect(const char* device, long baudrate) {

  // ڑ
  pimpl->hComm = CreateFile(device, GENERIC_READ | GENERIC_WRITE, 0, NULL,
			    OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (pimpl->hComm == INVALID_HANDLE_VALUE) {
    LPVOID lpMsg;
    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
		  FORMAT_MESSAGE_FROM_SYSTEM |
		  FORMAT_MESSAGE_IGNORE_INSERTS,
		  NULL, GetLastError(),
		  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		  (LPTSTR)&lpMsg, 0, NULL);
    char buffer[256];
    sprintf(buffer, "port(%s) open failed: %s\n", device, (char*)lpMsg);
    pimpl->error_message = buffer;
    LocalFree(lpMsg);
    return -1;
  }

  // {[[gݒ
  int ret = pimpl->changeBaudrate(baudrate);
  if (ret < 0) {
    return ret;
  }
  pimpl->error_message = "no error";

  return 0;
}


void SerialCtrl::disconnect(void) {
  pimpl->disconnect();
}


bool SerialCtrl::isConnected(void) {
  return (pimpl->hComm == INVALID_HANDLE_VALUE) ? false : true;
}


int SerialCtrl::changeBaudrate(long baudrate) {
  return pimpl->changeBaudrate(baudrate);
}


int SerialCtrl::send(const char* data, int size) {
  DWORD n;
  WriteFile(pimpl->hComm, data, size, &n, NULL);
  return n;
}


int SerialCtrl::recv(char* data, int size, int timeout) {
  if ((isConnected() == false) || (size <= 0)) {
    return 0;
  }

  // vMf[^΁AԂ
  int filled = SerialCtrl::size();
  if (filled >= size) {
    pimpl->ring_buffer.get(data, size);
    return size;
  }

  // Młf[^Ԃ
  int left = size - filled;
  pimpl->update(timeout, left);

  // size() ĂԂ update(0) Ă΂邽
  filled = pimpl->ring_buffer.size();
  pimpl->ring_buffer.get(data, filled);

  return filled;
}


int SerialCtrl::size(void) {
  pimpl->update(0);
  return pimpl->ring_buffer.size();
}


void SerialCtrl::clear(void) {
  pimpl->update(0);
  pimpl->ring_buffer.clear();
}


void SerialCtrl::skip(int timeout) {
  clear();

  char buffer[BUFSIZ];
  int total = 0;
  int n;

  enum { EachTimeout = 10 };
  do {
    n = recv(buffer, BUFSIZ, EachTimeout);
    total += EachTimeout;
  } while ((n > 0) && ((timeout > 0) && (total < timeout)));
}
