/*******************************************************************************
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.engine.simple;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.lang.StringUtils;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;
import benten.core.BentenConstants;
import benten.core.io.Files;
import blanco.tmx.BlancoTmxParser;
import blanco.tmx.valueobject.BlancoTmx;
import blanco.tmx.valueobject.BlancoTmxTu;
import blanco.tmx.valueobject.BlancoTmxTuv;

/**
 * このクラスは、シンプル TM エンジン・クラスです。
 * 
 * @author IGA Tosiki
 * @author YAMAMOTO Koji
 */
public class SimpleTmEngine implements BentenTmEngine {
	/**
	 * 翻訳元の言語。
	 */
	protected String fSourceLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳先の言語。
	 */
	protected String fTargetLang = "ja-JP";//$NON-NLS-1$

	/**
	 * 翻訳言語を設定。
	 * 
	 * @param source 翻訳元の言語。例: en-US。
	 * @param target 翻訳先の言語。例: ja-JP。
	 */
	public void setLang(final String source, final String target) {
		fSourceLang = source;
		fTargetLang = target;
	}

	/**
	 * TM データのリスト。
	 */
	private List<TmTuvEntry> fListTuvEntry = Collections.synchronizedList(new ArrayList<TmTuvEntry>());

	/**
	 * 最大検索数。
	 * 
	 */
	private static final int MAX_RESULT_SIZE = 100;

	private BenteTmComparator<BentenTmSearchResult> bentenTmComparator = new BenteTmComparator<BentenTmSearchResult>();;

	/**
	 * {@inheritDoc}
	 */
	public void loadTmx(final File dirTmx) throws IOException {
		final File[] fileTmxs = dirTmx.listFiles();
		if (fileTmxs == null) {
			throw new IllegalArgumentException("The pathname does not denote directory: " + dirTmx.toString());
		}

		for (int index = 0; index < fileTmxs.length; index++) {
			final File fileTmx = fileTmxs[index];
			if (fileTmx.isFile() == false) {
				continue;
			}
			if (fileTmx.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_TMX) == false) {
				// ファイルの拡張子が .tmx のものだけ処理します。
				continue;
			}

			final BlancoTmxParser parser = new BlancoTmxParser();
			final BlancoTmx tmx = parser.parse(fileTmx);

			for (BlancoTmxTu tu : tmx.getBody().getTuList()) {
				final TmTuvEntry entry = new TmTuvEntry();

				final List<String> sources = new ArrayList<String>();
				final List<String> targets = new ArrayList<String>();
				for (BlancoTmxTuv tuv : tu.getTuvList()) {
					// TODO 現在は、アンダースコアは対応していない。
					if (tuv.getLang().toLowerCase().startsWith(fSourceLang.toLowerCase())) {
						// 先頭に追加。
						sources.add(0, tuv.getSeg());
					} else if (tuv.getLang().toLowerCase().equals(getLanguageFromLocale(fSourceLang.toLowerCase()))) {
						// 末尾に追加。
						sources.add(tuv.getSeg());
					} else if (tuv.getLang().toLowerCase().startsWith(fTargetLang.toLowerCase())) {
						// 先頭に追加。
						targets.add(0, tuv.getSeg());
					} else if (tuv.getLang().toLowerCase().equals(getLanguageFromLocale(fTargetLang.toLowerCase()))) {
						targets.add(tuv.getSeg());
						// 末尾に追加。
					}
				}
				if (sources.size() == 0) {
					continue;
				}
				if (targets.size() == 0) {
					continue;
				}

				fListTuvEntry.add(entry);
				entry.source = sources.get(0);
				entry.sourceLower = sources.get(0).toLowerCase();
				entry.target = targets.get(0);
				entry.origin = Files.baseName(fileTmx);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public List<BentenTmSearchResult> fuzzySearch(final String target) {
		final List<BentenTmSearchResult> result = new ArrayList<BentenTmSearchResult>();
		for (int indexTm = 0; indexTm < fListTuvEntry.size(); indexTm++) {
			final TmTuvEntry tmEntry = fListTuvEntry.get(indexTm);

			int levenshteinDistance = StringUtils.getLevenshteinDistance(target, tmEntry.source);
			int targetLength = Math.max(target.length(), tmEntry.source.length());

			int quality = ((targetLength - levenshteinDistance) * 100) / targetLength;

			final BentenTmSearchResult tmSearchResult = new BentenTmSearchResult();
			tmSearchResult.setSource(tmEntry.source);
			tmSearchResult.setTarget(tmEntry.target);
			tmSearchResult.setOrigin(tmEntry.origin);
			tmSearchResult.setMatchQuality(Integer.toString(quality) + "%"); //$NON-NLS-1$

			if (result.size() >= MAX_RESULT_SIZE) {
				final BentenTmSearchResult currentLastResult = result.get(result.size() - 1);
				if (bentenTmComparator.compare(tmSearchResult, currentLastResult) > 0) {
					continue;
				}
				result.add(tmSearchResult);
				Collections.sort(result, bentenTmComparator);
				result.remove(result.size() - 1);
			} else {
				result.add(tmSearchResult);
			}

		}

		Collections.sort(result, bentenTmComparator);

		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	public void unload() {
		fListTuvEntry.clear();
	}

	/**
	 * ロケール文字列から言語部分を抽出。
	 * @param locale en-US など。
	 * @return 言語部分。en など。
	 */
	String getLanguageFromLocale(final String locale) {
		StringTokenizer tok = new StringTokenizer(locale, "-_");
		return tok.nextToken();
	}
}

/**
 * BentenTmSearchResult 用コンパレーター
 * 
 * @param <T>
 *            BentenTmSearchResult 型
 * 
 */
class BenteTmComparator<T> implements Comparator<BentenTmSearchResult> {

	public int compare(final BentenTmSearchResult tuLeft, final BentenTmSearchResult tuRight) {
		String left = tuLeft.getMatchQuality().trim();
		String right = tuRight.getMatchQuality().trim();
		if (left.endsWith("%")) { //$NON-NLS-1$
			left = left.substring(0, left.length() - 1);
		}
		if (right.endsWith("%")) { //$NON-NLS-1$
			right = right.substring(0, right.length() - 1);
		}
		return Integer.parseInt(right) - Integer.parseInt(left);
	}
}

/**
 * TM の tuv エントリーをあらわすクラス。
 */
class TmTuvEntry {
	/** ソースを小文字化したもの。 */
	public String sourceLower;
	/** ソース。 */
	public String source;
	/** ターゲット。 */
	public String target;
	/** 翻訳メモリーの由来。 */
	public String origin;
}