/*
**  bif_m.h
**  bif-c
**
**  Created by Joel Rees on 2009/07/22.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIF/M, as mechanically as possible.
**
*/

#if !defined BIF_M_H
#define BIF_M_H

#include <stddef.h>


#include "bifu_i.h"
#include "bifdp_a.h"


/* The FORTH standard values for storing TRUE and FALSE:
*/
#define	TRUE	(-1L)
#define	FALSE	0L
/* But for testing, any non-zero is true.
*/
#define isTRUE( b )	( (b) != FALSE )


/* The assembler liked to have macros in a separate file. 
** Most of this file probably belongs logicaly in bifdp_a.h .
** The object header definitions should eventually move to the symbol table.
*/


/*
00010 * EQUates and MACRO definitions for BIF
00020 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
01000 *
*/

/** NEXT will be emulated by returning to the vm interpret loop.
NEXT    ( --- )         jmp [,y++] (macro in bif.m)
        Causes the next definition to execute.
**
01010 * JMP to next icode word
01020 NEXT	MACRO
** Even when hand-constructing in assembly language, 
** we will not be using optimizations like this,
** only a return to the interpret loop or whatever called the routine definition.
** While it slows virtual code down, it allows better optimization of interactions 
** with stuff in other languages.
01030 	JMP [,Y++]
01040 	ENDM
*/
/* If we defined NEXT, here's what it would look like:
#define NEXT return
*/
/* 01050 * */


/* Again, the question of whether to do this with defines or C structs,
** but this one defines negative offsets, so the question is a little more 
** difficult. Still, I don't really want to spend time fighting with C.
** Eventually, I expect to cut this up into logical elements, 
** and maybe switch to ordinary positive offsets.
** Relative links or absolute?
** C pointers in the current standard really do not permit relative links.
** (It's the relocating loader having to patch links at load time for absolute-only CPUs.)
** If I want to use relative links, it must be within an array of bytes that I manage,
** whether the links are relative to a base or to the linked objects.
** If I want to use C (low level) addressing, I'm stuck with absolute.
** I could mix relative links between symbols with absolute links to low-level code, 
** I suppose.
** But then I would have to allocate UP myself, which is not necessarily evil.
** Or I could let the symbol table be absolute addressed in the C version
** and remind the implementer to use relative all through for the native implementation,
** where that's indicated.
**
** The symbol table may want to exist in its own space anyway, 
** to avoid the difficulties of C initialization math.
** In that case, I have to build the initial symbol table in COLD 
** (or maybe re-build it in WARM).
** If I'm building the table, I guess I can calculate the offset,
** the way I did it in the 6809 version.
*/

/* Moved typedef struct definition_header_s to bifu_i.h .
*/

/* 01053 * offset from name field to mode field
** 01056 MFAOFF	EQU 1 
*/
#define IFAOFF ( offsetof( definition_header_s, interpMode ) )	/* Interpretation mode only, needs a definition. */
#define MFAOFF ( offsetof( definition_header_s, allocMode ) )	/* Allocation mode only, needs a name change. */

/* 01057 * length of symbols, must be 2^n for mask
** 01058 NLMASK	EQU $1F 
*/
#define NLMASK 0x1f	/* max length 31 byte_t. (For now. Could use bit fields, but not yet.) */
#define NameLength_k ( NLMASK + 1 )

/* 01060 * offset to previous allocation link
** 01070 LFAOFF	EQU MFAOFF+1 
*/
#define LFAOFF ( offsetof( definition_header_s, allocLink ) )

/* 01080 * offset to owning VOCABULARY (genesis) link
** 01090 GFAOFF	EQU LFAOFF+2
*/
#define GFAOFF ( offsetof( definition_header_s, vocabLink ) )

/* 01100 LFTOFF	EQU GFAOFF+2	left child link */
#define LFTOFF ( offsetof( definition_header_s, leftLink ) )
/* 01110 RTOFF	EQU GFAOFF+4	right child link */
#define RTOFF ( offsetof( definition_header_s, rightLink ) )

/* 01120 * offset to code field
** 01130 CFAOFF	EQU GFAOFF+6
*/
#define CFAOFF ( offsetof( definition_header_s, codeLink ) )

/* 01140 * offset to parameter field (past Direct Page jump)
** 01150 PFAOFF	EQU CFAOFF+2 
*/
#define PFAOFF ( offsetof( definition_header_s, parameterLink ) )

/*
01160 * All links are relative to the name field (NFA)
01170 * Use PFAOFF (or PFA) with ;CODE!
** In my 6809 implementation, the first parameter was machine code.
** In leaf definitions, the machine code continued to the end.
** In non-leaf ("high-level") definitions, the machine code was a 
** subroutine call to the characteristic code.
** By using a subroutine call to the characteristic, 
** I automatically stored the pointer to the bytes following on the return stack,
** where the characteristic could use it as the actual parameter field pointer.
** In the initial C version, everything will be as if colon defined, I think.
** More specifically, the first parameter cell will be a pointer to the characteristic function,
** which will probably not even be in the same segment.
** Actual parameters will follow that.
** Eventually, I will want to put variable parameters in a different segment, as well.
01320 * usage and name length share a byte (name length < 32).
01330 * foreward blocks marked in the mode byte
01340 * usage and mode bits
*/

/* 01350 MFORE	EQU $80	part of FORWARD reference block (in mode byte -- MFA) */
#define MFORE	0x80
/* 01360 MCOMP	EQU $80	compile only (in length byte -- NFA) */
#define MCOMP	0x80
/* 01370 MIMM	EQU $40	precedence is IMMEDIATE execution (in length byte -- NFA) */
#define MIMM	0x40
/* 01380 MHID	EQU $20	SMUDGEd definition (in length byte -- NFA) */
#define MHID	0x20
/* 01390 * */
#define BOOT_VOC_MARK	1	/* To mark the initial vocabulary, to help prune. */
#define VOC_MEMO_MARK	2	/* To mark where we've been when pruning. */


/*
DOCOL   ( *** IP )      jsr <XCOL (see bif.m, bifdp.a)
        Characteristic of a colon (:) definition.  Begins execution of a
        high-level definition, i. e., nests the definition and begins
        processing icodes.  Mechanically, it pushes the IP (Y register)
        and loads the Parameter Field Address of the definition which
        called it into the IP.
**
01400 * initiate an icode list
01410 DOCOL	MACRO
01420 	JSR <XCOL
01430 	ENDM
*/
#define DOCOL	XCOL	/* maybe. We'll see when we get there. */
/* 01440 * */

/*
DOVAR   ( --- vadr )    jsr <XVAR (bif.m, bifdp.a)
        Characteristic of a VARIABLE.  A VARIABLE pushes its PFA address
        on the stack.  The parameter field of a VARIABLE is the actual
        allocation of the variable, so that pushing its address allows
        its contents to be @ed (fetched).  Ordinary arrays and strings
        that do not subscript themselves may be allocated by defining a
        variable and immediately ALLOTting the remaining space.
        VARIABLES are global to all users, and thus should have been
        hidden in resource monitors, but aren't.
**
01450 DOVAR	MACRO
01460 	JSR <XVAR
01470 	ENDM
*/
#define DOVAR	XVAR
/* 01480 * */

/*
DOCON   ( --- n )       jsr <XCON (bif.m, bifdp.a)
        Characteristic of a CONSTANT.  A CONSTANT simply loads its value
        from its parameter field and pushes it on the stack.
**
01490 DOCON	MACRO
01500 	JSR <XCON
01510 	ENDM
*/
#define DOCON	XCON
/* 01520 * */

/*
DOUSER  ( --- vadr )    jsr <XUSER (bif.m, bifdp.a)
        Characteristic of a per-USER variable. USER variables are
        similiar to VARIABLEs, but are allocated (by hand!) in the
        per-user table.  A USER variable's parameter field contains its
        offset in the per-user table.
**
01530 DOUSER	MACRO
01540 	JSR <XUSER
01550 	ENDM
*/
#define	DOUSER	XUSER
/* 01560 * */

/* non-standard:
DO1ARR  ( index --- eadr )      jsr <X1ARR (bif.m, bifdp.a)
        Characteristic of a linear array. Linear arrays take the top
        word on the stack as an index to the array, and return the
        address of the element indexed.  So this routine subtracts the
        base index of the array, limit checks the result, then
        multiplies by the size of the array elements.  If the index is
        out of bounds, it returns a NULL pointer (0).  At some point I
        intended to implement multi-dimensional arrays in a similar
        manner, but I haven't.  It would be a quick and interesting
        project for anyone interested.
**
01570 DO1ARR	MACRO
01580 	JSR <X1ARR
01590 	ENDM
*/
#define	DO1ARR	X1ARR
/* 01600 * */

/* non-standard:
DOUCON  ( --- n )       jsr <XUCON (bif.m, bifdp.a)
        Characteristic of a USER variable treated as a CONSTANT, i. e.,
        fetches the value stored at the specified offset in the per-user
        table.
**
01610 DOUCON	MACRO
01620 	JSR <XUCON
01630 	ENDM
*/
#define	DOUCON	XUCON
/* 01640 * */

/* non-standard, no DODCON header in the 6809 source code:
        ( --- d )       jsr <XDCON (bifdp.a)
        Characteristic of a double integer constant; the parameter field
        contains two words instead of one, both of which get pushed.
*/

/*
DOVOC   ( --- )         jsr <XVOC (bif.m, bifdp.a)
        Characteristic of a VOCABULARY.  A VOCABULARY stores a pointer
        to itself in the current interpretation ROOT per-USER variable.
        It contains a pointer to the definition at the root of its
        symbol table tree.  This allows the symbol table routines to
        treat the root as a leaf node.  This is also not standard FORTH!
**
01650 DOVOC	MACRO
01660 	JSR <XVOC
01670 	ENDM
*/
#define	DOVOC	XVOC
/* 01680 * */

/* (No DODOES header in the 6809 source code.)
        ( --- PFA )     ( *** IP )      jsr <XDOES (routine in bifdp.a)
        Characteristic of a DOES> defined word.  The characteristics of
        DOES> definitions are written in high-level icodes rather than
        machine level code. The first parameter word points to the
        high-level characteristic.  This routine's job is to push the
        IP, load the high level characteristic pointer in IP, and leave
        the address following the characteristic pointer on the stack so
        the parameter field can be accessed.
*/

/* non-standard (but often in fig sources):
ADDTOP  (MACRO in BIF.M) is not a characteristic; is used in several
        routines to add a value to the top of stack.
**
02000 * add D to top of stack
02010 ADDTOP	MACRO
02020 	ADDD ,U
02030 	STD ,U
02040 	JMP [,Y++]
02050 	ENDM
** This is going to have side effects that need to be watched.
** Probably don't really want to use it, anyway?
** We'll see.
** Note that, like the rest of this file, this belongs elsewhere. 
** (Probably BIFDP/A => bifdp_a.h.)
*/
#define ADDTOP( D )	( ( * SP ).integer += (D) )
#define SUBTOP( D )	( ( * SP ).integer -= (D) )

/* 02060 * */


#endif /* !defined BIF_M_H */
