;;;
;;; binary.struct
;;;
;;;  Copyright (c) 2007 Shinya Ando <shinya_ando@users.sourceforge.jp>
;;;
;;;  Redistribution and use in source and binary forms, with or without
;;;  modification, are permitted provided that the following conditions
;;;  are met:
;;;
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;
;;;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;
;;; $Id: struct.scm 35 2007-04-27 13:08:41Z shinya $

(define-module binary.struct
  (use binary.io)
  (use srfi-1)
  (use srfi-13)
  (use gauche.uvector)
  (use gauche.sequence)
  (export <bs:type> bs:make-type
          bs:reader-of bs:writer-of
          
          <bs:error>
          <bs:read-error> <bs:eof-error>
          <bs:marshal-error>
          <bs:field-not-found>
          
          bs:cstring bs:block
          bs:array bs:struct
          bs:struct-ref bs:struct-ref*
          bs:struct-set!
          bs:check bs:const
          bs:or
          bs:many bs:many2

          <bs:ra-input-port>
          ))

(select-module binary.struct)

;; ------------------------------------------------------------------
;; conditions

(define-condition-type <bs:error> <error> #f)

(define-condition-type <bs:read-error> <bs:error> #f)

(define-condition-type <bs:eof-error> <bs:read-error> #f)

(define-condition-type <bs:marshal-error> <bs:error> #f)

(define-condition-type <bs:field-not-found> <bs:error> #f)

;; ------------------------------------------------------------------
;; Type is a pair of reader and writer.
;;
;; Reader is a procedure takes one argument, which is input port.
;; Writer is a procedure takes two arguments, output data and output port.

(define-class <bs:type> ()
  ((reader :init-keyword :reader
           :init-value   #f)
   (writer :init-keyword :writer
           :init-value   #f)
   ))

(define (bs:make-type reader writer)
  (make <bs:type> :reader reader :writer writer))

(define (bs:reader-of type)
  (slot-ref (prepare-type type) 'reader))

(define (bs:writer-of type)
  (slot-ref (prepare-type type) 'writer))

(define (bs:size-of type)
  (slot-ref (prepare-type type) 'size))

(define (prepare-type type)
  (cond
   ((is-a? type <bs:type>)
    type)
   ((symbol? type)
    (or (hash-table-get type-alias-table type #f)
        (errorf <bs:error> "type alias ~a is not defined" type)))
   (else
    (errorf <bs:error> "bs:type required but got ~a" type))))

;; ------------------------------------------------------------------
;; alias of primitive type

(define (eof-check reader)
  (lambda (in)
    (let1 val (reader in)
      (if (eof-object? val)
          (errorf <bs:eof-error>)
          val))))

;; FIXME
(define (argument-check writer)
  (lambda (data out)
    (guard (exc
            ((<io-error> exc) (raise exc))
            ((<error> exc)
             (errorf <bs:marshal-error> (slot-ref exc 'message))))
           (writer data out))))

(define type-alias-table
  (hash-table
   'eq?

   (cons 'u8    (bs:make-type (eof-check read-u8) (argument-check write-u8)))
   (cons 'u16   (bs:make-type (eof-check read-u16) (argument-check write-u16)))
   (cons 'u32   (bs:make-type (eof-check read-u32) (argument-check write-u32)))
   (cons 'u64   (bs:make-type (eof-check read-u64) (argument-check write-u64)))
   (cons 'u16le (bs:make-type (eof-check (cut read-u16 <> 'little-endian))
                              (argument-check (cut write-u16 <> <> 'little-endian))))
   (cons 'u32le (bs:make-type (eof-check (cut read-u32 <> 'little-endian))
                              (argument-check (cut write-u32 <> <> 'little-endian))))
   (cons 'u64le (bs:make-type (eof-check (cut read-u64 <> 'little-endian))
                              (argument-check (cut write-u64 <> <> 'little-endian))))
   (cons 'u16be (bs:make-type (eof-check (cut read-u16 <> 'big-endian))
                              (argument-check (cut write-u16 <> <> 'big-endian))))
   (cons 'u32be (bs:make-type (eof-check (cut read-u32 <> 'big-endian))
                              (argument-check (cut write-u32 <> <> 'big-endian))))
   (cons 'u64be (bs:make-type (eof-check (cut read-u64 <> 'big-endian))
                              (argument-check (cut write-u64 <> <> 'big-endian))))
   
   (cons 's8    (bs:make-type (eof-check read-s8) (argument-check write-s8)))
   (cons 's16   (bs:make-type (eof-check read-s16) (argument-check write-s16)))
   (cons 's32   (bs:make-type (eof-check read-s32) (argument-check write-s32)))
   (cons 's64   (bs:make-type (eof-check read-s64) (argument-check write-s64)))
   (cons 's16le (bs:make-type (eof-check (cut read-s16 <> 'little-endian))
                              (argument-check (cut write-s16 <> <> 'little-endian))))
   (cons 's32le (bs:make-type (eof-check (cut read-s32 <> 'little-endian))
                              (argument-check (cut write-s32 <> <> 'little-endian))))
   (cons 's64le (bs:make-type (eof-check (cut read-s64 <> 'little-endian))
                              (argument-check (cut write-s64 <> <> 'little-endian))))
   (cons 's16be (bs:make-type (eof-check (cut read-s16 <> 'big-endian))
                              (argument-check (cut write-s16 <> <> 'big-endian))))
   (cons 's32be (bs:make-type (eof-check (cut read-s32 <> 'big-endian))
                              (argument-check (cut write-s32 <> <> 'big-endian))))
   (cons 's64be (bs:make-type (eof-check (cut read-s64 <> 'big-endian))
                              (argument-check (cut write-s64 <> <> 'big-endian))))

   (cons 'ber-integer (bs:make-type (eof-check read-ber-integer)
                                    (argument-check write-ber-integer)))

   (cons 'f16   (bs:make-type (eof-check read-f16) (argument-check write-f16)))
   (cons 'f32   (bs:make-type (eof-check read-f32) (argument-check write-f32)))
   (cons 'f64   (bs:make-type (eof-check read-f64) (argument-check write-f64)))
   (cons 'f16le (bs:make-type (eof-check (cut read-f16 <> 'little-endian))
                              (argument-check (cut write-f16 <> <> 'little-endian))))
   (cons 'f32le (bs:make-type (eof-check (cut read-f32 <> 'little-endian))
                              (argument-check (cut write-f32 <> <> 'little-endian))))
   (cons 'f64le (bs:make-type (eof-check (cut read-f64 <> 'little-endian))
                              (argument-check (cut write-f64 <> <> 'little-endian))))
   (cons 'f16be (bs:make-type (eof-check (cut read-f16 <> 'big-endian))
                              (argument-check (cut write-f16 <> <> 'big-endian))))
   (cons 'f32be (bs:make-type (eof-check (cut read-f32 <> 'big-endian))
                              (argument-check (cut write-f32 <> <> 'big-endian))))
   (cons 'f64be (bs:make-type (eof-check (cut read-f64 <> 'big-endian))
                              (argument-check (cut write-f64 <> <> 'big-endian))))
   ))

;; ------------------------------------------------------------------
;; basic structure types

;;
;; null terminated string
;;

(define (bs:cstring . opts)
  (define cstr-any-size
    (bs:make-type
     (lambda (in)
       (let1 buf (open-output-string)
         (let loop ()
           (let1 b (read-byte in)
             (if (eof-object? b)
                 (errorf <bs:eof-error> "got eof while reading cstring"))
             (if (= b 0)
                 (get-output-string buf)
                 (begin
                   (write-byte b buf)
                   (loop)))))))
     (lambda (data out)
       (write-block (string->u8vector data) out)
       (write-byte 0 out))
     ))
  
  (define (cstr-size-specified size must-terminate?)
    (bs:make-type
     (lambda (in)
       (let1 block
           (let loop ((str-acc ""))
             (let1 rest-size (- size (string-size str-acc))
               (if (= rest-size 0)
                   str-acc
                   (let1 str (read-block rest-size in)
                     (if (eof-object? str)
                         (errorf <bs:eof-error> "got eof while reading cstring"))
                     (loop (string-append str-acc str))))))
         (let1 str (or (string-scan block #\null 'before)
                       (if must-terminate?
                           (errorf <bs:read-error> "cstring requires null terminator")
                           block))
           (or (string-incomplete->complete str)
               str))
         ))
     (lambda (str out)
       (let1 str-size (string-size str)
         (if (> str-size size)
             (errorf <bs:marshal-error>
                     "string size (~d) larger than specified size (~d)"
                     str-size size))
         (if (and must-terminate? (= str-size size))
             (errorf <bs:marshal-error>
                     "no space for null terminater"))
         (write-block (string->u8vector str) out)
         (let loop ((rest (- size str-size)))
           (when (> rest 0)
             (write-byte 0 out)         ; fill by null
             (loop (- rest 1))))))
     ))
  
  (let-optionals* opts ((size #f)
                        (must-terminate? #t))
    (if size
        (cstr-size-specified size must-terminate?)
        cstr-any-size)))

;;
;; u8vector block
;;

(define (bs:block size)
  (bs:make-type
   (lambda (in)
     (if (= size 0)
         #u8()
         (let1 buf (make-u8vector size)
           (let loop ((offset 0))
             (let1 read-size (read-block! buf in offset)
               (if (eof-object? read-size)
                   (errorf <bs:eof-error> "got eof while reading block"))
               (if (= size (+ offset read-size))
                   buf
                   (loop (+ offset read-size))))))))
   (lambda (data out)
     (let* ((u8vec (uvector-alias <u8vector> data))
            (vec-size (u8vector-length u8vec)))
       (if (not (= size vec-size))
           (errorf <bs:marshal-error>
                   "block size (~d) mismatch specified size (~d)"
                   vec-size size))
       (write-block u8vec out)))
   ))

;;
;; array
;;

(define (bs:array size subtype)
  (bs:make-type
   (lambda (in)
     (let each-element ((rest size)
                        (result '()))
       (if (= rest 0)
           (reverse result)
           (let1 value ((bs:reader-of subtype) in)
             (if (eof-object? value)
                 (errorf <bs:eof-error> "got eof while reading array"))
             (each-element (- rest 1)
                           (cons value result))))))
   (lambda (seq out)
     (if (not (list? seq))
         (errorf <bs:marshal-error>
                 "array type requires list but got ~a"
                 seq))
     (let1 seq-size (length seq)
       (if (not (= seq-size size))
           (errorf <bs:marshal-error>
                   "array size (~d) mismatch specified size (~d)"
                   seq-size size)))
     (let1 subtype-writer (bs:writer-of subtype)
       (for-each
        (cut subtype-writer <> out)
        seq)))
   ))

;;
;; struct type
;;

(define-syntax make-struct-reader
  (syntax-rules ()
    ((_ ((name type) ...))
     (lambda (in)
       (let* ((name ((bs:reader-of type) in))
              ...)
         (list (cons (quote name) name)
               ...))))))

(define-syntax make-struct-writer
  (syntax-rules ()
    ((_ ((name type) ...))
     (lambda (data out)
       (guard (exc
               ((<bs:field-not-found> exc)
                (errorf <bs:marshal-error> "struct mismatch")))
              (let ((try-out (open-output-string))
                    (name (bs:struct-ref data (quote name)))
                    ...)
                ((bs:writer-of type) name try-out)
                ...
                (write-block (string->u8vector (get-output-string try-out))
                             out)
                ))))))

(define-syntax bs:struct
  (syntax-rules ()
    ((_ defs)
     (bs:make-type
      (make-struct-reader defs)
      (make-struct-writer defs)
      ))))

(define (bs:struct-ref struct name)
  (let1 pair (assq name struct)
    (if pair
        (cdr pair)
        (errorf <bs:field-not-found> "filed not found ~a" name))))

(define (bs:struct-set! struct name value)
  (let1 pair (assq name struct)
    (if pair
        (set-cdr! pair value)
        (errorf <bs:field-not-found> "filed not found ~a" name))))

(define (bs:struct-ref* struct . fields)
  (if (null? fields)
      struct
      (apply bs:struct-ref*
             (bs:struct-ref struct (car fields))
             (cdr fields))))

;;
;; value check
;;

(define (bs:check pred type)
  (bs:make-type
   (lambda (in)
     (let1 val ((bs:reader-of type) in)
       (if (not (pred val))
           (errorf <bs:read-error> "check fault got ~a" val)
           val)))
   (lambda (data out)
     (if (not (pred data))
         (errorf <bs:marshal-error> "check fault got ~a" data)
         ((bs:writer-of type) data out)))
   ))

;;
;; value specified type
;;

(define (bs:const val-required type . opts)
  (let-optionals* opts ((val-equal? equal?))
    (bs:check (pa$ val-equal? val-required) type)))

;;
;; or
;;

;; FIXME
(define (port-seekable? port)
  (port-tell port))

(define (bs:or . types)
  (bs:make-type
   (lambda (in)
     (if (not (port-seekable? in))
         (errorf <bs:error> "bs:or requires seekable port"))
     (let1 init-pos (port-tell in)
       (let try-loop ((rest-types types))
         (if (null? rest-types)
             (errorf <bs:read-error> "not match any types"))
         (port-seek in init-pos SEEK_SET)
         (guard (exc
                 ((<bs:read-error> exc)
                  (try-loop (cdr rest-types))))
                ((bs:reader-of (car rest-types)) in)))))
   (lambda (data out)
     (let try-loop ((rest-types types))
       (if (null? rest-types)
           (errorf <bs:marshal-error> "not match any types"))
       (let1 try-out (open-output-string)
         (guard (exc
                 ((<bs:marshal-error> exc)
                  (try-loop (cdr rest-types))))
                (let1 type (car rest-types)
                  ((bs:writer-of type) data try-out)
                  (write-block (string->u8vector (get-output-string try-out))
                               out))))))
   ))

;;
;; element number specified sequence
;;

(define (bs:many subtype . opts)
  (let-optionals* opts ((lower 0)
                        (upper #f))
    (bs:make-type
     (lambda (in)
       (if (not (port-seekable? in))
           (errorf <bs:error> "bs:many requires seekable port"))
       (let loop ((len 0)
                  (acc '()))
         (if (and upper (= len upper))
             (reverse acc)
             (let1 pos (port-tell in)
               (guard (exc
                       ((<bs:read-error> exc)
                        (if (< len lower)
                            (errorf <bs:read-error>
                                    "bs:many range specified ~a-~a but got ~a elements"
                                    lower upper len))
                        (port-seek in pos SEEK_SET)
                        (reverse acc)))
                      (loop (+ len 1)
                            (cons ((bs:reader-of subtype) in)
                                  acc)))))))
     (lambda (data out)
       (define (in-range? len)
         (and (< lower len)
              (or (not upper) (> upper len))))
       (if (not (in-range? (length data)))
           (errorf <bs:marshal-error> "bs:many range specified ~a-~a but length of data is ~a"
                   lower upper (length data)))
       (for-each
        (lambda (elem)
          ((bs:writer-of subtype) elem out))
        data))
     )))

;;
;; size specified sequence
;;

;; FIXME: this name
(define (bs:many2 size subtype)
  (bs:make-type
   (lambda (in)
     (let1 try-in (make <bs:ra-input-port>
                    :in (open-input-limited-length-port in size))
       (let loop ((acc '()))
         (let1 curr-pos (port-tell try-in)
           (cond
            ((= curr-pos size)
             (reverse acc))
            ((< curr-pos size)
             (let1 val ((bs:reader-of subtype) try-in)
               (if (eof-object? val)
                   (errorf <bs:eof-error> "bs:many2 got eof")
                   (loop (cons val acc)))))
            (else (errorf <bs:error>)))))))
   (lambda (data out)
     (let1 try-out (open-output-string)
       (let loop ((rest data))
         (let1 curr-pos (port-tell try-out)
           (cond
            ((null? rest)
             (if (= curr-pos size)
                 (write-block (string->u8vector (get-output-string try-out))
                              out)
                 (errorf <bs:marshal-error> "bs:many2 specified ~d bytes but data size is ~d bytes"
                         size curr-pos)))
            ((< curr-pos size)
             ((bs:writer-of subtype) (car rest) try-out)
             (loop (cdr rest)))
            ((>= curr-pos size)
             (errorf <bs:marshal-error> "bs:many2 specified ~d bytes but, write ~d bytes and rest:~a"
                     size curr-pos rest)))))))
   ))

;; ==================================================================

(use gauche.vport)
(use gauche.uvector)

;;
;; random accessible input port
;;

(define-class <bs:ra-input-port> (<virtual-input-port>)
  ((buf  :init-form (make-u8vector 1024))
   (in   :init-keyword :in)
   (out-ptr :init-value 0)
   (put-ptr :init-value 0)
   ))

(define-method initialize ((self <bs:ra-input-port>) initargs)
  (define (buffered-size)
    (- (slot-ref self 'put-ptr) (slot-ref self 'out-ptr)))
  (define (buffer-empty?)
    (= (buffered-size) 0))
  (define (buffer-realloc)
    (let1 old-buf (slot-ref self 'buf)
      (let1 new-buf (make-u8vector (* (u8vector-length old-buf)
                                      2))
        (u8vector-copy! new-buf 0 old-buf)
        (slot-set! self 'buf new-buf))))
  (define (fill-buffer size)
    (define (fill-aux)
      (while (< (u8vector-length (slot-ref self 'buf))
                (+ (slot-ref self 'put-ptr) size))
        (buffer-realloc))
      (let ((buf (slot-ref self 'buf))
            (in  (slot-ref self 'in)))
        (let loop ((rest size))
          (if (= rest 0)
              size
              (let1 read-size (read-block! buf in (slot-ref self 'put-ptr)
                                           (+ (slot-ref self 'put-ptr) rest))
                (if (eof-object? read-size)
                    (- size rest)
                    (begin
                      (slot-set! self 'put-ptr (+ (slot-ref self 'put-ptr)
                                                  read-size))
                      (loop (- rest read-size)))))))))
    (if (and (buffer-empty?)
             (eof-object? (peek-byte (slot-ref self 'in))))
        #f
        (fill-aux)))

  (next-method)
  (slot-set! self 'getb
             (lambda ()
               (if (and (buffer-empty?)
                        (not (fill-buffer 1)))
                   #f ;; EOF
                   (begin0
                     (u8vector-ref (slot-ref self 'buf)
                                   (slot-ref self 'out-ptr))
                     (slot-set! self 'out-ptr
                                (+ (slot-ref self 'out-ptr) 1))))))
  (slot-set! self 'gets
             (lambda (size)
               (if (and (< (buffered-size) size)
                        (not (fill-buffer (- size (buffered-size))))
                        (buffer-empty?))
                   #f ;; EOF
                   (let ((out-ptr (slot-ref self 'out-ptr))
                         (write-size (min size (buffered-size))))
                     (begin0
                       (u8vector->string (uvector-alias <u8vector> (slot-ref self 'buf)
                                                        out-ptr
                                                        (+ out-ptr write-size)))
                       (slot-set! self 'out-ptr (+ out-ptr write-size)))))))
  (slot-set! self 'seek
             (lambda (offset whence)
               (let ((op (slot-ref self 'out-ptr)))
                 (let1 new-pos
                     (cond
                      ((= whence SEEK_SET) offset)
                      ((= whence SEEK_CUR) (+ op offset))
                      ((= whence SEEK_END) #f)
                      (else (error "invalid seek whence")))
                   (if (>= new-pos (slot-ref self 'put-ptr))
                       (fill-buffer (- (+ new-pos 1)
                                       (slot-ref self 'put-ptr))))
                   (if (or (< new-pos 0)
                           (> new-pos (slot-ref self 'put-ptr)))
                       (error "seek error"))
                   (slot-set! self 'out-ptr new-pos)
                   new-pos))))
  )

;; ------------------------------------------------------------------
;; variable length

(define-class <bs:variable-length> ()
  ((fixed-length :init-keyword :fixed-length
                 :getter       fixed-length-of)
   ))

(define (size-fold kons knil clist)
  (let loop ((seed knil)
             (rest clist))
    (if (null? rest)
        seed
        (let ((elem (car rest))
              (seed-fixed? (number? seed))
              (seed-fixed  (if (number? seed)
                               seed
                               (fixed-length-of seed))))
          (cond
           ((is-a? elem <bs:variable-length>)
            (loop (bs:variable-length (kons (fixed-length-of elem) seed-fixed))
                  (cdr rest)))
           ((integer? elem)
            (if seed-fixed?
                (loop (kons elem seed-fixed) (cdr rest))
                (loop (bs:variable-length (kons elem seed-fixed)) (cdr rest))))
           (else (errorf "<bs:variable-length> or integer required")))))))

(define-method bs:size+ ((vl <bs:variable-length>) . rest)
  (size-fold + vl rest))

(define-method bs:size+ ((fixed <integer>) . rest)
  (size-fold + fixed rest))

(define-method bs:size* ((vl <bs:variable-length>) . rest)
  (size-fold * vl rest))

(define-method bs:size* ((fixed <integer>) . rest)
  (size-fold * fixed rest))

(define-method bs:size-min ((vl <bs:variable-length>) . rest)
  (size-fold min vl rest))

(define-method bs:size-min ((fixed <integer>) . rest)
  (size-fold min fixed rest))

(define (bs:variable-length fixed)
  (make <bs:variable-length> :fixed-length fixed))

(define-method write-object ((vl <bs:variable-length>) out)
  (format out "#<~a fixed-length:~d>"
          (class-name (class-of vl)) (fixed-length-of vl)))


(provide "binary/struct")
