;;
;; Test binary.struct
;;
;; $Id: test.scm 32 2007-04-27 12:26:18Z shinya $

(use gauche.test)
(use file.util)                         ; file-equal?

(test-start "binary.struct")

(use binary.struct)
(test-module 'binary.struct)

(define struct-type
  (bs:struct
   ((num1  'u8)
    (num2  'u16le)
    (count 'u8)
    (str   (bs:cstring 10))
    (arr   (bs:array count 'u8))
    )))

(define (test-write name expect type data)
  (if (is-a? expect <condition-meta>)
      (test* name #t
             (guard (exc
                     ((expect exc) #t)
                     (else exc))
                    (call-with-output-string
                      (lambda (out)
                        ((bs:writer-of type) data out)))))
      (test* name expect
             (call-with-output-string
               (lambda (out)
                 ((bs:writer-of type) data out))))))

(define (test-read name expect type serialized)
  (let1 in (open-input-string serialized)
    (if (is-a? expect <condition-meta>)
        (test* name #t
               (guard (exc
                       ((expect exc) #t)
                       (else exc))
                      ((bs:reader-of type) in)))
        (test* name expect ((bs:reader-of type) in)))))

(define struct-data
  '((num1  . 97)
    (num2  . 321)
    (count . 3)
    (str   . "abcdefg")
    (arr   . (1 2 3))))

(define serialized-data "aA\x01\x03abcdefg\0\0\0\x01\x02\x03")

(test* "struct ref" 321 (bs:struct-ref struct-data 'num2))

(test* "is instance of <bs:type>" #t (is-a? struct-type <bs:type>))

(test* "reader is a procedure" #t (procedure? (bs:reader-of struct-type)))

(test* "writer is a procedure" #t (procedure? (bs:writer-of struct-type)))

(test* "type alias" #t (procedure? (bs:reader-of 'u8)))

(test* "undefined type alias" *test-error* (bs:reader-of 'undefined-alias))

;; ------------------------------------------------------------------

(test-section "read")

(test-read "cstring" "abcdefg"
           (bs:cstring 10) "abcdefg\0xxx")
(test-read "cstring" <bs:read-error>
           (bs:cstring 10) "aa")
(test-read "cstring" "abc"
           (bs:cstring) "abc\0def")
(test-read "cstring" <bs:read-error>
           (bs:cstring) "")

(test-read "block" #u8(0 1 2 3)
           (bs:block 4) "\x00\x01\x02\x03\x04\x05")
(test-read "block" <bs:read-error>
           (bs:block 1000) "123")

(test-read "array" '(9 8 7 6)
           (bs:array 4 'u8) "\x09\x08\x07\x06\x05\x04")
(test-read "array" <bs:read-error>
           (bs:array 9 'u64) "xyz")

(test-read "structure" struct-data
           struct-type serialized-data)
(test-read "structure" <bs:read-error>
           struct-type "abc")

(test-read "check" #x10
           (bs:check even? 'u8) "\x10\xff")
(test-read "check" <bs:read-error>
           (bs:check even? 'u8) "\x11\xff")

(test-read "const" #x10
           (bs:const #x10 'u8) "\x10\xff")
(test-read "const" <bs:read-error>
           (bs:const #x10 'u8) "\x11\xff")

(test-read "or" #x02
           (bs:or (bs:check even? 'u8)
                  'u32be)
           "\x02\x01")
(test-read "or" #x01020304
           (bs:or (bs:check even? 'u8)
                  'u32be)
           "\x01\x02\x03\x04\x05")

(test-read "many" '(#x01 #x02 #x03)
           (bs:many 'u8 0 3)
           "\x01\x02\x03\x04")
(test-read "many" <bs:read-error>
           (bs:many 'u8 4)
           "\x01")

(test-read "many2" '(#x10 #x20 #x30)
           (bs:many2 12 'u32be)
           "\x00\x00\x00\x10\x00\x00\x00\x20\x00\x00\x00\x30\x01")
(test-read "many2" <bs:read-error>
           (bs:many2 9 'u32be)
           "\x00\x00\x00\x10\x00\x00\x00\x20\x01\x02\x03\x04")
(test-read "many2" '(#x10 #x20 #x01)
           (bs:many2 9 (bs:or 'u32be 'u8))
           "\x00\x00\x00\x10\x00\x00\x00\x20\x01\x02\x03\x04")
(test-read "many2" <bs:read-error>
           (bs:many2 3 'u8)
           "\x01\x02")

;; ------------------------------------------------------------------

(test-section "write")

(test-write "cstring" "abcdefg\0\0\0"
            (bs:cstring 10) "abcdefg")
(test-write "cstring" <bs:marshal-error>
            (bs:cstring 10) "xxxxxxxxxxxxxxx")
(test-write "cstring" "abc\0"
            (bs:cstring) "abc")

(test-write "block" "\x01\x02\x03"
            (bs:block 3) #u8(1 2 3))
(test-write "block" <bs:marshal-error>
            (bs:block 3) #u8(1 2 3 4 5))

(test-write "array" "\x09\x00\x00\x00\x08\x00\x00\x00"
            (bs:array 2 'u32le) '(9 8))
(test-write "array" <bs:marshal-error>
            (bs:array 4 'u8) '(1))

(test-write "structure" serialized-data
            struct-type struct-data)
(test-write "structure" <bs:marshal-error>
            struct-type
            '((num1 . 1)
              (num2 . 2)))

(test-write "check" "\x02"
            (bs:check even? 'u8) 2)
(test-write "check" <bs:marshal-error>
            (bs:check even? 'u8) 3)

(test-write "const" "\x02"
            (bs:const #x02 'u8) 2)
(test-write "const" <bs:marshal-error>
            (bs:const #x02 'u8) 3)

(test-write "or" "\x02"
            (bs:or (bs:check even? 'u8)
                   'u32le)
            2)
(test-write "or" "\x00\x00\x00\x03"
            (bs:or (bs:check even? 'u8)
                   'u32be)
            3)

(test-write "many" "\x01\x02\x03"
            (bs:many 'u8)
            '(1 2 3))
(test-write "many" <bs:marshal-error>
            (bs:many 'u8 1 2)
            '(1 2 3))
(test-write "many" <bs:marshal-error>
            (bs:many 'u8 4)
            '(1 2 3))

(test-write "many2" "\x00\x00\x00\x10\x00\x00\x00\x20\x00\x00\x00\x30"
            (bs:many2 12 'u32be)
            '(#x10 #x20 #x30))
(test-write "many2" <bs:marshal-error>
            (bs:many2 12 'u32be)
            '(#x10 #x20 #x30 #x04))

;; ------------------------------------------------------------------

(test-section "complex structure")

;; PNG file - example of chunk structure

(define png:identifier
  (bs:const #u8(#x89 #x50 #x4E #x47 #x0D #x0A #x1A #x0A)
            (bs:block 8)))

(define png:chunk-name (bs:cstring 4 #f))

(define png:IHDR
  (bs:struct
   ((data-length  'u32be)
    (chunk-name   (bs:const "IHDR" png:chunk-name))
    (image-width  'u32be)
    (image-height 'u32be)
    (bit-depth    'u8)
    (color-type   'u8)
    (compression  'u8)
    (filter       'u8)
    (interlace    'u8)
    (crc          'u32be))))

(define png:pallet-rgb
  (bs:struct
   ((R 'u8)
    (G 'u8)
    (B 'u8))))

(define png:PLTE
  (bs:struct
   ((data-length 'u32be)
    (chunk-name  (bs:const "PLTE" png:chunk-name))
    (pallet-data (bs:array (/ data-length 3) png:pallet-rgb))
    (crc         'u32be))))

(define png:IDAT
  (bs:struct
   ((data-length 'u32be)
    (chunk-name  (bs:const "IDAT" png:chunk-name))
    (data        (bs:block data-length))
    (crc         'u32be))))

(define png:IEND
  (bs:struct
   ((data-length 'u32be)
    (chunk-name  (bs:const "IEND" png:chunk-name))
    (crc         'u32be))))

(define (unknown-chunk-name? name)
  (not (member name '("IHDR" "PLTE" "IDAT" "IEND"))))

(define png:unknown-chunk
  (bs:struct
   ((data-length 'u32be)
    (chunk-name  (bs:check unknown-chunk-name? png:chunk-name))
    (data        (bs:block data-length))
    (crc         'u32be))))

(define png:file
  (bs:struct
   ((identifier png:identifier)
    (ihdr       png:IHDR)
    (chunks     (bs:many
                 (bs:or png:PLTE
                        png:IDAT
                        png:unknown-chunk)))
    (iend       png:IEND))))

(define png-path "test.png")

(define png-structuer
  (call-with-input-file png-path
      (bs:reader-of png:file)))

(test* "read png" 135 (bs:struct-ref* png-structuer 'ihdr 'image-width))

(define copied-png-path "test-copy.png")

(call-with-output-file copied-png-path
  (cut (bs:writer-of png:file) png-structuer <>))

(test* "write png" png-path copied-png-path file-equal?)

(sys-unlink copied-png-path)

(test-end)
